use std::env;
use asciimoji::{Asciimoji, read_asciimojis_from_file, search_name};
use asciimoji::downloader::download_yaml_file_from_repository;

use tabular::{Table, Row};

fn usage() {
    println!("ASCIImoji - {}\n{}\n", env!("CARGO_PKG_VERSION"), env!("CARGO_PKG_DESCRIPTION"));
    println!("usage:\tasciimoji <asciimoji name>\n");
    println!("Arguments:");
    println!("\t-l\tlist all the available asccimojis with their relative names")
}

fn print_asciimoji(config_home: &String) {
    let mut asciimojis: Vec<Asciimoji> = read_asciimojis_from_file(config_home).unwrap();
    asciimojis = asciimojis
        .into_iter()
        .map(|asciimoji| asciimoji.to_lowercase())
        .collect();

    let mut args = env::args();
    args.next();

    for arg in args {
        let searched = search_name(&asciimojis, &arg.to_lowercase()); 
        if searched.len() != 0{
            for ascimoji in searched {
                println!("{}", ascimoji);
            }
        } else {
            println!("Asciimoji for {} not found. Try 'asciimoji -l' to list all available asciimojis.", arg);
        }
    }
}

fn print_list_of_asciimoji(config_home: &String) {
    let mut asciimojis: Vec<Asciimoji> = read_asciimojis_from_file(config_home).unwrap();
    asciimojis = asciimojis
        .into_iter()
        .collect();
    let mut table = Table::new("{:<} {:<}");

    for asciimoji in asciimojis {
        let mut names = String::new();
        for name in asciimoji.names() {
            names.push_str(name);
            names.push_str(&", ");
        }

        table.add_row(Row::new()
            .with_cell(asciimoji.moji())
            .with_cell(names));
    }

    print!("{}", table);
}

fn main() {
    let config_home = env::var("XDG_CONFIG_HOME").or_else(|_| env::var("HOME").map(|home|format!("{}/.asciimojis", home)));
    let config_home = config_home.unwrap_or(String::from("asciimojis.yml"));

    match download_yaml_file_from_repository(config_home.as_str()) {
        Ok(_) => {},
        Err(e) => println!("Errore: {}", e)
    }

    let args: Vec<String> = env::args().collect();
    if args.len() == 1 {
        usage();
    } else {
        match &args[1][..] {
            "-l" => print_list_of_asciimoji(&config_home),
            _ => print_asciimoji(&config_home),
        }
    }

}
