//! Holds a dictionary of built-in ASCII art images.

/// When disclosing an image, the signature is shown last.
/// `str` must have at least 3 characters. Otherwise we would get
/// too many false positives.
#[cfg(not(test))]
pub const IMAGE_KNOWN_SIGNATURES: &[&str] = &[
    " VK", "VK ", "hjw", " ac", "ac ", "jgs", "snd", "mrf", "hjw", "DR J", "fsc", "ejm", "ejm98",
    " hh", "hh ", "jrei", "b'ger", "wtx", "fsc", "Asik", "Phs", "Sher", " jg", "jg ", "(PS)",
];
#[cfg(test)]
pub const IMAGE_KNOWN_SIGNATURES: &[&str] = &["jens", "lis"];

/// A collection of built-in images from whom one is chosen at the start of the game.
// first char of image lines must be '|'
pub const DEFAULT_IMAGES: &[&str] = &[
    r#"image: |1
     ,,,,,
    (o   o)
     /. .\
    (_____)
      : :
     ##O##'
   ,,,: :,,,
  _)\ : : /(____
 {  \     /  ___}
  \/)     ((/
   (_______)
     :   :
     :   :
    / \ / \
    """ """
"#,
    r#"image: |1
     |\_|X|_/|
    /         \
  =(  O     O  )=
   -\    o    /-
    / .-----. \
  /_ | o   o |_ \
 (U  |       |  U)
    _|_     _|_
   (   )---(   )
"#,
    r#"image: |1
        _.---._    /\\
     ./'       "--`\//
   ./              o \
  /./\  )______   \__ \
 ./  / /\ \   | \ \  \ \
    / /  \ \  | |\ \  \7
     "     "    "  "        VK
"#,
    r#"image: |1
        ,.
       (_|,.
      ,' /, )_______   _
   __j o``-'        `.'-)'
  (")                 \'
   `-j                |
     `-._(           /
 hjw    |_\  |--^.  /
       /_]'|_| /_)_/
          /_]'  /_]'
"#,
    r#"image: |1
        _
       [ ]
      (   )
       |>|
    __/===\__
   //| o=o |\\
 <]  | o=o |  [>
     \=====/
    / / | \ \
   <_________>
"#,
    r#"image: |1
                           (_)(_)
                           /     \
                          /       |
                         /   \  * |
           ________     /    /\__/
   _      /        \   /    /
  / \    /  ____    \_/    /
 //\ \  /  /    \         /
 V  \ \/  /      \       /
     \___/        \_____/
"#,
    r#"image: |1
          .-.
         (. .)__,')
         / V      )
   ()    \  (   \/
 <)-`\()  `._`._ \
   <).>=====<<==`'====
    C-'`(>          hjw
# Author: hjw
"#,
    r#"image: |1
  >(. )
   |  (     /)
   |   \___/ )
   (   ----- )  >@)_//   >@)_//  >@)_//  >@)_//
    \_______/    (__)     (__)    (__)    (__)
 ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~
"#,
    r#"image: |1
            __
            /(`o
      ,-,  //  \\
     (,,,) ||   V
    (,,,,)\//
    (,,,/w)-'
    \,,/w)
    `V/uu
      / |
      | |
      o o
      \ |
 \,/  ,\|,.  \,/
"#,
    r#"image: |1
 o
  \_/\o
 ( Oo)                    \|/
 (_=-)  .===O-  ~~Z~A~P~~ -O-
 /   \_/U'                /|\
 ||  |_/
 \\  |
 {K ||
  | PP
  | ||
  (__\\  ac
# Author: ac
"#,
    r#"image: |1
       ______
      /     /\
     /     /  \
    /_____/----\_    (
   "     "          ).
  _ ___          o (:') o
 (@))_))        o ~/~~\~ o
                 o  o  o
"#,
    r#"image: |1
                              _______     |\
                             |License|    | \
   _____                     | ~~*~~ |    |  \
  |     |  (((        .--.   |_______|    |
  |DrJRO| ~OvO~ __   (////)               |
  |     | ( _ )|==|   \__/                |
  |o    |  \_/ |_(|  /    \   _______     |
  |     | //|\\   \\//|  |\\  |__o__|     |
  |   __|//\_/\\ __\/ |__|//  |__o__|     |
  |  |==""//=\\""====|||||)   |__o__|     |
 _|__||_|_||_||_____||||||____|__o__|_____|
     ||  (_) (_)    ||||||                \
     []             [(_)(_)
"#,
    r#"image: |1
    _     _
   ( |_ _| )
    ( .". )
   _( (Y) )_
  / /,---.\ \
 / / | + | \ \
 \_)-"   "-(_/
   |_______|
   _)  |  (_
  (___,'.___)  hjw
# Art by Hayley Jane Wakenshaw
# (slightly modified)
"#,
    r#"image: |1
           |
         \ _ /
       -= (_) =-
         /   \         _\/_
           |           //o\  _\/_
    _____ _ __ __ ____ _ | __/o\\ _
  =-=-_-__=_-= _=_=-=_,-'|"'""-|-,_
   =- _=-=- -_=-=_,-"          |
 jgs =- =- -=.--"
# Art by Genoveva Galarza
"#,
    r#"image: |1
         __I__
    .-'"  .  "'-.
  .'  / . ' . \  '.
 /_.-..-..-..-..-._\ .---------------------------------.
          #  _,,_   ( I hear it might rain people today )
          #/`    `\ /'---------------------------------'
          / / 6 6\ \
          \/\  Y /\/       /\-/\
          #/ `'U` \       /a a  \               _
        , (  \   | \     =\ Y  =/-~~~~~~-,_____/ )
        |\|\_/#  \_/       '^--'          ______/
        \/'.  \  /'\         \           /
         \    /=\  /         ||  |---'\  \
    jgs  /____)/____)       (_(__|   ((__|
# Art by Joan Stark
"#,
    r#"image: |1
  [][][] /""\ [][][]
   |::| /____\ |::|
   |[]|_|::::|_|[]|
   |::::::__::::::|
   |:::::/||\:::::|
   |:#:::||||::#::|
  #%*###&*##&*&#*&##
 ##%%*####*%%%###*%*#
"#,
    r#"image: |1
   ,-~~-.___.
  / |  '     \
 (  )         0
  \_/-, ,----'
     ====           //
    /  \-'~;    /~~~(O)
   /  __/~|   /       |
 =(  _____| (_________|
"#,
    r#"image: |1
   \,`/ /
  _)..  `_
 ( __  -\
     '`.
    ( \>_-_,
    _||_ ~-/    W<
"#,
    r#"image: |1
             __:.__
            (_:..'"=
             ::/ o o\         AHAH!
            ;'-'   (_)     Spaceman Spiff      .
            '-._  ;-'        wins again !  _'._|\/:
            .:;  ;                .         '- '   /_
           :.. ; ;,                \       _/,    "_<
          :.|..| ;:                 \__   '._____  _)
          :.|.'| ||                            _/ /
 snd      :.|..| :'                           `;--:
          '.|..|:':       _               _ _ :|_\:
       .. _:|__| '.\.''..' ) ___________ ( )_):|_|:
 :....::''::/  | : :|''| "/ /_=_=_=_=_=/ :_[__'_\3_)
  ''''      '-''-'-'.__)-'
# Art by Shanaka Dias
"#,
    r#"image: |1
   _,                          _
 .'  `.                  ___.>"''-..-.
 `-.   ;           .--"""        .-._@;
    ;  !_.--..._ .'      /     .[_@'`'.
   ;            /       : .'  ; :_.._  `.
   :           ;        ;[   _T-"  `.'-. `-.
    \        .-:      ; `.`-=_,88p.   _.}.-"
     `-.__.-'   \    /L._ Y",P$T888;  ""
              .-'_.-'  / ;$$$$$$]8P;
              \ /     / / "Y$$P" ^"
      fsc      ;\_    `.\_._
               ]__\     \___;
"#,
    r#"image: |1
         _
       _<_/_
    __/    _>
   '\  '  |
     \___/
     /+++\
  o=|..|..|
    | o/..|
 0==|+++++|
  0======/
"#,
    r#"image: |1
         _../|_
       ='__   _~-.
            \'  ~-`\._
                  |/~`
    .    .    .    .    .
 _.`(._.`(._.`(._.`(._.`(._
"#,
    r#"image: |1
                         ____
                    .---'-    \
       .-----------/           \
      /           (         ^  |   __
 &   (             \        O  /  / .'
 '._/(              '-'  (.   (_.' /
      \                    \     ./
       |    |       |    |/ '._.'
        )   @).____\|  @ |
    .  /    /       (    | mrf
   \|, '_:::\  . ..  '_:::\ ..\).
# Art by Morfina
"#,
    r#"image: |1
            __n__n__
     .------`-\00/-'
    /  ##  ## (oo)
   / \## __   ./
      |//YY \|/
 snd  |||   |||
# Art by Shanaka Dias
"#,
    r#"image: |1
                        .-'~~~-.
                      .'o  oOOOo`.
                     :~~~-.oOo   o`.
                      `. \ ~-.  oOOo.
                        `.; / ~.  OO:
                        .'  ;-- `.o.'
                       ,'  ; ~~--'~
                       ;  ;
 _______\|/__________\\;_\\//___\|/________
"#,
    r#"image: |1
     ____
    (__  '.
     /_____)
    ()@ @ )))
     'C ,()(()
     ,.'_'.' \
  __/ )   (--'
 '._./     \
    (_._._._)
     _|| ||_
 mrf(__.).__)
"#,
    r#"image: |1
         o    .   _     .
           .     (_)         o
    o      ____            _       o
   _   ,-/   /)))  .   o  (_)   .
  (_)  \_\  ( e(     O             _
  o       \/' _/   ,_ ,  o   o    (_)
   . O    _/ (_   / _/      .  ,        o
      o8o/    \\_/ / ,-.  ,oO8/( -TT
     o8o8O | } }  / /   \Oo8OOo8Oo||     O
    Oo(""o8"""""""""""""""8oo""""""")
   _   `\`'                  `'   /'   o
  (_)    \                       /    _   .
       O  \           _         /    (_)
 o   .     `-. .----<(o)_--. .-'
    --------(_/------(_<_/--\_)--------hjw
"#,
    r#"image: |1
                 \||/
                 |  @___oo
       /\  /\   / (__,,,,|
      ) /^\) ^\/ _)
      )   /^\/   _)
      )   _ /  / _)
  /\  )/\/ ||  | )_)
 <  >      |(,,) )__)
  ||      /    \)___)\
  | \____(      )___) )___
   \______(_______;;; __;;;
"#,
    r#"image: |1
    (\{\
    { { \ ,~,
   { {   \)))  *
    { {  (((  /
     {/{/; ,\/
        (( '
         \` \
         (/  \
 ejm     `)  `\
"#,
    r#"image: |1
                     /
                ,.. /
              ,'   ';
   ,,.__    _,' /';  .
  :','  ~~~~    '. '~
 :' (   )         )::,
 '. '. .=----=..-~  .;'
  '  ;'  ::   ':.  '"
    (:   ':    ;)
     \\   '"  ./
      '"      '"
                DR J
"#,
    r#"image: |1
      __/\__
 . _  \\''//
 -( )-/_||_\
  .'. \_()_/
   |   | . \
   |mrf| .  \
  .'. ,\_____'.
"#,
    r#"image: |1
          _.-.
        ,'/ //\
       /// // /)
      /// // //|
     /// // ///
    /// // ///
   (`: // ///
    `;`: ///
    / /:`:/
   / /  `'
  / /
 (_/  hh
"#,
    r#"image: |1
  _____
 |A .  | _____
 | /.\ ||A ^  | _____
 |(_._)|| / \ ||A _  | _____
 |  |  || \ / || ( ) ||A_ _ |
 |____V||  .  ||(_'_)||( v )|
        |____V||  |  || \ / |
               |____V||  .  |
                      |____V| ejm98
"#,
    r#"image: |1
      !!!!\\\\
    '`!_  ||||
     ` \`-'''|
       `\   /
        )\  \
 ejm   /  \  \
           \|
"#,
    r#"image: |1
  ,~~--~~-.
 +      | |\
 || |~ |`,/-\
 *\_) \_) `-'#,
"#,
    r#"image: |1
  (.  \
   \  |
    \ |___(\--/)
  __/    (  . . )
 "'._.    '-.O.'
      '-.  \ "|\
         '.,,/'.,,mrf
"#,
    r#"image: |1
              __
    ,'```--'''  ``-''-.
  ,'            ,-- ,-'.
 (//            `"'| 'a \
   |    `;         |--._/
   \    _;-._,    /
    \__/\\   \__,'
     ||  `'   \|\\
     \\        \\`'
 hjw  `'        `'
"#,
    r#"image: |1
 \\             //
  \\\' ,      / //
   \\\//,   _/ //,
    \_-//' /  //<,
      \ ///  <//`
     /  >>  \\\`__/_
    /,)-^>> _\` \\\
    (/   \\ //\\
        // _//\\\\
       ((` ((
"#,
    r#"image: |1
 >o)
 (_>   <o)
       (_>
"#,
    r#"image: |1
               I~
           I~ /V\  I~
       I~ /V\ | | /V\  I~
  @ @ /V\ | |_|_|_| | /V\ @ @
 @@@@@| |_| |_/V\_| |_| |@@@@@
 @@@@@| | |_|_|_|_|_| | |@@@@@
 @@@@@|_|_V_V|   |V_V_|_|@@@@@
 _._._._._._._._._._._._._._._
 :::::::::::::|X|:::::::::::::
 Sher^
"#,
    r#"image: |1
  W                   __
 [ ]                 |::|
  E          ._.     |::|   ._.
  |\         |:| ._. |::|   |/|
  \ \\|/     |:|_|/| |::|_  |/|
   |-( )-    |:|"|/|_|::|\|_|/| _
   | V L     |:|"|/|||::|\|||/||:|
   \    `  ___   ~~~~~~~~~~~~~~~~~~~~
    |    \/  /     ~~~~ ~~~~ ~~~ ~~~
"#,
    r#"image: |1
       .___.
      /     \
     | O _ O |
     /  \_/  \
   .' /     \ `.
  / _|       |_ \
 (_/ |       | \_)
     \       /
    __\_>-<_/__
    ~;/     \;~
"#,
    r#"image: |1
   ,--.
   \  _\_
   _\/_|_\____.'\
 -(___.--._____(
      \   \
       \   \
        `--'
 jg
"#,
    r#"image: |1
  _..__.          .__.._
 .^"-.._ '-(\__/)-' _..-"^.
        '-.' oo '.-'
           `-..-'       fsc
"#,
    r#"image: |1
                          .
                      _.:/ )
    _              .-Q      `._
 '\(o7/'         o(.__         '-.
 `.( ).'           `_/    )
    H       ._      '-._.'         kOs
    w       ( \         /
             \ '.     .'   )
"#,
    r#"image: |1
    .
   / \__        .. _
   \.'  '._o    \_|_) ))
 __(  __ / /      ).
 \  _( ,/ /.____.' /
  '' '..-'        |
         \    _   (
          )v /-'._ )
         ////   |//
        // \\   //
       //   \\ ||\\
    --"------"-"--"--  mrf
"#,
    r#"image: |1
    a'!   _,,_   a'!   _,,_     a'!   _,,_
      \\_/    \    \\_/    \      \\_/    \.-,
       \, /-( /'-,  \, /-( /'-,    \, /-( /
       //\ //\\     //\ //\\       //\ //\\
 jrei
"#,
    r#"image: |1
      ,__,
 (/__/\oo/\__(/
   _/\/__\/\_
    _/    \_    b'ger
"#,
    r#"image: |1
           (    )
            (oo)
   )\.-----/(O O)
  # ;       / u
    (  .   |} )
     |/ `.;|/;
     "     " "
#unknown
"#,
    r#"image: |1
                 __
                /\/'-,
        ,--'''''   /"
  ____,'.  )       \___
 '"""""------'"""`-----'
#pb
"#,
    r#"image: |1
           _
    ______/ \-.   _         _ __ _         _    _
 .-/     (    o\_//        / |..| \       / >--< \
  |  ___  \_/\---'         \/ || \/       \|  \ |/
  |_||  |_||       wtx      |_''_|         |_||_|
"#,
    r#"image: |1
                   _J""-.
      .-""L_      /o )   \ ,';
 ;`, /   ( o\     \ ,'    ;  /
 \  ;    `, /      "-.__.'"\_;
 ;_/"`.__.-"
                             fsc
"#,
    r#"image: |1
      >=<                >=<
 ,.--'  ''-.        ,.--'  ''-.
 (  )  ',_.'        (  )  ',_.'
  Xx'xX              mn'mn`
                            Asik
"#,
    r#"image: |1
         .-' '-.
        /       \
       |,-,-,-,-,|
      ___   |
     _)_(_  |
     (/ \)  |
     _\_/_  /)
    / \_/ \//
    |(   )\/
    ||)_(
    |/   \
    n|   |
   / \   |
   |_|___|
      \|/
 jgs _/L\_
"#,
    r#"image: |1
  oo`'._..---.___..-
 (_,-.        ,..'`
      `'.    ;
         : :`
        _;_;      jrei
"#,
    r#"image: |1
        `.oo'    |    `oo.'
     ,.  (`-'    |    `-')  ,.
    '^\`-' )     |     ( `-'/^`
       c-L'-     |     -`_-)
 -bf-
"#,
    r#"image: |1
        .___,
     ___('v')___
     `"-\._./-"'
 hjm     ^ ^
"#,
    r#"image: |1
           __,---.__
        ,-'         `-.__
      &/           `._\ _\
      /               ''._
      |   ,             (")
 jrei |__,'`-..--|__|--''
"#,
    r#"image: |1
   ;     /        ,--.
  ["]   ["]  ,<  |__**|
 /[_]\  [~]\/    |//  |
  ] [   OOO      /o|__|   Phs
"#,
    r#"image: |1
       |
  m1a  |
       |
   /   |   \
   \   |   /
 .  --\|/--  ,
  '--|___|--'
  ,--|___|--,
 '  /\o o/\  `
   +   +   +
    `     '
"#,
    r#"image: |1
      ___`\`,`.
 ()    `-.--. |.
  \`-.-(e(e.' .1
   ;.___..~   _")
  <./|_|`-~.   |
            \  |
             | |
             | |
   (PS)      | /\
"#,
    r#"image: |1
      O>         _
     ,/)          )_
 -----<---<<<   )   )
      ``      ` _)
               jrei
"#,
    r#"image: |1
     _|\_
      ("}
   i_.-@-._ _
   8--,  .-`*
   I  /==\
   I  |   \
   I  /___\
 snd
"#,
    r#"image: |1
  ,__ ., __, ,,,,
  '--/,,\--'\*\%\*\
    //  \\\'\'%.\'%\
     '..'//'%\.\%/\\'.^
        \\'/'/%''/\'
          ||     ||
          "      "
#morfina
"#,
    r#"image: |1
      .------------------....___       ,;;;,
     /                      /  /'''---/;;;;;;,.,
 jgs \                      \__\...---\;;;;;;;"
      '------------------''''           ````
"#,
    r#"image: |1
       _______________________
      /   //=================/`"-._
     |   ||=================|      D
 jgs  \___\\_________________\__.-"
"#,
    r#"image: |1
      _________________________________________________
     |'|'|'|'|'|'|'|'|'|'|'|'|'|'|'|'|'|'|'|'|'|'|'|'|'|
 jgs |___1___2___3___4___5___6___7___8___9__10__11__12_|



"#,
    r#"image: |1
            _________________
    .--H--.|             jgs |
  _//_||  ||                 |
 [    -|  |'--;--------------'
 '-()-()----()"()^^^^^^^()"()'
"#,
    r#"image: |1
                 ___
      .--.      [ATM]
 .----'   '--.    |
 '-()-----()-'    |
"#,
    r#"image: |1
           .---;-,
        __/_,{)|__;._
     ."` _     :  _  `.  .:::;.    .::'
 jgs '--(_)------(_)--' `      '::'
"#,
    r#"image: |1
                (
                            )
 .------.------------------.
 |-_- = |                  #
 '------'------------------'
"#,
    r#"image: |1
   oOOOOOo
  ,|    oO
 //|     |
 \\|     |
   `-----`
"#,
    r#"image: |1
      _*
 .---/ `\
 |~~~~|`'
  \  /
   )(
  /__\
"#,
    r#"image: |1
  _
 | |
 |~|
 \_/
  |
 _|_
"#,
    r#"image: |1
        ,,
      W ()      , `)
      |-><       )  ,
 jgs  | )(\_v   (, ` )
"#,
    r#"image: |1
     -    C                 ,  .-.___
   -     /<               @/  /xx\XXX\
  -   __/\ `\            \/\  |xx|XXX|
     `    \, \_ =        _\<< |xx|XXX|
 jgs"""""""""""""""""""""""""""""""""""
"#,
    r#"image: |1
     .-.  .-.
    |   \/   |
    \        /
     `\    /`
 jgs   `\/`
"#,
    r#"image: |1
        __ __ __ __ __
      /__/__/__/__/__/|
     /__/__/__/__/__/|/
 jgs |__'__'__'__'__|/
"#,
    r#"image: |1
       __.-n-.
       \|'---'|-,
        |     | ))
        |     |-'
 jgs    '.___.'
"#,
    r#"image: |1
              ___
      `-._\ /     `~~"--.,_
     ------>|              `~~"--.,_
 jgs  _.-'/ '.____,,,,----"""~~```'
"#,
    r#"image: |1
     .--.
    /.-. '----------.
    \'-' .--"--""-"-'
 jgs '--'
"#,
    r#"image: |1
       _..._
     .'     '.
    /_.=._.=._\
    \  \   /  /
 jgs '.__.__.'
"#,
    r#"image: |1
       _   _
     .' `"` '.
    /_       _\
    \ `--`--` /
 jgs '.__.__.'
"#,
    r#"image: |1
     .-.
    ( . )
  .-.':'.-.
 (  =,!,=  )
  '-' | '-'
"#,
    r#"image: |1
      9::::=======
      |::::=======
      |===========
 jgs  |===========
"#,
    r#"image: |1
             ,--,_
      __    _\.---'-.
      \ '.-"     // o\
 jgs  /_.'-._    \\  /
             `"--(/"`
"#,
    r#"image: |1
             _,--,
          .-'---./_    __
         /o \\     "-.' /
         \  //    _.-'._\
 jgs      `"\)--"`
"#,
    r#"image: |1
   .=========.
  / (_)  (_) /|
 /-========-/ |
 |          |/
 '-========-'
"#,
    r#"image: |1
          .-.-.
       _ /     \ _
      / `\=====/` \
 jgs  '.__`---`__.'
"#,
    r#"image: |1
      _, ____
     \_ `  , `P
       |/--\ /
 jgs  "`   "`
"#,
    r#"image: |1
            _.._    _
          ."\__/"./`_\
        _/__<__>__\/
 jgs   `"/_/""""\_\\
"#,
    r#"image: |1
               _ __            _ __
              (_|_ '.         (_|_ '.
                  '-'             '-'
       _ _.-.         _ _.-.
 jgs  (_|__.'        (_|__.'
"#,
    r#"image: |1
      <(__)> | | |
      | \/ | \_|_/
      \^  ^/   |
      /\--/\  /|
 jgs /  \/  \/ |
"#,
    r#"image: |1
             ___
       ___ /`"""`8-.__
     /`"""`8-.__._    )
     \   ._     ) """""
 jgs """"  """"""
"#,
    r#"image: |1
      .---.        __
     /     '-.__.-'  `'-.
     \     .-'           '.__.'
 jgs  '---'
"#,
    r#"image: |1
      _
   __|_\_
 =u__.-.__)
    /_/
"#,
    r#"image: |1
         _h___   I
      __[_]___\_[]=_
      \,           /
 jgs^V^V^V^V^V^V^V^V^V^V^^''''""
"#,
    r#"image: |1
        _..-.._           *
   *  .'_/ _ \_'.
     /_ _| __|_ _\___  *
     | _ |_ _| _ |  /\
 jgs |___|___J___|__\/
"#,
    r#"image: |1
          _
       _.-H-._       _.-""-._
    .='_______'=. .=' _   _  '=.
     ||       ||[+]|_[X]__[X]_|_
 jgs_||_______||___[=_=-_-=_-=_=]
"#,
    r#"image: |1
  (`\/)
  _\|/
 (_X_,B
"#,
    r#"image: |1
              c._
    ."````"-"C  o'-.
  _/   \       _..'
 '-\  _/--.<<-'
    `\)     \)  jgs
"#,
    r#"image: |1
      _V_
     /___\
    |  |  |
    \o | o/
 jgs '-.-'
"#,
    r#"image: |1
         +
       .-"-.
      / RIP \
      |     |
 jgs \\  ___|//
"#,
    r#"image: |1
       .-""-.
      /)    (\   ,;;;,
      ( ' \' )  // \\\\
       \  = /   ( ' ' )
 jgs    )--(     \_=_/
"#,
    r#"image: |1
  \_(\_\
     \\_
    <`\\>
       ))
 jgs   (
"#,
    r#"image: |1
       __
      <- )
      /( \
      \_\_>
 jgs  " "
"#,
    r#"image: |1
        __
       ( ->
       / )\
      <_/_/
 jgs   " "
"#,
    r#"image: |1
      __ >>>
     (__)(_)()=
         >>>
 jgs      __ >>>
         (__)(_)()=
             >>>
"#,
    r#"image: |1
           .--.
     (\_/)/  _ \
     (o.o) _(   |____
    .- ".-`----'`"""")
 jgs             '--`
"#,
    r#"image: |1
        ()
    []_/<
     \-=\
 jgs _)(/_
"#,
    r#"image: |1
            ,;:'`'::'
         __||
   _____/LLLL\_
   \__________"|
 ~^~^~^~^~^~^~^~^~^~
"#,
    r#"image: |1
                       __T__I___...__7~
         ,_           `"|-=||==|==|==|
         [_`'---...,____|"_||__|__|__|_
         | `'---...__PHILOMENA D_______]
 jgs^~-~^-^~^'----~^~---~---------~^---'`~^-^~~^
"#,
    r#"image: |1
          __
       | /  \ |
      \_\\  //_/
       .'/()\'.
 jgs    \\  //
"#,
    r#"image: |1

       //  \\
      _\\()//_
     / //  \\ \
 jgs  | \__/ |
"#,
    r#"image: |1
  __
 /o \_____
 \__/-="="`
"#,
    r#"image: |1
      __
     / o\
     \_ /
      <|
      <|
 jgs  <|
       `
"#,
    r#"image: |1
   _ (.".) _
  '-'/. .\'-'
    ( o o )
 jgs `"-"`
"#,
    r#"image: |1
  __QQ
 (_)_">
 _)
"#,
    r#"image: |1
  /\-/\    /\-/\    /\-/\
 (=^Y^=)  (=^Y^=)  (=^Y^=)
  (>o<)    (>o<)    (>o<)
"#,
    r#"image: |1
                                                ________  h___
         __        __      _____       ___     |        | |  L|_
       _/ L\__   _| L\__  |    L\_   _/  L\__  |        |_|     |
      '-o---o-' '-o---o-' '-O---O-' '=o----o-' '-OO----`OO----O-'
 ^jgs^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
"#,
    r#"image: |1
      .,.,.,.,.,.,.
     [_           _]
     | '-.     .-' |
      \   \   /   /
 jgs   '.__\_/__.'
"#,
    r#"image: |1
            ____.-==-, _______  _______  _______  _______  _..._
           {"""""LILI|[" " "'"]['""'"""][''""''']["" """"][LI LI]
  ^#^#^#^#^/_OO====OO`'OO---OO''OO---OO''OO---OO''OO---OO`'OO-OO'^#^#^#^
 ^^^jgs^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
"#,
    r#"image: |1
         __         .' '.
       _/__)        .   .       .
      (8|)_}}- .      .        .
 jgs   `\__)    '. . ' ' .  . '
"#,
    r#"image: |1
           .' '.            __
  .        .   .           (__\_
   .         .         . -{{_(|8)
     ' .  . ' ' .  . '     (__/`
 jgs
"#,
    r#"image: |1
         _....._
 '.  .-'`     | `.
   ><---------|---)
 .'  `-.,_    |_,'
  jgs     `````
"#,
    r#"image: |1
       _....._
     .' |     `'-.  /
    (---|---------><
     '._|    _,.-'  \
 jgs     `````
"#,
    r#"image: |1
          _....._
        .'       `'-.  /
       (- J-E-S-U-S -><
        '._     _,.-'  \
 jgs       `````
"#,
    r#"image: |1
        _,..=xxxxxxxxxxxx,
       /L_Y.-"""""""""`,--n.
       .--'[=======]|L]J: []\
       |/ _ ''_ " " ||[ -_ 4 |
 jgs   '-(_)-(_)----'v'-(_)--'
"#,
    r#"image: |1
              _....._
      '.  .-'`       `.
        >< D A R W I N )
      .'  `-.,_     _,'
 jgs           `L`L`
"#,
    r#"image: |1
           _
          `-)
        8__/__
 jgs      (___)
"#,
    r#"image: |1
             _
            (-`
        "",  \
 jgs     (____)
"#,
    r#"image: |1
           .-""-.
          ((````)) .=.
        8  \\  // /._.`-
        \.__))((_//
 jgs     '--'  '-'
"#,
    r#"image: |1
       _ .-. _
      /o`\^/`o\
     |o o | o o|
      \o _|_ o/
 jgs   `(@I@)`
"#,
    r#"image: |1
        _   |~  _
       [_]--'--[_]
       |'|""`""|'|
       | | /^\ | |
 jgs   |_|_|I|_|_|
"#,
    r#"image: |1
                ,:',:`,:'
             __||_||_||_||__
        ____["""""""""""""""]____
        \ " '''''''''''''''''''' \
 ~~jgs~^~^~^^~^~^~^~^~^~^~^~~^~^~^^~~^~^
"#,
    r#"image: |1
         ,.----------.
        ((|          |
       .--\          /--.
      '._  '========'  _.'
 jgs     `""""""""""""`
"#,
    r#"image: |1
      _______
    .'_/_|_\_'.
    \`\  |  /`/
     `\\ | //'
 jgs   `\|/`
"#,
    r#"image: |1
                 /\__
        .--.----'  - \
       /    )    \___/
      |  '------.___)
 jgs   `---------`
"#,
    r#"image: |1
         _  m
       ,`.\/'>
       (`\<_/`
 jgs     `<<
"#,
    r#"image: |1
       .-.
     _( " )_
    (_  :  _)
      / ' \
 jgs (_/^\_)
"#,
    r#"image: |1
                                                     __.  .--,
 *-/___,  ,-/___,-/___,-/___,-/___,           _.-.=,{\/ _/  /`)
  `\ _ ),-/___,-/___,-/___,-/___, )     _..-'`-(`._(_.;`   /
   /< \\=`\ _ )`\ _ )`\ _ )`\ _ )<`--''`     (__\_________/___,
          /< <\ </ /< /< /< </ /<           (_____Y_____Y___,  jgs
"#,
    r#"image: |1
                                              __.  ,--,
   .-/___,-/___,-/___,-/___,           _.-.=,{\/ _/  /`)
 .-/___,-/___,-/___,-/___, )     _..-'`-(`._(_.;`   /
  `\ _ )`\ _ )`\ _ )`\ _ )<`--''`     (__\_________/___,
   /< <\ </ /< /< /< </ /<           (_____Y_____Y___,  jgs
"#,
    r#"image: |1
                       ,
        ,_     ,     .'<_
       _> `'-,'(__.-' __<
       >_.--(.. )  =;`
 jgs        `V-'`'\/``
"#,
    r#"image: |1
             .---. /)
           .'     `\/)
          /  _   .'`/|
 jgs   ,=(  __\ (__/_/==,
      `==================`
"#,
    r#"image: |1
       O      ~O
      <|\     /|\
       |   ~o/ | \o    ~o/    _o
       |\  /|  |\ |\   /|      |\
 jgs  / |  / \ |// >   / \    / >
"#,
    r#"image: |1
            ~O
            /|\
         ~o/ | \o    ~o/    _o
         /|  |\ |\   /|      |\
 jgs     / \ |// >   / \    / >
"#,
    r#"image: |1
          \--/
       /`-'  '-`\
      /          \
     /.'|/\  /\|'.\
 jgs       \/
"#,
    r#"image: |1
        (,_    ,_,    _,)
        /|\`-._( )_.-'/|\
       / | \`'-/ \-'`/ | \
      /  |_.'-.\ /.-'._|  \
 jgs /_.-'      "      `-._\
"#,
    r#"image: |1
       _
      /\`--.
     |o-|   )>=====o
 jgs  \/.--'
"#,
    r#"image: |1
       _   __    _   __
      ( `^` ))  ( `^` ))
      |     ||  |     ||
      |     ||  |     ||
 jgs  '-----'`  '-----'`
"#,
    r#"image: |1
      .-~~~~-. |\\_
   @_/        /  oo\_
     |    \   \   _(")
      \   /-| ||'--'
 jgs   \_\  \_\\
"#,
    r#"image: |1
       _//|.-~~~~-,
     _/66  \       \_@
    (")_   /   /   |
      '--'|| |-\  /
 jgs      //_/ /_/
"#,
    r#"image: |1
         _   ,_,   _
        / `'=) (='` \
       /.-.-.\ /.-.-.\
 jgs   `      "      `
"#,
    r#"image: |1
          ,_,
          )v(
          \_/
 jgs    ==="===
"#,
    r#"image: |1
                 ;'-.
     `;-._        )  '---.._
       >  `-.__.-'          `'.__
      /_.-'-._         _,   ^ ---)
 jgs  `       `'------/_.'----```
"#,
    r#"image: |1
               _
          .-.-.=\-.
 jgs      (_)=='(_)
"#,
    r#"image: |1
  ___               ___ ___  ___ ___  ___ ___
 |o o|             |ooo|ooo||ooo|o o||o o|o o|
 |o_o| ___ ___  ___|ooo|ooo||ooo|o_o||o_o|_o_|___
 |o  ||o  |o  ||o  |ooo|                 |o o|o  |
 |__o||__o|__o||__o|ooo| jgs             |_o_|__o|
"#,
    r#"image: |1
         _   ,_,   _
        / `'=) (='` \
       /.-.-.\ /.-.-.\
 jgs   `      "      `
"#,
    r#"image: |1
        (_    ,_,    _)
        / `'--) (--'` \
       /  _,-'\_/'-,_  \
 jgs  /.-'     "     '-.\
"#,
    r#"image: |1
                       __
     (\   .-.   .-.   /_")
      \\_//^\\_//^\\_//
 jgs   `"`   `"`   `"`
"#,
    r#"image: |1
             __
         _ ."--". _
        (_:/ oo \:_)
    /\,   _(`()`)_   ,/\
    \ `~~~,(`""`),~~~` /
 jgs `'---|/`""`\|---'`
"#,
    r#"image: |1
      ._,_,
    ,/'`'=.    ().-=-.()
    c _','|     |=. .=|
 jgs \\_//     (  _0_  )
        `"`       `"""""`
"#,
    r#"image: |1
              ,
            __)\_
      (\_.-'    a`-.
 jgs  (/~~````(/~^^`
"#,
    r#"image: |1
             ,
           __)\___
       _.-'      .`-.
 jgs .'/~~```"~z/~'"`
     ^^
"#,
    r#"image: |1
       ,       -,
       )\______) \___
 jgs  /.-.,_    ))  a`.
      `     "'/ /----""`
              "`
"#,
    r#"image: |1
           \   '
           /\ o
         >=)'>
           \/
 jgs       /
"#,
    r#"image: |1
         ,_
         /.\
        {>) `
 jgs    `((_,
          `"`
"#,
    r#"image: |1
          _         _
     \. _(9>       <6)_ ,/
 jgs  \==_)         (_==/  jgs
       -'=           ='-
"#,
    r#"image: |1
        _      _
       <')_,/ <') ,/
       (_==/  (_==/
 jgs    ='-    ='-
"#,
    r#"image: |1
          .-.
         ( (9>
         / )\
       ,/_/_/
 jgs     -"=
"#,
    r#"image: |1
    (\  }\   (\  }\   (\  }\
   (  \_('> (  \_('> (  \_('>
   (__(=_)  (__(=_)  (__(=_)
 jgs  -"=      -"=      -"=
"#,
    r#"image: |1
       _          _          _          _          _
     >(')____,  >(')____,  >(')____,  >(')____,  >(') ___,
       (` =~~/    (` =~~/    (` =~~/    (` =~~/    (` =~~/
 jgs~^~^`---'~^~^~^`---'~^~^~^`---'~^~^~^`---'~^~^~^`---'~^~^~
"#,
    r#"image: |1
                  ,-.
          ,      ( {o\
          {`"=,___) (`~
           \  ,_.-   )
 jgs  ~^~^~^`- ~^ ~^ '~^~^~^~
"#,
    r#"image: |1
      ,___,
      (9v9)
      (_^((\
 jgs^^^"^" \\^^^^
    ^^^^^^^^^^^^^
"#,
    r#"image: |1
      ,___,
      (6v6)
      (_^(_\
 jgs^^^"^" \\^^^^
    ^^^^^^^^^^^^^
"#,
    r#"image: |1
         (\___/)
         (o\ /o)
        /|:.V.:|\
 jgs    \\::::://
    -----`"" ""`-----
"#,
    r#"image: |1
  __,___@
 [_'^   )
   `//-\\
   ^^  ^^
"#,
    r#"image: |1
       @___,__
 jgs   (   ^'_]
       //-\\'
       ^^  ^^
"#,
    r#"image: |1
         _       _       _       _       _       _       _       _
      _-(_)-  _-(_)-  _-(_)-  _-(")-  _-(_)-  _-(_)-  _-(_)-  _-(_)-
    `(___)  `(___)  `(___)  `%%%%%  `(___)  `(___)  `(___)  `(___)
 jgs // \\   // \\   // \\   // \\   // \\   // \\   // \\   // \\
"#,
    r#"image: |1
  \_/
 '-0-'
 --0--
 .-0-.
"#,
    r#"image: |1
  \O/
 '-O-'
  /o\
   ^
"#,
    r#"image: |1
         _
        /_/_      .'''.
     =O(_)))) ...'     `.
 jgs    \_\              `.    .'''
                           `..'
"#,
    r#"image: |1
         ,   ,
        { \w/ }
         `>!<`
 jgs     (/^\)
         '   '
"#,
    r#"image: |1
     \_/-.--.--.--.--.--.
     (")__)__)__)__)__)__)
 jgs  ^ "" "" "" "" "" ""
"#,
    r#"image: |1
                            .    .
                             )  (
       _ _ _ _ _ _ _ _ _ _ _(.--.)
     {{ { { { { { { { { { { ( '_')
 jgs  >>>>>>>>>>>>>>>>>>>>>>>`--'>
"#,
    r#"image: |1
                       __
     (\   .-.   .-.   /_")
      \\_//^\\_//^\\_//
 jgs   `"`   `"`   `"`
"#,
    r#"image: |1
     ._q0p_.
     '=(_)='
      / V \
 jgs (_/^\_)
"#,
    r#"image: |1
     .W.
    (o|o)
 jgs `'`
"#,
    r#"image: |1
           (
            )
          _/
       _({_})_
 jgs    /(o)\
"#,
    r#"image: |1
         ,_,___,    ,
       _0/\|\_/ `--'
 jgs  .-'  ` `-.___,
"#,
    r#"image: |1
       |\=/|.-"""-.
       /6 6\       \
      =\_Y_/=  (_  ;\
 jgs    _U//_/-/__///
       /kit\      ((
"#,
    r#"image: |1
      \    /\
       )  ( ')
      (  /  )
 jgs   \(__)|
"#,
    r#"image: |1
       |\_/|
       |q p|   /}
       ( 0 )"""\
       |"^"`    |
 jgs   ||_/=\\__|
"#,
    r#"image: |1
       |\_/|
       |d b|    /}
       ( 0 )"""\'
       8===8    |
 jgs   ||_/=\\__|
"#,
    r#"image: |1
       .-"-.
      /|6 6|\
     {/(_0_)\}
      _/ ^ \_
 jgs (/_/^\_\)-'
"#,
    r#"image: |1
       .-"-.
      /|6 6|\
     {/(_0_)\}
      _/=o=\_
 jgs (/_/^\_\)-'
"#,
    r#"image: |1
           __
      (___()'`;
      /,    /`
 jgs  \\"--\\
"#,
    r#"image: |1
         __
        ()'`;
        /\|`
       /  |
 jgs (/_)_|_
"#,
    r#"image: |1
                                   .-.
      (___________________________()6 `-,
      (   ______________________   /''"`
 jgs  //\\                      //\\
      "" ""                     "" ""
"#,
    r#"image: |1
   _.-.
 '( ^{_}    (
   `~\`-----'\
      )_)---)_)jgs
"#,
    r#"image: |1
                  .-._
       )         {_}^ )'
      /`----------`/~`
 jgs (_(--------(_(
"#,
    r#"image: |1
     ,    /-.
    ((___/ __>
    /      }
    \ .--.(    ___
 jgs \\   \\  /___\
"#,
    r#"image: |1
        ."-.
       /X  | _o.----.    _
      /\_  \/ /  __  \_// )
 jgs  \__)-/_/\_____)____/
"#,
    r#"image: |1
       \\
        \\_
         (_)
        / )
 jgs  o( )_\_
"#,
    r#"image: |1
       \\
        \\_
         (')
 jgs    / )=
      o( )_
"#,
    r#"image: |1
          \\
           \\_
        .---(')
 jgs  o( )_-\_
"#,
    r#"image: |1
       \\
        \\_   \\
         (')   \\_
 jgs    / )=.---(')
      o( )o( )_-\_
"#,
    r#"image: |1
         \\      \\      \\      \\      \\      \\
        __()    __()    __()    __()    __()    __()
 jgs  o(_-\_  o(_-\_  o(_-\_  o(_-\_  o(_-\_  o(_-\_
"#,
    r#"image: |1
 ,____          (\=-,
 \ "=.`'-.______/ /^
  `-._.-"(=====' /
          \<'--\(
   jgs     ^^   ^^
"#,
    r#"image: |1
        (`-()_.-=-.
        /66  ,  ,  \
      =(o_/=//_(   /======`
 jgs      ~"` ~"~~`
"#,
    r#"image: |1
                    _ _
             .-"-._( ) )
            /  ,  ,  66\
     `======\   )_\\=\_o)=
 jgs         `~~"~ `"
"#,
    r#"image: |1
             _
            (c).-.
 jgs     .-" , _( )____)
"#,
    r#"image: |1
                 _
             .-.(c)
 jgs  (_____( )  , "-.
               `~  ~""`
"#,
    r#"image: |1
         _    __
        (,).-"  ".
 jgs   /',  _ (   )________,
       `^'"` """"'`
"#,
    r#"image: |1
 <'`,,-,,)-----
"#,
    r#"image: |1
 ---(___C'>
"#,
    r#"image: |1
 _____(,__,c'>
"#,
    r#"image: |1
 ___(m-mC'>
"#,
    r#"image: |1
   _()______
 /_'_/   /  `\            ,
    /'---\___/~~~~~~~~~~~~   jgs
   ~     ~~
"#,
    r#"image: |1
 ()-()
  \"/  jgs
   `
"#,
    r#"image: |1
 ()-().----.          .
  \"/` ___  ;________.'  jgs
   ` ^^   ^^
"#,
    r#"image: |1
   _  __
  (_)/   \
 <'_, ____)~~~~~~~     jgs
   ^^   ^^
"#,
    r#"image: |1
        ____()()
       /      @@
 `~~~~~\_;m__m._>o    jgs
"#,
    r#"image: |1
           ___
  _  _  .-'   '-.
 (.)(.)/         \   jgs
  /@@             ;
 o_\\-mm-......-mm`~~~~~~~~~~~~~~~~`
"#,
    r#"image: |1
         _
        : `            _..-=-=-=-.._.--.
         `-._ ___,..-'" -~~`         __')
 jgs         `'"---'"`>>"'~~"~"~~>>'`
 =====================```========```========
"#,
    r#"image: |1
       _  .----.
      (_\/      \_,
 jgs    'uu----uu~'
"#,
    r#"image: |1
          .==.
       __/-^-^\
      (' )^-^-^\)
 jgs   `^UU^^UU^
"#,
    r#"image: |1
          ,-"""-.
     66._/ \___/ \
    (____)_/___\__\_)
 jgs    /_//   \\_\
"#,
    r#"image: |1
         .-""""-,
        / \____/ \_.99
     (_/__/____\_(____)
 jgs   //_//    \\_\
"#,
    r#"image: |1
         ()-()
       .-(___)-.
        _<   >_
 jgs    \/   \/
"#,
    r#"image: |1
     =,    (\_/)    ,=
      /`-'--(")--'-'\
 jgs /     (___)     \
    /.-.-./ " " \.-.-.\
"#,
    r#"image: |1
      ()=()   ()-()   ()=()   ()-()
      ('Y')   (':')   (^;^)   ('&')
      q . p   d . b   C   C   c . c
 jgs  ()_()   ()_()   ()_()   ()=()
"#,
    r#"image: |1
          _,____ c--.
        /`  \   ` _^_\
 jgs`~~~\  _/---'\\  ^
         `~~~     ~~
"#,
    r#"image: |1
      .--p_____,_
     /_^_ `   /  `\
 jgs ^  //'---\_  /~~~
       ~~     ~~~`
"#,
    r#"image: |1
    oo  .--.  oo  .--.  oo  .--.
 jgs \\(____)_ \\(____)_ \\(____)_
      `~~~~~~~` `~~~~~~~` `~~~~~~~`
"#,
    r#"image: |1
    __  (\_
   (_ \ ( '>
     ) \/_)=
 jgs (_(_ )_
"#,
    r#"image: |1
                                  /'-\
                       .-=-.     /  " \
         _   .-'-.   .'  _  '.  |  .-\oo)
        ( \ / .-. \ /  .' '.  \ /  |
 ~jgs^~`~^~`~^~`~^~`~^~^~`^~^~`^~^~^~^~^~`
"#,
    r#"image: |1
                           _   _       _a_a
               _   _     _{.`=`.}_    {/ ''\_
         _    {.`'`.}   {.'  _  '.}  {|  ._oo)
        { \  {/ .-. \} {/  .' '.  \} {/  |
 ~jgs^~`~^~`~^~`~^~`~^~^~`^~^~`^~^~^~^~^~^~`^~~`
"#,
    r#"image: |1
    _\/_                 |                _\/_
    /o\\             \       /            //o\
     |                 .-'-.                |
    _|_______     --  /     \  --     ______|__
 jgs         `~^~^~^~^~^~^~^~^~^~^~^~`
"#,
    r#"image: |1
   ,\//,.\//\/.         ,\/,   ,\/.//,          ,~
   //o\\/o//o\\ ,.,.,   //o\   /o\\o\\          |\
     |   |  |  /###/#\    |     |  |           /| \
     |   |  |  |' '|:|    |`=.='|  |          /_|__\
 jgs^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^"""~~~~~~~~~~~~~~
"#,
    r#"image: |1
      _        _
    (\o/)    {\o/}
 jgs /_\      /_\
"#,
    r#"image: |1
              (_)L|J
       )      (") |     (
       ,(. A `/ \-|   (,`)
      )' (' \/\ / |  ) (.
 jgs (' ),).  _W_ | (,)' )
    ^^^^^^^^^^^^^^^^^^^^^^^
"#,
    r#"image: |1
                L|J(_)
       )    | (")      (
       ,(.  |`/ \- y  (,`)
      )' (' | \ /\/  ) (.
 jgs (' ),) | _W_   (,)' ).
"#,
    r#"image: |1
    ~~~;       ;~~~
  _/@@ '       ' @@\_
 (_   .C       C    _)
   ""::'        \<>/   jgs
    ''           `'
"#,
    r#"image: |1
      _//////
     // o o \
     (   7   )
 jgs  \ '=' /
       '---'
"#,
    r#"image: |1
   ,@@@@@@,
 ,@@@@@""@@,
 @@@@@ 6.6\@
 @@@C  _-_)@
 '@@@) (@@'
"#,
    r#"image: |1
        _....,_         _,...._
    _.-` _,..,_'.     .'_,..,_ `-._
     _,-`/ o \ '.     .' / o \`-,_
 jgs  '-.\___/.-`     `-.\___/.-'
"#,
    r#"image: |1
      _____________
     (_________ _.'``''-....
         (___.-' _,
          (_(_.-'  `
 jgs       (__)__,-"``'-----
"#,
    r#"image: |1
     ....-''``'._ _________)
             ,_  '-.___)
               `'-._)_)
 jgs -----'``"-,__(__)
"#,
    r#"image: |1
     oOOO() ()OOOo
     /  _)   (_  \
     |  (     )  |
 jgs \__)     (__/
"#,
    r#"image: |1
        .. ..
      .'  `  `.
    .'_.-...-._`.
     `.       .'
 jgs   `-...-'
"#,
    r#"image: |1
         .-.
        (0.0)
      '=.|m|.='
 jgs  .='`"``=.
"#,
    r#"image: |1
         _           _
        (_'---------'_)
 jgs    (_.=========._)
"#,
    r#"image: |1
        .-.-.
       (_\|/_)
       ( /|\ )       __
 jgs    '-'-'`-.__.-'  `-
"#,
    r#"image: |1
          _
        _ \'-_,#
       _\'--','`|
       \`---`  /
 jgs    `----'`
"#,
    r#"image: |1
    _____________
   |\ , , , , , /|
    \\ ` ' ` ' //
     \'._____.'/
 jgs  `'-----'`
"#,
    r#"image: |1
                x
     .-. _______|
     |=|/     /  \
     | |_____|_""_|
 jgs |_|_[X]_|____|
"#,
    r#"image: |1
              .----------------.
              |_I_I_I_I_I_I_I_I]___
      .::.    |  _          : ; _  )
 jgs  ':::'' ='-(_)----------=-(_)-'
"#,
    r#"image: |1
              .----------------.
              |_I_I_I_I_I_I_I_I]___
      .::.    |  _  SCHOOL  : ; _  )
 jgs  ':::'' ='-(_)----------=-(_)-'
"#,
    r#"image: |1
      _____________________________________
     |""|""|""|""|""|""|""|""|""|""|""|""|"|
     |  1  2  3  4  5  6  7  8  9 10 11 12 |
 jgs '-------------------------------------'
"#,
    r#"image: |1
       ,
      /(  ___________
     |  >:===========`
      )(
 jgs  ""
"#,
    r#"image: |1
                        (
       __________       )\
      /         /\______{,}
 jgs  \_________\/
"#,
    r#"image: |1
                         _    ,
  __   __   __   --  -,_/\\_~0_\ ___    __   __   __
               --    /  ___ \-  `___`"-,
             ---    `"-( @ )----( @ )---`
 jgs                    '-'      '-'
"#,
    r#"image: |1
        .--.
     .-'    `--.
 jgs '-()---()--'
"#,
    r#"image: |1
             _____
       _____|LI|_\\__
      [    _  [^   _ `)
 jgs  `"""(_)"""""(_)~
"#,
    r#"image: |1
                        __,
              __.-"-.__//
        _.-'~`         /
 jgs   '---(_)----(_)-'
"#,
    r#"image: |1
     .--------.__
     |::::::::|[_I___,
     |_.-.____I__.-~;|
 jgs  `(_)--------(_)"
"#,
    r#"image: |1
      .-------.___
      | ||||| |[_o\_
      | ^^^^^ |- `  )
 jgs  '-()------()-'
"#,
    r#"image: |1
        ___
       /___\
      ." | ".
      (o_|_o)
 jgs   u   u
"#,
    r#"image: |1
           ,~
           |\
          /| \
  ~^~ ^~ /_|__\~^~ ~^~
 ~^~^ ~ '======' ~^ ~^~
"#,
    r#"image: |1
             O__,     O__,
         /'._|\/______|\_/.'\
         \    /             /
 jgs~^~^~^~^^`~^~^~^~^~~^~^~^~^~^~^
"#,
    r#"image: |1
         _f_,_
 jgs    (_)`(_)
"#,
    r#"image: |1
       f_,
 jgs  ()\()
"#,
    r#"image: |1
           == __\
       __.-"\---|__
      /  \\_@\-'/  \
 jgs  \__/      \__/
"#,
    r#"image: |1
        )
       (_)
       |`|
       | |  _()
 jgs \_|_|_/`
"#,
    r#"image: |1
        .-------------------------------.
       /(_____,_________________________)\
      |      / . .__||] . . . . .         |
 jgs   \_____\______||]__________________/
"#,
    r#"image: |1
       ____ ___
       )  =\  =\
      /    =\  =\
      \      `-._`-._
 jgs   )__(`\____)___)
"#,
    r#"image: |1
   .--------------.
    \-.        .-/
     \ \      / /
 jgs  \ \    / /
       `-'--'-`
"#,
    r#"image: |1
           )\_ _/(
 jgs   --=(__/^\__)==-
"#,
    r#"image: |1
     ,___ ,__
      )  `\  `\
     (   _ '-._'-._
 jgs  )_( \____)___)
"#,
    r#"image: |1
      _,
   .-'_|      ,
  _| (_|     _|\
 (_|        (_|
"#,
    r#"image: |1
        _.+._
      (^\/^\/^)
       \@*@*@/
 jgs   {_____}
"#,
    r#"image: |1
                        .--.
        _              /____\
     ,_(')<     [B]    \~~~~/
 jgs \___)     [A][C]   '--'
"#,
    r#"image: |1
 O./
 O'\   jgs
"#,
    r#"image: |1
       __
      (__)   _,.-~>
       __`~~>_.-~`
 jgs  (__)~`  `~-.>
"#,
    r#"image: |1
     .-==============-.  .-==============-.
     |                |  |                |
    (|+              -| (|+              -|
     |                |  |                |
 jgs '-==============='  '-==============-'
"#,
    r#"image: |1
     .( * .
  .*  .  ) .
 . . POOF .* .
  '* . (  .) '
   ` ( . *
"#,
    r#"image: |1
           66666
         __|||||__   H A P P Y
        {._._._._.}     B I R T H D A Y  !
 jgs  __{._._._._.}__
     `~~~~~~~~~~~~~~~`
"#,
    r#"image: |1
          ,,,,,
         _|||||_
        {~*~*~*~}
      __{*~*~*~*}__
 jgs `-------------`
"#,
    r#"image: |1
        .:.      .:.         .:.
      _oOoOo   _oOoOo       oOoOo_
     [_|||||  [_|||||       |||||_]
       |||||    |||||       |||||
 jgs   ~~~~~    ~~~~~       ~~~~~
"#,
    r#"image: |1
         ___________
     _.'`.---------.`'-._
   `'-. : (")(")(") : _.'~`
 jgs   '-'---------'-'
"#,
    r#"image: |1
       _
      |-|  __
 jgs  |=| [Ll]
      "^" ====`o
"#,
    r#"image: |1
                _
           ~0  (_|
          |(_~|^~~|
          TT/_ T"T
 jgs^^^^^^^^^^^^^^^^^^^
"#,
    r#"image: |1
             _                          _
        ~0  (_|  . - ' - . _ . - ' - . |_)  O
       |(_~|^~~|                      |~~^|~_)|
       TT/_ T"T                        T"T _\HH
 jgs^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
"#,
    r#"image: |1
  .        }   ~ .__,~O     ` ~ _   ~  ^  ~  -
         `}` - =    /#/`-'     -   ~   =   ~  _ ~
    ~ .   }   ~ -   |^\   _ ~ _  - ~ -_  =  _
         }`  _____ /_  /____ - ~ _   ~ _
 jgs   }`   `~~~~~~~~~~~~~~~`_ = _ ~ -
"#,
    r#"image: |1
     -   \O                   ,  .-.___
   -     /\                 O/  /xx\XXX\
  -   __/\ `                /\  |xx|XXX|
     `    \, ()            ` << |xx|XXX|
 jgs^^^^^^^^`^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
"#,
    r#"image: |1
    O
   /\\>
 _/ < \
 =  '  `
"#,
    r#"image: |1
      ____
      \___\_.:::::::.___
 jgs  /___/ ':::::::'
"#,
    r#"image: |1
   \~~~/  ' .
    \_/  \~~~/
     |    \_/
   __|__   |
 jgs     __|__
"#,
    r#"image: |1
         \_|  __
           \_/__\            .--=='/~\
    ____,__/__,_____,______)/    /{~}}}
    -,------,----,-----,---,\'--' {{~}}
 jgs                         '--==.\}/
"#,
    r#"image: |1
                 ';.
       .---.,       \
      {}-.__,>=======;==================
       `----'      ,/
 jgs             .;'
"#,
    r#"image: |1
           _
          (_)
         (_)_)
 jgs       T
"#,
    r#"image: |1
   _____//  \\_____
 " ----( `\/` )----"
 jgs    `~``~`
"#,
    r#"image: |1
         \\\
  .---.  ///
 (:::::)(_)():
  `---'  \\\
  jgs    ///
"#,
    r#"image: |1
            (
            ___
         .-'_ =\
 jgs   c=<___\-_)
      ~~~~~~~~~~~~
"#,
    r#"image: |1
         _
        ( )   _
       (_` )_('>
       (__,~_)8
 jgs      _YY_
"#,
    r#"image: |1
        _[_]_
         (")
     `--( : )--'
       (  :  )
 jgs ""`-...-'""
"#,
    r#"image: |1
           ___       ___       ___       ___
          ('v')     ('v')     ('v')     ('v')
         ((   ))   ((   ))   ((   ))   ((   ))
 jgs   -/-"---"---/-"---"---/-"---"---/-"---"--
"#,
    r#"image: |1
          .-.   .-.   .-.   .-.   .-.
         ((_)) ((_)) ((_)) ((_)) ((_))
 jgs      '-'   '-'   '-'   '-'   '-'
"#,
    r#"image: |1
      _
     [_]|
     | ||.==.
     |_ ``""`)
 jgs  _)    (_
"#,
    r#"image: |1
        O
    |  /|
    |_(__\
     \  )\
 jgs  )( /_
"#,
    r#"image: |1
               )
              (
  _ ___________ )
 [_[___________#
"#,
    r#"image: |1
       ____
   __ /\../\ __
  (__|>:  :<|__)
 jgs  \/''\/
       ^^^^
"#,
    r#"image: |1
          ==
       <^\()/^>
        \/  \/
 jgs     /  \
         `''`
"#,
    r#"image: |1
        ==  ==      "we are each of us angels
     <^\()/\()/^>       with but one wing,
      \/  \/  \/       and can only fly by
       /  \/  \        embracing each other"
 jgs   `""``""`
"#,
    r#"image: |1
     (       ..:::[=--.  /o\             _
       )   .:::''      \ (")\           /_\
      (,`):::,(.        `/:\            I I
      ) (. )' ('         |:|`\       ,={_O_}
 jgs (,)' ). (' ),)     _/^|_  -.__.'   | |
"#,
    r#"image: |1
        .=.
       //"\\
      (/6 6\)
      )\ = /(
 jgs (_ ) ( _)
"#,
    r#"image: |1
            {}   _,
        ___/__\_//
       (_\_    _/
        -' `)(`
 jgs       ~""~
"#,
    r#"image: |1
                      \  '  /
         ~,             .-.          ^^
         /|        -=  (   )  =-       ^^
    ~^~ /_|\ ^~~-~^~^~-=~=~=-~^~^~^-~^~^~^~
 jgs~^~ ~^~^ ~-~^~~-~^~^~-=^~~^-~^~^~^-~^~^~^~
"#,
    r#"image: |1
       ~,  ^^             |          "The sun setting
       /|    ^^         \ _ /           is no less
      / |\           -=  ( )  =-      beautiful than
 ~^~ ~=====^~^~-~^~~^~^-=~=~=-~^~^~^~ the sun rising."
"#,
];

#[cfg(test)]
mod tests {
    use super::{DEFAULT_IMAGES, IMAGE_KNOWN_SIGNATURES};
    use crate::image::Image;

    #[test]
    fn test_parsing() {
        assert_eq!(
            DEFAULT_IMAGES
                .iter()
                .map(|i| { (i, Image::from_yaml(i)) })
                .inspect(|(i, e)| if e.is_err() {
                    eprintln!("Error:{:?}\n\nImage:\n{}", e, i)
                })
                .count(),
            320 // Number of built in images!
        )
    }

    #[test]
    fn test_not_all_leading_whitespace() {
        for (n, img) in DEFAULT_IMAGES.iter().enumerate() {
            if !img
                .lines()
                // The first line is always `image: |1`. We skip it.
                .skip(1)
                .map(|l| {
                    let mut it = l.chars();
                    // Skip first char in line.
                    let _ = it.next();
                    // We want the second char.
                    let second_char = it.next().unwrap_or(' ');
                    second_char
                })
                .any(|second_char| !second_char.is_whitespace())
            {
                panic!(
                    "Please remove leading useless whitespace in image no. {}:\n{}",
                    n, img
                );
            };
        }
    }

    /// Test if `IMAGE_KNOWN_SIGNATURES[i].len()` is >= 3.
    /// Otherwise we will have too many false positives.
    #[test]
    fn test_image_known_signatures() {
        for sig in IMAGE_KNOWN_SIGNATURES {
            assert!(sig.len() >= 3);
        }
    }
}
