use crate::jwt;
use failure::{Error, SyncFailure};
use std::result::Result as StdResult;

// A handy alias for `Result` that carries a generic error type.
pub type Result<T> = ::std::result::Result<T, Error>;

// The `jsonwebtoken` lib uses the deprecated `error_chain` lib for its error
// handling. This is a small fix so that the `failures` lib plays nicely with
// the errors generated by `error_chain`.
//
// See: https://github.com/rust-lang-nursery/failure/issues/109
pub trait ResultExt<T, E> {
    fn sync(self) -> StdResult<T, SyncFailure<E>>
    where
        Self: Sized,
        E: ::std::error::Error + Send + 'static;
}

impl<T, E> ResultExt<T, E> for StdResult<T, E> {
    fn sync(self) -> StdResult<T, SyncFailure<E>>
    where
        Self: Sized,
        E: ::std::error::Error + Send + 'static,
    {
        self.map_err(SyncFailure::new)
    }
}

// The types of errors a Validator may encounter.
#[derive(Fail, Debug)]
pub enum ValidatorError {
    #[fail(display = "JWT header did not contain a `kid` claim: {:?}", _0)]
    NoKIDFound(jwt::Header),

    #[fail(
        display = "JWT header did not contain a valid `kid` claim. As per \
                   ASAP spec, the `kid` claim must start with \"$iss/\" where $iss is the \
                   issuer (kid: {:?}, iss: {:?})",
        _0, _1
    )]
    InvalidKID(String, String),

    #[fail(display = "Received `None` when fetching from cache")]
    CacheError,

    #[fail(display = "Failed to retrieve public key from keyserver: {:?}", _0)]
    KeyserverError(String),

    #[fail(display = "Expired item: {:?}", _0)]
    ExpiredCache(String),

    #[fail(
        display = "Token contained a lifespan greater than the `max_lifespan` \
                   (hard limit of 3600 seconds)"
    )]
    InvalidLifespan,

    #[fail(display = "Immature jwt signature, nbf: {:?} exp: {:?}", _0, _1)]
    ImmatureSignature(i64, i64),

    #[fail(display = "Expired jwt signature, nbf: {:?} exp: {:?}", _0, _1)]
    ExpiredSignature(i64, i64),

    #[fail(display = "Duplicate `jti` encountered: {:?}", _0)]
    DuplicateJTI(String),

    #[fail(display = "Required claim not found in token: {:?}", _0)]
    ClaimNotFound(String),

    #[fail(
        display = "Resource server audience not found in `aud` claims of \
                   token {:?}",
        _0
    )]
    UnrecognisedAudience(Vec<String>),

    #[fail(
        display = "Unknown or unauthorized subject {:?}. The `sub` claim \
                   (or `iss`) must exist in `whitelisted_issuers` {:?}",
        _0, _1
    )]
    UnauthorizedSubject(String, Vec<String>),
}
