//! # Art
//!
//! A library for modeling artistic concepts.

pub mod kinds {
    /// The primary colors according to the RYB color model.
    #[derive(Debug, PartialEq)]
    pub enum PrimaryColor {
        Red,
        Yellow,
        Blue,
    }

    /// The secondary colors according to the RYB color model.
    #[derive(Debug, PartialEq)]
    pub enum SecondaryColor {
        Orange,
        Green,
        Purple,
    }
}

pub mod utils {
    use crate::kinds::*;

    /// Combines two primary colors in equal amounts to create
    /// a secondary color.
    /// # Example
    /// ```
    /// use art_lib::{SecondaryColor, PrimaryColor, mix};
    /// assert_eq!(SecondaryColor::Purple, mix(PrimaryColor::Red, PrimaryColor::Blue));
    /// assert_eq!(SecondaryColor::Purple, mix(PrimaryColor::Blue, PrimaryColor::Red));
    /// assert_eq!(SecondaryColor::Orange, mix(PrimaryColor::Red, PrimaryColor::Yellow));
    /// assert_eq!(SecondaryColor::Orange, mix(PrimaryColor::Yellow, PrimaryColor::Red));
    /// assert_eq!(SecondaryColor::Green, mix(PrimaryColor::Yellow, PrimaryColor::Blue));
    /// assert_eq!(SecondaryColor::Green, mix(PrimaryColor::Blue, PrimaryColor::Yellow));
    /// ```
    pub fn mix(c1: PrimaryColor, c2: PrimaryColor) -> SecondaryColor {
        match c1 {
            PrimaryColor::Blue => match c2 {
                PrimaryColor::Yellow => SecondaryColor::Green,
                PrimaryColor::Red => SecondaryColor::Purple,
                PrimaryColor::Blue => panic!("You can't mix the same colors"),
            },
            PrimaryColor::Yellow => match c2 {
                PrimaryColor::Yellow => panic!("You can't mix the same colors"),
                PrimaryColor::Red => SecondaryColor::Orange,
                PrimaryColor::Blue => SecondaryColor::Green,
            },
            PrimaryColor::Red => match c2 {
                PrimaryColor::Yellow => SecondaryColor::Orange,
                PrimaryColor::Red => panic!("You can't mix the same colors"),
                PrimaryColor::Blue => SecondaryColor::Purple,
            },
        }
    }
}

pub use kinds::PrimaryColor;
pub use kinds::SecondaryColor;
pub use utils::mix;

#[cfg(test)]
mod test {
    use super::*;

    #[test]
    #[should_panic(expected = "You can't mix the same colors")]
    fn test_yellow_yellow() {
        mix(PrimaryColor::Yellow, PrimaryColor::Yellow);
    }

    #[test]
    #[should_panic(expected = "You can't mix the same colors")]
    fn test_red_red() {
        mix(PrimaryColor::Red, PrimaryColor::Red);
    }

    #[test]
    #[should_panic(expected = "You can't mix the same colors")]
    fn test_blue_blue() {
        mix(PrimaryColor::Blue, PrimaryColor::Blue);
    }
}
