#[derive(Debug)]
pub enum OptError {
    InvalidOpt(String),
}

#[derive(Clone, Debug)]
pub struct Opt {
    pub name: String,
    pub short: String,
    pub long: String,
    pub desc: Option<String>,
    pub(crate) kind: OptKind,
}

impl Opt {
    /// Create a boolean opt that is present or not and does not accept additional arguments
    ///
    /// Example:
    /// ```rust
    /// use arkham::{Opt, App};
    /// App::new().opt(Opt::flag("verbose").short("v").long("verbose"));
    ///```
    pub fn flag(name: &str) -> Self {
        Self {
            name: name.into(),
            short: "".into(),
            long: "".into(),
            kind: OptKind::Flag,
            desc: None,
        }
    }

    /// Create a opt that accepts additioanl arguments
    ///
    /// Example:
    /// ```rust
    /// use arkham::{Opt, App};
    /// App::new().opt(Opt::scalar("user").short("u").long("user"));
    ///```
    pub fn scalar(name: &str) -> Self {
        Self {
            name: name.into(),
            short: "".into(),
            long: "".into(),
            kind: OptKind::String,
            desc: None,
        }
    }

    /// Sets the short flag that can be used with -x
    ///
    /// Example:
    /// ```rust
    /// use arkham::{Opt, App};
    /// App::new().opt(Opt::scalar("user").short("u").long("user"));
    ///```
    pub fn short(mut self, short: &str) -> Self {
        self.short = short.into();
        self
    }

    /// Sets the long flag that can be used with --xxxxx
    ///
    /// Example:
    /// ```rust
    /// use arkham::{Opt, App};
    /// App::new().opt(Opt::scalar("user").short("u").long("user"));
    ///```
    pub fn long(mut self, long: &str) -> Self {
        self.long = long.into();
        self
    }

    /// Sets the description for the option. This is displayed when listing via help commands
    ///
    /// Example:
    /// ```rust
    /// use arkham::{Opt, App};
    /// App::new()
    ///     .opt(
    ///         Opt::scalar("user")
    ///         .short("u")
    ///         .long("user")
    ///         .desc("The user to perform the action against")
    ///     );
    ///```
    pub fn desc(mut self, desc: &str) -> Self {
        self.desc = Some(desc.into());
        self
    }

    pub(crate) fn usage(&self) -> String {
        match self.kind {
            OptKind::Flag => {
                format!("-{}, --{}", self.short, self.long)
            }
            OptKind::String => {
                format!("-{} [value], --{} [value]", self.short, self.long)
            }
        }
    }
}

#[derive(Clone, Debug)]
pub(crate) enum OptKind {
    Flag,
    String,
}
