use crate::actions::ActionId;
use crate::ArchetypeError;
use std::fs;
use std::path::PathBuf;

#[derive(Debug, Deserialize, Serialize)]
pub struct ArchetypeConfig {
    #[serde(skip_serializing_if = "Option::is_none")]
    description: Option<String>,
    #[serde(skip_serializing_if = "Option::is_none")]
    authors: Option<Vec<String>>,
    #[serde(skip_serializing_if = "Option::is_none")]
    languages: Option<Vec<String>>,
    #[serde(skip_serializing_if = "Option::is_none")]
    frameworks: Option<Vec<String>>,
    #[serde(skip_serializing_if = "Option::is_none")]
    tags: Option<Vec<String>>,
    #[serde(skip_serializing_if = "Option::is_none", alias = "actions")]
    script: Option<Vec<ActionId>>,
}

impl ArchetypeConfig {
    pub fn new() -> ArchetypeConfig {
        ArchetypeConfig::default()
    }

    pub fn load<P: Into<PathBuf>>(path: P) -> Result<ArchetypeConfig, ArchetypeError> {
        let mut path = path.into();
        if path.is_dir() {
            let candidates = vec!["archetype.yml", "archetype.yaml"];
            for candidate in candidates {
                let config_file = path.join(candidate);
                if config_file.exists() {
                    path = config_file;
                }
            }
        }
        if path.is_dir() {
            Err(ArchetypeError::ArchetypeConfigMissing)
        } else if !path.exists() {
            Err(ArchetypeError::ArchetypeConfigNotFound { path })
        } else {
            let config = fs::read_to_string(&path)?;
            return match serde_yaml::from_str::<ArchetypeConfig>(&config) {
                Ok(config) => Ok(config),
                Err(source) => Err(ArchetypeError::YamlError { path, source }),
            }
        }
    }

    pub fn with_description(mut self, description: &str) -> ArchetypeConfig {
        self.description = Some(description.into());
        self
    }

    pub fn add_author(&mut self, author: &str) {
        let authors = self.authors.get_or_insert_with(|| vec![]);
        authors.push(author.into());
    }

    pub fn with_author(mut self, author: &str) -> ArchetypeConfig {
        self.add_author(author);
        self
    }

    pub fn authors(&self) -> &[String] {
        self.authors.as_ref().map(|v| v.as_slice()).unwrap_or_default()
    }

    pub fn with_language(mut self, language: &str) -> ArchetypeConfig {
        self.add_language(language);
        self
    }

    pub fn add_language(&mut self, language: &str) {
        let languages = self.languages.get_or_insert_with(|| Vec::new());
        languages.push(language.to_owned());
    }

    pub fn languages(&self) -> &[String] {
        self.languages.as_ref().map(|r| r.as_slice()).unwrap_or_default()
    }

    pub fn with_tag(mut self, tag: &str) -> ArchetypeConfig {
        self.add_tag(tag);
        self
    }

    pub fn add_tag(&mut self, tag: &str) {
        let tags = self.tags.get_or_insert_with(|| Vec::new());
        tags.push(tag.to_owned());
    }

    pub fn tags(&self) -> &[String] {
        self.tags.as_ref().map(|r| r.as_slice()).unwrap_or_default()
    }

    pub fn with_framework(mut self, framework: &str) -> ArchetypeConfig {
        self.add_framework(framework);
        self
    }

    pub fn add_framework(&mut self, framework: &str) {
        let frameworks = self.frameworks.get_or_insert_with(|| Vec::new());
        frameworks.push(framework.to_owned());
    }

    pub fn frameworks(&self) -> &[String] {
        self.frameworks.as_ref().map(|r| r.as_slice()).unwrap_or_default()
    }

    pub fn with_action(mut self, action: ActionId) -> ArchetypeConfig {
        self.add_action(action);
        self
    }

    pub fn add_action(&mut self, action: ActionId) {
        let actions = self.script.get_or_insert_with(|| Vec::new());
        actions.push(action);
    }

    pub fn actions(&self) -> &[ActionId] {
        self.script.as_ref().map(|r| r.as_slice()).unwrap_or_default()
    }
}

impl Default for ArchetypeConfig {
    fn default() -> Self {
        ArchetypeConfig {
            description: None,
            authors: None,
            languages: None,
            frameworks: None,
            tags: None,
            script: None,
        }
    }
}

#[cfg(test)]
mod tests {
    use super::*;
    use crate::config::variable::VariableType;
    use crate::config::VariableInfo;
    use linked_hash_map::LinkedHashMap;

    #[test]
    fn test_serialize_to_yaml() {
        let mut variables = LinkedHashMap::new();
        variables.insert(
            "name".to_owned(),
            VariableInfo::with_prompt("What is your first name?")
                .with_type(VariableType::Enum(vec!["DynamoDb".to_owned(), "JPA".to_owned()]))
                .build(),
        );

        let config = ArchetypeConfig::default()
            .with_description("Simple REST Service")
            .with_language("Java")
            .with_framework("Spring")
            .with_framework("Hessian")
            .with_tag("Service")
            .with_tag("REST")
            .with_action(ActionId::Set(variables));

        let output = serde_yaml::to_string(&config).unwrap();
        println!("{}", output);
    }
}
