use anyhow::{anyhow, Result};
use async_oneshot::{oneshot, Sender};
use concurrent_queue::ConcurrentQueue;
use std::future::Future;
use std::sync::atomic::{AtomicU8, Ordering};

const IDLE: u8 = 0;
const OPEN: u8 = 1;

/// async future thread safe queue
pub struct AQueue {
    deque: ConcurrentQueue<Sender<()>>,
    spin_loop_reader: async_channel::Receiver<()>,
    spin_loop_sender: async_channel::Sender<()>,
    state: AtomicU8,
    lock: AtomicU8,
}

impl Default for AQueue {
    #[inline]
    fn default() -> Self {
        let (spin_loop_sender, spin_loop_reader) = async_channel::unbounded();

        AQueue {
            deque: ConcurrentQueue::unbounded(),
            spin_loop_reader,
            spin_loop_sender,
            state: AtomicU8::new(IDLE),
            lock: AtomicU8::new(IDLE),
        }
    }
}

impl AQueue {
    #[inline]
    pub fn new() -> AQueue {
        AQueue::default()
    }

    #[inline]
    pub async fn run<A, T, S>(&self, call: impl FnOnce(A) -> T, arg: A) -> Result<S>
    where
        T: Future<Output = Result<S>>,
    {
        self.check_run(call(arg)).await
    }

    #[inline]
    async fn check_run<S, T>(&self, future: T) -> Result<S>
    where
        T: Future<Output = Result<S>>,
    {
        while self.lock.load(Ordering::Acquire) == OPEN {
            self.spin_loop_reader.recv().await?;
        }

        loop {
            if self.state.compare_exchange(IDLE, OPEN, Ordering::Acquire, Ordering::Acquire) == Ok(IDLE) {
                self.lock.store(IDLE, Ordering::Release);
                self.spin_loop_sender.try_send(())?;
                let result = future.await;

                self.lock.store(OPEN, Ordering::Release);
                return if let Ok(mut item) = self.deque.pop() {
                    self.state.store(IDLE, Ordering::Release);
                    item.send(()).map_err(|_| anyhow!("rx is close"))?;
                    result
                } else {
                    self.state.store(IDLE, Ordering::Release);
                    self.lock.store(IDLE, Ordering::Release);
                    self.spin_loop_sender.try_send(())?;
                    result
                };
            } else {
                let (sender, receiver) = oneshot();
                self.deque.push(sender).map_err(|err| anyhow!(err.to_string()))?;
                receiver.await.map_err(|_| anyhow!("tx is close"))?;
            }
        }
    }
}
