use super::*;
use std::cmp::Ordering;
use std::time::SystemTime;

/// Something that happens - created and used by [`State`].
#[derive(Debug, Clone, PartialEq, Eq, Hash)]
pub struct Event {

	pub(crate) page: Rc<Page>,
	pub(crate) start: Start,
	pub(crate) end: End,

}

impl Deref for Event {

	type Target = str;

	fn deref(&self) -> &Self::Target {
		
		self.slice()

	}

}

impl Event {

	/// `self.end.status`
	pub fn status(&self) -> &Status {
		
		&self.end.status
		
	}

	/// A slice from `self.start.place` to `self.end.place`.
	pub fn slice(&self) -> &str {

		let start = self.start.place.index;
		let end = self.end.place.index;
		let range = start .. end + 1;
		let string = self.page.string.as_ref();
		
		&string[range]

	}
	
	/// `self.start.goal`
	pub fn goal(&self) -> &Goal {
		
		&self.start.goal
		
	}

}

impl PartialOrd for Event {

	fn partial_cmp(&self, other: &Self) -> Option<std::cmp::Ordering> {
		
		Some(self.cmp(&other))

	}

}

impl Ord for Event {

	fn cmp(&self, other: &Self) -> std::cmp::Ordering {
		
		let start_comparison = self.start.time.cmp(&other.start.time);

		match start_comparison {

			Ordering::Equal => self.end.time.cmp(&other.end.time),
			_ => start_comparison

		}

	}

}

impl Display for Event {

	fn fmt(&self, f: &mut Formatter<'_>) -> std::fmt::Result {
		
		let color = self.end.status.color();
		let description = self.end.status.description();

		let mut stdout = stdout();
		
		// displaying the description of 'self.end.status'
		let _ = stdout.queue(SetForegroundColor(color));
		let _ = stdout.queue(SetAttribute(Attribute::Bold));
		
		write!(f, "\r\n{description}")?;

		let _ = stdout.queue(ResetColor);

		// if this event spans 0 characters, don't display any of 'self.page'
		if self.len() == 0 { return Ok(()); }
		
		let string: &str = self.page.deref();

		let range = self.start.place.index .. self.end.place.index + 1; 
		let line_range = self.start.place.line .. self.end.place.line + 1;

		struct Line {
	
			pub(crate) line_number: usize, 
			pub(crate) places: Vec::<Place>

		}

		let ref page = Rc::clone(&self.page);

		let mut lines = Vec::<Line>::default();

		// initializing 'lines'
		for line_number in line_range.clone() {

			let places = Vec::default();
			let line = Line { line_number, places };

			lines.push(line);

		}

		// populating 'places' in 'lines'
		for (index, character) in string.char_indices() {

			if character == '\n' { continue; }

			let place = Place::new(page, index);

			for line in lines.iter_mut() {

				let line_number = line.line_number;

				if line_range.contains(&line_number) {
		
					line.places.push(place.clone());
	
				}

			}
			
		}

		for line in lines {

			let line_number = line.line_number;

			let _ = stdout.queue(SetForegroundColor(Color::Blue));
			let _ = stdout.queue(SetAttribute(Attribute::Bold));
			let _ = stdout.queue(SetAttribute(Attribute::Dim));
			
			// writing the 'Page' path if there is one
			if let Some(path) = &self.page.path_maybe {
				
				let path = path.display();
				
				write!(f, "\n{path}")?;
				
			}
			
			// writing the line number indicator 
			let _ = write!(f, "\n{line_number} | ");
			
			let _ = stdout.queue(SetAttribute(Attribute::Reset));

			let places = line.places;

			// writing each line spanned by this event
			for place in places {

				let place_line_number = place.line;

				let character = place.character;

				// use color when displaying this character if it's included in 'range'
				if range.contains(&place.index) {

					let _ = stdout.queue(SetForegroundColor(color));
					let _ = stdout.queue(SetAttribute(Attribute::Bold));
					
					if character.is_whitespace() {
						
						let _ = stdout.queue(SetAttribute(Attribute::Underlined));

					}

				} 

				if line_number == place_line_number { 
					
					write!(f, "{character}")?; 
				
				}

				let _ = stdout.queue(SetAttribute(Attribute::Reset));

			}

		}
		
		Ok(())

	}
		
}
