
use std::time::SystemTime;

use super::*;
use super::event::*;
use super::goal::*;

/// Information that should persist between parses, type-checks, etc.
#[derive(Debug, Hash, PartialEq, Eq)]
pub struct State {
	
	pub(crate) history: Vec<Rc<Event>>,
	pub(crate) current_page: Rc<Page>,
	pub(crate) index: usize,
	
}

impl<'state> State {

	/// Creates a [`Page`] from `path`, assigning it to `self.current_page`.
	pub fn read_file(&mut self, path: &Path) -> Result<()> {

		let string_maybe = read_to_string(path);

		let read = Read::new(self, path);
		let goal = Goal::Read(read);

		let event = self.start_event(goal);

		let path_display = path.display();

		match string_maybe {

			Ok(string) => {

				let page = Page::from(string.as_str());
				let shared_page = Rc::new(page);

				self.current_page = shared_page;
				
				let description = format!("Opened '{path_display}'.");
				let status = Status::Success(description);

				self.end_event(event, status);
				
				Ok(())

			},
			
			Err(_) => {

				let description = format!("Could not open '{path_display}'.");
				let status = Status::Error(description);

				self.end_event(event, status);

				Err(())

			}

		}
		
	}

	/// Starts a parsing [`Event`].
	pub fn start_parse<T : 'static>(&self) -> Start {

		let parse = Parse::new::<T>();
		let goal = Goal::Parse(parse);

		self.start_event(goal)

	} 

	/// Starts and ends a new [`Event`].
	pub fn new_event(&mut self, status: Status) -> Receipt {

		let start = self.start_event(Goal::None);
		
		self.end_event(start, status)

	}

	/// Starts an [`Event`].
	pub(crate) fn start_event(&self, goal: Goal) -> Start  {
		
		let ref page = Rc::clone(&self.current_page);
		
		let place = Place::new(page, self.index);
		let time = SystemTime::now();

		Start { place, time, goal }

	}

	/// Advances `self.index` by the amount given.
	pub fn advance(&mut self, amount: usize) -> Result<()> {

		let string = self.current_page.string.as_ref();
		let string_length = string.len();

		for n in 0..amount {

			if string_length == n {

				let description = "Unexpectedly reached end of input.".to_string();
				let status = Status::Error(description);
				
				let _ = self.new_event(status);

				break;

			}

			self.index += 1;

		}

		Ok(())

	}

	/// Finishes an [`Event`], adding it to `self.history`.
	pub(crate) fn end_event(&mut self, start: Start, status: Status) -> Receipt {
		
		let time = SystemTime::now();
		let place = Place::new(&self.current_page, self.index);

		let end = End { time, place, status };
		
		let page = Rc::clone(&self.current_page);

		let event = Event { start, end, page };

		let shared_event = Rc::new(event);

		self.history.push(Rc::clone(&shared_event));

		Receipt::new(shared_event)

	}

	/// An iterator over `self.history`.
	pub fn history(&self) -> std::slice::Iter<'_, Rc<Event>> {

		self.history.deref().iter()

	}

	/// Every [`Event`] in `self` with `Status::Error(_)`.	
	pub fn errors(&self) -> Vec<&Rc<Event>> {

		self.history
		.iter()
		.filter_map(
		|event|
			if let Status::Error(_) = event.end.status { Some(event) } 
			else { None }
		).collect()

	}

}

impl From<Page> for State {

	fn from(page: Page) -> Self {

		let current_page = Rc::new(page);

		let index = 0;

		let history = Default::default();

		Self { current_page, index, history }

	}

} 

impl From<&str> for State {

	fn from(string: &str) -> Self {

		let page = Page::from(string);

		Self::from(page)

	}

} 

impl Display for State {

	fn fmt(&self, f: &mut Formatter<'_>) -> std::fmt::Result {
		
		for event in &self.history {

			write!(f, "{event}")?;

		}

		Ok(())

	}

}