
use super::*;
use super::event::*;
use super::goal::*;

/// Information that should persist between parses, type-checks, etc.
#[derive(Debug, Hash, PartialEq, Eq)]
pub struct State {
	
	pub(crate) history: Vec<Rc<Event>>,
	pub(crate) current_page: Rc<Page>,
	pub(crate) index: usize,
	
}

impl<'state> State {

	/// Pushes a [`Page`] to `self.pages` and returns it as a `Rc<Page>`.
	pub fn read_file(&mut self, path: &Path) -> Result<Rc<Page>> {

		let string_maybe = read_to_string(path);

		let read = Read::new(self, path);
		let goal = Goal::Read(read);

		let event = self.start_event(goal);

		let path_display = path.display();

		match string_maybe {

			Ok(string) => {

				let page = Page::from(string.as_str());
				let shared_page = Rc::new(page);
				
				let description = format!("Opened '{path_display}'.");
				let status = Status::Success(description);

				self.finish_event(event, status);
				
				Ok(shared_page)

			},
			
			Err(_) => {

				let description = format!("Could not open '{path_display}'.");
				let status = Status::Error(description);

				self.finish_event(event, status);

				Err(())

			}

		}
		
	}

	/// Starts a parsing [`Event`].
	pub fn start_parse<T : 'static>(&self) -> Event {

		let parse = Parse::new::<T>(self);
		let goal = Goal::Parse(parse);

		self.start_event(goal)

	} 

	/// Starts an [`Event`].
	pub(crate) fn start_event(&self, goal: Goal) -> Event  {
		
		let ref page = Rc::clone(&self.current_page);
		let start = Place::new(page, self.index);
		let span = start + 0;

		let span_maybe = Some(span);

		let status = Status::Unfinished;

		Event { goal, status, span_maybe }

	}

	/// Advances `self.index` by the amount given.
	pub fn advance(&mut self, amount: usize) -> Result<()> {

		let string = self.current_page.string.as_ref();
		let string_length = string.len();

		let description = "Didn't expect end of input.".to_string();
		let status = Status::Error(description);
		let span_maybe = None;
		let goal = Goal::None;
		let event = Event { span_maybe, goal, status};
		
		for n in 0..amount {

			if string_length == n {

				let shared_event = Rc::new(event);
				self.history.push(shared_event);
				break;

			}

			self.index += 1;

		}

		Ok(())

	}

	/// Finishes an [`Event`], adding it to `self.history`.
	pub(crate) fn finish_event(&mut self, mut event: Event, mut status: Status) -> Rc<Event>{

		if status.is_unfinished() {

			let description = "Cannot finish an event with 'Status::Unfinished'.".to_string();
			status = Status::Error(description);
			
		}

		event.status = status;
		
		if let Some(span) = event.span_maybe.borrow_mut() {

			let ref page = span.page;
			let index = self.index;

			span.end = Place::new(page, index);

		}

		let shared_event = Rc::new(event);

		self.history.push(Rc::clone(&shared_event));

		shared_event

	}

	/// An iterator over `self.history`.
	pub fn history(&self) -> std::slice::Iter<'_, Rc<Event>> {

		self.history.deref().iter()

	}

	/// Every [`Event`] in `self` with `Status::Error(_)`.	
	pub fn errors(&self) -> Vec<&Rc<Event>> {

		self.history
		.iter()
		.filter_map(
		|event|
			if let Status::Error(_) = event.status { Some(event) } 
			else { None }
		).collect()

	}

}

impl From<Page> for State {

	fn from(page: Page) -> Self {

		let current_page = Rc::new(page);

		let index = 0;

		let history = Default::default();

		Self { current_page, index, history }

	}

} 

impl From<&str> for State {

	fn from(string: &str) -> Self {

		let page = Page::from(string);

		Self::from(page)

	}

} 

impl Display for State {

	fn fmt(&self, f: &mut Formatter<'_>) -> std::fmt::Result {
		
		for event in &self.history {

			write!(f, "{event}")?;

		}

		Ok(())

	}

}