use std::collections::HashMap;
use std::io::stdout;

use super::*;


/// A reference to a portion of a [`Page`]
#[derive(Clone, PartialEq, Eq, Hash)]
pub struct Span {

	pub(crate) page: Rc<Page>,
	pub(crate) start: Place,
	pub(crate) end: Place,

}

impl Deref for Span {

	type Target = str;

	fn deref(&self) -> &Self::Target {
		
		self.slice()

	}

}

impl Span {

	/// `self.page`
	pub fn page(&self) -> Rc<Page> {

		Rc::clone(&self.page)

	}

	/// A slice from `self.start` to `self.end`.
	pub fn slice(&self) -> &str {

		let start = self.start.index;
		let end = self.end.index;
		let range = start .. end + 1;
		let string = self.page.string.as_ref();
		
		&string[range]

	}

	/// Writes the lines referenced by this span, also coloring `self.slice()`. 
	pub(crate) fn write_with_color(&self, formatter: &mut Formatter, color: Color) -> std::fmt::Result {

		let mut stdout = stdout();
		
		let string: &str = self.page.deref();
		let range = self.start.index .. self.end.index + 1; 
		let line_range = self.start.line .. self.end.line + 1;

		struct Line {

			pub(crate) line_number: usize, 
			pub(crate) places: Vec::<Place>

		}

		let ref page = Rc::clone(&self.page);

		let mut lines = Vec::<Line>::default();

		for line_number in line_range.clone() {

			let places = Vec::default();
			let line = Line { line_number, places };

			lines.push(line);

		}

		for (index, character) in string.char_indices() {

			if character == '\n' { continue; }

			let place = Place::new(page, index);

			for line in lines.iter_mut() {

				let line_number = line.line_number;

				if line_range.contains(&line_number) {
		
					line.places.push(place.clone());
	
				}

			}
			
		}

		for line in lines {

			let line_number = line.line_number;

			// writing line number indicator 
			let _ = stdout.queue(SetForegroundColor(Color::Blue));
			let _ = stdout.queue(SetAttribute(Attribute::Bold));
			let _ = stdout.queue(SetAttribute(Attribute::Dim));

			if let Some(path) = &self.page.path_maybe {

				let path = path.display();

				write!(formatter, "\n{path}")?;

			}
			
			let _ = write!(formatter, "\n{line_number} | ");

			let _ = stdout.queue(SetAttribute(Attribute::Reset));

			let places = line.places;

			for place in places {

				let place_line_number = place.line;

				let character = place.character;

				if range.contains(&place.index) {

					let _ = stdout.queue(SetForegroundColor(color));
					let _ = stdout.queue(SetAttribute(Attribute::Bold));
					
					if character.is_whitespace() {
						
						let _ = stdout.queue(SetAttribute(Attribute::Underlined));

					}

				} 

				if line_number == place_line_number { 
					
					write!(formatter, "{character}")?; 
				
				}

				let _ = stdout.queue(SetAttribute(Attribute::Reset));

			}

		}
		
		Ok(())

	}

}

impl Debug for Span {

	fn fmt(&self, f: &mut Formatter<'_>) -> std::fmt::Result {

		let start = self.start.index;
		let end = self.end.index;
		write!(f, "Span {{ {start}..{end} }}")
		
	}

}

impl Display for Span {

	fn fmt(&self, f: &mut Formatter<'_>) -> std::fmt::Result {

		self.write_with_color(f, Color::White)
		
	}

}


impl Ord for Span {

	fn cmp(&self, other: &Self) -> std::cmp::Ordering {
		
		self.start.index.cmp(&other.start.index)

	}

}

impl PartialOrd for Span {

	fn partial_cmp(&self, other: &Self) -> Option<std::cmp::Ordering> {
		
		Some(self.cmp(&other))

	}

}
