use super::*;
use std::cmp::Ordering;

/// Something that happens - created and used by [`State`].
#[derive(Debug, Clone, PartialEq, Eq, Hash)]
pub struct Event {

	pub(crate) span_maybe: Option<Span>,
	pub(crate) goal: Goal,
	pub(crate) status: Status,
	
}

impl Event {

	/// `self.status`
	pub fn status(&self) -> &Status {
		
		&self.status
		
	}
	
	/// `self.goal`
	pub fn goal(&self) -> &Goal {
		
		&self.goal
		
	}
	
	/// `self.span_maybe`
	pub fn span(&self) -> &Option<Span> {

		&self.span_maybe

	}

}

impl PartialOrd for Event {

	fn partial_cmp(&self, other: &Self) -> Option<std::cmp::Ordering> {
		
		Some(self.cmp(&other))

	}

}

impl Ord for Event {

	fn cmp(&self, other: &Self) -> std::cmp::Ordering {
		
		match &self.span_maybe {

			Some(span) => match &other.span_maybe {

				Some(other_span) => span.start.index.cmp(&other_span.start.index),				
				None => Ordering::Greater,

			},

			None => match &other.span_maybe {

				Some(_) => Ordering::Less,
				None => Ordering::Equal

			}

		}

	}

}


impl Display for Event {

	fn fmt(&self, f: &mut Formatter<'_>) -> std::fmt::Result {
		
		let color = self.status.color();
		let description = self.status.description();

		let mut stdout = stdout();
		
		let _ = stdout.queue(SetForegroundColor(color));
		let _ = stdout.queue(SetAttribute(Attribute::Bold));
		
		write!(f, "\r\n{description}")?;

		let _ = stdout.queue(ResetColor);

		if let Some(span) = &self.span_maybe {

			span.write_with_color(f, color)?;

		}

		Ok(())

	}

}
