use super::*;

/// A reference to a span of some `&str`.
#[derive(Clone, PartialEq, Eq, Hash)]
pub struct Span {

	pub(crate) lines: Range<usize>,
	pub(crate) columns: Range<usize>,
	pub(crate) range: Range::<usize>,
	pub(crate) string: Rc::<str>

}


impl Span {

	/// Assigns `self.range.end`.
	pub fn end(&mut self, end: usize) {
		
		self.range.end = end;

	}

	/// A slice of `self.string()`.
	pub fn slice(&self) -> &str {

		&self.string[self.range()]

	}

	/// The complete `&str` referenced by `self`.
	pub fn string(&self) -> &str {
		
		&self.string

	}

	/// The indices of `self.slice()`.
	pub fn range(&self) -> Range<usize> {

		self.range.clone()

	}

	/// Constructs a new `Span`.
	pub fn new(string: &str, range: Range<usize>) -> Span {
		
		let mut current_line = 1;
		let mut current_column = 1;

		let mut lines = Range::default();
		let mut columns = Range::default();
		
		for (index, character) in string.char_indices() {

			if range.start == index {

				lines.start = current_line;
				columns.start = current_column;

			}

			else if range.end == index {

				lines.end = current_line + 1;
				columns.end = current_column;
				break;

			}

			if character == '\n' {

				current_line += 1;
				current_column = 1;

			} 
			
			else {

				current_column += 1;

			}

		}

		let string = Rc::from(string);

		Self { string, range, lines, columns }
		
	}

}

impl Debug for Span {

	fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
		
		let slice = &self.slice();
		let start = self.range.start;
		let end = self.range.end;
		let line_start = self.lines.start;
		let line_end = self.lines.end;
		
		write!(f, "[{slice:?}, {start}..{end}, lines {line_start}..{line_end}]")

	}

}

impl Ord for Span {

	fn cmp(&self, other: &Self) -> std::cmp::Ordering {
		
		self.range.start.cmp(&other.range.start)

	}

}

impl PartialOrd for Span {

	fn partial_cmp(&self, other: &Self) -> Option<std::cmp::Ordering> {
		
		self.range.start.partial_cmp(&other.range.start)

	}

}

impl AsRef<str> for Span {

	fn as_ref(&self) -> &str {
	
		self.slice()
		
	}

}

impl Deref for Span {

	type Target = str;

	fn deref(&self) -> &Self::Target {
	
		self.slice()
		
	}

}

