


use super::*;


/// An error message.
#[derive(Clone, Debug, PartialEq, Eq, Hash)]
pub struct Error {

	pub(crate) next_maybe: Option<Rc<Error>>,
	pub(crate) span_maybe: Option<Span>,
	pub(crate) description: String,
	pub(crate) level: Level,

}

impl IntoIterator for Error {

	type Item = Rc<Error>;
	type IntoIter = std::vec::IntoIter<Self::Item>;

	fn into_iter(self) -> Self::IntoIter {
		
		let mut all_errors = Vec::default();

		let mut current_error = Rc::from(self);

		loop {

			all_errors.push(Rc::clone(&current_error));

			match &current_error.next_maybe {

				Some(next_error) => current_error = Rc::clone(&next_error),
				None => break

			}

		}

		all_errors.into_iter()

	}

}

impl Error {

	/// `&self.span_maybe`
	pub fn span(&self) -> &Option<Span> {

		&self.span_maybe

	}
	
	/// A constructor for `Message`.
	pub fn new(span_maybe: Option<Span>, level: Level, description: &str) -> Self {

		let description = description.to_string();
		let next_maybe = None;

		Self { next_maybe, span_maybe, description, level }		

	}
	
	/// `&self.description`.
	pub fn description(&self) -> &str {

		&self.description

	}

}

impl RangeBounds<usize> for Error {

	fn start_bound(&self) -> std::ops::Bound<&usize> {

		match &self.span_maybe {

			Some(span) => Bound::Included(&span.range.start),
			None => Bound::Unbounded

		}
		
	}
	
	fn end_bound(&self) -> std::ops::Bound<&usize> {
		
		match &self.span_maybe {
	
			Some(span) => Bound::Included(&span.range.end),
			None => Bound::Unbounded
	
		}		
		
	}

}


impl AddAssign<Error> for Error {

	fn add_assign(&mut self, error: Error) {
		
		self.next_maybe = Some(Rc::from(error))
		
	}

}

impl AddAssign<Error> for &'_ mut Error {

	fn add_assign(&mut self, error: Error) {
		
		self.next_maybe = Some(Rc::from(error))
		
	}

}

impl AddAssign<Rc<Error>> for Error {

	fn add_assign(&mut self, error: Rc<Error>) {
		
		self.next_maybe = Some(Rc::clone(&error))
		
	}

}

impl AddAssign<Rc<Error>> for &'_ mut Error {

	fn add_assign(&mut self, error: Rc<Error>) {
		
		self.next_maybe = Some(Rc::clone(&error))
		
	}

}

impl Display for Error {

	fn fmt(&self, f: &mut std::fmt::Formatter) -> std::fmt::Result {

		let mut stdout = stdout();

		// displays each message
		for error in self.clone().into_iter() {
				
			let message_color = match error.level {
				
				Level::Warning => Color::Yellow,
				Level::Error => Color::Red,
				Level::Note => Color::Blue,
				
			};
			 
			let _ = stdout.queue(SetForegroundColor(message_color));
			let _ = stdout.queue(SetAttribute(Attribute::Bold));
			
			writeln!(f, "\r\n{}", error.description)?;
			
			let _ = stdout.queue(ResetColor);
			
			if error.span_maybe.is_some() {

				let span = error.span_maybe.clone().unwrap();
				let string = span.string();
				let range = span.range(); 
				let lines = span.lines.clone();
				let mut current_line = 0;

				let mut should_write_line_indicator = true;

				for (index, character) in string.char_indices() {
					
					if character == '\n' || index == 0 { 
						
						current_line += 1;
						should_write_line_indicator = true; 
					
					}

					if should_write_line_indicator { 
						
						if lines.contains(&current_line) { 

							// writing line number indicator 
							let _ = stdout.queue(SetForegroundColor(Color::Blue));
							let _ = stdout.queue(SetAttribute(Attribute::Bold));
							let _ = stdout.queue(SetAttribute(Attribute::Dim));

							let new_line_maybe = if current_line == 1 { "" } else {"\r\n"};

							write!(f, "{new_line_maybe}{current_line} | ")?;

							if index == 0 {

								if range.contains(&index) {

									let _ = stdout.queue(SetForegroundColor(message_color));
									let _ = stdout.queue(SetAttribute(Attribute::Bold));
			
								} 
									
								write!(f, "{character}")?;
									
							};
							
							let _ = stdout.queue(SetAttribute(Attribute::Reset));

							should_write_line_indicator = false;
							
						}

						else { continue; }

					} 
				
					if range.contains(&index) {

						let _ = stdout.queue(SetForegroundColor(message_color));
						let _ = stdout.queue(SetAttribute(Attribute::Bold));
						
					} 
			
					
					if character != '\n' && lines.contains(&current_line) { write!(f, "{character}")?; }
					
					let _ = stdout.queue(SetAttribute(Attribute::Reset));
										
				}
		
			}
			
		}
		
		Ok(())
		
	}
	
}



impl Ord for Error {

	fn cmp(&self, other: &Self) -> std::cmp::Ordering {
		
		let self_start = match &self.span_maybe {

			Some(span) => span.range.start,
			None => 0

		};

		let other_start = match &other.span_maybe {

			Some(span) => span.range.start,
			None => 0

		};

		self_start.cmp(&other_start)

	}

}

impl PartialOrd for Error {

	fn partial_cmp(&self, other: &Self) -> Option<std::cmp::Ordering> {
		
		Some(self.cmp(&other))

	}

}