use std::ops::Range;
use std::ops::RangeBounds;
use std::ops::Bound;

use super::*;

/// An error message.
#[derive(Clone, Debug, PartialEq, Eq, Hash)]
pub struct Message {

	pub(crate) range_maybe: Option<Range<usize>>,
	pub(crate) description: String,
	pub(crate) level: Level,

}

impl Message {
	
	/// A constructor for `Message`.
	pub fn start(start: usize, level: Level, description: &str) -> Self {

		let mut message = Message::new(level, description);
		let range = start .. start + 1;
		let range_maybe = Some(range);

		message.range_maybe = range_maybe;

		message

	}
	
	/// A constructor for `Message`.
	pub fn new(level: Level, description: &str) -> Self {

		let description = description.to_string();
		let range_maybe = None;

		Self {

			description,
			level,
			range_maybe

		}


	}

	/// Assigns `self.range_maybe.end`.
	pub fn end(&mut self, end: usize) {

		match &mut self.range_maybe {

			Some(range) => range.end = end,
			None => { 
				
				let mut range = Range::<usize>::default();
				range.start = end - 1;
				range.end = end;

				self.range_maybe = Some(range);

			}

		}

	}

	/// Returns `self.description`.
	pub fn description(&self) -> String {

		self.description.clone()

	}

}

impl RangeBounds<usize> for Message {

	fn start_bound(&self) -> std::ops::Bound<&usize> {

		match &self.range_maybe {

			Some(range) => Bound::Included(&range.start),
			None => Bound::Unbounded

		}
		
	}

	fn end_bound(&self) -> std::ops::Bound<&usize> {

		match &self.range_maybe {

			Some(range) => Bound::Included(&range.end),
			None => Bound::Unbounded

		}
		
		
	}

}

impl Ord for Message {

	fn cmp(&self, other: &Self) -> std::cmp::Ordering {
		
		let self_start = match &self.range_maybe {

			Some(range) => range.start,
			None => 0

		};

		let other_start = match &other.range_maybe {

			Some(range) => range.start,
			None => 0

		};

		self_start.cmp(&other_start)

	}

}

impl PartialOrd for Message {

	fn partial_cmp(&self, other: &Self) -> Option<std::cmp::Ordering> {
		
		Some(self.cmp(&other))

	}

}