use std::path::PathBuf;
use super::*;

/// An error message.
#[derive(Clone, Debug, PartialEq, Eq, Hash)]
pub struct Error {

	pub(crate) next_maybe: Option<Rc<Error>>,
	pub(crate) path_maybe: Option<PathBuf>,
	pub(crate) span_maybe: Option<Span>,
	pub(crate) level: Level,
	pub(crate) description: String,

}

impl Default for Error {

	fn default() -> Self {
		
		let level = Level::Note;
		let description = "";
		let span_maybe = None;

		Error::new(span_maybe, level, description)
	
	}

}

impl IntoIterator for Error {

	type Item = Rc<Error>;
	type IntoIter = std::vec::IntoIter<Self::Item>;

	fn into_iter(self) -> Self::IntoIter {
		
		let mut all_errors = Vec::default();

		let mut current_error = Rc::from(self);

		loop {

			all_errors.push(Rc::clone(&current_error));

			match &current_error.next_maybe {

				Some(next_error) => current_error = Rc::clone(&next_error),
				None => break

			}

		}

		all_errors.into_iter()

	}

}

impl Error {

	/// `&self.span_maybe`
	pub fn span(&self) -> &Option<Span> {

		&self.span_maybe

	}

	/// `self.level`
	pub fn level(&self) -> Level {

		self.level

	}
	
	/// A constructor for `Error`.
	pub fn new(span_maybe: Option<Span>, level: Level, description: &str) -> Self {

		let next_maybe = None;
		let path_maybe = None;
		let description = description.to_string();

		Self { next_maybe, span_maybe, level, description, path_maybe }		

	}
	
	/// `&self.description`.
	pub fn description(&self) -> &str {

		&self.description

	}

}

impl RangeBounds<usize> for Error {

	fn start_bound(&self) -> std::ops::Bound<&usize> {

		match &self.span_maybe {

			Some(span) => Bound::Included(&span.start.index),
			None => Bound::Unbounded

		}
		
	}
	
	fn end_bound(&self) -> std::ops::Bound<&usize> {
		
		match &self.span_maybe {
	
			Some(span) => Bound::Included(&span.end.index),
			None => Bound::Unbounded
	
		}		
		
	}

}


impl AddAssign<Error> for Error {

	fn add_assign(&mut self, error: Error) {
		
		self.next_maybe = Some(Rc::from(error))
		
	}

}

impl AddAssign<Error> for &'_ mut Error {

	fn add_assign(&mut self, error: Error) {
		
		self.next_maybe = Some(Rc::from(error))
		
	}

}

impl AddAssign<Rc<Error>> for Error {

	fn add_assign(&mut self, error: Rc<Error>) {
		
		self.next_maybe = Some(Rc::clone(&error))
		
	}

}

impl AddAssign<Rc<Error>> for &'_ mut Error {

	fn add_assign(&mut self, error: Rc<Error>) {
		
		self.next_maybe = Some(Rc::clone(&error))
		
	}

}

impl Display for Error {

	fn fmt(&self, f: &mut std::fmt::Formatter) -> std::fmt::Result {

		let mut stdout = stdout();

		// displays each message
		for error in self.clone().into_iter() {
				
			let message_color = error.level.color();
			 
			let _ = stdout.queue(SetForegroundColor(message_color));
			let _ = stdout.queue(SetAttribute(Attribute::Bold));
			
			write!(f, "\r\n{}", error.description)?;
			
			let _ = stdout.queue(ResetColor);
			
			if let Some(span) = &error.span_maybe {

				span.write_with_color(f, message_color)?;

			}
			
		}
		
		Ok(())
		
	}
	
}

impl Ord for Error {

	fn cmp(&self, other: &Self) -> std::cmp::Ordering {
		
		use std::cmp::Ordering;

		if self.span().is_none() { Ordering::Greater }
		else if other.span().is_none() { Ordering::Less }
		else {

			let self_span = self.span().clone().unwrap();
			let other_span = other.span().clone().unwrap();

			self_span.cmp(&other_span)

		}
	}

}

impl PartialOrd for Error {

	fn partial_cmp(&self, other: &Self) -> Option<std::cmp::Ordering> {
		
		Some(self.cmp(&other))

	}

}