use super::*;

/// An `Rc<str>` and possibly a file path.  
#[derive(Clone, Debug, PartialEq, Eq, PartialOrd, Ord, Hash)]
pub struct Page {

	pub(crate) path_maybe: Option<PathBuf>,
	pub(crate) string: Rc<str>

}

impl Page {

	pub fn from_file(&self, path: &Path) -> Result<Self> {

		use std::fs::read_to_string;

		let string_maybe = read_to_string(path);

		match string_maybe {

			Ok(string) => {
				
				let path_buffer = path.to_owned();
				let path_maybe = Some(path_buffer);

				let string = Rc::from(string);

				let page = Page { path_maybe, string };

				Ok(page)

			},

			Err(_) => {

				let path_display = path.display();
				let description = &format!("Couldn't open '{path_display}'");
				let level = Level::Error;
				let error = Error::new(None, level, description);

				Err(error)

			}

		}


	} 

	pub fn path(&self) -> &Option<PathBuf> {

		&self.path_maybe

	}

}


impl From<&str> for Page {

	fn from(string: &str) -> Self {
		
		let string = Rc::from(string);
		let path_maybe = None;

		Self { path_maybe, string }

	}

}

impl Deref for Page {

	type Target = str;

	fn deref(&self) -> &Self::Target {
		
		self.string.deref()

	}

}
