
use std::fmt::Display;
use std::io::stdout;
use std::ops::AddAssign;
use std::ops::Deref;
use crossterm::*;
use crossterm::style::*;

use super::*;

/// A collection of [`Message`].
/// 
/// To use, initialize the `Error` with a `&str`, and add a `Message`.
/// 
/// 
/// ```
/// 
/// let mut error = Error::from("Lorem ipsum dolor sit amet");
/// 
/// let mut ref warning = Message::start(0, Level::Warning, "Something's wrong here.");
/// 
/// error += warning;
/// 
/// println!("{}", warning);
/// 
/// ```
/// 
#[derive(Clone, Debug, PartialEq, Eq, Hash)]
pub struct Error<'a> {
	
	pub(crate) string: &'a str,
	pub(crate) messages: Vec<Message>

}

impl<'a> From<&'a str> for Error<'a> {

	fn from(string: &'a str) -> Self {
		
		let messages = Vec::default();
		
		Self { string, messages }

	}

}

impl<'a> Deref for Error<'a> {

	type Target = [Message];

	fn deref(&self) -> &Self::Target {
		
		self.messages.deref()

	}

}

impl Extend<Message> for Error<'_> {

	fn extend<T>(&mut self, iterator: T) 
	where T: IntoIterator<Item = Message> {
		
		for message in iterator {

			if self.contains(&message) {}

			else { self.messages.push(message) }

		}

	}

}

impl<'string> AddAssign<Error<'string>> for Error<'string> {

	fn add_assign(&mut self, error: Error) {
		
		for message in error.into_iter() {

			self.extend([message.clone()]);

		}
		
	}

}
impl<'string> AddAssign<Error<'string>> for &'_ mut Error<'string> {

	fn add_assign(&mut self, error: Error) {
		
		for message in error.into_iter() {

			self.extend([message.clone()]);

		}
		
	}

}

impl AddAssign<Message> for Error<'_> {

	fn add_assign(&mut self, message: Message) {
		
		self.extend([message]);

	}

}

impl AddAssign<Message> for &'_ mut Error<'_> {

	fn add_assign(&mut self, message: Message) {
		
		self.extend([message]);

	}

}

impl Display for Error<'_> {

	fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
		
		fn get_line_number(string: &str, index : usize) -> usize {

			let mut line = 1;

			'line_number: for (n, character) in string.char_indices() {

				if n == index { break 'line_number; }
				if character == '\n' { line += 1; }

			}

			line

		}

		let mut stdout = stdout();

		// displays each message
		for message in &self.messages {
				
			let message_color = match message.level {
				
				Level::Warning => Color::Yellow,
				Level::Error => Color::Red,
				Level::Note => Color::Blue,
				
			};
			 
			let _ = stdout.queue(SetForegroundColor(message_color));
			let _ = stdout.queue(SetAttribute(Attribute::Bold));
			
			writeln!(f, "\n{}", message.description)?;
			
			let _ = stdout.queue(ResetColor);
			
			if (&message.range_maybe).is_some() {
				
				let range = message.range_maybe.clone().unwrap(); 
	
				let message_start_line_number = get_line_number(self.string, range.start);
				let message_end_line_number = get_line_number(self.string, range.end) + 1;
				
				let message_line_numbers = message_start_line_number.. message_end_line_number;
				
				let mut line_number = 1;
				
				// is true initially, and is reset to true after every '\n'
				let mut should_write_line_indicator = true;
	
				for (index, character) in self.string.char_indices() {
					
					// only lines containing characters contained in `message.range`
					let should_write_line = message_line_numbers.contains(&line_number);
	
					if character == '\n' { 
						
						line_number += 1;
					
					}
					
					if should_write_line && should_write_line_indicator {
						
						// ... otherwise write the line number indicator 
						let _ = stdout.queue(SetForegroundColor(Color::Blue));
						let _ = stdout.queue(SetAttribute(Attribute::Bold));
						
						write!(f, " {} | ", line_number)?;
						
						let _ = stdout.queue(ResetColor);
						
					}
					
					should_write_line_indicator = false;
					
					if should_write_line {
						
						// if the message's range includes the character at this index,
						// print the color associated with it's `Level` ...
						if range.contains(&index) {
							
							if character.is_whitespace() {
	
								let _ = stdout.queue(SetAttribute(Attribute::Underlined));
		
							}
	
							let _ = stdout.queue(SetForegroundColor(message_color));
							
							write!(f, "{}", character)?;
							
							let _ = stdout.queue(ResetColor);
							
						}
						
						// otherwise print it without coloring
						else {
							
							write!(f, "{}", character)?;
							
						}
						
					}
	
					if character == '\n' { 
						
						should_write_line_indicator = true;
					
					}
					
				}
			
			}
			
		}
		
		Ok(())
		
	}
	
}


impl<'string> Ord for Error<'string> {

	fn cmp(&self, other: &Self) -> std::cmp::Ordering {

		if self.iter().len() == 0 || other.iter().len() == 0 {

			std::cmp::Ordering::Equal

		} else {

			let self_minimum = self.iter().min().unwrap();
			let other_minimum = other.iter().min().unwrap();
			
			self_minimum.cmp(&other_minimum)

		}


	}

}

impl<'string> PartialOrd for Error<'string> {

	fn partial_cmp(&self, other: &Self) -> Option<std::cmp::Ordering> {
		
		Some(self.cmp(&other))

	}

}