pub mod event;

#[macro_use]
extern crate lazy_static;

use backoff::{future::retry, ExponentialBackoff, ExponentialBackoffBuilder};
use jsonschema::JSONSchema;
use reqwest::{
    multipart::{Form, Part},
    Response, StatusCode,
};
use serde::Deserialize;
use std::time::Duration;
use std::{cell::RefCell, pin::Pin};
use std::{collections::HashMap, io::Write};
use uuid::Uuid;

use crate::event::ApptrailEvent;

// Change the alias to `Box<error::Error>`.
type Result<T> = std::result::Result<T, Box<dyn std::error::Error>>;

static EVENT_SCHEMA_RAW: &'static str = include_str!("raw-event-schema.json");

lazy_static! {
    static ref EVENT_SCHEMA: JSONSchema = get_event_schema();
}

fn get_event_schema() -> JSONSchema {
    let schema_value = serde_json::from_str::<serde_json::Value>(EVENT_SCHEMA_RAW).unwrap();
    let schema = JSONSchema::compile(&schema_value).unwrap();
    schema
}

fn default_exponential_backoff() -> ExponentialBackoff {
    ExponentialBackoffBuilder::new()
        .with_initial_interval(Duration::from_millis(200))
        .with_max_elapsed_time(Some(Duration::from_millis(4500)))
        .with_multiplier(1.2)
        .with_randomization_factor(0.2)
        .build()
}

#[derive(Clone)]
/// The Apptrail Application Events Client lets you send audit logs from your Rust applications
/// to your customers.
///
/// ## Learn more
/// - [Working with events](https://apptrail.com/docs/applications/guide/working-with-events)
/// - [SDK Reference](http://apptrail.com/docs/applications/guide/working-with-events/using-the-events-sdk/application-events-sdk-rust)
pub struct ApptrailEventsClient {
    pub region: String,

    base_api_url: String,
    api_key: String,
    application_id: String,
    req_client: reqwest::Client,

    upload_url: RefCell<Option<String>>,
    form_data: RefCell<Option<HashMap<String, String>>>,
}

fn parse_application_id(api_key_str: String) -> Result<String> {
    let api_key_bytes = base64::decode_config(api_key_str, base64::URL_SAFE_NO_PAD)?;
    let api_key_parsed = std::str::from_utf8(&api_key_bytes)?;
    let parts = api_key_parsed.split(',').collect::<Vec<_>>();
    parts
        .get(0)
        .map_or_else(|| Err("Invalid API Key.".into()), |s| Ok(s.to_string()))
}

impl ApptrailEventsClient {
    /// Create a new events client. Instantiate this once at the top of your application and reuse it.
    ///
    /// ## Arguments
    /// #### region
    ///
    /// The Apptrail region to send events to. Create a single instance per region. Regions are specified as strings, e.g. `us-west-2`.
    /// [Learn more about Apptrail regions](https://apptrail.com/docs/applications/guide/regions).
    ///
    /// #### api_key
    ///
    /// Your Apptrail secret API Key. You can generate and retrieve API Keys from the Apptrail Dashboard.
    /// [Learn more about managing API Keys](https://apptrail.com/docs/applications/guide/dashboard/managing-api-keys).
    pub fn new<T>(region: T, api_key: T) -> Result<Self>
    where
        T: ToString,
    {
        let application_id = match parse_application_id(api_key.to_string()) {
            Ok(aid) => aid,
            Err(_) => return Err("Invalid API Key.".into()),
        };

        Ok(Self {
            region: region.to_string(),
            api_key: api_key.to_string(),
            base_api_url: format!(
                "https://events.{}.apptrail.com/applications/session",
                region.to_string()
            ),
            application_id: application_id.to_string(),
            req_client: reqwest::Client::new(),
            upload_url: RefCell::new(None),
            form_data: RefCell::new(None),
        })
    }

    async fn refresh_post_policy(&mut self) -> Result<()> {
        let op = || async {
            let res = (&self.req_client)
                .get(&self.base_api_url)
                .bearer_auth(&self.api_key)
                .send()
                .await?;
            match res.status() {
                s if s.is_client_error() => {
                    res.error_for_status().map_err(backoff::Error::Permanent)
                }
                s if s.is_success() => Ok(res),
                _ => res
                    .error_for_status()
                    .map_err(|e| backoff::Error::Transient {
                        err: e,
                        retry_after: None,
                    }),
            }
        };
        let res_result: std::result::Result<Response, reqwest::Error> =
            retry(default_exponential_backoff(), op).await;

        let data = res_result?.json::<SessionData>().await?;
        Pin::new(&mut self.upload_url).set(RefCell::new(Some(data.uploadUrl)));
        Pin::new(&mut self.form_data).set(RefCell::new(Some(data.form)));

        Ok(())
    }

    fn is_session_empty(&self) -> bool {
        self.upload_url.borrow().is_none() || self.form_data.borrow().is_none()
    }

    /// Send a single audit log to Apptrail.
    pub async fn put_event(&mut self, event: &ApptrailEvent) -> Result<()> {
        self.put_events(&vec![event]).await
    }

    /// Log multiple audit events to Apptrail. You can pass up to 1000 events to this method. To log
    /// more events, make multiple calls to this method.
    pub async fn put_events(&mut self, events: &Vec<&ApptrailEvent>) -> Result<()> {
        if events.len() == 0 || events.len() > 1000 {
            return Err("Can put between 0 and 1000 events in a single call.".into());
        }
        if self.is_session_empty() {
            self.refresh_post_policy().await?
        }
        // todo: better way lol
        if self.is_session_empty() {
            return Err("Invalid session".into());
        }

        let mut body_buf = Vec::<u8>::new();
        for event in events.into_iter() {
            let event_value = serde_json::to_value(event)?;
            if !EVENT_SCHEMA.is_valid(&event_value) {
                let message = EVENT_SCHEMA
                    .validate(&event_value)
                    .err()
                    .and_then(|mut err| err.next().map(|e| e.to_string()))
                    .unwrap_or_else(|| "".to_string());
                return Err(format!("Invalid event format: {}.", message).into());
            }
            body_buf.write(&serde_json::to_vec(&event_value)?)?;
            body_buf.write("\n".as_bytes())?;
        }

        let application_id: String = self.application_id.to_owned();
        let form_data = &self.form_data.borrow().as_ref().unwrap().to_owned();

        let create_form = || {
            let form: Form = form_data
                .to_owned()
                .into_iter()
                .fold(Form::new(), |form, (k, v)| form.text(k, v));
            let filename = Uuid::new_v4().to_hyphenated().to_string() + ".jsonl";
            let s3_key = format!("{}/{}", application_id.to_owned(), filename);

            let form: Form = form.text("key", s3_key);

            let body = std::str::from_utf8(&body_buf)?.to_owned();
            let file_part = Part::text(body)
                .file_name(filename)
                .mime_str("application/jsonlines")
                .unwrap();
            let form: Form = form.part("file", file_part);
            Result::Ok(form)
        };

        let upload_url = &self.upload_url.borrow().as_ref().unwrap().to_owned();
        let do_upload = || async {
            let form = create_form()
                .map_err(|_| backoff::Error::Permanent("Failed to create form".into()));
            let res = (&self.req_client)
                .post(upload_url)
                .multipart(form?)
                .send()
                .await
                .map_err(|e| e.to_string())?;

            match res.status() {
                s if s.is_success() => Ok(res),
                StatusCode::FORBIDDEN => {
                    let refresh_result = self.to_owned().refresh_post_policy().await;
                    if let Err(_) = refresh_result {
                        Err(backoff::Error::Permanent(
                            "Failed to refresh session".into(),
                        ))
                    } else {
                        Err(backoff::Error::Transient {
                            err: "Expired session".into(),
                            retry_after: None,
                        })
                    }
                }
                s if s.is_client_error() => res
                    .error_for_status()
                    .map_err(|e| backoff::Error::Permanent(e.to_string())),

                _ => res
                    .error_for_status()
                    .map_err(|e| backoff::Error::Transient {
                        err: e.to_string(),
                        retry_after: None,
                    }),
            }
        };
        let res_result: std::result::Result<Response, String> =
            retry(default_exponential_backoff(), do_upload).await;
        match res_result {
            Ok(_) => {
                println!("Apptrail: Succesfully put {} events.", events.len());
                Ok(())
            }
            Err(_) => Err("Failed to put event.".into()),
        }
    }
}

impl std::fmt::Display for ApptrailEventsClient {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        write!(f, "ApptrailEventsClient {{ region: {} }}", &self.region)
    }
}

#[derive(Deserialize)]
struct SessionData {
    uploadUrl: String,
    form: HashMap<String, String>,
}
