use serde::{Deserialize, Serialize};
use std::collections::HashMap;

/// An Apptrail event is a record of a an activity performed on your customer resources in
/// your application.
///
/// @see https://apptrail.com/docs/applications/guide/working-with-events/overview
#[derive(Serialize, Deserialize, Debug)]
pub struct ApptrailEvent {
    /// Actor specifies the identity that performed the action being logged. This may be a user
    /// or machine client. Consumers can filter based on this attribute.
    #[serde(rename = "actor")]
    #[serde(skip_serializing_if = "Option::is_none")]
    pub actor: Option<Actor>,

    /// The context object contains fields that give customers additional information about the
    /// environment in and source from which the activity recorded occurred
    #[serde(rename = "context")]
    #[serde(skip_serializing_if = "Option::is_none")]
    pub context: Option<Context>,

    /// Any additional custom information as a JSON object can be included in this field.
    #[serde(rename = "eventDetails")]
    #[serde(skip_serializing_if = "Option::is_none")]
    pub event_details: Option<HashMap<String, Option<serde_json::Value>>>,

    /// The name of the event as a string. You should represent unique events with unique names.
    /// For example if you are adding audit logs to an API, event names could correspond to your
    /// API methods.
    #[serde(rename = "eventName")]
    pub event_name: String,

    /// The time the event occurred. Formatted as an ISO 8601 formatted string, e.g.
    /// 2021-12-06T16:09:38Z.
    #[serde(rename = "eventTime")]
    pub event_time: String,

    /// If applicable, contains information about the relevant resources that the action this
    /// event is recording is related to.
    #[serde(rename = "resources")]
    #[serde(skip_serializing_if = "Option::is_none")]
    pub resources: Option<Vec<Resource>>,

    /// Tags are key value metadata that you associate with events. Users can search and filter
    /// events based on these tags.
    #[serde(rename = "tags")]
    #[serde(skip_serializing_if = "Option::is_none")]
    pub tags: Option<HashMap<String, String>>,

    /// The ID of the tenant this event is associated with and will be delivered to. For more on
    /// tenants @see https://apptrail.com/docs/applications/guide/working-with-tenants.
    #[serde(rename = "tenantId")]
    pub tenant_id: String,
}

/// Actor specifies the identity that performed the action being logged. This may be a user
/// or machine client. Consumers can filter based on this attribute.
#[derive(Serialize, Deserialize, Debug)]
pub struct Actor {
    /// Additional arbitrary metadata related to the actor.
    #[serde(rename = "details")]
    #[serde(skip_serializing_if = "Option::is_none")]
    pub details: Option<HashMap<String, Option<serde_json::Value>>>,

    /// A string representing the identity of the actor.
    #[serde(rename = "id")]
    pub id: String,
}

/// The context object contains fields that give customers additional information about the
/// environment in and source from which the activity recorded occurred
#[derive(Serialize, Deserialize, Debug)]
pub struct Context {
    /// The IP address the activity/request being recorded was made from.
    #[serde(rename = "sourceIpAddress")]
    #[serde(skip_serializing_if = "Option::is_none")]
    pub source_ip_address: Option<String>,

    /// The agent through which the request was made, e.g. `Mozilla/5.0` or
    /// `python-requests/2.20.1`.
    #[serde(rename = "userAgent")]
    #[serde(skip_serializing_if = "Option::is_none")]
    pub user_agent: Option<String>,
}

/// A resource that was accessed, modified, or is related to the event being recorded.
#[derive(Serialize, Deserialize, Debug)]
pub struct Resource {
    /// Additional arbitrary metadata related to the resource.
    #[serde(rename = "details")]
    #[serde(skip_serializing_if = "Option::is_none")]
    pub details: Option<HashMap<String, Option<serde_json::Value>>>,

    /// A unique identifier for this resource.
    #[serde(rename = "id")]
    pub id: String,
}
