# Apptrail Application Events SDK for Rust

You can use the Apptrail Application Events SDK for Rust to send audit logs from your Rust applications to your customers.

## Learn more

- [Working with events](https://apptrail.com/docs/applications/guide/working-with-events/overview)
- [Full SDK Reference](https://apptrail.com/docs/applications/guide/working-with-events/using-the-events-sdk/application-events-sdk-rust)
- [Developer Guide](https://apptrail.com/docs/applications/guide)
- [Apptrail](https://apptrail.com)

## Notes and tips

- Requires Rust edition 2018.
- Instantiate the client once at the top of your application and reuse it to prevent unnecessary recreation.
- The client uses async and is tested with Tokio runtime, and should work with other async runtimes.

## Installing

Add the dependency to your `Cargo.toml`.

```bash
apptrail-application-events-sdk="*"
```

## Instantiating client

```rust
use apptrail_application_events_sdk::ApptrailEventsClient;

let my_api_key: String = loadMySecretApiKey();
let my_region = "us-west-2".to_string();

let apptrail_client = ApptrailEventsClient::new(my_region, my_api_key);
```

## Sending an event

```rust
use apptrail_application_events_sdk::event::{Actor, ApptrailEvent, Context, Resource};

let event = ApptrailEvent {
    tenant_id: "cust_MGY4MmYzNDMtZjEwOC00OWI".to_string(),
    event_name: "CreateFoo".to_string(),
    event_time: "2022-01-26T06:01:00Z".to_string(),
    actor: Some(Actor {
        id: "acct_MmRlODllZDctM2I0Yi0".to_string(),
        details: Some(HashMap::from([
            ("type".to_string(), Some(json!("account"))),
            ("name".to_string(), Some(json!("API Access"))),
        ])),
    }),
    resources: Some(vec![Resource {
        id: "repo_YWI5NjkzY2UtNzI1Ny00N".to_string(),
        details: Some(HashMap::from([(
            "repositoryType".to_string(),
            Some(json!("V2")),
        )])),
    }]),
    context: Some(Context {
        source_ip_address: Some("103.6.179.245".to_string()),
        user_agent: Some("Apache-HttpClient/4.5.3 (Java/11.0.11)".to_string()),
    }),
    event_details: Some(HashMap::from([(
        "request".to_string(),
        Some(json!({
          "repositoryName": "my-repository",
        })),
    )])),
    tags: Some(HashMap::from([("severity".to_string(), "LOW".to_string())])),
};

apptrail_client.put_event(&event).await;
```

## Handling errors

As a best practice, you should handle errors while sending events, especially if you are sending critical logs. The Events client includes automatic retries with backoff, but errors can happen due to rare server issues or client side issues.

You can choose what to do with failing events. For example, you may sideline them to disk, or a dead letter queue for retry or remediation.

```rust
let result = events_client.put_event(event).await;
match result {
  Ok(_) => (),
  Err(e) => {
    // handle error
    println!("{}", e);
  }
}
```
