pub mod base {

	pub trait Entity {
		//! Layer Supertype
		//! A type that acts as the supertype for all types in its
		//! layer.
	}

	pub trait Gateway {
		//! Gateway
		//! An object that encapsulates access to an external system or
		//! resource
		type Connection;

		fn new(connection: &Self::Connection) -> Self;
	}
}

pub mod domain_logic {

	pub trait TransactionScript {
		//! Transaction Script
		//! Organizes business logic by procedures where each procedure
		//! handles a single request from the presentation.
		fn execute(self: &Self) -> bool;
	}

	pub trait TableModule {
		//! A single instance that handles the business logic for
		//! all rows in a database table or view.
	}

	pub trait ServiceLayer {
		//! Defines an application’s boundary with a layer of services
		//! that establishes a set of available operations and
		//! coordinates the application’s response in each operation.
	}
}

pub mod object_relational {

	use super::base::Gateway;

	pub mod structural {

		pub trait IdentityField {
			//! Identity Field
			//! Saves a database ID field in an object to maintain identity
			//! between an in-memory object and a database row.
			type IdType;

			fn id(self: &Self) -> Self::IdType;
		}
	}

	pub mod behavioral {

		use super::{structural::IdentityField, Gateway};

		pub trait UnitOfWork: Gateway {
			//! Maintains a list of objects affected by a business
			//! transaction and coordinates the writing out of changes and
			//! the resolution of concurrency problems.
		}

		pub trait IdentityMap {
			//! Ensures that each object gets loaded only once by keeping
			//! every loaded object in a map. Looks up objects using
			//! the map when referring to them.
			type IdType;

			type Model: IdentityField;

			type Error;

			fn get(self: Self, id: &Self::IdType) -> Result<Self::Model, Self::Error>;

			fn set(
				self: Self,
				id: &Self::IdType,
				model: Self::Model,
			) -> Result<Self::Model, Self::Error>;
		}

		pub trait LazyLoad {
			//! An object that doesn’t contain all of the data you need
			//! but knows how to get it.
		}
	}
}

pub mod data_source {

	use super::base::Gateway;

	pub trait TableGateway: Gateway {
		//! Table Gateway
		//! An object that acts as a Gateway to a database table.
		//! One instance handles all the rows in the table.
		type Model;

		type Params;

		type Error;

		fn create_table(self: &Self) -> Result<(), Self::Error>;

		fn drop_table(self: &Self) -> Result<(), Self::Error>;

		// CRUD operations
		fn insert(self: &Self, params: &Self::Params) -> Result<i64, Self::Error>;

		fn find(self: &Self, params: &Self::Params) -> Option<Self::Model>;

		fn update(self: &Self, params: &Self::Params) -> Result<(), Self::Error>;

		fn delete(self: &Self, params: &Self::Params) -> Result<(), Self::Error>;
	}

	pub trait RowDataGateway: Gateway {
		//! An object that acts as a Gateway (466) to a single record in
		//! a data source. There is one instance per row.
		type Model;

		type Params;

		type Error;

		fn insert(params: &Self::Params) -> Result<i64, Self::Error>;

		fn find(params: &Self::Params) -> Option<Self::Model>;

		fn delete(params: &Self::Params) -> Result<(), Self::Error>;
	}

	pub trait ActiveRecord: Gateway {
		//! An object that wraps a row in a database table or view,
		//! encapsulates the database access, and adds domain logic on
		//! that data.
		type Model;

		type Params;

		type Error;

		fn insert<T: Sized>(self: &Self) -> Result<T, Self::Error>;

		fn update(self: &Self) -> Result<(), Self::Error>;

		fn delete(self: &Self) -> Result<(), Self::Error>;
	}

	pub trait DataMapper {
		//! A layer of Mappers that moves data between objects
		//! and a database while keeping them independent of
		//! each other and the mapper itself.
		type Model;

		type Params;

		type Error;

		fn insert(model: &Self::Model) -> Result<(), Self::Error>;

		fn update(model: &Self::Model) -> Result<(), Self::Error>;

		fn delete(model: &Self::Model) -> Result<(), Self::Error>;
	}
}
