//! # Apparat
//!
//! A lightweight, event-driven behavioral state machine
//!
//! Notable features:
//!
//! - No unsafe, unless you actively enable it (see "Feature flags" below)
//! - No-std compatible
//! - Small and fast to compile
//!   - Fewer than 250 lines of code
//!   - No dependencies
//!   - No procedural macros (just a declarative one)
//! - Very ergonomic despite the manageable amount of macro magic
//! - Highly flexible, suitable for many use cases
//! - No dynamic dispatch, enables lots of compiler optimizations
//!
//! **Note:** I am still experimenting with this a bit so while it's below version 1.0, there might be breaking API changes in point releases. If you want to be sure, specify an exact version number in your Cargo.toml. In point-point-releases there won't be breaking changes ever.
//!
//! ## Feature flags
//!
//! ### "unsafe" (disabled by default)
//!
//! This feature flag facilitates more compiler optimizations in some cases (verified using *cargo asm*). This is achieved by using `core::hint::unreachable_unchecked()` in one place, where the author is certain enough that it's sound. Nevertheless, use this at your own risk. If you're curious, have a look at the code - I've written a safety-comment about my assumptions there. If you think I missed something, please file an issue.
//!
//! ## Architecture and usage
//!
//! ### Types you provide
//!
//! - An arbitrary number of types that represent the different states*
//! - A context type that is mutably accessible while events are handled and during transitions.
//! - An event type for the state machine to handle
//! - An output type that is returned whenever an event is handled. If this is not needed, the unit type can be used.
//!
//! \* *The data within the state types is exclusively accessible in the respective state. It's getting dropped on transitions and moved when events are handled. These moves might get optimized out in some cases, but generally, if you want the best possible performance, the state types should be rather small and cheap to move.  
//! If it's impossible to keep them small and you are handling a lot of events without transitioning, consider putting the bigger data in a `Box` within the state type. Alternatively you could make that bigger data a part of the context type which won't get moved or dropped at all. But if you do the latter, the data will of course also be accessible from other states.*
//!
//! **Note:** All provided types must implement `Debug`.
//!
//! ### Entities that are generated
//!
//! - A wrapper enum with variants for all provided state types
//! - An implementation of the `ApparatWrapper` trait for the wrapper enum
//! - An implementation of the `ApparatState` trait for the wrapper enum. The enum delegates all method-calls of that trait to the inner state objects.
//! - Implementations of the `Wrap` trait for all provided states (for convenience)
//!
//! ### Traits
//!
//! #### `ApparatState<StateWrapper>`
//!
//! This trait must be implemented for all state types. The only required method is `handle`, which takes an event and returns a `Handled<StateWrapper>`. That's just the next state wrapped in a `StateWrapper` alongside an output value. To construct this return type, we can first call `.wrap()` on our state and then `.with_output(...)` on the wrapper. If our output type implements `Default`, we can also use `.default_output()` instead.
//!
//! There are two other methods in the `ApparatState` trait: `init` and `is_init`. These form an initialization mechanism: After constructing an `Apparat` and after handling any event, `init` is called on the current state, until `is_init` returns `true`. This way, a single event can potentially trigger multiple, context dependent transitions. This happens in a while loop without any recursion. If a state doesn't need that, these methods can just be ignored.
//!
//! #### `TransitionFrom<OtherState, ContextData>`
//!
//! The `TransitionFrom` trait can be used to define specific transitions between states. The `TransitionTo` trait is then automatically implemented, so we can call `.transition` using the turbofish syntax. This design is similar to `From` and `Into` in the standard library, but `TransitionFrom` and `TransitionInto` can also mutate the provided context as a side effect. `TransitionInto` is also recommended for trait bounds.
//!
//! #### `Wrap<StateWrapper>`
//!
//! The `Wrap<StateWrapper>` trait provides a `wrap` method to turn individual state objects into a `StateWrapper`. This is preferred over using `into` because it's more concise and enables type inference in more cases. `Wrap` is automatically implemented for all state types by the macro.
//!
//! ## Example
//!
//! For a slightly more complete example, have a look at *counter.rs* in the *examples* directory.
//!
//! ```rust
//! //! This state machine switches from `StateA` to `StateB` on a single event but
//! //! then needs three events to switch back to `StateA`. Additionally it keeps
//! //! track of how often it got toggled back from `StateB` to `StateA`.
//!
//! use apparat::prelude::*;
//!
//! // Define the necessary types
//! // --------------------------
//!
//! // States
//!
//! #[derive(Debug, Default)]
//! pub struct StateA;
//!
//! #[derive(Debug, Default)]
//! pub struct StateB {
//!     ignored_events: usize,
//! }
//!
//! // Context
//!
//! // Data that survives state transitions and can be accessed in all states
//! #[derive(Debug, Default)]
//! pub struct ContextData {
//!     toggled: usize,
//! }
//!
//! // Auto-generate the state wrapper and auto-implement traits
//! // ---------------------------------------------------------
//!
//! // In this example we are just using the unit type for `event` and `output`
//! // because we are only handling one kind of event and we don't care about values
//! // being returned when events are handled.
//! build_wrapper! {
//!     states: [StateA, StateB],
//!     wrapper: MyStateWrapper, // This is just an identifier we can pick
//!     context: ContextData,
//!     event: (),
//!     output: (),
//! }
//!
//! // Define transitions
//! // ------------------
//!
//! impl TransitionFrom<StateB> for StateA {
//!     fn transition_from(_prev: StateB, ctx: &mut ContextData) -> Self {
//!         // Increase toggled value
//!         ctx.toggled += 1;
//!         println!("B -> A          | toggled: {}", ctx.toggled);
//!         StateA::default()
//!     }
//! }
//!
//! impl TransitionFrom<StateA> for StateB {
//!     fn transition_from(_prev: StateA, ctx: &mut ContextData) -> Self {
//!         println!("A -> B          | toggled: {}", ctx.toggled);
//!         StateB::default()
//!     }
//! }
//!
//! // Implement the `ApparatState` trait for all states
//! // -------------------------------------------------
//!
//! impl ApparatState for StateA {
//!     type Wrapper = MyStateWrapper;
//!
//!     fn handle(self, _event: (), ctx: &mut ContextData) -> Handled<MyStateWrapper> {
//!         println!("A handles event | toggled: {}", ctx.toggled);
//!         // Transition to `StateB`
//!         let state_b = self.transition::<StateB>(ctx);
//!         // Now we need to wrap that `state_b` in a `MyStateWrapper`...
//!         let state_b_wrapped = state_b.wrap();
//!         // ... and add an output value to turn it into a `Handled<...>`.
//!         state_b_wrapped.default_output()
//!         // If we would need a different output value or our output type wouldn't
//!         // implement `Default` we would have to use `.with_output(...)` instead.
//!     }
//! }
//!
//! impl ApparatState for StateB {
//!     type Wrapper = MyStateWrapper;
//!
//!     fn handle(mut self, _event: (), ctx: &mut ContextData) -> Handled<MyStateWrapper> {
//!         println!("B handles event | toggled: {}", ctx.toggled);
//!         if self.ignored_events == 2 {
//!             self.transition::<StateA>(ctx).wrap().default_output()
//!         } else {
//!             self.ignored_events += 1;
//!             self.wrap().default_output()
//!         }
//!     }
//! }
//!
//! // Run the machine
//! // ---------------
//!
//! fn main() {
//!     let mut apparat = Apparat::new(StateA::default().wrap(), ContextData::default());
//!
//!     // Handle some events
//!     for _ in 0..10 {
//!         apparat.handle(());
//!     }
//! }
//! ```

#![no_std]
#![warn(clippy::pedantic)]
#![allow(clippy::inline_always, clippy::needless_doctest_main)]

pub mod prelude {
    pub use crate::{
        build_wrapper, Apparat, ApparatState, ApparatTrait, ApparatWrapper,
        ApparatWrapperDefaultOutput, Handled, TransitionFrom, TransitionTo, Wrap,
    };
}

/// This trait is used to associate all the types used together in an `Apparat`
/// with the state wrapper enum, so users of the library dont' need to specify
/// all these types every time they implement the `ApparatState` trait for one
/// of their state types.
pub trait ApparatWrapper: Sized {
    type Context;
    type Event;
    type Output;

    fn with_output(self, output: Self::Output) -> Handled<Self> {
        Handled::new(self, output)
    }
}

pub trait ApparatWrapperDefaultOutput: ApparatWrapper {
    fn default_output(self) -> Handled<Self>;
}

impl<Wrapper: ApparatWrapper> ApparatWrapperDefaultOutput for Wrapper
where
    Wrapper::Output: Default,
{
    #[inline]
    fn default_output(self) -> Handled<Self> {
        self.into()
    }
}

pub trait ApparatTrait<Wrapper>
where
    Wrapper: ApparatWrapper + ApparatState<Wrapper = Wrapper>,
{
    fn new(state: Wrapper, ctx: Wrapper::Context) -> Self;
    fn handle(&mut self, event: Wrapper::Event) -> Wrapper::Output;
}

/// The actual state machine that handles your events and manages their
/// initialization and transitions
#[derive(Debug)]
pub struct Apparat<Wrapper: ApparatWrapper> {
    state: Option<Wrapper>,
    ctx: Wrapper::Context,
}

impl<Wrapper> ApparatTrait<Wrapper> for Apparat<Wrapper>
where
    Wrapper: ApparatWrapper + ApparatState<Wrapper = Wrapper>,
{
    fn new(mut state: Wrapper, mut ctx: Wrapper::Context) -> Self {
        while !state.is_init(&ctx) {
            state = state.init(&mut ctx);
        }
        Self {
            state: Some(state),
            ctx,
        }
    }

    #[inline]
    fn handle(&mut self, event: Wrapper::Event) -> Wrapper::Output {
        #[cfg(feature = "unsafe")]
        // Safety: `self.state` can't be accessed directly from outside of this
        // module. The only methods accessing it are this one and `new`. Both
        // ultimately leave a `Some` value in `self.state`. Panicking while it's
        // `None` would only lead to undefined behavior if there would be an
        // unsafe `Drop` implementation for `Apparat` that assumes that it's
        // `Some`, which isn't the case. Since this method takes `&mut self`
        // there also can't be an overlapping call from a different thread
        // thanks to the almighty borrow checker <3.
        let state = self
            .state
            .take()
            .unwrap_or_else(|| unsafe { core::hint::unreachable_unchecked() });

        #[cfg(not(feature = "unsafe"))]
        // The performance impact is negligible in most cases, but using this
        // "safe" version does sometimes lead to slightly less optimized code.
        let state = self.state.take().expect("https://xkcd.com/2200/");

        let mut handled = state.handle(event, &mut self.ctx);
        while !handled.state.is_init(&self.ctx) {
            handled.state = handled.state.init(&mut self.ctx);
        }
        self.state = Some(handled.state);
        handled.result
    }
}

/// A trait that must be implemented by all provided state types. Have a look at
/// the readme or the examples for details.
pub trait ApparatState: Sized {
    type Wrapper: ApparatWrapper + From<Self>;

    #[inline(always)]
    fn init(
        self,
        _ctx: &mut <<Self as ApparatState>::Wrapper as ApparatWrapper>::Context,
    ) -> Self::Wrapper {
        Self::Wrapper::from(self)
    }

    fn handle(
        self,
        _event: <<Self as ApparatState>::Wrapper as ApparatWrapper>::Event,
        _ctx: &mut <<Self as ApparatState>::Wrapper as ApparatWrapper>::Context,
    ) -> Handled<<Self as ApparatState>::Wrapper>;

    #[inline(always)]
    fn is_init(&self, _ctx: &<<Self as ApparatState>::Wrapper as ApparatWrapper>::Context) -> bool {
        true
    }
}

/// This type is being returned whenever an event is handled by a state type. It
/// contains the new state alongside an output value that will be returned  to
/// the caller of the `handle` method.
#[derive(Debug)]
pub struct Handled<Wrapper: ApparatWrapper> {
    pub state: Wrapper,
    pub result: Wrapper::Output,
}

impl<Wrapper: ApparatWrapper> Handled<Wrapper> {
    #[inline]
    pub fn new(state: Wrapper, result: Wrapper::Output) -> Self {
        Self { state, result }
    }
}

impl<Wrapper> Handled<Wrapper>
where
    Wrapper: ApparatWrapper,
    Wrapper::Output: Default,
{
    #[inline]
    pub fn new_default(state: Wrapper) -> Self {
        Self {
            state,
            result: Wrapper::Output::default(),
        }
    }
}

impl<Wrapper> From<Wrapper> for Handled<Wrapper>
where
    Wrapper: ApparatWrapper,
    Wrapper::Output: Default,
{
    #[inline(always)]
    fn from(wrapper: Wrapper) -> Self {
        Self::new_default(wrapper)
    }
}

/// Define transitions between states. These transition functions can access the
/// shared context/data mutably.
pub trait TransitionFrom<FromState: ApparatState> {
    fn transition_from(
        prev: FromState,
        _ctx: &mut <<FromState as ApparatState>::Wrapper as ApparatWrapper>::Context,
    ) -> Self;
}

/// Similar to `Into` in std, this is mostly for convenience and should not get
/// implemented manually. Implement `TransitionFrom` instead and use
/// `TransitionInto` in trait bounds, when needed.
pub trait TransitionTo<Context>: Sized + ApparatState {
    #[inline(always)]
    fn transition<Next>(
        self,
        ctx: &mut <<Self as ApparatState>::Wrapper as ApparatWrapper>::Context,
    ) -> Next
    where
        Next: ApparatState<Wrapper = Self::Wrapper> + TransitionFrom<Self>,
    {
        Next::transition_from(self, ctx)
    }
}

// Blanket implementation so state types can transition into themselves
impl<T: ApparatState> TransitionTo<T> for T {}

/// An alternative to `std::Into` for turning state types into the respective
/// state wrapper enum. This is preferred over `Into` because it provides more
/// reliable type inference in the context of apparat.
pub trait Wrap<Wrapper> {
    fn wrap(self) -> Wrapper;
}

/// Generate an enum that wraps all provided state types. Additionally all
/// necessary traits are implemented for it, so the wrapper can be used within
/// an `Apparat` state machine.
#[macro_export]
macro_rules! build_wrapper {
    (
        states: [$state1:ident, $($state:ident),* $(,)*],
        wrapper: $wrapper:ident,
        context: $context:ty,
        event: $event:ty,
        output: $output:ty
        $(,)*
    ) => {
        #[derive(Debug)]
        pub enum $wrapper {
            $state1($state1),
            $(
                $state($state)
            ),*
        }

        impl ::apparat::ApparatWrapper for $wrapper {
            type Context = $context;
            type Event = $event;
            type Output = $output;
        }

        impl ::apparat::ApparatState for $wrapper {
            type Wrapper = Self;

            #[inline]
            fn init(self, ctx: &mut $context) -> Self {
                match self {
                    $wrapper::$state1(state) => state.init(ctx),
                    $(
                        $wrapper::$state(state) => state.init(ctx)
                    ),*
                }
            }

            #[inline]
            fn handle(self, event: $event, ctx: &mut $context) -> Handled<$wrapper> {
                match self {
                    $wrapper::$state1(state) => state.handle(event, ctx),
                    $(
                        $wrapper::$state(state) => state.handle(event, ctx)
                    ),*
                }
            }

            #[inline]
            fn is_init(&self, ctx: &$context) -> bool {
                match self {
                    $wrapper::$state1(state) => state.is_init(ctx),
                    $(
                        $wrapper::$state(state) => state.is_init(ctx)
                    ),*
                }
            }
        }

        impl ::std::convert::From<$state1> for $wrapper {
            #[inline(always)]
            fn from(state: $state1) -> $wrapper {
                $wrapper::$state1(state)
            }
        }

        // These manual implementations are needed (instead of a blanket impl) so
        // the wrapper type can get inferred when wrap is called on a state
        // object.
        impl ::apparat::Wrap<$wrapper> for $state1 {
            #[inline(always)]
            fn wrap(self) -> $wrapper {
                $wrapper::$state1(self)
            }
        }

        $(
            impl ::std::convert::From<$state> for $wrapper {
                #[inline(always)]
                fn from(state: $state) -> $wrapper {
                    $wrapper::$state(state)
                }
            }

            impl ::apparat::Wrap<$wrapper> for $state {
                #[inline(always)]
                fn wrap(self) -> $wrapper {
                    $wrapper::$state(self)
                }
            }
        )*
    };
}
