use std::fs;

/// Type of token
#[derive(PartialEq)]
pub enum DocTokenType {
    Title,
    Author,
    Date,
    Comment,
    Newline,

    NHeader,
    Header,

    Bold,
    Italic,
    BoldItalic,
    Text,
}

/// Document token
/// Generated by `preprocess_file`
pub struct DocToken {
    /// type of token
    pub token_type: DocTokenType,

    /// content of the token (if exists)
    pub content: String,
}

pub struct DocTokenList(pub Vec<DocToken>);

impl DocTokenList {
    /// Pre-processes a file given by `path`
    /// Returns `DocTokenList`
    pub fn from_file(path: String) -> DocTokenList {
        // file contents
        let contents = fs::read_to_string(path).expect("Could not open file!!!");
        let lines: Vec<&str> = contents.lines().collect();

        DocTokenList::from_lines(lines)
    }

    pub fn from_lines(lines: Vec<&str>) -> DocTokenList {
        let mut rvec = Vec::<DocToken>::new();
        for line in lines {
            let l = String::from(line);
            let cmd = l.split(';').next().unwrap();
            let mut cont: String;
            if let Some(v) = l.split(';').nth(1) {
                cont = String::from(v);
            } else {
                cont = String::new();
            }

            // switch based off the command
            let token_type = match cmd {
                "title" => DocTokenType::Title,

                "author" => DocTokenType::Author,

                "date" => DocTokenType::Date,

                // comment
                "#" => DocTokenType::Comment,

                "newline" | "nl" => DocTokenType::Newline,

                "nheader" | "nhead" => DocTokenType::NHeader,

                "header" | "head" => DocTokenType::Header,

                "bold" | "bl" => DocTokenType::Bold,

                "italic" | "il" => DocTokenType::Italic,

                "bold-italic" | "bi" => DocTokenType::BoldItalic,

                // everything else is assumed to be text
                _ => {
                    // empty line
                    if l.is_empty() {
                        DocTokenType::Newline
                    } else {
                        cont = l;
                        DocTokenType::Text
                    }
                }
            };
            rvec.push(DocToken {
                token_type,
                content: cont,
            });
        }
        DocTokenList(rvec)
    }
}
