use crate::processor::{DocTokenList, DocTokenType};

// generator type enum for options
#[derive(Clone, clap::ArgEnum, Debug)]
pub enum GeneratorType {
    ANSI,
    HTML,
}

/// Generates text from tokens using ansi escape sequences
/// All output should be viewable in anything that interprets
/// ansi escapes, eg terminal emulators
pub fn generate_ansi(tokens: DocTokenList) -> String {
    // return string
    let mut rstr = String::new();

    // header number for numbered headers
    let mut header_number = 0;

    // iterates over each token
    for token in tokens.0.iter() {
        // content to be used in tags
        let content = token.content.trim();

        // matches token type to corresponding format
        let tty_tok = match token.token_type {
            DocTokenType::Title => format!("\t\t\x1b[1m{} \x1b[0m\n\n", content),

            DocTokenType::Author => format!("\t\t\x1b[3m{} \x1b[0m\n", content),

            DocTokenType::Date => format!("\t\t\x1b[0m{} \n", content),

            DocTokenType::Text => format!("{} ", content),

            DocTokenType::Newline => String::from("\n"),

            DocTokenType::NHeader => {
                header_number += 1;
                format!("\t\x1b[1m{}. {}\x1b[0m\n", header_number, content)
            }

            DocTokenType::Header => format!("\t\x1b[1m{}\x1b[0m\n", content),

            DocTokenType::Bold => format!("\x1b[1m{}\x1b[0m ", content),

            DocTokenType::Italic => format!("\x1b[3m{}\x1b[0m ", content,),

            DocTokenType::BoldItalic => format!("\x1b[1;3m{}\x1b[0m", content),

            _ => String::from(""),
        };

        // Pushes token to return string
        rstr.push_str(tty_tok.as_str());
    }

    // pushes a final newline to return string
    rstr.push('\n');

    // returns string
    rstr
}

/// Generates html from DocTokenList
/// All html should be viewable in anything that can view html files
pub fn generate_html(tokens: DocTokenList) -> String {
    //NOTE: much of the code is shared with "generate_ansi"
    // header text for inside of <head> tag
    let mut header_text = String::new();

    // body text for inside of <body> tag
    let mut body_text = String::new();

    // header number for numbered headers
    let mut header_number = 0;

    // variable for if a new paragraph is being started
    let mut is_new_paragraph = true;

    for token in tokens.0.iter() {
        let content = token.content.trim();

        // adds title to head and body
        if token.token_type == DocTokenType::Title {
            header_text.push_str(format!("<title>{}</title>", content).as_str());

            body_text
                .push_str(format!("<h1 style=\"text-align:center;\">{}</h1>", content).as_str());
        } else {
            let content = token.content.trim();
            // matches token type to corresponding format
            let html_content = match token.token_type {
                DocTokenType::Author => {
                    format!("<h2 style=\"text-align:center;\"><i>{}</i></h2>", content)
                }

                DocTokenType::Date => {
                    format!("<h3 style=\"text-align:center;\">{}</h3><br/>", content)
                }

                DocTokenType::Text => {
                    if is_new_paragraph {
                        is_new_paragraph = false;
                        format!("</p><p>{} ", content)
                    } else {
                        format!("{} ", content)
                    }
                }

                DocTokenType::Newline => {
                    is_new_paragraph = true;
                    String::from("<br/>")
                }

                DocTokenType::NHeader => {
                    header_number += 1;
                    is_new_paragraph = true;
                    format!("<h2>{}. {}</h2>", header_number, content)
                }

                DocTokenType::Header => format!("<h2>{}</h2>", content),

                DocTokenType::Bold => format!("<b>{}</b> ", content),

                DocTokenType::Italic => format!("<i>{}</i> ", content,),

                DocTokenType::BoldItalic => format!("<b><i>{}</i></b> ", content),

                _ => String::from(""),
            };

            body_text.push_str(html_content.as_str());
        }
    }

    // NOTE/FIXME: creates malformed html in the body due to the <p> tags
    format!(
        "\
<!DOCTYPE html>\
<html>\
<head>
{}
</head>
<body>
<p>
{}
</p>
</body>
</html>",
        header_text, body_text
    )
}
