use std::fmt;

use crate::TokenKind;

/// A token generated by the lexer.
#[derive(Clone)]
pub struct Token {
    pub(crate) kind: TokenKind,
    pub(crate) data: String,
    pub(crate) index: usize,
}

impl Token {
    pub(crate) fn new(kind: TokenKind, data: String) -> Self {
        Self {
            kind,
            data,
            index: 0,
        }
    }

    /// Get a reference to the token's kind.
    pub fn kind(&self) -> TokenKind {
        self.kind
    }

    /// Get a reference to the token's data.
    pub fn data(&self) -> &str {
        self.data.as_str()
    }

    /// Get a reference to the token's loc.
    pub fn index(&self) -> usize {
        self.index
    }
}

impl fmt::Debug for Token {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        let start = self.index;
        let end = self.index + self.data.len();

        match &self.kind {
            TokenKind::Whitespace => {
                write!(f, "WHITESPACE@{}:{} {:?}", start, end, self.data)
            }
            TokenKind::Comment => {
                write!(f, "COMMENT@{}:{} {:?}", start, end, self.data)
            }
            TokenKind::Bang => {
                write!(f, "BANG@{}:{} {:?}", start, end, self.data)
            }
            TokenKind::Dollar => {
                write!(f, "DOLLAR@{}:{} {:?}", start, end, self.data)
            }
            TokenKind::Amp => {
                write!(f, "AMP@{}:{} {:?}", start, end, self.data)
            }
            TokenKind::Spread => {
                write!(f, "SPREAD@{}:{} {:?}", start, end, self.data)
            }
            TokenKind::Colon => {
                write!(f, "COLON@{}:{} {:?}", start, end, self.data)
            }
            TokenKind::Comma => {
                write!(f, "COMMA@{}:{} {:?}", start, end, self.data)
            }
            TokenKind::Eq => {
                write!(f, "EQ@{}:{} {:?}", start, end, self.data)
            }
            TokenKind::At => {
                write!(f, "AT@{}:{} {:?}", start, end, self.data)
            }
            TokenKind::LParen => {
                write!(f, "L_PAREN@{}:{} {:?}", start, end, self.data)
            }
            TokenKind::RParen => {
                write!(f, "R_PAREN@{}:{} {:?}", start, end, self.data)
            }
            TokenKind::LBracket => {
                write!(f, "L_BRACKET@{}:{} {:?}", start, end, self.data)
            }
            TokenKind::RBracket => {
                write!(f, "R_BRACKET@{}:{} {:?}", start, end, self.data)
            }
            TokenKind::LCurly => {
                write!(f, "L_CURLY@{}:{} {:?}", start, end, self.data)
            }
            TokenKind::RCurly => {
                write!(f, "R_CURLY@{}:{} {:?}", start, end, self.data)
            }
            TokenKind::Pipe => {
                write!(f, "PIPE@{}:{} {:?}", start, end, self.data)
            }
            TokenKind::Eof => {
                write!(f, "EOF@{}:{}", start, start)
            }

            // composite nodes
            TokenKind::Name => {
                write!(f, "NAME@{}:{} {:?}", start, end, self.data)
            }
            TokenKind::StringValue => {
                write!(f, "STRING_VALUE@{}:{} {:?}", start, end, self.data)
            }
            TokenKind::Int => {
                write!(f, "INT@{}:{} {:?}", start, end, self.data)
            }
            TokenKind::Float => {
                write!(f, "FLOAT@{}:{} {:?}", start, end, self.data)
            }
        }
    }
}
