use crate::photon_decode;

use crate::photon_decode::Message;
use crate::photon_decode::EventData;
use crate::photon_decode::OperationRequest;
use crate::photon_decode::OperationResponse;
use crate::photon_decode::Value;
use crate::photon_decode::Parameters;

#[test]
fn test_single_event_message()
{
    let header = vec![
		0x00, 0x01, 			// PeerID
		0x01,                   // CrcEnabled
		0x01,                   // CommandCount
		0x00, 0x00, 0x00, 0x01, // Timestamp
		0x00, 0x00, 0x00, 0x01, // Challenge
    ];

	let photon_command = vec![
		0x06,                   // Type
		0x01,                   // ChannelID
		0x01,                   // Flags
		0x04,                   // ReservedByte
		0x00, 0x00, 0x00, 0x1a, // Length
		0x00, 0x00, 0x00, 0x01, // ReliableSequenceNumber
    ];

	let message = vec![
		0x01,							// ??
		0x04,							// MsgType (Event)
										// Payload
		0x64, 0x00, 0x02, 				// Key
		0x00, 0x73, 					// String type
		0x00, 0x05,						// String length
		0x74, 0x65, 0x73, 0x74, 0x31	//test1
	];

	let mut photon = photon_decode::Photon::new();

	let mut decoded = photon.try_decode(
		&[&header[..], &photon_command[..], &message[..]].concat()).expect("Shold works").into_iter();
	let msg = decoded.next().expect("Shold contain one msg").expect("Should decode event");

	let parameters: Parameters =
		[(0, Value::String("test1".to_owned()))]
    	.iter().cloned().collect();
	assert_eq!(msg, Message::Event(EventData{ code: 100,  parameters}));
}

#[test]
fn test_fragmented_event_message()
{
    let header = vec![
		0x00, 0x01, 			// PeerID
		0x01,                   // CrcEnabled
		0x01,                   // CommandCount
		0x00, 0x00, 0x00, 0x01, // Timestamp
		0x00, 0x00, 0x00, 0x01, // Challenge
    ];

    let mut photon = photon_decode::Photon::new();

	let photon_command = vec![
		0x08,                   // Type
		0x01,                   // ChannelID
		0x01,                   // Flags
		0x04,                   // ReservedByte
		0x00, 0x00, 0x00, 0x22, // Length
		0x00, 0x00, 0x00, 0x01, // ReliableSequenceNumber
		0x00, 0x00, 0x00, 0x01, // SequenceNumber
		0x00, 0x00, 0x00, 0x03, // FragmentCount
		0x00, 0x00, 0x00, 0x01, // FragmentNumber
		0x00, 0x00, 0x00, 0x1a, // TotalLength
		0x00, 0x00, 0x00, 0x01, // OperationLength
    ];

	let message = vec![
		0x01,							// ??
		0x04,							// MsgType (Event)
										// Payload
	];

    let decoded = photon.decode(
		&[&header[..], &photon_command[..], &message[..]].concat());
	assert!(decoded.len() == 0);

	let photon_command = vec![
		0x08,                   // Type
		0x01,                   // ChannelID
		0x01,                   // Flags
		0x04,                   // ReservedByte
		0x00, 0x00, 0x00, 0x27, // Length
		0x00, 0x00, 0x00, 0x01, // ReliableSequenceNumber
		0x00, 0x00, 0x00, 0x01, // SequenceNumber
		0x00, 0x00, 0x00, 0x03, // FragmentCount
		0x00, 0x00, 0x00, 0x02, // FragmentNumber
		0x00, 0x00, 0x00, 0x1a, // TotalLength
		0x00, 0x00, 0x00, 0x01, // OperationLength
    ];

	let message = vec![
										// Payload
		0x64, 0x00, 0x02, 				// Key
		0x00, 0x73, 					// String type
		0x00, 0x05,						// String length
	];

    let decoded = photon.decode(
		&[&header[..], &photon_command[..], &message[..]].concat());
	assert!(decoded.len() == 0);

	let photon_command = vec![
		0x08,                   // Type
		0x01,                   // ChannelID
		0x01,                   // Flags
		0x04,                   // ReservedByte
		0x00, 0x00, 0x00, 0x25, // Length
		0x00, 0x00, 0x00, 0x01, // ReliableSequenceNumber
		0x00, 0x00, 0x00, 0x01, // SequenceNumber
		0x00, 0x00, 0x00, 0x03, // FragmentCount
		0x00, 0x00, 0x00, 0x03, // FragmentNumber
		0x00, 0x00, 0x00, 0x1a, // TotalLength
		0x00, 0x00, 0x00, 0x01, // OperationLength
    ];

	let message = vec![
		0x74, 0x65, 0x73, 0x74, 0x31	//test1
	];

    let decoded = photon.decode(
		&[&header[..], &photon_command[..], &message[..]].concat());
	assert_eq!(decoded.len(), 1);

	let parameters: Parameters =
		[(0, Value::String("test1".to_owned()))]
    	.iter().cloned().collect();
	assert_eq!(decoded[0], Message::Event(EventData{ code: 100,  parameters}));
}


#[test]
fn test_two_event_messages()
{
	let payload = vec![
		0x00, 0x00, 
		0x00, 
		0x02, 
		0x72, 0xe9, 0xe2, 0x84, 
		0x11, 0x63, 0x88, 0x88,

		0x07, 
		0x00, 
		0x00, 
		0x00, 
		0x00, 0x00, 0x00, 0x45, 
		0x00, 0x00, 0x16, 0x8e, 
		0x00, 0x00, 0x45, 0x2b, 
		0xf3, 0x04, 0x02, 0x00, 0x02, 0x00, 0x6c, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa, 0xb7, 0x36, 0x01, 0x78, 0x00, 0x00, 0x00, 0x20, 0x03, 0x6b, 0xc9, 0x04, 0xc1, 0xe2, 0x62, 0xd7, 0x08, 0x6b, 0xb0, 0x91, 0xc1, 0x2a, 0x80, 0x1b, 0xc2, 0x26, 0x00, 0x00, 0xb0, 0x40, 0xd8, 0xe6, 0x53, 0xc1, 0x8f, 0xcd, 0xc, 0xc2, 0x00, 0x00,

		0x06, 
		0x00, 
		0x01, 
		0x00, 
		0x00, 0x00, 0x00, 0x1b, 
		0x00, 0x00, 0x16, 0x8f, 
		0xf3, 0x04, 0x01, 0x00, 0x02, 0x00, 0x69, 0x00, 0xa, 0xf6, 0xdf, 0xfc, 0x6b, 0x00, 0x01
	];

    let mut photon = photon_decode::Photon::new();

    let decoded = photon.decode(&payload);

	assert!(decoded.len() == 2);
	let parameters: Parameters =
		[
			(0x01, Value::ByteArray(vec![
				3, 107, 201, 4, 193, 226, 98, 215, 8, 107, 176, 145, 193, 
				42, 128, 27, 194, 38, 0, 0, 176, 64, 216, 230, 83, 193, 
				143, 205, 12, 194, 0, 0])),	
			(0x00, Value::Long(702262))
		]
    	.iter().cloned().collect();
	assert_eq!(decoded[0], Message::Event(EventData{ code: 2,  parameters}));

	let parameters: Parameters =
		[
			(0x00, Value::Integer(718559)),
			(0xfc, Value::Short(1))
		]
    	.iter().cloned().collect();
	assert_eq!(decoded[1], Message::Event(EventData{ code: 1,  parameters}));

}

#[test]
fn test_request_message()
{
	let payload = vec![
		0x7d, 0xd3, 
		0x00, 
		0x01, 
		0x00, 0x04, 0xcc, 0xb6, 
		0x11, 0x63, 0x88, 0x88, 
		
		0x06, 
		0x00, 
		0x01, 
		0x04, 
		0x00, 0x00, 0x00, 0x2a, 
		0x00, 0x00, 0x00, 0x19, 
		0xf3, 0x02, 0x01, 0x00, 0x03, 0x00, 
		0x79, 0x00, 0x02, 0x66, 0xc1, 0xd3, 0x61, 0x20, 0xc2, 0x27, 0xc, 0xcd, 0x01, 0x62, 
		0x00, 0xfd, 					// key
		0x73, 							// String type
		0x00, 0x05,						// String length
		0x74, 0x65, 0x73, 0x74, 0x31	// test1
	];

    let mut photon = photon_decode::Photon::new();
    let decoded = photon.decode(&payload);

	assert!(decoded.len() == 1);
	let parameters: Parameters =
		[
			(0x00, Value::Array(vec![Value::Float(-26.422424), Value::Float(-41.7625)])),
			(0x01, Value::Byte(0)),
			(0xfd, Value::String("test1".to_owned()))
		
		]
    	.iter().cloned().collect();
	assert_eq!(decoded[0],  Message::Request(OperationRequest { code: 1, parameters}));
}

#[test]
fn test_response_message()
{
	let payload = vec![
		0x00, 0x00, 
		0x00, 
		0x02, 
		0x72, 0xea, 0x1d, 0x6a, 
		0x11, 0x63, 0x88, 0x88, 
		
		0x01, 
		0x00, 
		0x00, 
		0x00, 
		0x00, 0x00, 0x00, 0x14, 
		0x00, 0x00, 0x00, 0x00, 
		0x00, 0x00, 0x00, 0x24, 0x00, 0x04, 0xd7, 0x1a, 
		
		0x06, 
		0x00, 
		0x01, 
		0x00, 
		0x00, 0x00, 0x00, 0xb2, 
		0x00, 0x00, 0x17, 0x71, 
		0xf3, 0x03, 0x01, 0x00, 0x00, 0x2a, 0x00, 0xa, 0x00, 0x69, 0x18, 0xf2, 0x79, 0x68, 0x01, 0x73, 0x00, 0x5d, 0x40, 0x49, 0x53, 0x4c, 0x41, 0x4e, 0x44, 0x40, 0x65, 0x66, 0x33, 0x37, 0x39, 0x34, 0x64, 0x61, 0x2d, 0x38, 0x66, 0x38, 0x33, 0x2d, 0x34, 0x33, 0x34, 0x30, 0x2d, 0x62, 0x66, 0x35, 0x32, 0x2d, 0x36, 0x38, 0x31, 0x31, 0x37, 0x61, 0x30, 0x36, 0x37, 0x32, 0x65, 0x34, 0x7c, 0x57, 0x65, 0x64, 0x72, 0x6f, 0x77, 0x79, 0x63, 0x7a, 0x4a, 0x61, 0x6b, 0x75, 0x62, 0x7c, 0x36, 0x33, 0x37, 0x30, 0x37, 0x35, 0x33, 0x37, 0x38, 0x38, 0x30, 0x33, 0x31, 0x38, 0x33, 0x39, 0x33, 0x35, 0x7c, 0x57, 0x65, 0x64, 0x72, 0x6f, 0x77, 0x79, 0x63, 0x7a, 0x4a, 0x61, 0x6b, 0x75, 0x62, 0x02, 0x79, 0x00, 0x00, 0x78, 0x03, 0x79, 0x00, 0x00, 0x73, 0x04, 0x79, 0x00, 0x00, 0x73, 0x05, 0x78, 0x00, 0x00, 0x00, 0x00, 0x06, 0x79, 0x00, 0x01, 0x79, 0x00, 0x01, 0x69, 0xff, 0xff, 0xff, 0xff, 	0x07, 0x79, 0x00, 0x01, 0x79, 0x00, 0x01, 0x69, 0xff, 0xff, 0xff, 0xff, 0x08, 0x78, 0x00, 0x00, 0x00, 0x00, 0xfd, 0x6b, 0x00, 0xa6
	];

    let mut photon = photon_decode::Photon::new();

    let decoded = photon.decode(&payload);

	assert_eq!(decoded.len(), 1);
	let parameters: Parameters =
		[
			(0x00, Value::Integer(418543976)),
			(0x01, Value::String("@ISLAND@ef3794da-8f83-4340-bf52-68117a0672e4|WedrowyczJakub|637075378803183935|WedrowyczJakub".to_owned())),
			(0x02, Value::Array(vec![])),
			(0x03, Value::Array(vec![])),
			(0x04, Value::Array(vec![])),
			(0x05, Value::ByteArray(vec![])),
			(0x06, Value::Array(vec![Value::Array(vec![Value::Integer(4294967295)])])),
			(0x07, Value::Array(vec![Value::Array(vec![Value::Integer(4294967295)])])),
			(0x08, Value::ByteArray(vec![])),
			(0xFD, Value::Short(166)),
		]
    	.iter().cloned().collect();
	assert_eq!(decoded[0],  Message::Response(OperationResponse { code: 1, return_code: 0, debug_message: "None".to_owned(), parameters}));
}

#[test]
fn test_example(){
	use photon_decode::{Photon, Message};

	let mut photon = Photon::new();
    let photon_packet = vec![
		0x00, 0x01, 			// PeerID
		0x01,                   // CrcEnabled
		0x00,                   // CommandCount
		0x00, 0x00, 0x00, 0x01, // Timestamp
		0x00, 0x00, 0x00, 0x01, // Challenge
    ];

	for message in photon.decode(&photon_packet).iter() {
		match message {
			Message::Event(_) => {
				// use event
			},
			Message::Request(_) => {
				// use request
			},
			Message::Response(_) => {
				// use response
			}
		}
	}
}