#![cfg(target_arch = "wasm32")]

use js_sys::Date;
use std::fmt;

pub struct ElapsedMilliseconds {
    initial_timestamp: f64,
}

impl crate::ElapsedMilliseconds for ElapsedMilliseconds {
    fn start() -> Self {
        Self {
            initial_timestamp: Date::now(),
        }
    }
    fn elapsed(&self) -> u128 {
        let elapsed_millis = (self.initial_timestamp - Date::now()).trunc();
        if elapsed_millis < 0.0 {
            return 0;
        }
        return elapsed_millis as u128;
    }
}

impl fmt::Debug for super::DebuggerError {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        #[cfg(not(debug_assertions))]
        return write!(f, "{}", obfstr::obfstr!("Something went wrong"));
        #[cfg(debug_assertions)]
        match self {
            super::DebuggerError::InstructionNotAvailable => {
                write!(f, "Can not use JS eval to execute debugger statement")
            }
            super::DebuggerError::TamperedEnvironment => {
                write!(f, "JS eval behavior is different than expected, maybe it was changed")
            }
        }
    }
}

pub(crate) fn debugger_instruction() -> Result<(), super::DebuggerError> {
    match js_sys::eval(obfstr::obfstr!("debugger;")) {
        Ok(result) => {
            if result.is_undefined() {
                Ok(())
            } else {
                // If the result is not undefined maybe someone changed eval function.
                Err(super::DebuggerError::TamperedEnvironment)
            }
        }
        // CSP of a website can forbid eval.
        Err(_) => Err(super::DebuggerError::InstructionNotAvailable),
    }
}

pub fn is_debugger_present() -> bool {
    return false;
}
