//! # `android-logd-logger`
//!
//! [![Crates.io][crates-badge]][crates-url]
//! [![Build Status][actions-badge]][actions-url]
//! [![Docs][docs-badge]][docs-url]
//!
//! [docs-badge]: https://docs.rs/android-logd-logger/badge.svg
//! [docs-url]: https://docs.rs/android-logd-logger
//! [crates-badge]: https://img.shields.io/crates/v/android-logd-logger.svg
//! [crates-url]: https://crates.io/crates/android-logd-logger
//! [actions-badge]: https://github.com/flxo/android-logd-logger/workflows/CI/badge.svg
//! [actions-url]: https://github.com/flxo/android-logd-logger/actions?query=workflow%3ACI+branch%3Amaster
//!
//! This logger writes logs to the Android `logd`, a system service with
//! multiple ringbuffers for logs and evens. This is normally done
//! via `liblog` (a native Android lib). Instead of using `liblog`, this crate
//! writes directly to the `logd` socket with the trivial protocol below.
//! This logger is written in pure Rust without any need for ffi.
//!
//! [log]: https://docs.rs/log/*/log/
//! [`error!`]: https://docs.rs/log/*/log/macro.error.html
//! [`warn!`]: https://docs.rs/log/*/log/macro.warn.html
//! [`info!`]: https://docs.rs/log/*/log/macro.info.html
//! [`debug!`]: https://docs.rs/log/*/log/macro.debug.html
//! [`trace!`]: https://docs.rs/log/*/log/macro.trace.html
//!
//! On non Android system the log output is printed to stdout in the default
//! format of `logcat`.
//!
//! # Usage
//!
//! Add this to your Cargo.toml
//!
//! ```toml
//! [dependencies]
//! android-logd-logger = "0.2.1"
//! ```
//!
//! Initialize the logger with a fixed `tag` and the module path included
//! in the log payload.
//!
//! ```
//! # use log::*;
//!
//! fn main() {
//!     android_logd_logger::builder()
//!         .parse_filters("debug")
//!         .tag("log_tag")
//!         .prepend_module(true)
//!         .init();
//!
//!     trace!("trace message: is not logged");
//!     debug!("debug message");
//!     info!("info message");
//!     warn!("warn message");
//!     error!("error message");
//! }
//! ```
//!
//! To write android logd "events" use `event` or `event_now`, e.g:
//!
//! ```
//! android_logd_logger::write_event_now(1, "test").unwrap();
//! ```
//!
//! # Configuration
//!
//! Writing to the logd socket is a single point of synchronization for threads.
//! The `android-logd-logger` can be configured with the `tls` feature to maintain
//! one socket per thread *or* use a single socket for the whole process.
//! Use the features `tls` if you want less interference between threads but pay
//! for one connection per thread.
//!
//! # License
//!
//! Licensed under either of
//!
//!  * Apache License, Version 2.0, ([LICENSE-APACHE](LICENSE-APACHE) or http://www.apache.org/licenses/LICENSE-2.0)
//!  * MIT license ([LICENSE-MIT](LICENSE-MIT) or http://opensource.org/licenses/MIT)

#![deny(missing_docs)]

use bytes::{BufMut, Bytes, BytesMut};
use env_logger::filter::{Builder as FilterBuilder, Filter};
use log::{LevelFilter, Log, Metadata, SetLoggerError};
use std::{fmt, io, iter::FromIterator, time::SystemTime};
use thiserror::Error;

#[allow(dead_code)]
#[cfg(not(target_os = "windows"))]
mod logd;
mod thread;

/// Max log entry len
const LOGGER_ENTRY_MAX_LEN: usize = 5 * 1024;

/// Error
#[derive(Error, Debug)]
pub enum Error {
    /// IO error
    #[error("IO error")]
    Io(#[from] io::Error),
    /// The supplied event data exceed the maximum length
    #[error("Event exceeds maximum size")]
    EventSize,
}

//#[cfg(target_os = "android")]
/// Log priority as defined by logd
#[derive(Clone, Copy, Debug)]
#[repr(u8)]
enum Priority {
    _Unknown = 0,
    _Default = 1,
    Verbose = 2,
    Debug = 3,
    Info = 4,
    Warn = 5,
    Error = 6,
    _Fatal = 7,
    _Silent = 8,
}

impl std::fmt::Display for Priority {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        let c = match self {
            Priority::_Unknown => 'U',
            Priority::_Default | Priority::Debug => 'D',
            Priority::Verbose => 'V',
            Priority::Info => 'I',
            Priority::Warn => 'W',
            Priority::Error => 'E',
            Priority::_Fatal => 'F',
            Priority::_Silent => 'S',
        };
        f.write_str(&c.to_string())
    }
}

impl From<log::Level> for Priority {
    fn from(l: log::Level) -> Priority {
        match l {
            log::Level::Error => Priority::Error,
            log::Level::Warn => Priority::Warn,
            log::Level::Info => Priority::Info,
            log::Level::Debug => Priority::Debug,
            log::Level::Trace => Priority::Verbose,
        }
    }
}

/// Log buffer ids
#[derive(Clone, Copy, Debug)]
#[repr(u8)]
pub enum Buffer {
    /// The main log buffer. This is the only log buffer available to apps.
    Main,
    /// The radio log buffer
    Radio,
    /// The event log buffer.
    Events,
    /// The system log buffer.
    System,
    /// The crash log buffer.
    Crash,
    /// The statistics log buffer.
    Stats,
    /// The security log buffer.
    Security,
    /// User defined Buffer
    Custom(u8),
}

impl From<Buffer> for u8 {
    fn from(b: Buffer) -> u8 {
        match b {
            Buffer::Main => 0,
            Buffer::Radio => 1,
            Buffer::Events => 2,
            Buffer::System => 3,
            Buffer::Crash => 4,
            Buffer::Stats => 5,
            Buffer::Security => 6,
            Buffer::Custom(id) => id,
        }
    }
}

/// Tag mode
#[derive(Debug, Clone)]
enum TagMode {
    /// Use the records target metadata as tag
    Target,
    /// Use root module as tag. The target field contains the module path
    /// if not overwritten. Use the root module as tag. e.g a target of
    /// `crate::module::submodule` will be `crate`.
    TargetStrip,
    /// Custom fixed tag string
    Custom(String),
}

impl Default for TagMode {
    fn default() -> Self {
        TagMode::TargetStrip
    }
}

/// Builder for initializing logger
///
/// The builder is used to initialize the logging framework for later use.
/// It provides
#[derive(Default)]
pub struct Builder {
    filter: FilterBuilder,
    tag: TagMode,
    prepend_module: bool,
    buffer: Option<Buffer>,
}

impl<'a> Builder {
    /// Initializes the log builder with defaults.
    ///
    /// # Examples
    ///
    /// Create a new builder and configure filters and style:
    ///
    /// ```
    /// # use log::LevelFilter;
    /// # use android_logd_logger::Builder;
    ///
    /// let mut builder = Builder::new();
    /// builder.filter(None, LevelFilter::Info).init();
    /// ```
    ///
    /// [`filter`]: #method.filter
    pub fn new() -> Builder {
        Builder::default()
    }

    /// Use a specific android log buffer. Defaults to the main buffer
    /// is used as tag (if present).
    ///
    /// # Examples
    ///
    /// ```
    /// # use android_logd_logger::Builder;
    /// # use android_logd_logger::Buffer;
    ///
    /// let mut builder = Builder::new();
    /// builder.buffer(Buffer::Crash)
    ///     .init();
    /// ```
    pub fn buffer(&mut self, buffer: Buffer) -> &mut Self {
        self.buffer = Some(buffer);
        self
    }

    /// Use a specific log tag. If no tag is set the module path
    /// is used as tag (if present).
    ///
    /// # Examples
    ///
    /// ```
    /// # use android_logd_logger::Builder;
    ///
    /// let mut builder = Builder::new();
    ///
    /// builder.tag("foo")
    ///     .init();
    /// ```
    pub fn tag(&mut self, tag: &str) -> &mut Self {
        self.tag = TagMode::Custom(tag.to_string());
        self
    }

    /// Use the target string as tag
    ///
    /// # Examples
    ///
    /// ```
    /// # use android_logd_logger::Builder;
    ///
    /// let mut builder = Builder::new();
    /// builder.tag_target().init();
    /// ```
    pub fn tag_target(&mut self) -> &mut Self {
        self.tag = TagMode::Target;
        self
    }

    /// Use the target string as tag and strip off ::.*
    ///
    /// # Examples
    ///
    /// ```
    /// # use android_logd_logger::Builder;
    ///
    /// let mut builder = Builder::new();
    /// builder.tag_target_strip().init();
    /// ```
    pub fn tag_target_strip(&mut self) -> &mut Self {
        self.tag = TagMode::TargetStrip;
        self
    }

    /// Prepend module to log message.
    ///
    /// If set true the Rust module path is prepended to the log message.
    ///
    /// # Examples
    ///
    /// ```
    /// # use android_logd_logger::Builder;
    ///
    /// let mut builder = Builder::new();
    /// builder.prepend_module(true).init();
    /// ```
    pub fn prepend_module(&mut self, prepend_module: bool) -> &mut Self {
        self.prepend_module = prepend_module;
        self
    }

    /// Adds a directive to the filter for a specific module.
    ///
    /// # Examples
    ///
    /// Only include messages for warning and above for logs in `path::to::module`:
    ///
    /// ```
    /// # use log::LevelFilter;
    /// # use android_logd_logger::Builder;
    ///
    /// let mut builder = Builder::new();
    ///
    /// builder.filter_module("path::to::module", LevelFilter::Info).init();
    /// ```
    pub fn filter_module(&mut self, module: &str, level: LevelFilter) -> &mut Self {
        self.filter.filter_module(module, level);
        self
    }

    /// Adds a directive to the filter for all modules.
    ///
    /// # Examples
    ///
    /// Only include messages for warning and above for logs in `path::to::module`:
    ///
    /// ```
    /// # use log::LevelFilter;
    /// # use android_logd_logger::Builder;
    ///
    /// let mut builder = Builder::new();
    /// builder.filter_level(LevelFilter::Info).init();
    /// ```
    pub fn filter_level(&mut self, level: LevelFilter) -> &mut Self {
        self.filter.filter_level(level);
        self
    }

    /// Adds filters to the logger.
    ///
    /// The given module (if any) will log at most the specified level provided.
    /// If no module is provided then the filter will apply to all log messages.
    ///
    /// # Examples
    ///
    /// Only include messages for warning and above for logs in `path::to::module`:
    ///
    /// ```
    /// # use log::LevelFilter;
    /// # use android_logd_logger::Builder;
    ///
    /// let mut builder = Builder::new();
    /// builder.filter(Some("path::to::module"), LevelFilter::Info).init();
    /// ```
    pub fn filter(&mut self, module: Option<&str>, level: LevelFilter) -> &mut Self {
        self.filter.filter(module, level);
        self
    }

    /// Parses the directives string in the same form as the `RUST_LOG`
    /// environment variable.
    ///
    /// See the module documentation for more details.
    pub fn parse_filters(&mut self, filters: &str) -> &mut Self {
        self.filter.parse(filters);
        self
    }

    /// Initializes the global logger with the built logd logger.
    ///
    /// This should be called early in the execution of a Rust program. Any log
    /// events that occur before initialization will be ignored.
    ///
    /// # Errors
    ///
    /// This function will fail if it is called more than once, or if another
    /// library has already initialized a global logger.
    pub fn try_init(&mut self) -> Result<(), SetLoggerError> {
        let logger = self.build();

        let max_level = logger.filter.filter();
        let r = log::set_boxed_logger(Box::new(logger));

        if r.is_ok() {
            log::set_max_level(max_level);
        }

        r
    }

    /// Initializes the global logger with the built logger.
    ///
    /// This should be called early in the execution of a Rust program. Any log
    /// events that occur before initialization will be ignored.
    ///
    /// # Panics
    ///
    /// This function will panic if it is called more than once, or if another
    /// library has already initialized a global logger.
    pub fn init(&mut self) {
        self.try_init()
            .expect("Builder::init should not be called after logger initialized");
    }

    fn build(&mut self) -> Logger {
        let buffer = self.buffer.unwrap_or(Buffer::Main);
        let filter = self.filter.build();
        let prepend_module = self.prepend_module;
        Logger::new(buffer, filter, self.tag.clone(), prepend_module).expect("Failed to build logger")
    }
}

struct Logger {
    filter: Filter,
    tag: TagMode,
    prepend_module: bool,
    #[allow(unused)]
    buffer_id: Buffer,
}

impl Logger {
    pub fn new(buffer_id: Buffer, filter: Filter, tag: TagMode, prepend_module: bool) -> Result<Logger, io::Error> {
        Ok(Logger {
            filter,
            tag,
            prepend_module,
            buffer_id,
        })
    }
}

impl Log for Logger {
    fn enabled(&self, metadata: &Metadata) -> bool {
        self.filter.enabled(metadata)
    }

    fn log(&self, record: &log::Record) {
        if !self.filter.matches(record) {
            return;
        }

        let args = record.args().to_string();
        let message = if let Some(module_path) = record.module_path() {
            if self.prepend_module {
                [module_path, &args].join(": ")
            } else {
                args
            }
        } else {
            args
        };

        let priority: Priority = record.metadata().level().into();
        let tag = match &self.tag {
            TagMode::Target => record.target(),
            TagMode::TargetStrip => record
                .target()
                .split_once("::")
                .map(|(tag, _)| tag)
                .unwrap_or_else(|| record.target()),
            TagMode::Custom(tag) => tag.as_str(),
        };

        #[cfg(target_os = "android")]
        logd::log(tag, self.buffer_id, priority, &message);

        #[cfg(not(target_os = "android"))]
        {
            let now = ::time::OffsetDateTime::now_utc();
            println!(
                "{}.{} {} {} {} {}: {}",
                now.format("%Y-%m-%d %T"),
                &now.format("%N")[..3],
                std::process::id(),
                thread::id(),
                priority,
                tag,
                message
            );
        }
    }

    #[cfg(not(target_os = "android"))]
    fn flush(&self) {
        use std::io::Write;
        io::stdout().flush().ok();
    }

    #[cfg(target_os = "android")]
    fn flush(&self) {}
}

/// Returns a default [`Builder`] for configuration and initialization of logging.
///
/// With the help of the [`Builder`] the logging is configured.
/// The tag, filter and buffer can be set.
/// Additionally it is possible to set whether the modul path appears in a log message.
///
/// After a call to [`init`](Builder::init) the global logger is initialized with the configuration.
pub fn builder() -> Builder {
    Builder::default()
}

/// Event tag
pub type EventTag = u32;

/// Event data
#[derive(Debug, Clone, PartialEq)]
pub struct Event {
    /// Timestamp
    pub timestamp: SystemTime,
    /// Tag
    pub tag: EventTag,
    /// Value
    pub value: EventValue,
}

/// Event's value
#[derive(Debug, PartialEq, Clone)]
pub enum EventValue {
    /// Void value
    Void,
    /// Int value
    Int(i32),
    /// Long value
    Long(i64),
    /// Float value
    Float(f32),
    /// String value
    String(String),
    /// List of values
    List(Vec<EventValue>),
}

impl EventValue {
    /// Serialied size
    pub fn serialized_size(&self) -> usize {
        match self {
            &EventValue::Void => 0,
            EventValue::Int(_) | EventValue::Float(_) => 1 + 4,
            EventValue::Long(_) => 1 + 8,
            EventValue::String(s) => 1 + 4 + s.as_bytes().len(),
            EventValue::List(l) => 1 + 1 + l.iter().map(EventValue::serialized_size).sum::<usize>(),
        }
    }

    /// Serialize the event value into bytes
    pub fn as_bytes(&self) -> Bytes {
        const EVENT_TYPE_INT: u8 = 0;
        const EVENT_TYPE_LONG: u8 = 1;
        const EVENT_TYPE_STRING: u8 = 2;
        const EVENT_TYPE_LIST: u8 = 3;
        const EVENT_TYPE_FLOAT: u8 = 4;

        let mut buffer = BytesMut::with_capacity(self.serialized_size());
        match self {
            EventValue::Void => (),
            EventValue::Int(num) => {
                buffer.put_u8(EVENT_TYPE_INT);
                buffer.put_i32_le(*num);
            }
            EventValue::Long(num) => {
                buffer.put_u8(EVENT_TYPE_LONG);
                buffer.put_i64_le(*num);
            }
            EventValue::Float(num) => {
                buffer.put_u8(EVENT_TYPE_FLOAT);
                buffer.put_f32_le(*num);
            }
            EventValue::String(string) => {
                buffer.put_u8(EVENT_TYPE_STRING);
                buffer.put_u32_le(string.len() as u32);
                buffer.put(string.as_bytes());
            }
            EventValue::List(values) => {
                buffer.put_u8(EVENT_TYPE_LIST);
                buffer.put_u8(values.len() as u8);
                values.iter().for_each(|value| buffer.put(value.as_bytes()));
            }
        };
        buffer.freeze()
    }
}

impl From<()> for EventValue {
    fn from(_: ()) -> Self {
        EventValue::Void
    }
}

impl From<i32> for EventValue {
    fn from(v: i32) -> Self {
        EventValue::Int(v)
    }
}

impl From<i64> for EventValue {
    fn from(v: i64) -> Self {
        EventValue::Long(v)
    }
}

impl From<f32> for EventValue {
    fn from(v: f32) -> Self {
        EventValue::Float(v)
    }
}

impl From<&str> for EventValue {
    fn from(v: &str) -> Self {
        EventValue::String(v.to_string())
    }
}

impl<T> FromIterator<T> for EventValue
where
    T: Into<EventValue>,
{
    fn from_iter<I: IntoIterator<Item = T>>(iter: I) -> Self {
        EventValue::List(iter.into_iter().map(Into::into).collect())
    }
}

impl<T> From<Vec<T>> for EventValue
where
    T: Into<EventValue>,
{
    fn from(mut v: Vec<T>) -> Self {
        EventValue::List(v.drain(..).map(|e| e.into()).collect())
    }
}

/// Write an event with the timestamp now to `Buffer::Events`
/// ```
/// use android_logd_logger::{write_event, write_event_now, Error, Event, EventValue};
/// android_logd_logger::builder().init();
///
/// write_event_now(1, "test").unwrap();
///
/// let value: Vec<EventValue> = vec![1.into(), "one".into(), 123.3.into()].into();
/// write_event_now(2, value).unwrap();
/// ```
pub fn write_event_now<T: Into<EventValue>>(tag: EventTag, value: T) -> Result<(), Error> {
    write_event(&Event {
        timestamp: SystemTime::now(),
        tag,
        value: value.into(),
    })
}

/// Write an event with the timestamp now to buffer
/// ```
/// use android_logd_logger::{write_event_buffer_now, Buffer, Error, Event, EventValue};
/// android_logd_logger::builder().init();
///
/// write_event_buffer_now(Buffer::Stats, 1, "test").unwrap();
///
/// let value: Vec<EventValue> = vec![1.into(), "one".into(), 123.3.into()].into();
/// write_event_buffer_now(Buffer::Stats, 2, value).unwrap();
/// ```
pub fn write_event_buffer_now<T: Into<EventValue>>(log_buffer: Buffer, tag: EventTag, value: T) -> Result<(), Error> {
    write_event_buffer(
        log_buffer,
        &Event {
            timestamp: SystemTime::now(),
            tag,
            value: value.into(),
        },
    )
}

/// Write an event to `Buffer::Events`
/// ```
/// use android_logd_logger::{write_event, Error, Event, EventValue};
/// android_logd_logger::builder().init();
///
/// write_event(&Event {
///     timestamp: std::time::SystemTime::now(),
///     tag: 1,
///     value: "blah".into(),
/// }).unwrap();
/// ```
pub fn write_event(event: &Event) -> Result<(), Error> {
    write_event_buffer(Buffer::Events, event)
}

/// Write an event to an explicit buffer
/// ```
/// use android_logd_logger::{write_event_buffer, Buffer, Error, Event, EventValue};
/// android_logd_logger::builder().init();
///
/// write_event_buffer(Buffer::Stats, &Event {
///     timestamp: std::time::SystemTime::now(),
///     tag: 1,
///     value: "blah".into(),
/// }).unwrap();
/// ```
pub fn write_event_buffer(log_buffer: Buffer, event: &Event) -> Result<(), Error> {
    if event.value.serialized_size() > (LOGGER_ENTRY_MAX_LEN - 1 - 2 - 4 - 4 - 4) {
        return Err(Error::EventSize);
    }

    #[cfg(target_os = "android")]
    logd::write_event(log_buffer, event);

    #[cfg(not(target_os = "android"))]
    println!("buffer: {:?}, event: {:?}", log_buffer, event);

    Ok(())
}
