use crate::parser::program::ctx_accounts_ident;
use crate::{Ix, IxArg};
use syn::parse::{Error as ParseError, Result as ParseResult};
use syn::spanned::Spanned;

// Parse all non-state ix handlers from the program mod definition.
pub fn parse(program_mod: &syn::ItemMod) -> ParseResult<Vec<Ix>> {
    let mod_content = &program_mod
        .content
        .as_ref()
        .ok_or_else(|| ParseError::new(program_mod.span(), "program content not provided"))?
        .1;

    mod_content
        .iter()
        .filter_map(|item| match item {
            syn::Item::Fn(item_fn) => Some(item_fn),
            _ => None,
        })
        .map(|method: &syn::ItemFn| {
            let mut args: Vec<IxArg> = method
                .sig
                .inputs
                .iter()
                .map(|arg: &syn::FnArg| match arg {
                    syn::FnArg::Typed(arg) => {
                        let ident = match &*arg.pat {
                            syn::Pat::Ident(ident) => &ident.ident,
                            _ => {
                                return Err(ParseError::new(
                                    arg.pat.span(),
                                    "expected argument name",
                                ))
                            }
                        };
                        Ok(IxArg {
                            name: ident.clone(),
                            raw_arg: arg.clone(),
                        })
                    }
                    syn::FnArg::Receiver(_) => Err(ParseError::new(
                        arg.span(),
                        "expected a typed argument not self",
                    )),
                })
                .collect::<ParseResult<_>>()?;
            // Remove the Context argument
            let anchor = args.remove(0);
            let anchor_ident = ctx_accounts_ident(&anchor.raw_arg)?;

            Ok(Ix {
                raw_method: method.clone(),
                ident: method.sig.ident.clone(),
                args,
                anchor_ident,
            })
        })
        .collect::<ParseResult<Vec<Ix>>>()
}
