
#### Slice

```rust
~ implicit
= slice &[1 2 3]

~ explicit
= slice &[u8]
    [1 2 3]

~ access by index
println!("{}", slice[1])
```














The `?` postfix operator works like it does in Rust.

#### Option `<>`

`Option<T>` has the shorthand `<T>`

```rust
= option <3>

~ match
?! option
    <3> println!("Some(3)"
    <>  println!("None"

~ if let
?= <3> option
    println!("Some(3)"
    ^: "None"
```

#### Result `?<>`

`Result<T, E>` has the shorthand `?<T E>`.

- `Ok<T>` has the shorthand `#<T>`
- `Err<E>` has the shorthand `!<E>`

*Empty braces imply the `unit` type.*

```rust
= ok <&str># "works"
= err <&str>! "errdout"

:= check status <bool> >> <i32, {}>?
    ?= b <status>
        println!("{}" b
        > <123>#
    ^   > <>!
```

However, quite often we do not need detailed error handling.
We can automate usage with `??<T>`. This will automate `Result` with the `anyhow` crate.

```rust
:= get_config >> <HashMap>??
    = config std,fs,read_to_string? "config.json"
    = map HashMap serde_json,from_str? &config
    -> map
```


### Shell Commands

```rust
(cmd)   ~ returns @()
(cmd)0  ~ returns status code as u8
(cmd)1  ~ returns "stdout"
(cmd)2  ~ returns "stderr"

~ redirection
(cmd < stdin)
(cmd > stdout)
(cmd 1> stdout)
(cmd 2> stderr)
(cmd 3> both)

~ noclobber
(cmd >? stdout)

~ append
(cmd >> stdout)
(cmd 1>> stdout)
(cmd 2>> stderr)
(cmd 3>> both)

~ piping
(cmd | stdout)
(cmd 1| stdout)
(cmd 2| stderr)
(cmd 3| both)

~ logical and
(cmd1 && cmd2)

~ logical or
(cmd1 || cmd2)

~ grouping commands
(cmd (1) (2) )
(cmd
    (1)
    (2)

~ interpolation
= cache "~/.cache/analog/"
(rm -rf {cache})
```

Shell commands
--------------

We can run shell commands with `()`.

```rust
(sudo reboot)
```

Syntax within `()` differs from the rest of the language.
It is more akin to the syntax of POSIX shells.

### `@()`

`(@)` is a builtin alias for `{u8 String String}`. This is returned by every shell command and equates to `{status code, stdout, stderr}`.

The return type can be specified with a numerical postfix.

```rust
(cmd)   ~ returns (@)
(cmd)0  ~ returns status code as u8
(cmd)1  ~ returns "stdout"
(cmd)2  ~ returns "stderr"
```

### Redirection

Redirection is accomplished with `>` and `<`.

```rust
(cmd < stdin)
(cmd > stdout)
(cmd 1> stdout)
(cmd 2> stderr)
(cmd 3> both)
```

We can choose not to clobber a file if it exists with `>?`.

```rust
(cmd >? stdout)
```

To append rather than overwrite, use `>>`.

```rust
(cmd >> stdout)
(cmd 1>> stdout)
(cmd 2>> stderr)
(cmd 3>> both)
```

### Piping

Piping is accomplished with `|`.

```rust
(cmd | stdout)
(cmd 1| stdout)
(cmd 2| stderr)
(cmd 3| both)
```

### Combinators

We can chain short circuiting commands with `&&`.

Whenever there is the potential for multiple commands results, the return type will be `![@()]`.

```rust
(cmd1 && cmd2)
```

Logical "or" operations can be accomplished with `||`.

```rust
(cmd1 || cmd2)
```

"or" operations can also be accomplished through a catch-all.

```rust
? (cmd1) % (cmd2)
```

### Status codes

We can easily make a decision about what to do `if` a command succeeds or fails with `?`.

`== true` is implied.

```rust
? (uname)
    :: "command succeeded!"
    %: "command failed"
```

We can also `match` on the returned status code.

```rust
? (uname)0
    == 0 :: "command succeeded!"
    == 20 :: "this is ok."
    == 127 :: "file not found"
    %: "command failed."
```

### Command grouping

We can group commands with additional `()`.

This is equivalent to using the short-circuiting `&&` shell operator.

```rust
( (cmd a) (cmd b) )   ~ returns ![@()]
( (cmd a) (cmd b) )0  ~ returns ![u8] of status codes
( (cmd a) (cmd b) )1  ~ returns ![String] of stdout
( (cmd a) (cmd b) )2  ~ returns ![String] of stderr
```

Conditionals will return `true` if status code for *ALL* commands is `0`.

```rust
? ( (cmd a) (cmd b) )
```

We can expand commands in groups to avoid repetition.

```rust
(cmd (a) (b) )
```

Groups also provide access to an indented syntax.
The closing `)` is implied through indentation.

```rust
(cmd
    (a)
    (b)
```

### Reading input

`!<` is a special operator which stops code execution to read a line of input from stdin.

Arguments are received as `![String]` which are then accessed with the defined variable.

```rust
!< reading

:: "first argument " reading[0]
:: "third argument " reading[2]
:: "every argument " [ reading ]
```

### Interpolation

Interpolation in shell commands can be accomplished by placing a variable within `{}`.

```rust
= bin "~/.cargo/bin/analog"
= cache "~/.cache/analog/"
= config "~/.config/analog/"

:= uninstall_analog
    (rm -rf {bin} {cache} {config})
```

### Example

Let"s make a backup script...

We start with separate commands.

```rust
(rsync -avhP ~/.config/ /media/usb/backup/.config/)
(rsync -avhP ~/.local/ /media/usb/backup/.local/)
(rsync -avhP ~/.vim/ /media/usb/backup/.vim/)
```

Lets expand them...

```rust
(rsync -avhP
    (~/.config/ /media/usb/backup/.config/)
    (~/.local/ /media/usb/backup/.local/)
    (~/.vim/ /media/usb/backup/.vim/)
```

Now lets interpolate...

```rust
backup = "/media/usb/backup/"

(rsync -avhP
    (~/.config/ {backup}.config/)
    (~/.local/ {backup}.local/)
    (~/.vim/ {backup}.vim/)
```

And reduce error prone repetition with a loop...

```rust
backup = "/media/usb/backup/"
folders = []
    ".config"
    ".local"
    ".vim"

|f| folders
    (rsync -avhP ~/{f}/ {backup}{f}/)
```
We can also declare and initialize an anonymous struct with `@{}`

```rust
= recipe @{}
    onions  2
    peppers 4
    rice    true
```

### Repetition

We can repeat a line of code `n` number of times with `|n|` placed at the end of a statement.

```rust
= test "how to repeat"

;test.uppercase
;test.pop |6|
;test.push_str "be"
```


### Debug

We can use the `:.` operator to print any type which implements `std.debug.Debug`.

```rust
= keymap @[char u8]
    `a` 1
    `b` 2
    `c` 3

~ print debug
:. keymap

~ print debug (without newline)
:, keymap
```

### Iterative

The `::` operator provides a special syntax which allows `[]` to iterate over
applicable types.

```rust
= letters [char] "a" "b" "c"

:= print_abc
    ~ prints newline separated
    :: ":: " [ letters ]

    ~ anything within brackets will iterate
    :: [ ":: " letters ]

    ~ we can prevent newlines after entire iteration
    :; [ ":; " letters ]

    ~ or during...
    ;: ";: " [ letters ]

    ~ or both
    ;; ";; " [ letters ]
```

```rust
letters = [ "a" "b" "c" ]

print_abc =
    :: "iterative printing"
    :: ":: " [ letters ] ~ newline separated
    :: [ ":: " letters ] ~ anything within brackets will iterate
    :; ":;" [ letters ]  ~ prevent newlines after iteration
    ;: ";:" [ letters ]  ~ prevent newlines during iteration
    ;; ";;" [ letters ]  ~ prevent newlines during and after iteration
```

#[]   ~ HashMap

### HashMap

```rust
= map @[str u8]
    'to' 2
    'at' 48

:: map['to']
```

= v #[]     ~ HashMap

`|k v| hashmap` can be read as "for `k`, `v` in hashmap".

```rust
hmap = #[]
    "a" 9
    "b" 21
    "c" 99

|k v| hmap
    :: "key: " k
    :+ v
    :: "value: " v
```


#### HashMap

```rust
:= cats >> &str
    'cats and cats and cats'

~ implicit
= hashmap #[]
    22 'round'
    44 'about'
    88 :cats

~ explicit
= hashmap #[char &str]
    'a' 'found'
    'b' 'around'
    'c' :cats

~ access value by key
:: hashmap['a']
```

```rust
= hmap @[]
    `a` 13
    `b` 24
    `c` 99

|k v| hmap
    :: "key: " k
    :: "value: " v
```





:= select<`short, `long> -> &str`short
    - s1 &str`short
    - s2 &str`long
    - second bool
    <<< `long `short
    ? second > s2 %> s1

let outer = Stringprintln!from("Long living ref");
let longer = &outer;
{
    let inner = Stringprintln!from("Short living ref");
    let shorter = &inner;

    assert_eq!(select(shorter, longer, false), shorter);
    assert_eq!(select(shorter, longer, true), longer);
}









`@true` and `@false` allow typecasting to a `bool`.

```rust
type      | falsey | truthy
----------|--------|----------------
string    | empty  | not empty
int/float | 0      | not 0
vec       | empty  | not empty
hashmap   | empty  | not empty
```

Checking the truthiness of an integer...

```rust
theme = 0

? theme == true
    :: 'custom'
    %: 'default'

~ statement above will print 'default'
```

`analog` specific:

```rust
()  ~ shell command
[!] ~ read line of input

::  ~ println
:;  ~ print
:.  ~ debug println
:,  ~ debug print
;:  ~ |print| println
;;  ~ |print| print
```

#### Comparison Operators

```rust
<   ~ less than
<=  ~ less than or equal to
>   ~ greater than
>=  ~ greater than or equal to
==  ~ equal
!=  ~ not equal
%=  ~ remainder
```

We can also use boolean comparisons the same as Rust.

``` rust
!b     ~ logical not
a | b  ~ logical or
a & b  ~ logical and
a ^ b  ~ logical xor
```

### Varadic
:= f **       ~ takes any number of arguments

Varadic functions can be defined with `**` as an argument.

`@*` is used to access `![]` of arguments from a varadic function and can be
accessed by index with `@n`.

```rust
:= print_args **
    :: @0       ~ prints the first argument
    :: @2       ~ prints the third argument
    :: [ @* ]   ~ prints ["args"]

~ set parameters can still be defined
:= always show **
    :: show ' <param args> '
    :: [ @* ]
```

This syntax is mirrored when accepting arguments from a command line invocation.

`#*` is used to access `![]` of arguments and can be accessed by index with
`#n`.

```rust
:= print
    :: #0
    :: #2
    :: [ #* ]
```

### Abstract this in?

fn color_print(text: &str, bold: bool=false, underline: bool = false) {
    // ASCII STUFF
}

Could be called as

color_print("Hello", underline=true);

### Shell command printing

Printing to the terminal is not implicit;
We can define what prints to our terminal with the following postfixes.

```rust
(cmd)    ~ prints nothing
(cmd):   ~ prints stdout
(cmd);   ~ prints stderr
(cmd):;  ~ prints stdout and stderr
```

### Array type inference
You can specify the type of a vec in order to prevent needing to encapsulate each value.

```rust
vec_char = '[ a b c ]
vec_strs = "[ quotes can be used "to include whitespace" ]

easy = "[ you can see how \"easy\" this is ... ]
hard = [ "over" "having" "to" "do" "this" "..."]
```

Function calls can be used within these vec declarations with the `!` operator.

```rust
berry = "Elderberries"

result = "[ !berry are delicious ]

escape = "[ this means we must escape \! and \" "when needed\!" ]
```

### Asynchronous shell commands
You can run groups of commands asynchronously with `[ ]`s. Since this is asynchronous, all commands are run.

```rust
[ (cmd1) (cmd2) ]   ~ returns [@cmd]
&[ (cmd1) (cmd2) ]  ~ returns [@&]
![ (cmd1) (cmd2) ]  ~ returns ["stdout"]
!![ (cmd1) (cmd2) ] ~ returns ["stderr"]

~ conditional which returns @true if status code of ALL commands is 0
?[ (cmd1) (cmd2) ]

~ commands can still be expanded...
$[cmd -p (1) (2) ]
```
