Shell commands
--------------

`analog` makes working with shell commands easy.

```rust
(sudo reboot)
```

Syntax within `()` differs from the rest of the language.
It is more akin to the syntax of POSIX shells.

### Running commands

We run shell commands within `( )`s.

`#cmd` is a type alias for `{ #u8 #str #str }` which equates to
`{ status code, stdout, stderr }`.

The return type can be specified with a numerical postfix.

```rust
(cmd)   ~ returns #cmd
(cmd)0  ~ returns status code as #u8
(cmd)1  ~ returns 'stdout'
(cmd)2  ~ returns 'stderr'
```

### Redirection

Redirection is accomplished similar to most POSIX shells.

```rust
(cmd < stdin)
(cmd > stdout)
(cmd 1> stdout)
(cmd 2> stderr)
(cmd 3> both)
```

To append rather than overwrite, use `>>`.

```rust
(cmd >> stdout)
(cmd 1>> stdout)
(cmd 2>> stderr)
(cmd 3>> both)
```

We can choose not to clobber a file if it exists with `>?`.

```rust
(cmd >? stdout)
```

### Piping

Piping is also similar to most POSIX shells.

```rust
(cmd | stdout)
(cmd 1| stdout)
(cmd 2| stderr)
(cmd 3| both)
```

### Combinators

We can chain short circuiting commands with `&&`.

Whenever there is the potential for multiple commands results, the return type will be `[#cmd]`.

```rust
(cmd1 && cmd2)
```

Logical "or" operations can be accomplished with `||`.

```rust
(cmd1 || cmd2)
```

"or" operations can also be accomplished through a catch-all.

```rust
? (cmd1) % (cmd2)
```

### Status codes

We can easily make a decision about what to do if a command succeeds or fails with `?`.

Additionally, `== #true` is implied and does not have to be written.

```rust
? (uname)
    :: "command succeeded!"
    %: "command failed"
```

We can also match on the returned status code.

```rust
? (uname)0
    == 0 :: "command succeeded!"
    == 20 :: "this is ok."
    == 127 :: "file not found"
    %: "command failed."
```

### Command grouping

We can group commands with additional `()`.

This is equivalent to using the short-circuiting `&&` shell operator.

```rust
( (cmd a) (cmd b) )   ~ returns [#cmd]
( (cmd a) (cmd b) )0  ~ returns status codes as [#u8]
( (cmd a) (cmd b) )1  ~ returns [#str] of stdout
( (cmd a) (cmd b) )2  ~ returns [#str] of stderr

~ conditional which returns #true if status code for ALL commands is 0
? ( (cmd a) (cmd b) )
```

We can expand commands in groups to avoid repetition.

```rust
(cmd (a) (b) )
```

Groups also provide access to an encapsulated indented syntax.

```rust
(cmd
    (a)
    (b)
)
```

### Reading input

`[!] var` is a special operator which stops code execution to read a line of input from stdin.

Arguments are received as `[#str]` of arguments which are then accessed with the defined variable.

```rust
[!] reading

:: "first argument " reading[0]
:: "third argument " reading[2]
:: "every argument " [ reading ]
```

### Interpolation

Interpolation in shell commands is accomplished by placing a variable within `{}`.

Interpolations cannot be nested.

```rust
= bin '~/.cargo/bin/analog'
= cache '~/.cache/analog/'
= config '~/.config/analog/'

:= uninstall_analog
    (rm -rf {bin} {cache} {config})
```

### Example

Let's make a backup script...

We start with separate commands.

```rust
(rsync -avhP ~/.config/ /media/usb/backup/.config/)
(rsync -avhP ~/.local/ /media/usb/backup/.local/)
(rsync -avhP ~/.vim/ /media/usb/backup/.vim/)
```

Lets expand them...

```rust
(rsync -avhP
    (~/.config/ /media/usb/backup/.config/)
    (~/.local/ /media/usb/backup/.local/)
    (~/.vim/ /media/usb/backup/.vim/)
)
```

Now lets interpolate...

```rust
backup = '/media/usb/backup/'

(rsync -avhP
    (~/.config/ {backup}.config/)
    (~/.local/ {backup}.local/)
    (~/.vim/ {backup}.vim/)
)
```

And reduce repetition with a loop...

```rust
backup = '/media/usb/backup/'
folders = []
    '.config'
    '.local'
    '.vim'

|f| folders
    (rsync -avhP ~/{f}/ {backup}{f}/)
```
