//#[macro_use]
//extern crate matches;

extern crate sesdiff;

use std::str::FromStr;
use analiticcl::*;
use analiticcl::test::*;


#[test]
fn test0001_alphabet() {
    let (alphabet, alphabet_size) = get_test_alphabet();
    assert_eq!(alphabet.len(), 27);
}

#[test]
fn test0002_primes() {
    //tests whether the primes are really prime
    //(since they're hard coded and we don't want accidental typos)
    for prime in PRIMES {
        for i in 2..*prime {
            assert!(*prime % i != 0);
        }
    }
}

#[test]
fn test0102_hash_hash() {
    let (alphabet, alphabet_size) = get_test_alphabet();

    //this is a hash that would overflow any normal 64-bit int, but it should hash fine
    assert_eq!(AnaValue::empty(), AnaValue::from(1 as usize));
}

#[test]
fn test0103_hash_basic() {
    let (alphabet, alphabet_size) = get_test_alphabet();

    assert_eq!("a".anahash(&alphabet), AnaValue::from(2 as usize));
    assert_eq!("b".anahash(&alphabet), AnaValue::from(3 as usize));
    assert_eq!("c".anahash(&alphabet), AnaValue::from(5 as usize));
    assert_eq!("ab".anahash(&alphabet), AnaValue::from((2*3) as usize));
    assert_eq!("ba".anahash(&alphabet), AnaValue::from((3*2) as usize));
    assert_eq!("ab".anahash(&alphabet), "ba".anahash(&alphabet));
    assert_eq!("abc".anahash(&alphabet), AnaValue::from((2*3*5) as usize));
    assert_eq!("abcabcabc".anahash(&alphabet), AnaValue::from((2*3*5*2*3*5*2*3*5) as usize));
}

#[test]
fn test0103_hash_alphabet_equivalence() {
    let (alphabet, alphabet_size) = get_test_alphabet();

    //the alphabet may define multiple values that map to the same
    //the provided example alphabet does so for case-differences
    //and periods and commas:

    assert_eq!("abc".anahash(&alphabet), "ABC".anahash(&alphabet));
    assert_eq!("abc".anahash(&alphabet), "bAc".anahash(&alphabet));
    assert_eq!("a.b".anahash(&alphabet), "a,b".anahash(&alphabet));
}

#[test]
fn test0104_hash_big() {
    let (alphabet, alphabet_size) = get_test_alphabet();

    //this is a hash that would overflow any normal 64-bit int, but it should hash fine
    assert!("xyzxyzxyzxyzxyzxyzxyzxyzxyzxyzxyzxyzxyzxyzxyzxyzxyzxyzxyzxyzxyzxyzxyzxyz".anahash(&alphabet) > AnaValue::empty());
}


#[test]
fn test0105_hash_anagram() {
    let (alphabet, alphabet_size) = get_test_alphabet();
    assert_eq!("stressed".anahash(&alphabet),"desserts".anahash(&alphabet) );
    assert_eq!("dormitory".anahash(&alphabet),"dirtyroom".anahash(&alphabet) );
    assert_eq!("presents".anahash(&alphabet),"serpents".anahash(&alphabet) );
}

#[test]
fn test0106_hash_insertion() {
    let (alphabet, alphabet_size) = get_test_alphabet();

    let ab = "ab".anahash(&alphabet);
    let c = "c".anahash(&alphabet);
    let abc = "abc".anahash(&alphabet);

    assert_eq!(ab.insert(&c), abc);
    assert_eq!(c.insert(&ab), abc);
}

#[test]
fn test0107_hash_containment() {
    let (alphabet, alphabet_size) = get_test_alphabet();

    let ab = "ab".anahash(&alphabet);
    let c = "c".anahash(&alphabet);
    let abc = "abc".anahash(&alphabet);

    assert_eq!(abc.contains(&c), true);
    assert_eq!(abc.contains(&ab), true);
    assert_eq!(abc.contains(&abc), true);

    //counter-examples that should evaluate to false:
    assert_eq!(c.contains(&abc), false);
    assert_eq!(ab.contains(&c), false);
    assert_eq!(ab.contains(&abc), false);
}

#[test]
fn test0108_hash_deletion() {
    let (alphabet, alphabet_size) = get_test_alphabet();

    let ab = "ab".anahash(&alphabet);
    let b = "b".anahash(&alphabet);
    let c = "c".anahash(&alphabet);
    let abc = "abc".anahash(&alphabet);
    let ac = "ac".anahash(&alphabet);
    let x = "x".anahash(&alphabet);

    assert_eq!(abc.delete(&c), Some(ab));
    assert_eq!(abc.delete(&b), Some(ac));

    //counter-examples that should return None
    assert_eq!(c.delete(&abc), None);
    assert_eq!(abc.delete(&x), None);
}


#[test]
fn test0108_hash_upper_bound() {
    let (alphabet, alphabet_size) = get_test_alphabet();

    let ab = "ab".anahash(&alphabet);
    let abc = "abc".anahash(&alphabet);
    let x = "x".anahash(&alphabet);

    assert_eq!(abc.alphabet_upper_bound(alphabet_size), (2,3)); //indices 0,1,2 -> a,b,c   3 -> 3 characters
    assert_eq!(ab.alphabet_upper_bound(alphabet_size), (1,2));
    assert_eq!(x.alphabet_upper_bound(alphabet_size), (23,1));
}



#[test]
fn test0201_iterator_parents() {
    let (alphabet, alphabet_size) = get_test_alphabet();
    let anavalue: AnaValue = "house".anahash(&alphabet);
    let mut chars: Vec<AnaValue> = Vec::new();
    let mut deletions: Vec<AnaValue> = Vec::new();
    for deletion in anavalue.iter_parents(alphabet_size) {
       chars.push(AnaValue::character(deletion.charindex));
       deletions.push(deletion.value.clone());
    }
    assert_eq!(chars.len(), 5, "Checking length of results",);
    assert_eq!(chars.get(0).unwrap(), &"u".anahash(&alphabet));
    assert_eq!(chars.get(1).unwrap(), &"s".anahash(&alphabet));
    assert_eq!(chars.get(2).unwrap(), &"o".anahash(&alphabet));
    assert_eq!(chars.get(3).unwrap(), &"h".anahash(&alphabet));
    assert_eq!(chars.get(4).unwrap(), &"e".anahash(&alphabet));
    assert_eq!(deletions.get(0).unwrap(), &"hose".anahash(&alphabet));
    assert_eq!(deletions.get(1).unwrap(), &"houe".anahash(&alphabet));
    assert_eq!(deletions.get(2).unwrap(), &"huse".anahash(&alphabet));
    assert_eq!(deletions.get(3).unwrap(), &"ouse".anahash(&alphabet));
    assert_eq!(deletions.get(4).unwrap(), &"hous".anahash(&alphabet));
}

#[test]
fn test0202_iterator_parents_dup() {
    //This one has duplicate letters, but no duplicate
    //anagram output will be generated, we do only
    //1 deletion
    let (alphabet, alphabet_size) = get_test_alphabet();
    let anavalue: AnaValue = "pass".anahash(&alphabet);
    let mut chars: Vec<AnaValue> = Vec::new();
    let mut deletions: Vec<AnaValue> = Vec::new();
    for deletion in anavalue.iter_parents(alphabet_size) {
       chars.push(AnaValue::character(deletion.charindex));
       deletions.push(deletion.value.clone());
    }
    assert_eq!(chars.len(),3, "Checking length of results",);
    assert_eq!(chars.get(0).unwrap(), &"s".anahash(&alphabet));
    assert_eq!(chars.get(1).unwrap(), &"p".anahash(&alphabet));
    assert_eq!(chars.get(2).unwrap(), &"a".anahash(&alphabet));
    assert_eq!(deletions.get(0).unwrap(), &"pas".anahash(&alphabet));
    assert_eq!(deletions.get(1).unwrap(), &"ass".anahash(&alphabet));
    assert_eq!(deletions.get(2).unwrap(), &"pss".anahash(&alphabet));
}

#[test]
fn test0203_iterator_recursive_singlebeam() {
    let (alphabet, alphabet_size) = get_test_alphabet();
    let anavalue: AnaValue = "house".anahash(&alphabet);
    let mut chars: Vec<AnaValue> = Vec::new();
    let mut depths: Vec<_> = Vec::new();
    let mut deletions: Vec<AnaValue> = Vec::new();
    for (deletion, depth) in anavalue.iter(alphabet_size) {
       chars.push(AnaValue::character(deletion.charindex));
       deletions.push(deletion.value.clone());
       depths.push(depth);
    }
    assert_eq!(chars.len(), 5, "Checking length of results",);
    assert_eq!(chars.get(0).unwrap(), &"u".anahash(&alphabet));
    assert_eq!(chars.get(1).unwrap(), &"s".anahash(&alphabet));
    assert_eq!(chars.get(2).unwrap(), &"o".anahash(&alphabet));
    assert_eq!(chars.get(3).unwrap(), &"h".anahash(&alphabet));
    assert_eq!(chars.get(4).unwrap(), &"e".anahash(&alphabet));
    assert_eq!(deletions.get(0).unwrap(), &"hose".anahash(&alphabet));
    assert_eq!(deletions.get(1).unwrap(), &"hoe".anahash(&alphabet));
    assert_eq!(deletions.get(2).unwrap(), &"he".anahash(&alphabet));
    assert_eq!(deletions.get(3).unwrap(), &"e".anahash(&alphabet));
    assert_eq!(deletions.get(4).unwrap(), &AnaValue::empty());
    assert_eq!(depths, &[1,2,3,4,5]);
}

#[test]
fn test0203_iterator_recursive() {
    //depth first by default
    let (alphabet, alphabet_size) = get_test_alphabet();
    let anavalue: AnaValue = "abcd".anahash(&alphabet);
    let mut depths: Vec<_> = Vec::new();
    let mut deletions: Vec<AnaValue> = Vec::new();
    for (deletion, depth) in anavalue.iter_recursive(alphabet_size, &SearchParams::default()) {
       deletions.push(deletion.value.clone());
       depths.push(depth);
    }
    let mut iter = deletions.iter();
    assert_eq!(iter.next().unwrap(), &"abc".anahash(&alphabet));
    assert_eq!(iter.next().unwrap(), &"ab".anahash(&alphabet));
    assert_eq!(iter.next().unwrap(), &"a".anahash(&alphabet));
    assert_eq!(iter.next().unwrap(), &AnaValue::empty());
    assert_eq!(iter.next().unwrap(), &"b".anahash(&alphabet));
    assert_eq!(iter.next().unwrap(), &AnaValue::empty());
    assert_eq!(iter.next().unwrap(), &"ac".anahash(&alphabet));
    assert_eq!(iter.next().unwrap(), &"a".anahash(&alphabet));
    assert_eq!(iter.next().unwrap(), &AnaValue::empty());
    assert_eq!(iter.next().unwrap(), &"c".anahash(&alphabet));
    assert_eq!(iter.next().unwrap(), &AnaValue::empty());
    assert_eq!(iter.next().unwrap(), &"bc".anahash(&alphabet));
    assert_eq!(iter.next().unwrap(), &"b".anahash(&alphabet));
    assert_eq!(iter.next().unwrap(), &AnaValue::empty());
    assert_eq!(iter.next().unwrap(), &"c".anahash(&alphabet));
    assert_eq!(iter.next().unwrap(), &AnaValue::empty());
    assert_eq!(iter.next().unwrap(), &"abd".anahash(&alphabet));
    assert_eq!(iter.next().unwrap(), &"ab".anahash(&alphabet));
    assert_eq!(iter.next().unwrap(), &"a".anahash(&alphabet));
    //.. and more
}

#[test]
fn test0203_iterator_recursive_no_empty_leaves() {
    //depth first by default
    let (alphabet, alphabet_size) = get_test_alphabet();
    let anavalue: AnaValue = "abcd".anahash(&alphabet);
    let mut depths: Vec<_> = Vec::new();
    let mut deletions: Vec<AnaValue> = Vec::new();
    for (deletion, depth) in anavalue.iter_recursive(alphabet_size, &SearchParams {
        allow_empty_leaves: false,
        ..Default::default()}) {
       deletions.push(deletion.value.clone());
       depths.push(depth);
    }
    let mut iter = deletions.iter();
    assert_eq!(iter.next().unwrap(), &"abc".anahash(&alphabet));
    assert_eq!(iter.next().unwrap(), &"ab".anahash(&alphabet));
    assert_eq!(iter.next().unwrap(), &"a".anahash(&alphabet));
    assert_eq!(iter.next().unwrap(), &"b".anahash(&alphabet));
    assert_eq!(iter.next().unwrap(), &"ac".anahash(&alphabet));
    assert_eq!(iter.next().unwrap(), &"a".anahash(&alphabet));
    assert_eq!(iter.next().unwrap(), &"c".anahash(&alphabet));
    assert_eq!(iter.next().unwrap(), &"bc".anahash(&alphabet));
    assert_eq!(iter.next().unwrap(), &"b".anahash(&alphabet));
    assert_eq!(iter.next().unwrap(), &"c".anahash(&alphabet));
    assert_eq!(iter.next().unwrap(), &"abd".anahash(&alphabet));
    assert_eq!(iter.next().unwrap(), &"ab".anahash(&alphabet));
    assert_eq!(iter.next().unwrap(), &"a".anahash(&alphabet));
    //.. and more
}

#[test]
fn test0203_iterator_recursive_no_duplicates() {
    //depth first by default
    let (alphabet, alphabet_size) = get_test_alphabet();
    let anavalue: AnaValue = "abcd".anahash(&alphabet);
    let mut depths: Vec<_> = Vec::new();
    let mut deletions: Vec<AnaValue> = Vec::new();
    for (deletion, depth) in anavalue.iter_recursive(alphabet_size, &SearchParams {
        allow_empty_leaves: false,
        allow_duplicates: false,
        ..Default::default()}) {
       deletions.push(deletion.value.clone());
       depths.push(depth);
    }
    let mut iter = deletions.iter();
    assert_eq!(iter.next().unwrap(), &"abc".anahash(&alphabet));
    assert_eq!(iter.next().unwrap(), &"ab".anahash(&alphabet));
    assert_eq!(iter.next().unwrap(), &"a".anahash(&alphabet));
    assert_eq!(iter.next().unwrap(), &"b".anahash(&alphabet));
    assert_eq!(iter.next().unwrap(), &"ac".anahash(&alphabet));
    assert_eq!(iter.next().unwrap(), &"c".anahash(&alphabet));
    assert_eq!(iter.next().unwrap(), &"bc".anahash(&alphabet));
    assert_eq!(iter.next().unwrap(), &"abd".anahash(&alphabet));
    //.. and more
}

#[test]
fn test0203_iterator_recursive_bfs() {
    //depth first by default
    let (alphabet, alphabet_size) = get_test_alphabet();
    let anavalue: AnaValue = "abcd".anahash(&alphabet);
    let mut depths: Vec<_> = Vec::new();
    let mut deletions: Vec<AnaValue> = Vec::new();
    for (deletion, depth) in anavalue.iter_recursive(alphabet_size, &SearchParams {
        breadthfirst: true,
        ..Default::default()}) {
       deletions.push(deletion.value.clone());
       depths.push(depth);
    }
    let mut iter = deletions.iter();
    let mut dpit = depths.iter();
    assert_eq!(iter.next().unwrap(), &"abc".anahash(&alphabet));
    assert_eq!(dpit.next().unwrap(), &1);
    assert_eq!(iter.next().unwrap(), &"abd".anahash(&alphabet));
    assert_eq!(dpit.next().unwrap(), &1);
    assert_eq!(iter.next().unwrap(), &"acd".anahash(&alphabet));
    assert_eq!(dpit.next().unwrap(), &1);
    assert_eq!(iter.next().unwrap(), &"bcd".anahash(&alphabet));
    assert_eq!(dpit.next().unwrap(), &1);

    assert_eq!(iter.next().unwrap(), &"ab".anahash(&alphabet));
    assert_eq!(dpit.next().unwrap(), &2);
    assert_eq!(iter.next().unwrap(), &"ac".anahash(&alphabet));
    assert_eq!(dpit.next().unwrap(), &2);
    assert_eq!(iter.next().unwrap(), &"bc".anahash(&alphabet));
    assert_eq!(dpit.next().unwrap(), &2);

    assert_eq!(iter.next().unwrap(), &"ab".anahash(&alphabet));
    assert_eq!(dpit.next().unwrap(), &2);
    assert_eq!(iter.next().unwrap(), &"ad".anahash(&alphabet));
    assert_eq!(dpit.next().unwrap(), &2);
    assert_eq!(iter.next().unwrap(), &"bd".anahash(&alphabet));
    assert_eq!(dpit.next().unwrap(), &2);

    assert_eq!(iter.next().unwrap(), &"ac".anahash(&alphabet));
    assert_eq!(dpit.next().unwrap(), &2);
    assert_eq!(iter.next().unwrap(), &"ad".anahash(&alphabet));
    assert_eq!(dpit.next().unwrap(), &2);
    assert_eq!(iter.next().unwrap(), &"cd".anahash(&alphabet));
    assert_eq!(dpit.next().unwrap(), &2);

    assert_eq!(iter.next().unwrap(), &"bc".anahash(&alphabet));
    assert_eq!(dpit.next().unwrap(), &2);
    assert_eq!(iter.next().unwrap(), &"bd".anahash(&alphabet));
    assert_eq!(dpit.next().unwrap(), &2);
    assert_eq!(iter.next().unwrap(), &"cd".anahash(&alphabet));
    assert_eq!(dpit.next().unwrap(), &2);

    assert_eq!(iter.next().unwrap(), &"a".anahash(&alphabet));
    assert_eq!(dpit.next().unwrap(), &3);
    assert_eq!(iter.next().unwrap(), &"b".anahash(&alphabet));
    assert_eq!(dpit.next().unwrap(), &3);

    assert_eq!(iter.next().unwrap(), &"a".anahash(&alphabet));
    assert_eq!(dpit.next().unwrap(), &3);
    assert_eq!(iter.next().unwrap(), &"c".anahash(&alphabet));
    assert_eq!(dpit.next().unwrap(), &3);

    //
    //.. and way more duplicates!
}

#[test]
fn test0203_iterator_recursive_bfs_no_duplicates() {
    //depth first by default
    let (alphabet, alphabet_size) = get_test_alphabet();
    let anavalue: AnaValue = "abcd".anahash(&alphabet);
    let mut depths: Vec<_> = Vec::new();
    let mut deletions: Vec<AnaValue> = Vec::new();
    for (deletion, depth) in anavalue.iter_recursive(alphabet_size, &SearchParams {
        breadthfirst: true,
        allow_duplicates: false,
        allow_empty_leaves: false,
        ..Default::default()}) {
       deletions.push(deletion.value.clone());
       depths.push(depth);
    }
    let mut iter = deletions.iter();
    let mut dpit = depths.iter();
    assert_eq!(iter.next().unwrap(), &"abc".anahash(&alphabet));
    assert_eq!(dpit.next().unwrap(), &1);
    assert_eq!(iter.next().unwrap(), &"abd".anahash(&alphabet));
    assert_eq!(dpit.next().unwrap(), &1);
    assert_eq!(iter.next().unwrap(), &"acd".anahash(&alphabet));
    assert_eq!(dpit.next().unwrap(), &1);
    assert_eq!(iter.next().unwrap(), &"bcd".anahash(&alphabet));
    assert_eq!(dpit.next().unwrap(), &1);

    assert_eq!(iter.next().unwrap(), &"ab".anahash(&alphabet));
    assert_eq!(dpit.next().unwrap(), &2);
    assert_eq!(iter.next().unwrap(), &"ac".anahash(&alphabet));
    assert_eq!(dpit.next().unwrap(), &2);
    assert_eq!(iter.next().unwrap(), &"bc".anahash(&alphabet));
    assert_eq!(dpit.next().unwrap(), &2);

    assert_eq!(iter.next().unwrap(), &"ad".anahash(&alphabet));
    assert_eq!(dpit.next().unwrap(), &2);
    assert_eq!(iter.next().unwrap(), &"bd".anahash(&alphabet));
    assert_eq!(dpit.next().unwrap(), &2);

    assert_eq!(iter.next().unwrap(), &"cd".anahash(&alphabet));
    assert_eq!(dpit.next().unwrap(), &2);

    assert_eq!(iter.next().unwrap(), &"a".anahash(&alphabet));
    assert_eq!(dpit.next().unwrap(), &3);
    assert_eq!(iter.next().unwrap(), &"b".anahash(&alphabet));
    assert_eq!(dpit.next().unwrap(), &3);
    assert_eq!(iter.next().unwrap(), &"c".anahash(&alphabet));
    assert_eq!(dpit.next().unwrap(), &3);
    assert_eq!(iter.next().unwrap(), &"d".anahash(&alphabet));
    assert_eq!(dpit.next().unwrap(), &3);
    assert_eq!(iter.next(), None); //all done!
    assert_eq!(dpit.next(), None);
}

#[test]
fn test0203_iterator_recursive_bfs_max_dist() {
    //depth first by default
    let (alphabet, alphabet_size) = get_test_alphabet();
    let anavalue: AnaValue = "abcd".anahash(&alphabet);
    let mut depths: Vec<_> = Vec::new();
    let mut deletions: Vec<AnaValue> = Vec::new();
    for (deletion, depth) in anavalue.iter_recursive(alphabet_size, &SearchParams {
        breadthfirst: true,
        allow_duplicates: false,
        allow_empty_leaves: false,
        max_distance: Some(3),
        ..Default::default()}) {
       deletions.push(deletion.value.clone());
       depths.push(depth);
    }
    let mut iter = deletions.iter();
    let mut dpit = depths.iter();
    assert_eq!(iter.next().unwrap(), &"abc".anahash(&alphabet));
    assert_eq!(dpit.next().unwrap(), &1);
    assert_eq!(iter.next().unwrap(), &"abd".anahash(&alphabet));
    assert_eq!(dpit.next().unwrap(), &1);
    assert_eq!(iter.next().unwrap(), &"acd".anahash(&alphabet));
    assert_eq!(dpit.next().unwrap(), &1);
    assert_eq!(iter.next().unwrap(), &"bcd".anahash(&alphabet));
    assert_eq!(dpit.next().unwrap(), &1);

    assert_eq!(iter.next().unwrap(), &"ab".anahash(&alphabet));
    assert_eq!(dpit.next().unwrap(), &2);
    assert_eq!(iter.next().unwrap(), &"ac".anahash(&alphabet));
    assert_eq!(dpit.next().unwrap(), &2);
    assert_eq!(iter.next().unwrap(), &"bc".anahash(&alphabet));
    assert_eq!(dpit.next().unwrap(), &2);

    assert_eq!(iter.next().unwrap(), &"ad".anahash(&alphabet));
    assert_eq!(dpit.next().unwrap(), &2);
    assert_eq!(iter.next().unwrap(), &"bd".anahash(&alphabet));
    assert_eq!(dpit.next().unwrap(), &2);

    assert_eq!(iter.next().unwrap(), &"cd".anahash(&alphabet));
    assert_eq!(dpit.next().unwrap(), &2);

    assert_eq!(iter.next().unwrap(), &"a".anahash(&alphabet));
    assert_eq!(dpit.next().unwrap(), &3);
    assert_eq!(iter.next().unwrap(), &"b".anahash(&alphabet));
    assert_eq!(dpit.next().unwrap(), &3);
    assert_eq!(iter.next().unwrap(), &"c".anahash(&alphabet));
    assert_eq!(dpit.next().unwrap(), &3);
    assert_eq!(iter.next().unwrap(), &"d".anahash(&alphabet));
    assert_eq!(dpit.next().unwrap(), &3);
    assert_eq!(iter.next(), None); //all done!
    assert_eq!(dpit.next(), None);
}

#[test]
fn test0203_iterator_recursive_bfs_max_dist2() {
    //depth first by default
    let (alphabet, alphabet_size) = get_test_alphabet();
    let anavalue: AnaValue = "abcd".anahash(&alphabet);
    let mut depths: Vec<_> = Vec::new();
    let mut deletions: Vec<AnaValue> = Vec::new();
    for (deletion, depth) in anavalue.iter_recursive(alphabet_size, &SearchParams {
        breadthfirst: true,
        allow_duplicates: false,
        allow_empty_leaves: false,
        max_distance: Some(2),
        ..Default::default()}) {
       deletions.push(deletion.value.clone());
       depths.push(depth);
    }
    let mut iter = deletions.iter();
    let mut dpit = depths.iter();
    assert_eq!(iter.next().unwrap(), &"abc".anahash(&alphabet));
    assert_eq!(dpit.next().unwrap(), &1);
    assert_eq!(iter.next().unwrap(), &"abd".anahash(&alphabet));
    assert_eq!(dpit.next().unwrap(), &1);
    assert_eq!(iter.next().unwrap(), &"acd".anahash(&alphabet));
    assert_eq!(dpit.next().unwrap(), &1);
    assert_eq!(iter.next().unwrap(), &"bcd".anahash(&alphabet));
    assert_eq!(dpit.next().unwrap(), &1);

    assert_eq!(iter.next().unwrap(), &"ab".anahash(&alphabet));
    assert_eq!(dpit.next().unwrap(), &2);
    assert_eq!(iter.next().unwrap(), &"ac".anahash(&alphabet));
    assert_eq!(dpit.next().unwrap(), &2);
    assert_eq!(iter.next().unwrap(), &"bc".anahash(&alphabet));
    assert_eq!(dpit.next().unwrap(), &2);

    assert_eq!(iter.next().unwrap(), &"ad".anahash(&alphabet));
    assert_eq!(dpit.next().unwrap(), &2);
    assert_eq!(iter.next().unwrap(), &"bd".anahash(&alphabet));
    assert_eq!(dpit.next().unwrap(), &2);

    assert_eq!(iter.next().unwrap(), &"cd".anahash(&alphabet));
    assert_eq!(dpit.next().unwrap(), &2);
    assert_eq!(iter.next(), None); //all done!
    assert_eq!(dpit.next(), None);
}

#[test]
fn test0301_normalize_to_alphabet() {
    let (alphabet, _alphabet_size) = get_test_alphabet();
    assert_eq!(&"a".normalize_to_alphabet(&alphabet), &[0]);
    assert_eq!(&"b".normalize_to_alphabet(&alphabet), &[1]);
}

#[test]
fn test0302_levenshtein() {
    let (alphabet, _alphabet_size) = get_test_alphabet();
    assert_eq!(levenshtein(&"a".normalize_to_alphabet(&alphabet), &"a".normalize_to_alphabet(&alphabet),99), Some(0));
    assert_eq!(levenshtein(&"a".normalize_to_alphabet(&alphabet), &"b".normalize_to_alphabet(&alphabet),99), Some(1));
    //substitution
    assert_eq!(levenshtein(&"ab".normalize_to_alphabet(&alphabet), &"ac".normalize_to_alphabet(&alphabet),99), Some(1));
    //insertion
    assert_eq!(levenshtein(&"a".normalize_to_alphabet(&alphabet), &"ab".normalize_to_alphabet(&alphabet),99), Some(1));
    //deletion
    assert_eq!(levenshtein(&"ab".normalize_to_alphabet(&alphabet), &"a".normalize_to_alphabet(&alphabet),99), Some(1));
    //transposition
    assert_eq!(levenshtein(&"ab".normalize_to_alphabet(&alphabet), &"ba".normalize_to_alphabet(&alphabet),99), Some(2));

    assert_eq!(levenshtein(&"abc".normalize_to_alphabet(&alphabet), &"xyz".normalize_to_alphabet(&alphabet),99), Some(3));
}

#[test]
fn test0303_damereau_levenshtein() {
    let (alphabet, _alphabet_size) = get_test_alphabet();
    assert_eq!(damerau_levenshtein(&"a".normalize_to_alphabet(&alphabet), &"a".normalize_to_alphabet(&alphabet),99), Some(0));
    assert_eq!(damerau_levenshtein(&"a".normalize_to_alphabet(&alphabet), &"b".normalize_to_alphabet(&alphabet),99), Some(1));
    //substitution
    assert_eq!(damerau_levenshtein(&"ab".normalize_to_alphabet(&alphabet), &"ac".normalize_to_alphabet(&alphabet),99), Some(1));
    //insertion
    assert_eq!(damerau_levenshtein(&"a".normalize_to_alphabet(&alphabet), &"ab".normalize_to_alphabet(&alphabet),99), Some(1));
    //deletion
    assert_eq!(damerau_levenshtein(&"ab".normalize_to_alphabet(&alphabet), &"a".normalize_to_alphabet(&alphabet),99), Some(1));
    //transposition (this is where the difference with normal levenshtein is)
    assert_eq!(damerau_levenshtein(&"ab".normalize_to_alphabet(&alphabet), &"ba".normalize_to_alphabet(&alphabet),99), Some(1));

    assert_eq!(damerau_levenshtein(&"abc".normalize_to_alphabet(&alphabet), &"xyz".normalize_to_alphabet(&alphabet),99), Some(3));
}

#[test]
fn test0304_lcslen() {
    let (alphabet, _alphabet_size) = get_test_alphabet();
    assert_eq!(longest_common_substring_length(&"test".normalize_to_alphabet(&alphabet), &"testable".normalize_to_alphabet(&alphabet)), 4);
    assert_eq!(longest_common_substring_length(&"fasttest".normalize_to_alphabet(&alphabet), &"testable".normalize_to_alphabet(&alphabet)), 4);
    assert_eq!(longest_common_substring_length(&"abcdefhij".normalize_to_alphabet(&alphabet), &"def".normalize_to_alphabet(&alphabet)), 3);
    assert_eq!(longest_common_substring_length(&"def".normalize_to_alphabet(&alphabet), &"abcdefhij".normalize_to_alphabet(&alphabet)), 3);
}

#[test]
fn test0304_prefixlen() {
    let (alphabet, _alphabet_size) = get_test_alphabet();
    assert_eq!(common_prefix_length(&"test".normalize_to_alphabet(&alphabet), &"testable".normalize_to_alphabet(&alphabet)), 4);
    assert_eq!(common_prefix_length(&"testable".normalize_to_alphabet(&alphabet), &"test".normalize_to_alphabet(&alphabet)), 4);
    assert_eq!(common_prefix_length(&"fasttest".normalize_to_alphabet(&alphabet), &"testable".normalize_to_alphabet(&alphabet)), 0);
    assert_eq!(common_prefix_length(&"fasttest".normalize_to_alphabet(&alphabet), &"test".normalize_to_alphabet(&alphabet)), 0);
}

#[test]
fn test0304_suffixlen() {
    let (alphabet, _alphabet_size) = get_test_alphabet();
    assert_eq!(common_suffix_length(&"test".normalize_to_alphabet(&alphabet), &"testable".normalize_to_alphabet(&alphabet)), 0);
    assert_eq!(common_suffix_length(&"testable".normalize_to_alphabet(&alphabet), &"test".normalize_to_alphabet(&alphabet)), 0);
    assert_eq!(common_suffix_length(&"fasttest".normalize_to_alphabet(&alphabet), &"testable".normalize_to_alphabet(&alphabet)), 0);
    assert_eq!(common_suffix_length(&"fasttest".normalize_to_alphabet(&alphabet), &"test".normalize_to_alphabet(&alphabet)), 4);
}


#[test]
fn test0400_model_load() {
    let (alphabet, _alphabet_size) = get_test_alphabet();
    let model = VariantModel::new_with_alphabet(alphabet, Weights::default(), true);
}

#[test]
fn test0401_model_build() {
    let (alphabet, _alphabet_size) = get_test_alphabet();
    let mut model = VariantModel::new_with_alphabet(alphabet, Weights::default(), true);
    let lexicon: &[&str] = &["rites","tiers", "tires","tries","tyres","rides","brides","dire"];
    for text in lexicon.iter() {
        model.add_to_vocabulary(text,None,None, 0);
    }
    model.build();
    assert!(model.has(&"rites"));
    for text in lexicon.iter() {
        assert!(model.has(text));
        assert!(model.get(text).is_some());
    }
    assert!(!model.has(&"unknown"));
    assert!(model.get(&"unknown").is_none());
}

#[test]
fn test0402_model_anagrams() {
    let (alphabet, _alphabet_size) = get_test_alphabet();
    let mut model = VariantModel::new_with_alphabet(alphabet, Weights::default(), true);
    let lexicon: &[&str] = &["rites","tiers", "tires","tries","tyres","rides","brides","dire"];
    for text in lexicon.iter() {
        model.add_to_vocabulary(text,None,None, 0);
    }
    model.build();
    assert!(model.has(&"rites"));
    assert_eq!(model.get_anagram_instances(&"rites").iter().map(|item| item.text.clone()).collect::<Vec<String>>(),
             &["rites","tiers","tires","tries"]
    );
}

#[test]
fn test0403_model_anagrams() {
    let (alphabet, _alphabet_size) = get_test_alphabet();
    let mut model = VariantModel::new_with_alphabet(alphabet, Weights::default(), true);
    let lexicon: &[&str] = &["rites","tiers", "tires","tries","tyres","rides","brides","dire"];
    for text in lexicon.iter() {
        model.add_to_vocabulary(text,None,None, 0);
    }
    model.build();
    model.find_variants("rite", 2, 2, 10, 0.0, StopCriterion::Exhaustive, None);
}

#[test]
fn test0404_score_test() {
    let (alphabet, _alphabet_size) = get_test_alphabet();
    let mut model = VariantModel::new_with_alphabet(alphabet, Weights::default(), true);
    let lexicon: &[&str] = &["huis","huls"];
    for text in lexicon.iter() {
        model.add_to_vocabulary(text,None,None, 0);
    }
    model.build();
    let results = model.find_variants("huys", 2, 2, 10, 0.0, StopCriterion::Exhaustive, None);
    //results are a bit indeterministic due to sort_unstable
    //(order of equal-scoring elements is not fixed)
    //we just check if we get two results with the same score
    assert_eq!( results.len(), 2);
    assert_ne!( results.get(0).unwrap().0, results.get(1).unwrap().0 );
    assert_eq!( results.get(0).unwrap().1, results.get(1).unwrap().1 );
}


#[test]
fn test0501_confusable_found_in() {
    let confusable =  Confusable::new("-[y]+[i]",1.1).expect("valid script");
    eprintln!("confusable: {:?}", confusable);
    let huis_script = sesdiff::shortest_edit_script("huys","huis", false, false, false);
    eprintln!("huis_script: {:?}", huis_script);
    let huls_script = sesdiff::shortest_edit_script("huys","huls", false, false, false);
    eprintln!("huls_script: {:?}", huls_script);
    assert!(confusable.found_in(&huis_script), "confusable should be found in huys->huis");
    assert!(!confusable.found_in(&huls_script),"confusable should not be found in huys->huls");
}

#[test]
fn test0502_confusable_test() {
    let (alphabet, _alphabet_size) = get_test_alphabet();
    let mut model = VariantModel::new_with_alphabet(alphabet, Weights::default(), true);
    let lexicon: &[&str] = &["huis","huls"];
    for text in lexicon.iter() {
        model.add_to_vocabulary(text,None,None, 0);
    }
    model.add_to_confusables("-[y]+[i]",1.1).expect("added to confusables");
    model.build();
    let results = model.find_variants("huys", 2, 2, 10, 0.0, StopCriterion::Exhaustive, None);
    assert_eq!( model.decoder.get(results.get(0).unwrap().0 as usize).unwrap().text, "huis");
    assert_eq!( model.decoder.get(results.get(1).unwrap().0 as usize).unwrap().text, "huls");
    assert!( results.get(0).unwrap().1 > results.get(1).unwrap().1, "score of huis should be greater than that of huls" );
}

#[test]
fn test0503_confusable_test2() {
    let (alphabet, _alphabet_size) = get_test_alphabet();
    let mut model = VariantModel::new_with_alphabet(alphabet, Weights::default(), true);
    let lexicon: &[&str] = &["huis","huls"];
    for text in lexicon.iter() {
        model.add_to_vocabulary(text,None,None, 0);
    }
    model.add_to_confusables("-[y]+[i]",1.1).expect("added to confusables");
    model.build();
    let results = model.find_variants("Huys", 2, 2, 10, 0.0, StopCriterion::Exhaustive, None);
    assert_eq!( model.decoder.get(results.get(0).unwrap().0 as usize).unwrap().text, "huis");
    assert_eq!( model.decoder.get(results.get(1).unwrap().0 as usize).unwrap().text, "huls");
    assert!( results.get(0).unwrap().1 > results.get(1).unwrap().1, "score of huis should be greater than that of huls" );
}

#[test]
fn test0504_confusable_nomatch() {
    let (alphabet, _alphabet_size) = get_test_alphabet();
    let mut model = VariantModel::new_with_alphabet(alphabet, Weights::default(), false);
    let lexicon: &[&str] = &["huis","huls"];
    for text in lexicon.iter() {
        model.add_to_vocabulary(text,None,None, 0);
    }
    model.add_to_confusables("-[y]+[p]",1.1).expect("added to confusables");
    model.build();
    let results = model.find_variants("Huys", 2, 2, 10, 0.0, StopCriterion::Exhaustive, None);
    assert_eq!( results.len() , 2 );
    assert_eq!( results.get(0).unwrap().1,results.get(1).unwrap().1, "score of huis should be equal to that of huls" );
}
