//! Unless otherwise noted, the names used are the section headers from the datasheet without the
//! word "register". Where appropriate, there will also be a corresponding `*Value` type with valid
//! values to write to that register.
use num_enum::{IntoPrimitive, TryFromPrimitive};

/// The I2C address for the Grid-EYE sensor.
///
/// The names chosen are *not* because of the actual address values, but because of what the
/// address selection pin is connected to (ground or VDD).
#[derive(Clone, Copy, Debug, PartialEq, IntoPrimitive, TryFromPrimitive)]
#[repr(u8)]
pub enum Address {
    Low = 0x68,
    High = 0x69,
}

#[cfg(test)]
mod address_test {
    use super::Address;
    use core::convert::TryFrom;

    #[test]
    fn try_low() {
        let low = Address::try_from(0x68);
        assert!(low.is_ok());
        assert_eq!(low.unwrap(), Address::Low);
    }

    #[test]
    fn try_high() {
        let high = Address::try_from(0x69);
        assert!(high.is_ok());
        assert_eq!(high.unwrap(), Address::High);
    }

    #[test]
    fn try_undefined() {
        let zero = Address::try_from(0);
        assert!(zero.is_err());
    }
}

/// The registers used to access the Grid-EYE.
#[derive(Clone, Copy, Debug, PartialEq, IntoPrimitive, TryFromPrimitive)]
#[repr(u8)]
pub enum Register {
    /// See [PowerControlValue] for details on values
    /// Read/write.
    PowerControl = 0x00,

    /// See [ResetValue] for details on values.
    /// Write-only.
    Reset = 0x01,

    /// See [FrameRateValue] for details on values.
    /// Read/write.
    FrameRate = 0x02,

    /// See [InterruptControlValue] for details on values.
    /// Read/write.
    InterruptControl = 0x03,

    /// The status flags for the device. The flags will remain set until cleared by writing to
    /// [StatusClear].
    /// Read-only.
    Status = 0x04,

    /// Clear one or both of the status flags for the device.
    /// Write-only.
    StatusClear = 0x05,

    /// Whether or not the pixel values should be overaged or not.
    /// Read/write.
    Average = 0x07,

    /// A semi-documented register that is written to when changing the moving average mode.
    /// Write-only.
    AverageData = 0x1F,

    /// The upper temperature limit for triggering an interrupt. This is in the
    /// [pixel temperature](#pixel_temperatures) format.
    /// Read/write.
    InterruptLevelHighLower = 0x08,

    /// The lower temperature limit for triggering an interrupt. This is in the
    /// [pixel temperature](#pixel_temperatures) format.
    /// Read/write.
    InterruptLevelLowLower = 0x0A,

    /// The hysteresis limit for triggering an interrupt. The datasheet isn't very clear as to how
    /// this value is used. This is in the [pixel temperature](#pixel_temperatures) format.
    /// Read/write.
    InterruptHysteresisLower = 0x0C,

    /// The sensor has an on-board thermistor useful for for calibrating the output of the camera.
    /// This is in the [thermistor temperature](#thermistor_temperature) format.
    /// Read-only.
    ThermistorLower = 0x0E,

    /// The first register of the interrupt table. It continues up through `0x17` with each
    /// register representing a bitfield of whether or not that pixel in the image has triggered an
    /// interrupt. The first bit of the first register is pixel 1, with the rest of the pixels
    /// following in order from there (see [identifying pixels](grideye#identifying_pixels)\).
    /// Read-only.
    InterruptTableStart = 0x10,

    /// The lower bits of the temperature of pixel 1. The upper bits are in the register following
    /// this one, with the lower bits for pixel 2 after that and so on up through `0xFF`.
    /// This is in the [pixel temperature](#pixel_temperature) format.
    PixelTemperatureStart = 0x80,
}

/// On power on, the device is in [normal](PowerControlValue::Normal) mode. There are no
/// restrictions for leaving normal mode. In [sleep](PowerControlValue::Sleep) mode, the only
/// valid mode to change to is normal mode. When leaving sleep mode the value of other
/// registers are undefined, and must be reset with this procedure:
///
/// 1. Set [Register::PowerControl] to [`Normal`](PowerControlValue::Normal).
/// 2. Wait 50 milliseconds.
/// 3. Set [Register::Reset] to [`Initial`](ResetValue::Initial).
/// 4. Wait 2 milliseconds.
/// 5. Set [Register::Reset] to [`Flag`](ResetValue::Flag).
///
/// The datasheet has a nice state diagram summarizing this.
#[derive(Clone, Copy, Debug, PartialEq, IntoPrimitive, TryFromPrimitive)]
#[repr(u8)]
pub enum PowerControlValue {
    /// Normal operation.
    Normal = 0x00,

    /// Sleep mode.
    Sleep = 0x10,

    /// Standby mode with the pixel temperature only being updated every 60 seconds.
    Standby60 = 0x20,

    /// Standby mode with the pixel temperature only being updated every 10 seconds.
    Standby10 = 0x21,
}

#[cfg(test)]
mod power_control_value_tests {
    use super::PowerControlValue;
    use core::convert::TryFrom;

    #[test]
    fn try_normal() {
        let normal = PowerControlValue::try_from(0x0);
        assert!(normal.is_ok());
        assert_eq!(normal.unwrap(), PowerControlValue::Normal);
    }

    #[test]
    fn try_sleep() {
        let sleep = PowerControlValue::try_from(0x10);
        assert!(sleep.is_ok());
        assert_eq!(sleep.unwrap(), PowerControlValue::Sleep);
    }

    #[test]
    fn try_standby_60() {
        let standby_60 = PowerControlValue::try_from(0x20);
        assert!(standby_60.is_ok());
        assert_eq!(standby_60.unwrap(), PowerControlValue::Standby60);
    }

    #[test]
    fn try_standby_10() {
        let standby_10 = PowerControlValue::try_from(0x21);
        assert!(standby_10.is_ok());
        assert_eq!(standby_10.unwrap(), PowerControlValue::Standby10);
    }

    #[test]
    fn unknown() {
        let unknown = PowerControlValue::try_from(u8::MAX);
        assert!(unknown.is_err());
    }
}

/// There are two kinds of soft reset available by writing to [Register::Reset]. Only use
/// [ResetValue::Initial] when changing from [sleep](PowerControlValue::sleep) to
/// [normal](PowerControlValue::Normal) mode. The datasheet emphasizes that other values are not to
/// be used.
#[derive(Clone, Copy, Debug, PartialEq, IntoPrimitive, TryFromPrimitive)]
#[repr(u8)]
pub enum ResetValue {
    /// Resets [Register::Status] and the interrupt table.
    Flag = 0x30,

    /// A [Flag] reset as well as re-reading sensor adjustment values.
    Initial = 0x3F,
}

#[cfg(test)]
mod reset_value_tests {
    use super::ResetValue;
    use core::convert::TryFrom;

    #[test]
    fn try_flag() {
        let flag = ResetValue::try_from(0x30);
        assert!(flag.is_ok());
        assert_eq!(flag.unwrap(), ResetValue::Flag);
    }

    #[test]
    fn try_initial() {
        let initial = ResetValue::try_from(0x3F);
        assert!(initial.is_ok());
        assert_eq!(initial.unwrap(), ResetValue::Initial);
    }

    #[test]
    fn try_unknown() {
        let unknown = ResetValue::try_from(0);
        assert!(unknown.is_err());
    }
}

/// The frame rate of the camera. Internally, the sensor always runs at 10 FPS, but in 1 FPS mode
/// it is averaging the frames internally to reduce noise.
/// The names are slightly different from those used in the datasheet to conform to Rust's
/// identifier rules.
#[derive(Clone, Copy, Debug, PartialEq, IntoPrimitive, TryFromPrimitive)]
#[repr(u8)]
pub enum FrameRateValue {
    /// 1 FPS.
    Fps1 = 1,

    /// 10 FPS.
    #[num_enum(alternatives = [10])]
    Fps10 = 0,
}

#[cfg(test)]
mod frame_rate_tests {
    use super::FrameRateValue;
    use core::convert::TryFrom;

    #[test]
    fn try_1() {
        let fps = FrameRateValue::try_from(1);
        assert!(fps.is_ok());
        assert_eq!(fps.unwrap(), FrameRateValue::Fps1);
    }

    #[test]
    fn try_0() {
        let fps = FrameRateValue::try_from(0);
        assert!(fps.is_ok());
        assert_eq!(fps.unwrap(), FrameRateValue::Fps10);
    }

    #[test]
    fn try_10() {
        let fps = FrameRateValue::try_from(10);
        assert!(fps.is_ok());
        assert_eq!(fps.unwrap(), FrameRateValue::Fps10);
    }

    #[test]
    fn try_unknown() {
        let unknown = FrameRateValue::try_from(20);
        assert!(unknown.is_err());
    }
}

/// How interrupts should be triggered.
#[derive(Clone, Copy, Debug, PartialEq, IntoPrimitive)]
#[repr(u8)]
pub enum InterruptControlMode {
    /// Interrupt triggered when an absolute temperature threshold is crossed.
    Absolute = 0x2,

    /// Interrupt triggered when the change in temperature has exceeded a threshold.
    Difference = 0x0,
}

impl From<u8> for InterruptControlMode {
    fn from(value: u8) -> Self {
        match value & 0x2 {
            0x2 => Self::Absolute,
            _ => Self::Difference,
        }
    }
}

impl Default for InterruptControlMode {
    // The initial value for the interrupt control register is 0x00
    fn default() -> Self {
        Self::Difference
    }
}

#[cfg(test)]
mod interrupt_control_mode_tests {
    use super::InterruptControlMode;

    #[test]
    fn from_int() {
        assert_eq!(
            InterruptControlMode::from(0),
            InterruptControlMode::Difference
        );
        assert_eq!(
            InterruptControlMode::from(1),
            InterruptControlMode::Difference
        );
        assert_eq!(
            InterruptControlMode::from(2),
            InterruptControlMode::Absolute
        );
    }

    #[test]
    fn default() {
        assert_eq!(
            InterruptControlMode::default(),
            InterruptControlMode::Difference
        );
    }
}

/// A bitfield for controlling interrupts on the device.
#[derive(Clone, Copy, Debug, Default, PartialEq)]
pub struct InterruptControlValue {
    /// Controls when interrupts are generated.
    mode: InterruptControlMode,

    /// If interrupts are enabled. This setting is only applicable when in
    /// normal or one of the standby [modes][PowerControlValue].
    enabled: bool,
}

impl InterruptControlValue {
    /// Get the current interrupt mode.
    pub fn mode(&self) -> InterruptControlMode {
        self.mode
    }

    /// Set the interrupt mode.
    pub fn set_mode(&mut self, mode: InterruptControlMode) {
        self.mode = mode
    }

    /// Test if interrupts are enabled.
    pub fn enabled(&self) -> bool {
        self.enabled
    }

    /// Enable interrupts.
    pub fn enable(&mut self) {
        self.enabled = true
    }

    /// Disable interrupts.
    pub fn disable(&mut self) {
        self.enabled = false
    }
}

impl From<u8> for InterruptControlValue {
    fn from(value: u8) -> Self {
        /* There are two bits relevant to the interrupt control, 0 and 1.
         * Bit 0 controls whether or not interrupts are enabled
         * Bit 1 controls the interrupt trigger mode.
         */
        Self {
            mode: InterruptControlMode::from(value),
            enabled: (value & 0x1) == 1,
        }
    }
}

impl From<InterruptControlValue> for bool {
    fn from(value: InterruptControlValue) -> Self {
        value.enabled
    }
}

impl From<InterruptControlValue> for u8 {
    fn from(value: InterruptControlValue) -> Self {
        (value.mode as u8) | (value.enabled as u8)
    }
}

#[cfg(test)]
mod interrupt_control_value_tests {
    use super::{InterruptControlMode, InterruptControlValue};

    #[test]
    fn from_byte() {
        let disabled_difference = InterruptControlValue::from(0x0);
        assert_eq!(disabled_difference.mode(), InterruptControlMode::Difference);
        assert!(!disabled_difference.enabled());
        let enabled_difference = InterruptControlValue::from(0x1);
        assert_eq!(enabled_difference.mode(), InterruptControlMode::Difference);
        assert!(enabled_difference.enabled());
        let disabled_absolute = InterruptControlValue::from(0x2);
        assert_eq!(disabled_absolute.mode(), InterruptControlMode::Absolute);
        assert!(!disabled_absolute.enabled());
        let enabled_absolute = InterruptControlValue::from(0x3);
        assert_eq!(enabled_absolute.mode(), InterruptControlMode::Absolute);
        assert!(enabled_absolute.enabled());
    }

    // Test that setting bits outside of the defined values are ignored
    #[test]
    fn from_byte_ignore_extra() {
        let disabled_difference = InterruptControlValue::from(0x10);
        assert_eq!(disabled_difference.mode(), InterruptControlMode::Difference);
        assert!(!disabled_difference.enabled());
        let enabled_difference = InterruptControlValue::from(0x11);
        assert_eq!(enabled_difference.mode(), InterruptControlMode::Difference);
        assert!(enabled_difference.enabled());
        let disabled_absolute = InterruptControlValue::from(0x12);
        assert_eq!(disabled_absolute.mode(), InterruptControlMode::Absolute);
        assert!(!disabled_absolute.enabled());
        let enabled_absolute = InterruptControlValue::from(0x13);
        assert_eq!(enabled_absolute.mode(), InterruptControlMode::Absolute);
        assert!(enabled_absolute.enabled());
    }

    #[test]
    fn to_bool() {
        // NOTE: there isn't a public constructor besides the [From] traits, so use that.
        let disabled = InterruptControlValue::from(0x0);
        assert!(!bool::from(disabled));
        let enabled = InterruptControlValue::from(0x1);
        assert!(bool::from(enabled));
    }

    #[test]
    fn to_byte() {
        // Just round-trip byte values.
        for byte in 0..4 {
            let value = InterruptControlValue::from(byte);
            assert_eq!(byte, u8::from(value));
        }
    }

    #[test]
    fn set_enabled() {
        let mut value = InterruptControlValue::default();
        // Default is to be disabled
        assert!(!value.enabled());
        // Disabling something already disabled doesn't enable it.
        value.disable();
        assert!(!value.enabled());
        // Disabled -> Enabled
        value.enable();
        assert!(value.enabled());
        // Enabling something already enabled doesn't disable it.
        value.enable();
        assert!(value.enabled());
        // Enabled -> Disabled
        value.disable();
        assert!(!value.enabled());
    }

    #[test]
    fn set_mode() {
        let mut value = InterruptControlValue::default();
        // Default is difference mode
        assert_eq!(value.mode(), InterruptControlMode::Difference);
        // Difference -> Absolute
        value.set_mode(InterruptControlMode::Absolute);
        assert_eq!(value.mode(), InterruptControlMode::Absolute);
        // Absolute -> Difference
        value.set_mode(InterruptControlMode::Difference);
        assert_eq!(value.mode(), InterruptControlMode::Difference);
    }
}

#[derive(Clone, Copy, Debug, PartialEq)]
#[repr(u8)]
enum StatusBits {
    TemperatureOverflow = 0x4,
    Interrupt = 0x2,
}

/// A bitfield for the status flags of the device. This value is read from [Register::Status] and
/// written to [Register::StatusClear].
#[derive(Clone, Copy, Debug, Default, PartialEq)]
pub struct StatusValue {
    /// Set when the analog to digital conversion for the camera sensor overflows (in other words,
    /// the camera sees something too hot).
    temperature_overflow: bool,

    /// Set when an interrupt has fired.
    interrupt: bool,
}

impl StatusValue {
    pub fn new(overflow: bool, interrupt: bool) -> Self {
        Self {
            temperature_overflow: overflow,
            interrupt,
        }
    }

    pub fn temperature_overflow(&self) -> bool {
        self.temperature_overflow
    }
    pub fn interrupt(&self) -> bool {
        self.interrupt
    }
}

impl From<u8> for StatusValue {
    fn from(value: u8) -> Self {
        // The two bits we're concerned with are 1 and 2 (still 0-indexed).
        // Bit 1 is the interrupt flag, and bit 2 is the overflow flag.
        let temperature_overflow = (value & StatusBits::TemperatureOverflow as u8)
            == StatusBits::TemperatureOverflow as u8;
        let interrupt = (value & StatusBits::Interrupt as u8) == StatusBits::Interrupt as u8;
        Self {
            temperature_overflow,
            interrupt,
        }
    }
}

impl From<StatusValue> for u8 {
    fn from(value: StatusValue) -> Self {
        let mut bitfield = 0u8;
        if value.temperature_overflow {
            bitfield |= StatusBits::TemperatureOverflow as u8;
        }
        if value.interrupt {
            bitfield |= StatusBits::Interrupt as u8;
        }
        bitfield
    }
}

#[cfg(test)]
mod test_status_value {
    use super::{StatusBits, StatusValue};

    #[test]
    fn create_and_access() {
        for overflow in [true, false].iter() {
            for interrupt in [true, false].iter() {
                let value = StatusValue::new(*overflow, *interrupt);
                assert_eq!(value.temperature_overflow(), *overflow);
                assert_eq!(value.interrupt(), *interrupt);
            }
        }
    }

    #[test]
    fn byte_no_flags() {
        let value = StatusValue::from(0x0);
        assert!(!value.temperature_overflow());
        assert!(!value.interrupt());
        assert_eq!(u8::from(value), 0x0);
    }

    #[test]
    fn byte_only_overflow() {
        let value = StatusValue::from(StatusBits::TemperatureOverflow as u8);
        assert!(value.temperature_overflow());
        assert!(!value.interrupt());
        assert_eq!(u8::from(value), StatusBits::TemperatureOverflow as u8);
    }

    #[test]
    fn byte_only_interrupt() {
        let value = StatusValue::from(StatusBits::Interrupt as u8);
        assert!(!value.temperature_overflow());
        assert!(value.interrupt());
        assert_eq!(u8::from(value), StatusBits::Interrupt as u8);
    }

    #[test]
    fn byte_both_flags() {
        let both_bits = StatusBits::Interrupt as u8 | StatusBits::TemperatureOverflow as u8;
        let value = StatusValue::from(both_bits);
        assert!(value.temperature_overflow());
        assert!(value.interrupt());
        assert_eq!(u8::from(value), both_bits);
    }

    #[test]
    fn ignore_undefined() {
        let all_set = StatusValue::from(0xF);
        assert!(all_set.temperature_overflow());
        assert!(all_set.interrupt());
        assert_eq!(
            u8::from(all_set),
            StatusBits::Interrupt as u8 | StatusBits::TemperatureOverflow as u8
        );
        let none_set = StatusValue::from(0x9);
        assert!(!none_set.temperature_overflow());
        assert!(!none_set.interrupt());
        assert_eq!(u8::from(none_set), 0x0);
    }
}

/// Value for [Register::Average], controlling whether or not the values presented by the sensor
/// are processed as a moving average of previous frames or not. Frankly the documentation of this
/// feature is lacking, and it's unclear to me what it actually does while the different frame
/// rates are set.
#[derive(Clone, Copy, Debug, PartialEq, IntoPrimitive)]
#[repr(u8)]
pub enum AverageValue {
    Disabled = 0x00,
    Enabled = 0x20,
}

impl From<bool> for AverageValue {
    fn from(value: bool) -> Self {
        if value {
            Self::Enabled
        } else {
            Self::Disabled
        }
    }
}

impl From<u8> for AverageValue {
    fn from(value: u8) -> Self {
        match value & 0x20 {
            0x20 => Self::Enabled,
            _ => Self::Disabled,
        }
    }
}

impl From<AverageValue> for bool {
    fn from(value: AverageValue) -> Self {
        value == AverageValue::Enabled
    }
}

impl Default for AverageValue {
    fn default() -> Self {
        Self::Disabled
    }
}

#[cfg(test)]
mod average_tests {
    use super::AverageValue;

    #[test]
    fn from_bool() {
        assert_eq!(AverageValue::from(false), AverageValue::Disabled);
        assert_eq!(AverageValue::from(true), AverageValue::Enabled);
    }

    #[test]
    fn from_byte() {
        // First the two expected values
        assert_eq!(AverageValue::from(0x0), AverageValue::Disabled);
        assert_eq!(AverageValue::from(0x20), AverageValue::Enabled);
        // And that undefined bits are ignored
        assert_eq!(AverageValue::from(0xF), AverageValue::Disabled);
        assert_eq!(AverageValue::from(0xFF), AverageValue::Enabled);
    }

    #[test]
    fn to_bool() {
        assert!(!bool::from(AverageValue::Disabled));
        assert!(bool::from(AverageValue::Enabled));
    }
}
