use std::io::{self, BufRead};

use base64::encode;
use clap::Parser;
use console::style;
use reqwest::ClientBuilder;
use serde::{Deserialize, Serialize};
use serde_json::{self, json};

#[derive(Parser, Debug)]
#[clap(author, version, about, long_about = None)]
struct Args {
    /// Show additional output
    #[clap(short, long)]
    verbose: bool,

    /// Output as json
    #[clap(short, long)]
    json: bool,
}

#[derive(Default, Debug, Clone, PartialEq, Serialize, Deserialize)]
pub struct Response {
    pub packager: String,
    pub packages: Vec<Package>,
    pub probability: f64,
}

#[derive(Default, Debug, Clone, PartialEq, Serialize, Deserialize)]
pub struct Package {
    pub homepage: Option<String>,
    pub name: String,
    pub package: String,
    pub summary: String,
    pub updated_at: String,
    pub url: String,
    pub version: String,
}

#[tokio::main]
async fn main() -> Result<(), Box<dyn std::error::Error>> {
    let args = Args::parse();
    let input = read_all_lines_from_stdin().unwrap();

    let client = ClientBuilder::new()
        .user_agent("Alpakr CLI (rust)")
        .build()
        .unwrap();

    let encoded = encode(&input.trim().to_string());
    let payload = json!({ "fragment": encoded });
    if args.verbose {
        eprintln!("payload = {}", payload);
    }
    let response = client
        .post("https://api.alpakr.code.boutique/v1/lookup")
        .header("Content-Type", "application/json")
        .body(payload.to_string())
        .send()
        .await?;

    let text = response.text().await.unwrap();
    if args.verbose {
        eprintln!("status response = {:?}", text);
    }

    let response: Response = serde_json::from_str(&text)?;

    if args.json {
        println!("{}", serde_json::to_string_pretty(&response)?);
    } else {
        for package in response.packages {
            println!(
                "{} -> {}",
                style(package.name).green(),
                style(package.summary)
            );
            println!("{}", style(package.url).dim());
            println!("");
        }
    }

    Ok(())
}

fn read_all_lines_from_stdin() -> io::Result<String> {
    let stdin = io::stdin();
    let mut lines = String::new();
    for line in stdin.lock().lines() {
        lines.push_str(&line?);
        lines.push_str("\n");
    }
    Ok(lines.trim().trim().to_owned())
}
