/*!
  This module contains all known tarantool constants.
*/

use num_derive::{FromPrimitive, ToPrimitive};

/**
  Every error tarantool code is ERROR_BITMASK | error_number.
*/
pub const ERROR_BITMASK: isize = 1 << 15;

/**
  It represents all known tarantool response codes.

  It also implements num_derive::{FromPrimitive, ToPrimitive},
  so you can also convert it to int types.
*/
#[derive(
  Debug, Clone, Copy,
  PartialEq, Eq,
  FromPrimitive, ToPrimitive,
)]
pub enum Code {
  Ok = 0,

  ErrorUnknown                        = ERROR_BITMASK,
  ErrorIllegalParams                  = ERROR_BITMASK | 1,
  ErrorMemoryIssue                    = ERROR_BITMASK | 2,
  ErrorTupleFound                     = ERROR_BITMASK | 3,
  ErrorTupleNotFound                  = ERROR_BITMASK | 4,
  ErrorUnsupported                    = ERROR_BITMASK | 5,
  ErrorNonmaster                      = ERROR_BITMASK | 6,
  ErrorReadonly                       = ERROR_BITMASK | 7,
  ErrorInjection                      = ERROR_BITMASK | 8,
  ErrorCreateSpace                    = ERROR_BITMASK | 9,
  ErrorSpaceExists                    = ERROR_BITMASK | 10,
  ErrorDropSpace                      = ERROR_BITMASK | 11,
  ErrorAlterSpace                     = ERROR_BITMASK | 12,
  ErrorIndexType                      = ERROR_BITMASK | 13,
  ErrorModifyIndex                    = ERROR_BITMASK | 14,
  ErrorLastDrop                       = ERROR_BITMASK | 15,
  ErrorTupleFormatLimit               = ERROR_BITMASK | 16,
  ErrorDropPrimaryKey                 = ERROR_BITMASK | 17,
  ErrorKeyPartType                    = ERROR_BITMASK | 18,
  ErrorExactMatch                     = ERROR_BITMASK | 19,
  ErrorInvalidMsgpack                 = ERROR_BITMASK | 20,
  ErrorProcRet                        = ERROR_BITMASK | 21,
  ErrorTupleNotArray                  = ERROR_BITMASK | 22,
  ErrorFieldType                      = ERROR_BITMASK | 23,
  ErrorIndexPartTypeMismatch          = ERROR_BITMASK | 24,
  ErrorUpdateSplice                   = ERROR_BITMASK | 25,
  ErrorUpdateArgType                  = ERROR_BITMASK | 26,
  ErrorFormatMismatchIndexPart        = ERROR_BITMASK | 27,
  ErrorUnknownUpdateOp                = ERROR_BITMASK | 28,
  ErrorUpdateField                    = ERROR_BITMASK | 29,
  ErrorFunctionTxActive               = ERROR_BITMASK | 30,
  ErrorKeyPartCount                   = ERROR_BITMASK | 31,
  ErrorProcLua                        = ERROR_BITMASK | 32,
  ErrorNoSuchProc                     = ERROR_BITMASK | 33,
  ErrorNoSuchTrigger                  = ERROR_BITMASK | 34,
  ErrorNoSuchIndexID                  = ERROR_BITMASK | 35,
  ErrorNoSuchSpace                    = ERROR_BITMASK | 36,
  ErrorNoSuchFieldNo                  = ERROR_BITMASK | 37,
  ErrorExactFieldCount                = ERROR_BITMASK | 38,
  ErrorFieldMissing                   = ERROR_BITMASK | 39,
  ErrorWalIo                          = ERROR_BITMASK | 40,
  ErrorMoreThanOneTuple               = ERROR_BITMASK | 41,
  ErrorAccessDenied                   = ERROR_BITMASK | 42,
  ErrorCreateUser                     = ERROR_BITMASK | 43,
  ErrorDropUser                       = ERROR_BITMASK | 44,
  ErrorNoSuchUser                     = ERROR_BITMASK | 45,
  ErrorUserExists                     = ERROR_BITMASK | 46,
  ErrorPasswordMismatch               = ERROR_BITMASK | 47,
  ErrorUnknownRequestType             = ERROR_BITMASK | 48,
  ErrorUnknownSchemaObject            = ERROR_BITMASK | 49,
  ErrorCreateFunction                 = ERROR_BITMASK | 50,
  ErrorNoSuchFunction                 = ERROR_BITMASK | 51,
  ErrorFunctionExists                 = ERROR_BITMASK | 52,
  ErrorBeforeReplaceRet               = ERROR_BITMASK | 53,
  ErrorMultistatementTransaction      = ERROR_BITMASK | 54,
  ErrorTriggerExists                  = ERROR_BITMASK | 55,
  ErrorUserMax                        = ERROR_BITMASK | 56,
  ErrorNoSuchEngine                   = ERROR_BITMASK | 57,
  ErrorReloadCfg                      = ERROR_BITMASK | 58,
  ErrorCfg                            = ERROR_BITMASK | 59,
  ErrorSavepointEmptyTx               = ERROR_BITMASK | 60,
  ErrorNoSuchSavepoint                = ERROR_BITMASK | 61,
  ErrorUnknownReplica                 = ERROR_BITMASK | 62,
  ErrorReplicasetUUIDMismatch         = ERROR_BITMASK | 63,
  ErrorInvalidUUID                    = ERROR_BITMASK | 64,
  ErrorReplicasetUUIDIsRo             = ERROR_BITMASK | 65,
  ErrorInstanceUUIDMismatch           = ERROR_BITMASK | 66,
  ErrorReplicaIDIsReserved            = ERROR_BITMASK | 67,
  ErrorInvalidOrder                   = ERROR_BITMASK | 68,
  ErrorMissingRequestField            = ERROR_BITMASK | 69,
  ErrorIdentifier                     = ERROR_BITMASK | 70,
  ErrorDropFunction                   = ERROR_BITMASK | 71,
  ErrorIteratorType                   = ERROR_BITMASK | 72,
  ErrorReplicaMax                     = ERROR_BITMASK | 73,
  ErrorInvalidXlog                    = ERROR_BITMASK | 74,
  ErrorInvalidXlogName                = ERROR_BITMASK | 75,
  ErrorInvalidXlogOrder               = ERROR_BITMASK | 76,
  ErrorNoConnection                   = ERROR_BITMASK | 77,
  ErrorTimeout                        = ERROR_BITMASK | 78,
  ErrorActiveTransaction              = ERROR_BITMASK | 79,
  ErrorCursorNoTransaction            = ERROR_BITMASK | 80,
  ErrorCrossEngineTransaction         = ERROR_BITMASK | 81,
  ErrorNoSuchRole                     = ERROR_BITMASK | 82,
  ErrorRoleExists                     = ERROR_BITMASK | 83,
  ErrorCreateRole                     = ERROR_BITMASK | 84,
  ErrorIndexExists                    = ERROR_BITMASK | 85,
  ErrorSessionClosed                  = ERROR_BITMASK | 86,
  ErrorRoleLoop                       = ERROR_BITMASK | 87,
  ErrorGrant                          = ERROR_BITMASK | 88,
  ErrorPrivGranted                    = ERROR_BITMASK | 89,
  ErrorRoleGranted                    = ERROR_BITMASK | 90,
  ErrorPrivNotGranted                 = ERROR_BITMASK | 91,
  ErrorRoleNotGranted                 = ERROR_BITMASK | 92,
  ErrorMissingSnapshot                = ERROR_BITMASK | 93,
  ErrorCantUpdatePrimaryKey           = ERROR_BITMASK | 94,
  ErrorUpdateIntegerOverflow          = ERROR_BITMASK | 95,
  ErrorGuestUserPassword              = ERROR_BITMASK | 96,
  ErrorTransactionConflict            = ERROR_BITMASK | 97,
  ErrorUnsupportedPriv                = ERROR_BITMASK | 98,
  ErrorLoadFunction                   = ERROR_BITMASK | 99,
  ErrorFunctionLanguage               = ERROR_BITMASK | 100,
  ErrorRtreeRect                      = ERROR_BITMASK | 101,
  ErrorProcC                          = ERROR_BITMASK | 102,
  ErrorUnknownRtreeIndexDistanceType  = ERROR_BITMASK | 103,
  ErrorProtocol                       = ERROR_BITMASK | 104,
  ErrorUpsertUniqueSecondaryKey       = ERROR_BITMASK | 105,
  ErrorWrongIndexRecord               = ERROR_BITMASK | 106,
  ErrorWrongIndexParts                = ERROR_BITMASK | 107,
  ErrorWrongIndexOptions              = ERROR_BITMASK | 108,
  ErrorWrongSchemaVersion             = ERROR_BITMASK | 109,
  ErrorMemtxMaxTupleSize              = ERROR_BITMASK | 110,
  ErrorWrongSpaceOptions              = ERROR_BITMASK | 111,
  ErrorUnsupportedIndexFeature        = ERROR_BITMASK | 112,
  ErrorViewIsRo                       = ERROR_BITMASK | 113,
  ErrorNoTransaction                  = ERROR_BITMASK | 114,
  ErrorSystem                         = ERROR_BITMASK | 115,
  ErrorLoading                        = ERROR_BITMASK | 116,
  ErrorConnectionToSelf               = ERROR_BITMASK | 117,
  ErrorKeyPartIsTooLong               = ERROR_BITMASK | 118,
  ErrorCompression                    = ERROR_BITMASK | 119,
  ErrorCheckpointInProgress           = ERROR_BITMASK | 120,
  ErrorSubStmtMax                     = ERROR_BITMASK | 121,
  ErrorCommitInSubStmt                = ERROR_BITMASK | 122,
  ErrorRollbackInSubStmt              = ERROR_BITMASK | 123,
  ErrorDecompression                  = ERROR_BITMASK | 124,
  ErrorInvalidXlogType                = ERROR_BITMASK | 125,
  ErrorAlreadyRunning                 = ERROR_BITMASK | 126,
  ErrorIndexFieldCountLimit           = ERROR_BITMASK | 127,
  ErrorLocalInstanceIDIsReadOnly      = ERROR_BITMASK | 128,
  ErrorBackupInProgress               = ERROR_BITMASK | 129,
  ErrorReadViewAborted                = ERROR_BITMASK | 130,
  ErrorInvalidIndexFile               = ERROR_BITMASK | 131,
  ErrorInvalidRunFile                 = ERROR_BITMASK | 132,
  ErrorInvalidVylogFile               = ERROR_BITMASK | 133,
  ErrorCheckpointRollback             = ERROR_BITMASK | 134,
  ErrorVyQuotaTimeout                 = ERROR_BITMASK | 135,
  ErrorPartialKey                     = ERROR_BITMASK | 136,
  ErrorTruncateSystemSpace            = ERROR_BITMASK | 137,
  ErrorLoadModule                     = ERROR_BITMASK | 138,
  ErrorVinylMaxTupleSize              = ERROR_BITMASK | 139,
  ErrorWrongDdVersion                 = ERROR_BITMASK | 140,
  ErrorWrongSpaceFormat               = ERROR_BITMASK | 141,
  ErrorCreateSequence                 = ERROR_BITMASK | 142,
  ErrorAlterSequence                  = ERROR_BITMASK | 143,
  ErrorDropSequence                   = ERROR_BITMASK | 144,
  ErrorNoSuchSequence                 = ERROR_BITMASK | 145,
  ErrorSequenceExists                 = ERROR_BITMASK | 146,
  ErrorSequenceOverflow               = ERROR_BITMASK | 147,
  ErrorNoSuchIndexName                = ERROR_BITMASK | 148,
  ErrorSpaceFieldIsDuplicate          = ERROR_BITMASK | 149,
  ErrorCantCreateCollation            = ERROR_BITMASK | 150,
  ErrorWrongCollationOptions          = ERROR_BITMASK | 151,
  ErrorNullablePrimary                = ERROR_BITMASK | 152,
  ErrorNoSuchFieldNameInSpace         = ERROR_BITMASK | 153,
  ErrorTransactionYield               = ERROR_BITMASK | 154,
  ErrorNoSuchGroup                    = ERROR_BITMASK | 155,
  ErrorSQLBindValue                   = ERROR_BITMASK | 156,
  ErrorSQLBindType                    = ERROR_BITMASK | 157,
  ErrorSQLBindParameterMax            = ERROR_BITMASK | 158,
  ErrorSQLExecute                     = ERROR_BITMASK | 159,
  ErrorUpdateDecimalOverflow          = ERROR_BITMASK | 160,
  ErrorSQLBindNotFound                = ERROR_BITMASK | 161,
  ErrorActionMismatch                 = ERROR_BITMASK | 162,
  ErrorViewMissingSQL                 = ERROR_BITMASK | 163,
  ErrorForeignKeyConstraint           = ERROR_BITMASK | 164,
  ErrorNoSuchModule                   = ERROR_BITMASK | 165,
  ErrorNoSuchCollation                = ERROR_BITMASK | 166,
  ErrorCreateFkConstraint             = ERROR_BITMASK | 167,
  ErrorDropFkConstraint               = ERROR_BITMASK | 168,
  ErrorNoSuchConstraint               = ERROR_BITMASK | 169,
  ErrorConstraintExists               = ERROR_BITMASK | 170,
  ErrorSQLTypeMismatch                = ERROR_BITMASK | 171,
  ErrorRowIDOverflow                  = ERROR_BITMASK | 172,
  ErrorDropCollation                  = ERROR_BITMASK | 173,
  ErrorIllegalCollationMix            = ERROR_BITMASK | 174,
  ErrorSQLNoSuchPragma                = ERROR_BITMASK | 175,
  ErrorSQLCantResolveField            = ERROR_BITMASK | 176,
  ErrorIndexExistsInSpace             = ERROR_BITMASK | 177,
  ErrorInconsistentTypes              = ERROR_BITMASK | 178,
  ErrorSQLSyntaxWithPos               = ERROR_BITMASK | 179,
  ErrorSQLStackOverflow               = ERROR_BITMASK | 180,
  ErrorSQLSelectWildcard              = ERROR_BITMASK | 181,
  ErrorSQLStatementEmpty              = ERROR_BITMASK | 182,
  ErrorSQLKeywordIsReserved           = ERROR_BITMASK | 183,
  ErrorSQLSyntaxNearToken             = ERROR_BITMASK | 184,
  ErrorSQLUnknownToken                = ERROR_BITMASK | 185,
  ErrorSQLParserGeneric               = ERROR_BITMASK | 186,
  ErrorSQLAnalyzeArgument             = ERROR_BITMASK | 187,
  ErrorSQLColumnCountMax              = ERROR_BITMASK | 188,
  ErrorHexLiteralMax                  = ERROR_BITMASK | 189,
  ErrorIntLiteralMax                  = ERROR_BITMASK | 190,
  ErrorSQLParserLimit                 = ERROR_BITMASK | 191,
  ErrorIndexDefUnsupported            = ERROR_BITMASK | 192,
  ErrorCkDefUnsupported               = ERROR_BITMASK | 193,
  ErrorMultikeyIndexMismatch          = ERROR_BITMASK | 194,
  ErrorCreateCkConstraint             = ERROR_BITMASK | 195,
  ErrorCkConstraintFailed             = ERROR_BITMASK | 196,
  ErrorSQLColumnCount                 = ERROR_BITMASK | 197,
  ErrorFuncIndexFunc                  = ERROR_BITMASK | 198,
  ErrorFuncIndexFormat                = ERROR_BITMASK | 199,
  ErrorFuncIndexParts                 = ERROR_BITMASK | 200,
  ErrorNoSuchFieldName                = ERROR_BITMASK | 201,
  ErrorFuncWrongArgCount              = ERROR_BITMASK | 202,
  ErrorBootstrapReadonly              = ERROR_BITMASK | 203,
  ErrorSQLFuncWrongRetCount           = ERROR_BITMASK | 204,
  ErrorFuncInvalidReturnType          = ERROR_BITMASK | 205,
  ErrorSQLParserGenericWithPos        = ERROR_BITMASK | 206,
  ErrorReplicaNotAnon                 = ERROR_BITMASK | 207,
  ErrorCannotRegister                 = ERROR_BITMASK | 208,
  ErrorSessionSettingInvalidValue     = ERROR_BITMASK | 209,
  ErrorSQLPrepare                     = ERROR_BITMASK | 210,
  ErrorWrongQueryID                   = ERROR_BITMASK | 211,
  ErrorSequenceNotStarted             = ERROR_BITMASK | 212,
  ErrorNoSuchSessionSetting           = ERROR_BITMASK | 213,
  ErrorUncommittedForeignSyncTxns     = ERROR_BITMASK | 214,
  ErrorSyncMasterMismatch             = ERROR_BITMASK | 215,
  ErrorSyncQuorumTimeout              = ERROR_BITMASK | 216,
  ErrorSyncRollback                   = ERROR_BITMASK | 217,
  ErrorTupleMetadataIsTooBig          = ERROR_BITMASK | 218,
  ErrorXlogGap                        = ERROR_BITMASK | 219,
  ErrorTooEarlySubscribe              = ERROR_BITMASK | 220,
  ErrorSQLCantAddAutoinc              = ERROR_BITMASK | 221,
  ErrorQuorumWait                     = ERROR_BITMASK | 222,

}

#[allow(dead_code)]
impl Code {
  /// Is shortcut for code != Code::Ok
  pub fn is_err(self) -> bool { self != Code::Ok }
}

impl Default for Code {
  fn default() -> Self { Code::Ok }
}

/**
  It represents all known tarantool request types.

  It also implements num_derive::{FromPrimitive, ToPrimitive},
  so you can also convert it to int types.
*/
#[derive(
  Debug, Clone, Copy,
  PartialEq, Eq,
  FromPrimitive, ToPrimitive,
)]
pub enum RequestType {
  Select          = 0x01,
  Insert          = 0x02,
  Replace         = 0x03,
  Update          = 0x04,
  Delete          = 0x05,
  Call16          = 0x06,
  Auth            = 0x07,
  Eval            = 0x08,
  Upsert          = 0x09,
  Call            = 0x0a,
  Execute         = 0x0b,
  Nop             = 0x0c,
  Prepare         = 0x0d,
  Confirm         = 0x28,
  Rollback        = 0x29,
  Ping            = 0x40,
  Join            = 0x41,
  Subscribe       = 0x42,
  VoteDeprecated  = 0x43,
  Vote            = 0x44,
  FetchSnapshot   = 0x45,
  Register        = 0x46,
}

/**
  It represents all known tarantool protocol fields.

  It also implements num_derive::{FromPrimitive, ToPrimitive},
  so you can also convert it to int types.
*/
#[derive(
  Debug, Clone, Copy, Hash,
  PartialEq, Eq,
  FromPrimitive, ToPrimitive,
)]
pub enum Field {
  RequestType   = 0x00,
  Sync          = 0x01,
  ReplicaID     = 0x02,
  LSN           = 0x03,
  Timestamp     = 0x04,
  SchemaVersion = 0x05,
  ServerVersion = 0x06,
  GroupID       = 0x07,
  TSN           = 0x08,
  Flags         = 0x09,
  SpaceID       = 0x10,
  IndexID       = 0x11,
  Limit         = 0x12,
  Offset        = 0x13,
  Iterator      = 0x14,
  IndexBase     = 0x15,
  Key           = 0x20,
  Tuple         = 0x21,
  FunctionName  = 0x22,
  UserName      = 0x23,
  InstanceUUID  = 0x24,
  ClusterUUID   = 0x25,
  Vclock        = 0x26,
  Expr          = 0x27,
  Ops           = 0x28,
  Ballot        = 0x29,
  TupleMeta     = 0x2a,
  Options       = 0x2b,
  Data          = 0x30,
  Error24       = 0x31,
  Metadata      = 0x32,
  BindMetadata  = 0x33,
  BindCount     = 0x34,
  SqlText       = 0x40,
  SqlBind       = 0x41,
  SqlInfo       = 0x42,
  StmtID        = 0x43,
  ReplicaAnon   = 0x50,
  IDFilter      = 0x51,
  Error         = 0x52,
  Term          = 0x53,
}

/**
  It represents all known tarantool iterator variants.

  It also implements num_derive::{FromPrimitive, ToPrimitive},
  so you can also convert it to int types.
*/
#[derive(
  Debug, Clone, Copy,
  PartialEq, Eq,
  FromPrimitive, ToPrimitive,
)]
pub enum Iterator {
  Eq            = 0,
	Req           = 1,
	All           = 2,
	Lt            = 3,
	Le            = 4,
	Ge            = 5,
	Gt            = 6,
	BitsAllSet    = 7,
	BitsAnySet    = 8,
	BitsAllNotSet = 9,
}
