use std::sync::Arc;

use super::commands::SubscribeCommand;
use super::command_results::SubscribeCommandResult;
use super::queries::LoginQuery;
use super::query_results::LoginQueryResult;
use all4art_authservice_domain::factories::UserFactory;
use all4art_authservice_persistence::user_repository::{UserRepository, self};
use log::info;
use async_trait::async_trait;

#[async_trait]
pub trait QueryHandler<T> { 
    async fn retrieve(&self) -> Result<T, std::io::Error>;
}

#[async_trait]
pub trait CommandHandler<T> { 
    async fn execute(&self) -> Result<T, std::io::Error>;
}

pub struct SubscribeCommandHandler { 
    user_repository: Arc<dyn UserRepository + Send + Sync>,
    user_factory: Arc<dyn UserFactory + Send + Sync>,
    command: SubscribeCommand
}

impl SubscribeCommandHandler { 
    pub fn new(command: SubscribeCommand, 
               user_repository: Arc<dyn UserRepository + Send + Sync>,
               user_factory: Arc<dyn UserFactory + Send + Sync>) -> Self { 
       SubscribeCommandHandler {command: command, user_repository: user_repository, user_factory: user_factory} 
    }
}

#[async_trait]
impl CommandHandler<SubscribeCommandResult> for SubscribeCommandHandler { 
    async fn execute(&self) -> Result<SubscribeCommandResult, std::io::Error> { 
        let hashed_password = self.command.clear_text_password().hashed_password()?;
        let user = self.user_factory.create(self.command.public_key(), hashed_password)?;
        let _ = self.user_repository.create_user(user.clone()).await?;
        Ok(SubscribeCommandResult::new(user))
    }
}

pub struct LoginQueryHandler {
    user_repository: Arc<dyn UserRepository + Send + Sync>,
    query: LoginQuery
}

impl LoginQueryHandler { 
    pub fn new(query: LoginQuery, user_repository: Arc<dyn UserRepository + Send + Sync>) -> Self { 
        LoginQueryHandler {query: query, user_repository: user_repository}
    }
     
}

#[async_trait]
impl QueryHandler<LoginQueryResult> for LoginQueryHandler { 
    async fn retrieve(&self) -> Result<LoginQueryResult, std::io::Error> { 
        let user = self.user_repository.retrieve_user(self.query.public_key()).await?;
        let token = user.authenticate(self.query.clear_text_password())?;
        Ok(LoginQueryResult::new(token))
    }
}

