use std::{sync::Arc};

use all4art_authservice_domain::{factories::UserFactory, user};
use all4art_authservice_persistence::user_repository::{UserRepository, self};

use crate::{commands::SubscribeCommand, command_results::SubscribeCommandResult, queries::LoginQuery, handlers::{QueryHandler, SubscribeCommandHandler, LoginQueryHandler}, query_results::LoginQueryResult};

use super::handlers::CommandHandler;

pub trait CommandHandlerFactory { 
    fn create_subscribe_command_handler(&self, command: SubscribeCommand) 
    -> Arc<dyn CommandHandler<SubscribeCommandResult>>;
}

pub trait QueryHandlerFactory { 
    fn create_login_query_handler(&self, query: LoginQuery) -> Arc<dyn QueryHandler<LoginQueryResult>>;
}

pub struct CommandHandlerFactoryImpl { 
    user_repository: Arc<dyn UserRepository>,
    user_factory: Arc<dyn UserFactory>
}

impl CommandHandlerFactory for CommandHandlerFactoryImpl { 
    fn create_subscribe_command_handler(&self, command: SubscribeCommand) 
    -> Arc<dyn CommandHandler<SubscribeCommandResult>> {
        Arc::new(SubscribeCommandHandler::new(command, self.user_repository.clone(), self.user_factory.clone()))
    }
}

impl CommandHandlerFactoryImpl {
    pub fn new(user_repository: Arc<dyn UserRepository>, user_factory: Arc<dyn UserFactory>) 
    -> impl CommandHandlerFactory { 
        CommandHandlerFactoryImpl{user_repository: user_repository, user_factory: user_factory}
    }
}

pub struct QueryHandlerFactoryImpl { 
    user_repository: Arc<dyn UserRepository>
}

impl QueryHandlerFactoryImpl { 
    pub fn new(user_repository: Arc<dyn UserRepository>) -> impl QueryHandlerFactory { 
        QueryHandlerFactoryImpl{user_repository: user_repository}        
    }
}

impl QueryHandlerFactory for QueryHandlerFactoryImpl { 
    fn create_login_query_handler(&self, query: LoginQuery) -> Arc<dyn QueryHandler<LoginQueryResult>> { 
        Arc::new(LoginQueryHandler::new(query, self.user_repository.clone()))
    }
}
