// Copyright 2021 Google LLC
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

//! Miscellaneous escaping functions.

use std::fmt;

/// The error returned by `unescape_utf8_literal()`.
#[derive(Clone, Copy, Debug)]
pub enum UnescapeError {
  UnexpectedEof,
  UnknownEscape(usize, char),
  BadHexDigit(usize, char),
  BadUnicode(usize, u32),
}

impl fmt::Display for UnescapeError {
  fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
    match self {
      UnescapeError::UnexpectedEof => write!(f, "input ended during escape"),
      UnescapeError::UnknownEscape(_, c) => {
        write!(f, "unknown escape sequence: '\\{}'", c)
      }
      UnescapeError::BadHexDigit(_, c) => {
        write!(f, "bad hex digit while parsing escape: '{}'", c)
      }
      UnescapeError::BadUnicode(_, x) => {
        write!(f, "bad Unicode codepoint generated by escape: {:#x}", x)
      }
    }
  }
}

/// Unescapes the string `s`.
///
/// This function recognizes the following escape sequences:
/// - `\0`, a NUL character.
/// - `\n`, a newline.
/// - `\t`, a tab character.
/// - `\r`, a carriage return.
/// - `\xNN`, a hex-encoded ASCII character.
/// - `\uNNNN`, a hex-encoded Unicode codepoint in the Basic Multilingual
///   Plane.
/// - `\UNNNNNNNN`, any hex-encoded Unicode codepoint.
pub fn unescape_utf8_literal(s: &str) -> Result<String, UnescapeError> {
  let mut buf = String::with_capacity(s.len());
  let mut chars = s.char_indices();
  while let Some((_, c)) = chars.next() {
    if c != '\\' {
      buf.push(c);
      continue;
    }

    let (escape_idx, escape) = match chars.next() {
      Some(c) => c,
      None => return Err(UnescapeError::UnexpectedEof),
    };

    fn parse_hex(
      chars: &mut impl Iterator<Item = (usize, char)>,
      digits: usize,
    ) -> Result<u32, UnescapeError> {
      fn hex_digit(c: char) -> Option<u32> {
        let nybble = match c {
          '0'..='9' => (c as u8) - b'0',
          'a'..='f' => (c as u8) - b'a' + 10,
          'A'..='F' => (c as u8) - b'A' + 10,
          _ => return None,
        };
        Some(nybble as u32)
      }

      let mut total: u32 = 0;

      for _ in 0..digits {
        let nybble = match chars.next().map(|(i, c)| (i, c, hex_digit(c))) {
          Some((_, _, Some(n))) => n,
          Some((idx, c, _)) => return Err(UnescapeError::BadHexDigit(idx, c)),
          None => return Err(UnescapeError::UnexpectedEof),
        };

        total <<= 4;
        total |= nybble;
      }

      Ok(total)
    }

    let unescaped: char = match escape {
      '"' | '\'' | '\\' => escape,
      '0' => '\0',
      'n' => '\n',
      't' => '\t',
      'r' => '\r',
      'b' => '\x08',
      'f' => '\x0c',
      'x' => {
        let codepoint = parse_hex(&mut chars, 2)?;
        match std::char::from_u32(codepoint) {
          Some(c) => c,
          None => return Err(UnescapeError::BadUnicode(escape_idx, codepoint)),
        }
      }
      'u' => {
        let codepoint = parse_hex(&mut chars, 4)?;
        match std::char::from_u32(codepoint) {
          Some(c) => c,
          None => return Err(UnescapeError::BadUnicode(escape_idx, codepoint)),
        }
      }
      'U' => {
        let codepoint = parse_hex(&mut chars, 8)?;
        match std::char::from_u32(codepoint) {
          Some(c) => c,
          None => return Err(UnescapeError::BadUnicode(escape_idx, codepoint)),
        }
      }
      _ => return Err(UnescapeError::UnknownEscape(escape_idx, escape)),
    };

    buf.push(unescaped)
  }

  Ok(buf)
}

/// Checks whether `s` is a valid Alkyne identifier, including `_`.
pub fn is_identifier(s: &str) -> bool {
  for (i, c) in s.chars().enumerate() {
    match (i, c) {
      (0, '0'..='9') => return false,
      (_, '0'..='9') | (_, 'a'..='z') | (_, 'A'..='Z') | (_, '_') => {}
      _ => return false,
    }
  }
  true
}

/// Escapes the string `s` according to Alkyne syntax.
///
/// The quotation marks are included, unless `maybe_id` is true and the string
/// is a valid Alkyne identifier other than `_`.
pub fn escape_alkyne_string(s: &str, maybe_id: bool, buf: &mut String) {
  if maybe_id && s != "_" && is_identifier(s) {
    buf.push_str(s);
    return;
  }

  let use_single_quotes = s.contains('"') && !s.contains('\'');
  if use_single_quotes {
    buf.push('\'');
  } else {
    buf.push('"');
  }

  for c in s.chars() {
    use std::fmt::Write;
    match c {
      '\x00' => buf.push_str("\\0"),
      '\x08' => buf.push_str("\\b"),
      // Tab.
      '\t' => buf.push_str("\\t"),
      '\n' => buf.push_str("\\n"),
      // Form feed.
      '\x0c' => buf.push_str("\\f"),
      '\r' => buf.push_str("\\r"),
      '"' if !use_single_quotes => buf.push_str("\\\""),
      '\'' if use_single_quotes => buf.push_str("\\'"),
      '\\' => buf.push_str("\\\\"),
      c if !c.is_control() => buf.push(c),
      c if (c as u32) < 256 => {
        let _ = write!(buf, "\\x{:02x}", c as u32);
      }
      c if (c as u32) < 65536 => {
        let _ = write!(buf, "\\u{:04x}", c as u32);
      }
      c => {
        let _ = write!(buf, "\\U{:08x}", c as u32);
      }
    }
  }

  if use_single_quotes {
    buf.push('\'');
  } else {
    buf.push('"');
  }
}

/// Escapes the string `s` according to JSON syntax.
///
/// The quotation marks are included.
pub fn escape_json_string(s: &str, buf: &mut String) {
  buf.push('"');
  for c in s.encode_utf16() {
    match c {
      // Backspace.
      0x08 => buf.push_str("\\b"),
      // Tab.
      0x09 => buf.push_str("\\t"),
      // Newline.
      0x0a => buf.push_str("\\n"),
      // Form feed.
      0x0c => buf.push_str("\\f"),
      // Carriage return.
      0x0d => buf.push_str("\\r"),
      // Double quote.
      0x22 => buf.push_str("\\\""),
      // Backslash.
      0x5c => buf.push_str("\\\\"),
      // Printable ASCII.
      0x20..=0x7e => buf.push(c as u8 as char),
      // Everything else, as \uXXXX.
      _ => {
        use std::fmt::Write;
        let _ = write!(buf, "\\u{:04x}", c);
      }
    }
  }
  buf.push('"');
}
