//! One-time symmetric authentication for short messages.
//!
//! This module corresponds to the [`crypto_onetimeauth`
//! API](https://doc.libsodium.org/advanced/poly1305) from Sodium.
//!
//! One-time authentication is similar in purpose to standard symmetric authentication (available
//! in the [`symmetric::auth`](crate::symmetric::auth) API). However, it has the major caveat that
//! a given key can only be used to authenticate a single message. Using the same key to
//! authentication two or more messages can result in an attacker recovering the key. With this
//! obvious caveat in mind, the benefit of one-time authentication is that it is generally much
//! faster than a full HMAC calculation, as used in standard symmetric message authentication.
//!
//! One-time authentication is therefore best suited to online protocols where lots of short
//! messages need to be authenticated. A common pattern would be to establish a shared secret key
//! between all parties, then use the [`hash::kdf` module](crate::hash::kdf) to derive an
//! authentication key for each message, and authenticate the messages with this API.
//!
//! # Algorithm Details
//! This construction is referred to as a [Message Authentication
//! Code](https://en.wikipedia.org/wiki/Message_authentication_code) (MAC).
//! In particular, this API uses [Poly1305](https://en.wikipedia.org/wiki/Poly1305).
//!
//! # Security Considerations
//! A single key must *never* be used to generate an authentication tag for more than one message.
//! If there is any possibility of using the same key to authenticate multiple messages, you should
//! be using the [`auth` API](crate::symmetric::auth).
//!
//! A common, but dangerous, mistake is to try to verify the authenticity of a message by generating
//! an authentication tag for the message yourself via [`authenticate`], and naively comparing the
//! tag you calculate with the tag you received from the message's sender. This opens the door to
//! [timing attacks](https://en.wikipedia.org/wiki/Timing_attack). The [`verify`] and
//! [`Multipart::verify`] functions use a constant-time comparison between the tags, and should be
//! used whenever you want to verify a tag, rather than comparing tags yourself.
//!
//! The length of an authentication tag generated by this API may be insufficient for the
//! authentication of very long inputs (e.g: large file transfers).
//!
//! Poly1305 is explicitly *not* a hash function, unlike some other message authentication schemes.
//!
//! ## Secret Data
//! * Each per-message one-time-auth key ([`Key`]) must only be known to parties who should be able
//!   to both create and verify tags
//! * The internal state of the [`Multipart`] struct should be kept secret
//!
//! ## Non-Secret Data
//! * Authentication tags ([`Tag`]) are not sensitive, and do not reveal anything about the content
//!   of the authenticated message to an attacker
//!
//! # Examples
//! Generating a one-time authentication tag for a message and verifying its validity (using
//! [`authenticate`] and [`verify`]):
//!
//! ```rust
//! use alkali::symmetric::one_time_auth;
//!
//! const MESSAGE: &'static str = "Here's a message to authenticate. It can be of any length.";
//!
//! // Sender side:
//!
//! // Generate a random, one-time key for the authentication. This will need to be shared with the
//! // message receiver somehow - in practise, it may be deterministically derived from a parent
//! // key.
//! let key = one_time_auth::Key::generate().unwrap();
//! // The tag returned by `authenticate` is what proves the message's authenticity, and should be
//! // transmitted alongside the message.
//! let tag = one_time_auth::authenticate(MESSAGE.as_bytes(), &key).unwrap();
//!
//!
//! // ...
//!
//!
//! // Receiver side:
//! // We assume `key` is somehow known to the receiver.
//!
//! // The `verify` function checks that `tag` is a valid authentication tag for the message and
//! // key, thereby proving the message's authenticity. An error is returned if the tag is not valid
//! // (i.e: the message is inauthentic).
//! one_time_auth::verify(MESSAGE.as_bytes(), &tag, &key).expect("Authentication failed!");
//! ```
//!
//! If you are transmitting/receiving a message in chunks, it may make more sense to use the
//! streaming API, which allows you to specify the message to authenticate in multiple parts (uses
//! [`Multipart`]):
//!
//! ```rust
//! use alkali::symmetric::one_time_auth;
//!
//! // Sender side:
//!
//! let key = one_time_auth::Key::generate().unwrap();
//! let mut state = one_time_auth::Multipart::new(&key).unwrap();
//! state.update(b"Here's the first part");
//! state.update(b"... And the second!");
//! let tag = state.authenticate();
//!
//!
//! // ...
//!
//!
//! // Receiver side:
//! // We assume `key` is somehow known to the receiver.
//!
//! // Now let's verify the tag we just generated. The contents don't have to be added to the
//! // authentication stream in the same chunks as they were in the original authentication tag
//! // calculation: Each piece of the message added to the state is simply concatenated in the tag
//! // calculation.
//! let mut state = one_time_auth::Multipart::new(&key).unwrap();
//! state.update(b"Here");
//! state.update(b"'s the first ");
//! state.update(b"part... And the ");
//! state.update(b"second!");
//! state.verify(&tag).expect("Authentication failed!");
//! ```

use thiserror::Error;

/// Error type returned if something went wrong in the `one_time_auth` module.
#[derive(Clone, Copy, Debug, Eq, Error, PartialEq)]
pub enum OneTimeAuthError {
    /// Failed to authenticate a message.
    ///
    /// The provided tag is not correct for this message + key.
    ///
    /// This may indicate an attempted forgery, a transmission error, or that you're using a
    /// different key to the one used by the message sender. In any case, the authenticity of the
    /// message can't be verified, and it should not be trusted.
    #[error("authentication failed")]
    AuthenticationFailed,
}

/// The [Poly1305](https://en.wikipedia.org/wiki/Poly1305) Message Authentication Code.
pub mod poly1305 {
    use super::OneTimeAuthError;
    use crate::{assert_not_err, mem, require_init, unexpected_err, AlkaliError};
    use libsodium_sys as sodium;
    use std::marker::PhantomData;
    use std::ptr;

    /// The length of a one-time key to authenticate a message, in bytes.
    pub const KEY_LENGTH: usize = sodium::crypto_onetimeauth_poly1305_KEYBYTES as usize;

    /// The length of an authentication tag, in bytes.
    ///
    /// No matter the length of the message to authenticate, the calculated tag is of this fixed
    /// length.
    pub const TAG_LENGTH: usize = sodium::crypto_onetimeauth_poly1305_BYTES as usize;

    mem::hardened_buffer! {
        /// One-time secret key for message authentication.
        ///
        /// There are no *technical* constraints on the contents of a key, but it should be
        /// indistinguishable from random noise. In general, a key will be derived as part of an
        /// online protocol, but it can be randomly generated using [`Key::generate`] if need be.
        ///
        /// A secret key should not be made public.
        ///
        /// A one-time-auth key must never be used to authenticate more than one message.
        ///
        /// This is a [hardened buffer type](https://docs.rs/alkali#hardened-buffer-types), and will
        /// be zeroed on drop. A number of other security measures are taken to protect its
        /// contents. This type in particular can be thought of as roughly equivalent to a `[u8;
        /// KEY_LENGTH]`, and implements [`std::ops::Deref`] so it can be used like it is an
        /// `&[u8]`. This struct uses heap memory while in scope, allocated using Sodium's [secure
        /// memory utilities](https://doc.libsodium.org/memory_management).
        pub Key(KEY_LENGTH);
    }

    impl Key {
        /// Generate a new, random key for use in one-time authentication.
        pub fn generate() -> Result<Self, AlkaliError> {
            require_init()?;

            let mut key = Self::new_empty()?;
            unsafe {
                // SAFETY: This function expects a pointer to a region of memory sufficient to store
                // a key for this algorithm. We have defined the `Key` type using the
                // `crypto_onetimeauth_KEYBYTES` constant from Sodium, so it definitely has the
                // correct amount of space allocated to store the key. The `Key::inner_mut` method
                // simply returns a mutable pointer to its backing memory. The Sodium documentation
                // specifies that `crypto_onetimeauth_KEYBYTES` random bytes will be written
                // starting at the provided pointer. This is a valid representation of
                // `[u8; KEY_LENGTH]`, so `key` is in a valid state following this function call.
                // The `Key::inner_mut` method simply returns a mutable pointer to its backing
                // memory.
                sodium::crypto_onetimeauth_poly1305_keygen(key.inner_mut() as *mut libc::c_uchar);
            }
            Ok(key)
        }
    }

    /// A one-time authentication tag for a message.
    ///
    /// This authentication tag is what proves the authenticity of a message, so it should be
    /// transmitted along with the message to be authenticated. Authentication tags are not
    /// sensitive, and may be transmitted in the clear.
    pub type Tag = [u8; TAG_LENGTH];

    /// Streaming one-time authentication API, for long/multi-part message authentication.
    ///
    /// This can be used to calculate an authentication tag for a message which is received in
    /// portions, or which is too large to fit into memory (although please see the security
    /// considerations in the module documentation if using this API to authenticate very large
    /// messages).
    ///
    /// This struct uses heap memory while in scope, allocated using Sodium's [secure memory
    /// utilities](https://doc.libsodium.org/memory_management).
    ///
    /// # Security Considerations
    /// The inner state of this struct should not be made public: It can be used to calculate
    /// authentication tags for the associated key. None of the methods exposed here will reveal
    /// any of its inner state, so this shouldn't be something that you need to worry about.
    #[derive(Debug)]
    pub struct Multipart {
        state: ptr::NonNull<sodium::crypto_onetimeauth_poly1305_state>,
        _marker: PhantomData<sodium::crypto_onetimeauth_poly1305_state>,
    }

    impl Multipart {
        /// Create a new instance of the struct.
        ///
        /// The provided [`Key`] should be the one-time symmetric key to use for
        /// authentication/verification. Remember that a given key must only be used to authenticate
        /// a single message.
        pub fn new(key: &Key) -> Result<Self, AlkaliError> {
            require_init()?;

            let mut state = unsafe {
                // SAFETY: This call to malloc will allocate the memory required for a
                // `crypto_onetimeauth_state` type, outside of Rust's memory management. The
                // associated memory is always freed in the corresponding `drop` call, unless
                // initialisation fails, in which case it is freed before `Multipart::new` returns,
                // and not used again. We never free the memory in any other place in this struct,
                // and drop can only be called once, so a double-free is not possible. We never
                // expose a pointer to the allocated memory directly. See the drop implementation
                // for more reasoning on safety.
                mem::malloc()?
            };

            let init_result = unsafe {
                // SAFETY: This function initialises a `crypto_onetimeauth_state` struct. The first
                // argument should be a pointer to a region of memory sufficient to store such a
                // struct. We pass a pointer to a region of memory sufficient to store the struct,
                // allocated above. The type of `state` is a `NonNull` pointer, and the unsafe block
                // above will return early if allocation failed, so this pointer is valid for use
                // here. The next argument specifies the key to use for the authentication. We have
                // defined the `Key` type to have length equal to `crypto_onetimeauth_KEYBYTES`, so
                // this many bytes can be read from `key` without an over-read, and this is how many
                // bytes this function will attempt to read from this pointer. The `Key::inner`
                // method simply returns an immutable pointer to the type's backing memory. Sodium's
                // documentation specifies that after this function is called, the memory pointed to
                // by `state` is correctly initialised, and is a valid representation of a
                // `crypto_onetimeauth_state` struct which can be used with other functions from
                // Sodium.
                sodium::crypto_onetimeauth_poly1305_init(
                    state.as_mut(),
                    key.inner() as *const libc::c_uchar,
                )
            };

            // This return value is not possible in the current implementation of
            // `crypto_onetimeauth_init` in Sodium, but could be in the future.
            if init_result != 0 {
                unsafe {
                    // SAFETY: The memory we free here was allocated previously in this function
                    // using Sodium's allocator, and has not yet been freed, so it is valid to free
                    // it here. The `unexpected_err!` macro below will always panic, so this
                    // function will not return, and an instance of `Self` is never initialised,
                    // preventing a double-free or use-after-free.
                    mem::free(state);
                }
                unexpected_err!("crypto_onetimeauth_poly1305_init");
            }

            Ok(Self {
                state,
                _marker: PhantomData,
            })
        }

        /// Try to clone this Multipart state.
        ///
        /// This function initialises a new instance of this struct, in the same state as the
        /// current one, so any data written to be authenticated in the current struct will also be
        /// used in the authentication tag calculation for the new struct.
        ///
        /// The same [`Key`] used to initiate the original [`Multipart`] instance will be used to
        /// authenticate any data added to the new instance.
        ///
        /// # Security Considerations
        /// Remember that any [`Key`] must only be used to authenticate a single message! Cloning a
        /// Multipart state and authenticating two different messages violates this rule, so care
        /// should be taken to ensure this does not occur.
        pub fn try_clone(&self) -> Result<Self, AlkaliError> {
            // We do not use `require_init` here, as it must be called to initialise a `Multipart`
            // struct.

            let state = unsafe {
                // SAFETY: This call to malloc() will allocate the memory required for a
                // `crypto_onetimeauth_state` type, outside of Rust's memory management. The
                // associated memory is always freed in the corresponding `drop` call for the
                // Multipart struct. We never free the memory in any other place in this struct, and
                // drop can only be called once, so a double-free is not possible. We never expose a
                // pointer to the allocated memory directly. See the drop implementation for more
                // reasoning on safety.
                let mut state = mem::malloc()?;

                // SAFETY: We have called `malloc` to allocate sufficient space for one
                // `crypto_onetimeauth_state` struct at each of the two pointers used here:
                // `self.state` was allocated in a call to `Self::new`, and `state` was allocated
                // above, so both are valid for reads/writes of
                // `size_of::<crypto_onetimeauth_state>` bytes. We have just allocated a fresh
                // region of memory for `state`, so it definitely doesn't overlap with `self.state`.
                // To initialise an instance of `Self`, `self.state` must be a valid representation
                // of a `crypto_onetimeauth_state` struct. No methods within `Self` would cause
                // `self.state` to point to an invalid representation of a
                // `crypto_onetimeauth_state` struct. Therefore, after the copy, `state` must also
                // point to a valid representation of a `crypto_onetimeauth_state` struct, and can
                // be used with the multipart one-time-auth functions from Sodium.
                ptr::copy_nonoverlapping(self.state.as_ptr(), state.as_mut(), 1);

                state
            };

            Ok(Self {
                state,
                _marker: std::marker::PhantomData,
            })
        }

        /// Add message contents to be authenticated.
        pub fn update(&mut self, chunk: &[u8]) {
            // We do not use `require_init` here, as it must be called to initialise a `Multipart`
            // struct.

            let update_result = unsafe {
                // SAFETY: The first argument to this function is a pointer to a
                // `crypto_onetimeauth_state` struct. `self.state` must be correctly initialised,
                // and point to a valid representation of a `crypto_onetimeauth_state` struct to
                // instantiate the `Multipart` struct, so it is in the correct state to use with
                // this function. The next two arguments specify the portion of the message to add
                // to the authentication tag calculation, and its length. We use `chunk.len()` to
                // specify the length of the message to read from `chunk`, so `chunk` is definitely
                // valid for reads of the specified length.
                sodium::crypto_onetimeauth_poly1305_update(
                    self.state.as_mut(),
                    chunk.as_ptr(),
                    chunk.len() as libc::c_ulonglong,
                )
            };

            assert_not_err!(update_result, "crypto_onetimeauth_poly1305_update");
        }

        /// Calculate a one-time authentication tag for the specified message.
        ///
        /// Equivalent to [`authenticate`] for single-part messages.
        ///
        /// # Security Considerations
        /// Do not use this method to *verify* an existing authentication tag for a message, as
        /// naïve comparison of authentication tags gives rise to a timing attack. Instead, use the
        /// [`Multipart::verify`] method, which verifies an authentication tag in constant time.
        pub fn authenticate(mut self) -> Tag {
            // We do not use `require_init` here, as it must be called to initialise a `Multipart`
            // struct.

            let mut tag = [0u8; TAG_LENGTH];

            let finalise_result = unsafe {
                // SAFETY: The first argument to this function is a pointer to a
                // `crypto_onetimeauth_state` struct. `self.state` must be correctly initialised,
                // and point to a valid representation of a `crypto_onetimeauth_state` struct to
                // instantiate the `Multipart` struct, so it is in the correct state to use with
                // this function. The next argument specifies the destination to which the
                // authentication tag should be written. Sodium will write a tag of
                // `crypto_onetimeauth_BYTES` to this destination. We have defined the `Tag` type
                // to be `crypto_onetimeauth_BYTES` bytes long, so it is valid for writes of this
                // length.
                sodium::crypto_onetimeauth_poly1305_final(self.state.as_mut(), tag.as_mut_ptr())
            };
            assert_not_err!(finalise_result, "crypto_onetimeauth_poly1305_final");

            tag
        }

        /// Verify the provided tag is correct for the specified message.
        ///
        /// Returns a [`OneTimeAuthError::AuthenticationFailed`] if verification of the one-time
        /// authentication tag failed.
        ///
        /// Equivalent to [`verify`] for single-part messages.
        pub fn verify(mut self, tag: &Tag) -> Result<(), AlkaliError> {
            // We do not use `require_init` here, as it must be called to initialise a `Multipart`
            // struct.

            let mut actual_tag = [0u8; TAG_LENGTH];

            let finalise_result = unsafe {
                // SAFETY: The first argument to this function is a pointer to a
                // `crypto_onetimeauth_state` struct. `self.state` must be correctly initialised,
                // and point to a valid representation of a `crypto_onetimeauth_state` struct to
                // instantiate the `Multipart` struct, so it is in the correct state to use with
                // this function. The next argument specifies the destination to which the
                // authentication tag should be written. Sodium will write a tag of
                // `crypto_onetimeauth_BYTES` to this destination. We have defined the `Tag` type
                // to be `crypto_onetimeauth_BYTES` bytes long, so it is valid for writes of this
                // length.
                sodium::crypto_onetimeauth_poly1305_final(
                    self.state.as_mut(),
                    actual_tag.as_mut_ptr(),
                )
            };
            assert_not_err!(finalise_result, "crypto_onetimeauth_poly1305_final");

            if mem::eq(tag, &actual_tag)? {
                Ok(())
            } else {
                Err(OneTimeAuthError::AuthenticationFailed.into())
            }
        }
    }

    impl Drop for Multipart {
        fn drop(&mut self) {
            // We do not use `require_init` here, as it must be called to initialise a `Multipart`
            // struct.

            unsafe {
                // SAFETY:
                // * Is a double-free possible in safe code?
                //   * No: We only free in `drop`, which cannot be called manually, and is called
                //     exactly once when the struct is actually dropped. Once the value is dropped,
                //     there's no way to call the method again to cause a double free.
                // * Is a use-after-free possible in safe code?
                //   * No: We only ever free `self.state` on drop, and after drop, none of the
                //     type's methods are accessible.
                // * Is a memory leak possible in safe code?
                //   * Yes: If the user uses something like `Box::leak()`, `ManuallyDrop`, or
                //     `std::mem::forget`, the destructor will not be called even though the struct
                //     is dropped. However, it is documented that in these cases heap memory may be
                //     leaked, so this is expected behaviour. In addition, certain signal interrupts
                //     or using panic=abort behaviour will mean the destructor is not called.
                //     There's little we can do about this, but a failure to free is probably
                //     reasonable in such cases. In any other case, `drop` will be called, and the
                //     memory freed.
                // `self.state` was allocated in the `Multipart` constructor using Sodium's
                // allocator, so it is correct to free it using Sodium's allocator.
                mem::free(self.state);
            }
        }
    }

    /// Compute the one-time authentication tag for the given message and key.
    ///
    /// Returns the one-time authentication tag, which is non-secret, and is used to verify the
    /// authenticity of the message.
    ///
    /// # Security Considerations
    /// A single key must *never* be used to generate an authentication tag for more than one
    /// message.
    ///
    /// Do not use this function to *verify* an existing authentication tag for a message, as naïve
    /// comparison of authentication tags gives rise to a timing attack. Instead, use the [`verify`]
    /// function, which verifies an authentication tag in constant time.
    ///
    /// The length of an authentication tag generated by this API may be insufficient for the
    /// authentication of very long inputs (e.g: large file transfers).
    pub fn authenticate(message: &[u8], key: &Key) -> Result<Tag, AlkaliError> {
        require_init()?;

        let mut tag = [0u8; TAG_LENGTH];

        let auth_result = unsafe {
            // SAFETY: The first argument to this function is a pointer to which the calculated
            // authentication tag will be written. We have defined the `tag` buffer to be
            // `crypto_onetimeauth_BYTES` bytes long, the length of a one-time-auth tag as defined
            // in Sodium, so it is of the correct size to store the tag. The next two arguments
            // specify the message to authenticate and its length. We use `message.len()` to specify
            // the length of the message, so `message` is clearly valid for reads of this length.
            // The final argument specifies the key to use for this message's authentication. We
            // define the `Key` type to be `crypto_onetimeauth_KEYBYTES` long, so it is of the
            // expected size for use in this function. The `Key::inner` method simply returns a
            // pointer to the backing memory.
            sodium::crypto_onetimeauth_poly1305(
                tag.as_mut_ptr(),
                message.as_ptr(),
                message.len() as libc::c_ulonglong,
                key.inner() as *const libc::c_uchar,
            )
        };
        assert_not_err!(auth_result, "crypto_onetimeauth_poly1305");

        Ok(tag)
    }

    /// Verify that the provided one-time authentication tag is valid for a given message and key.
    ///
    /// Returns a [`OneTimeAuthError::AuthenticationFailed`] if verification of the authentication
    /// tag failed.
    pub fn verify(message: &[u8], tag: &Tag, key: &Key) -> Result<(), AlkaliError> {
        require_init()?;

        let verification_result = unsafe {
            // SAFETY: The first argument to this function is the authentication tag to verify. We
            // have defined the `Tag` type to be `crypto_onetimeauth_BYTES` bytes long, the length
            // of a one-time-auth tag in Sodium, so it is of the expected size for use here. The
            // next two arguments specify the message to verify. We use `message.len()` to specify
            // the length of the message, so clearly `message` is valid for reads of this length,
            // and an over-read will not occur. The final argument specifies the key to use for this
            // message's verification. We define the `Key` type to be `crypto_onetimeauth_KEYBYTES`
            // long, so it is of the expected size for use in this function. The `Key::inner` method
            // simply returns a pointer to the backing memory.
            sodium::crypto_onetimeauth_poly1305_verify(
                tag.as_ptr(),
                message.as_ptr(),
                message.len() as libc::c_ulonglong,
                key.inner() as *const libc::c_uchar,
            )
        };

        if verification_result == 0 {
            Ok(())
        } else {
            Err(OneTimeAuthError::AuthenticationFailed.into())
        }
    }

    #[cfg(test)]
    mod tests {
        use super::{authenticate, verify, Key, Multipart};
        use crate::random::fill_random;
        use crate::AlkaliError;

        #[test]
        fn key_generation() -> Result<(), AlkaliError> {
            let _key = Key::generate()?;
            Ok(())
        }

        #[test]
        fn auth_and_verify() -> Result<(), AlkaliError> {
            let key_a = Key::generate()?;
            let key_b = Key::generate()?;
            let key_c = Key::generate()?;
            let key_d = Key::generate()?;

            let buf_a = [];
            let mut buf_b = [0; 16];
            let mut buf_c = [0; 1024];
            let mut buf_d = [0; 1 << 20];

            fill_random(&mut buf_b)?;
            fill_random(&mut buf_c)?;
            fill_random(&mut buf_d)?;

            let tag_a = authenticate(&buf_a, &key_a)?;
            let tag_b = authenticate(&buf_b, &key_b)?;
            let tag_c = authenticate(&buf_c, &key_c)?;
            let tag_d = authenticate(&buf_d, &key_d)?;

            verify(&buf_a, &tag_a, &key_a)?;
            verify(&buf_b, &tag_b, &key_b)?;
            verify(&buf_c, &tag_c, &key_c)?;
            verify(&buf_d, &tag_d, &key_d)?;

            fill_random(&mut buf_b)?;
            fill_random(&mut buf_c)?;
            fill_random(&mut buf_d)?;

            assert!(verify(&buf_b, &tag_b, &key_b).is_err());
            assert!(verify(&buf_c, &tag_c, &key_c).is_err());
            assert!(verify(&buf_d, &tag_d, &key_d).is_err());

            Ok(())
        }

        #[test]
        fn single_part_test_vectors() -> Result<(), AlkaliError> {
            let mut key = Key::new_empty()?;
            key.copy_from_slice(&[
                0xee, 0xa6, 0xa7, 0x25, 0x1c, 0x1e, 0x72, 0x91, 0x6d, 0x11, 0xc2, 0xcb, 0x21, 0x4d,
                0x3c, 0x25, 0x25, 0x39, 0x12, 0x1d, 0x8e, 0x23, 0x4e, 0x65, 0x2d, 0x65, 0x1f, 0xa4,
                0xc8, 0xcf, 0xf8, 0x80,
            ]);
            let message = [
                0x8e, 0x99, 0x3b, 0x9f, 0x48, 0x68, 0x12, 0x73, 0xc2, 0x96, 0x50, 0xba, 0x32, 0xfc,
                0x76, 0xce, 0x48, 0x33, 0x2e, 0xa7, 0x16, 0x4d, 0x96, 0xa4, 0x47, 0x6f, 0xb8, 0xc5,
                0x31, 0xa1, 0x18, 0x6a, 0xc0, 0xdf, 0xc1, 0x7c, 0x98, 0xdc, 0xe8, 0x7b, 0x4d, 0xa7,
                0xf0, 0x11, 0xec, 0x48, 0xc9, 0x72, 0x71, 0xd2, 0xc2, 0x0f, 0x9b, 0x92, 0x8f, 0xe2,
                0x27, 0x0d, 0x6f, 0xb8, 0x63, 0xd5, 0x17, 0x38, 0xb4, 0x8e, 0xee, 0xe3, 0x14, 0xa7,
                0xcc, 0x8a, 0xb9, 0x32, 0x16, 0x45, 0x48, 0xe5, 0x26, 0xae, 0x90, 0x22, 0x43, 0x68,
                0x51, 0x7a, 0xcf, 0xea, 0xbd, 0x6b, 0xb3, 0x73, 0x2b, 0xc0, 0xe9, 0xda, 0x99, 0x83,
                0x2b, 0x61, 0xca, 0x01, 0xb6, 0xde, 0x56, 0x24, 0x4a, 0x9e, 0x88, 0xd5, 0xf9, 0xb3,
                0x79, 0x73, 0xf6, 0x22, 0xa4, 0x3d, 0x14, 0xa6, 0x59, 0x9b, 0x1f, 0x65, 0x4c, 0xb4,
                0x5a, 0x74, 0xe3, 0x55, 0xa5,
            ];

            let tag = authenticate(&message, &key)?;
            assert_eq!(
                tag,
                [
                    0xf3, 0xff, 0xc7, 0x70, 0x3f, 0x94, 0x00, 0xe5, 0x2a, 0x7d, 0xfb, 0x4b, 0x3d,
                    0x33, 0x05, 0xd9,
                ]
            );
            verify(&message, &tag, &key)?;

            Ok(())
        }

        #[test]
        fn multi_part_test_vectors() -> Result<(), AlkaliError> {
            let mut key = Key::new_empty()?;
            key.copy_from_slice(&[
                0xee, 0xa6, 0xa7, 0x25, 0x1c, 0x1e, 0x72, 0x91, 0x6d, 0x11, 0xc2, 0xcb, 0x21, 0x4d,
                0x3c, 0x25, 0x25, 0x39, 0x12, 0x1d, 0x8e, 0x23, 0x4e, 0x65, 0x2d, 0x65, 0x1f, 0xa4,
                0xc8, 0xcf, 0xf8, 0x80,
            ]);

            let mut state = Multipart::new(&key)?;
            state.update(&[
                0x8e, 0x99, 0x3b, 0x9f, 0x48, 0x68, 0x12, 0x73, 0xc2, 0x96, 0x50, 0xba, 0x32, 0xfc,
                0x76, 0xce, 0x48, 0x33, 0x2e, 0xa7, 0x16, 0x4d, 0x96, 0xa4, 0x47, 0x6f, 0xb8, 0xc5,
                0x31, 0xa1, 0x18, 0x6a, 0xc0, 0xdf, 0xc1, 0x7c, 0x98, 0xdc, 0xe8, 0x7b, 0x4d, 0xa7,
                0xf0, 0x11, 0xec, 0x48, 0xc9, 0x72, 0x71, 0xd2, 0xc2, 0x0f, 0x9b, 0x92, 0x8f, 0xe2,
                0x27, 0x0d, 0x6f, 0xb8, 0x63, 0xd5, 0x17, 0x38, 0xb4, 0x8e, 0xee, 0xe3, 0x14, 0xa7,
                0xcc, 0x8a, 0xb9, 0x32, 0x16, 0x45, 0x48, 0xe5, 0x26, 0xae, 0x90, 0x22, 0x43, 0x68,
                0x51, 0x7a, 0xcf, 0xea, 0xbd, 0x6b, 0xb3, 0x73, 0x2b, 0xc0, 0xe9, 0xda, 0x99, 0x83,
                0x2b, 0x61,
            ]);
            state.update(&[]);
            state.update(&[
                0xca, 0x01, 0xb6, 0xde, 0x56, 0x24, 0x4a, 0x9e, 0x88, 0xd5, 0xf9, 0xb3, 0x79, 0x73,
                0xf6, 0x22, 0xa4, 0x3d, 0x14, 0xa6, 0x59, 0x9b, 0x1f, 0x65, 0x4c, 0xb4, 0x5a, 0x74,
                0xe3, 0x55, 0xa5,
            ]);

            let tag = state.authenticate();
            assert_eq!(
                tag,
                [
                    0xf3, 0xff, 0xc7, 0x70, 0x3f, 0x94, 0x00, 0xe5, 0x2a, 0x7d, 0xfb, 0x4b, 0x3d,
                    0x33, 0x05, 0xd9,
                ]
            );

            let mut state = Multipart::new(&key)?;
            state.update(&[
                0x8e, 0x99, 0x3b, 0x9f, 0x48, 0x68, 0x12, 0x73, 0xc2, 0x96, 0x50, 0xba, 0x32, 0xfc,
                0x76, 0xce, 0x48, 0x33, 0x2e, 0xa7, 0x16, 0x4d, 0x96, 0xa4, 0x47, 0x6f, 0xb8, 0xc5,
                0x31, 0xa1, 0x18, 0x6a, 0xc0, 0xdf, 0xc1, 0x7c, 0x98, 0xdc, 0xe8, 0x7b, 0x4d, 0xa7,
                0xf0, 0x11, 0xec, 0x48, 0xc9, 0x72, 0x71, 0xd2, 0xc2, 0x0f, 0x9b, 0x92, 0x8f, 0xe2,
                0x27, 0x0d, 0x6f, 0xb8, 0x63, 0xd5, 0x17, 0x38, 0xb4, 0x8e, 0xee, 0xe3, 0x14, 0xa7,
                0xcc, 0x8a, 0xb9, 0x32, 0x16, 0x45, 0x48, 0xe5, 0x26, 0xae, 0x90, 0x22, 0x43, 0x68,
                0x51, 0x7a, 0xcf, 0xea, 0xbd, 0x6b, 0xb3, 0x73, 0x2b, 0xc0, 0xe9, 0xda, 0x99, 0x83,
                0x2b, 0x61,
            ]);
            state.update(&[]);
            state.update(&[
                0xca, 0x01, 0xb6, 0xde, 0x56, 0x24, 0x4a, 0x9e, 0x88, 0xd5, 0xf9, 0xb3, 0x79, 0x73,
                0xf6, 0x22, 0xa4, 0x3d, 0x14, 0xa6, 0x59, 0x9b, 0x1f, 0x65, 0x4c, 0xb4, 0x5a, 0x74,
                0xe3, 0x55, 0xa5,
            ]);

            state.verify(&[
                0xf3, 0xff, 0xc7, 0x70, 0x3f, 0x94, 0x00, 0xe5, 0x2a, 0x7d, 0xfb, 0x4b, 0x3d, 0x33,
                0x05, 0xd9,
            ])?;

            Ok(())
        }
    }
}

pub use poly1305::*;
