//! One-time symmetric authentication for short messages.
//!
//! This module corresponds to the [`crypto_onetimeauth`
//! API](https://doc.libsodium.org/advanced/poly1305) from Sodium.
//!
//! One-time authentication is similar in purpose to standard symmetric authentication (available
//! in the [`symmetric::auth`](crate::symmetric::auth) API). However, it has the major caveat that
//! a given key can only be used to authenticate a single message. Using the same key to
//! authentication two or more messages can result in an attacker recovering the key. The benefit
//! of one-time authentication is that it is generally much faster than a full HMAC calculation, as
//! used in normal authentication.
//!
//! One-time authentication is best suited to online protocols where lots of short messages need to
//! be authenticated. A common pattern would be to establish a shared secret key between all
//! parties, then use the [`hash::kdf` module](crate::hash::kdf) to derive an authentication key
//! for each message, and authenticate the messages with this API.
//!
//! # Algorithm Details
//! This construction is referred to as a [Message Authentication
//! Code](https://en.wikipedia.org/wiki/Message_authentication_code) (MAC).
//! In particular, this API uses [Poly1305](https://en.wikipedia.org/wiki/Poly1305).
//!
//! # Security Considerations
//! A single key must *never* be used to generate an authentication tag for more than one message.
//!
//! A common, but dangerous, mistake is to try to verify a tag by generating the tag again yourself
//! via [`authenticate`], and naively comparing the tag you calculate with the other tag. This
//! opens the door to [timing attacks](https://en.wikipedia.org/wiki/Timing_attack). The [`verify`]
//! and [`Multipart::verify`] functions use a constant-time comparison between the tags, and should
//! be used whenever you want to verify a tag, rather than comparing tags yourself.
//!
//! The length of an authentication tag generated by this API may be insufficient for the
//! authentication of very long inputs (e.g: large file transfers).
//!
//! Poly1305 is explicitly *not* a hash function, unlike many classic authentication schemes.
//!
//! # Examples
//! Generating a one-time authentication tag for a message and verifying its validity (using
//! [`authenticate`] and [`verify`]):
//!
//! ```rust
//! use alkali::symmetric::one_time_auth;
//! // Generate a random, one-time key for the authentication
//! let key = one_time_auth::Key::generate().unwrap();
//! let message = b"Here's a message we wish to authenticate.";
//! let tag = one_time_auth::authenticate(message, &key).unwrap();
//!
//! // ...
//!
//! match one_time_auth::verify(message, &tag, &key) {
//!     Ok(_) => println!("Authentication succeeded!"),
//!     Err(alkali::AlkaliError::OneTimeAuthError(
//!         one_time_auth::OneTimeAuthError::AuthenticationFailed,
//!     )) => {
//!         panic!("Uh-oh, message altered!")
//!     }
//!     Err(_) => panic!("Some other error occurred"),
//! }
//! ```
//!
//! If you're receiving a message in chunks, it may make more sense to use the streaming API, which
//! allows you to specify the message to authenticate in multiple parts (using [`Multipart`]):
//!
//! ```rust
//! use alkali::symmetric::one_time_auth;
//!
//! // Here we specify a pre-existing key, rather than generating a new one (in this case just 32
//! // 0xdb bytes, which is obviously not secure).
//! let key = one_time_auth::Key::try_from(&[0xdb; 32]).unwrap();
//! let mut state = one_time_auth::Multipart::new(&key).unwrap();
//! state.update(b"Here's the first part");
//! state.update(b"... And the second!");
//! let tag = state.authenticate();
//!
//! // ...
//!
//! // Now let's verify the tag we just generated (switching up the chunks we add to the state):
//! let mut state = one_time_auth::Multipart::new(&key).unwrap();
//! state.update(b"Here");
//! state.update(b"'s the first ");
//! state.update(b"part... And the ");
//! state.update(b"second!");
//! assert!(state.verify(&tag).is_ok());
//! ```

use crate::{hardened_buffer, mem, require_init, AlkaliError};
use libsodium_sys as sodium;
use std::marker::PhantomData;
use std::ptr;
use thiserror::Error;

/// Error type returned if something went wrong in the `one_time_auth` module.
#[derive(Clone, Copy, Debug, Eq, Error, PartialEq)]
pub enum OneTimeAuthError {
    /// Failed to authenticate a message.
    ///
    /// This may indicate an attempted forgery, a transmission error, or that you're using the
    /// wrong key. In any case, the authenticity of the message can't be verified, and it should
    /// not be trusted.
    #[error("authentication failed")]
    AuthenticationFailed,
}

/// The length of a key to authenticate a message, in bytes.
pub const KEY_LENGTH: usize = sodium::crypto_onetimeauth_poly1305_KEYBYTES as usize;

/// The length of an authentication tag, in bytes.
pub const TAG_LENGTH: usize = sodium::crypto_onetimeauth_poly1305_BYTES as usize;

hardened_buffer! {
    /// Secret key for one-time authentication.
    ///
    /// There are no technical constraints on the contents of a key, but it should be
    /// indistinguishable from random data. In general, a key will be derived as part of an online
    /// protocol, but it can be generated randomly using [`Key::generate`] if need be.
    ///
    /// A secret key should not be made public.
    ///
    /// This is a [hardened buffer type](https://docs.rs/alkali#hardened-buffer-types), and will be
    /// zeroed on drop. A number of other security measures are taken to protect its contents.
    Key(KEY_LENGTH);
}

impl Key {
    /// Generate a new, random key for use in one-time authentication.
    pub fn generate() -> Result<Self, AlkaliError> {
        require_init()?;

        let mut key = Self::new_empty()?;
        unsafe {
            // SAFETY: This function expects a pointer to a region of memory sufficient to store a
            // key for this algorithm. We have defined the `Key` type based on the
            // crypto_onetimeauth_KEYBYTES constant from Sodium, so it definitely has the correct
            // amount of space allocated to store the key. The `Key::inner_mut` method simply
            // returns a mutable pointer to the backing memory.
            sodium::crypto_onetimeauth_poly1305_keygen(key.inner_mut() as *mut libc::c_uchar);
        }
        Ok(key)
    }
}

/// A one-time authentication tag for a message.
pub type Tag = [u8; TAG_LENGTH];

/// Streaming one-time authentication API, for long/multi-part message authentication.
///
/// This is used to calculate an authentication tag for a message which is received in portions, or
/// which is too large to fit into memory (although please see the security considerations if using
/// this API for very large messages).
///
/// # Security Considerations
/// A single key must *never* be used to generate an authentication tag for more than one message.
///
/// A common, but dangerous, mistake is to try to verify a tag by generating the tag again yourself
/// via [`Multipart::authenticate`], and naively comparing the tag you calculate with the other
/// tag. This opens the door to [timing attacks](https://en.wikipedia.org/wiki/Timing_attack). The
/// [`Multipart::verify`] method uses a constant-time comparison between the tags, and should be
/// used whenever you want to verify a tag, rather than comparing tags yourself.
///
/// The length of an authentication tag generated by this API may be insufficient for the
/// authentication of very long inputs (e.g: large file transfers).
#[derive(Debug)]
pub struct Multipart {
    state: ptr::NonNull<sodium::crypto_onetimeauth_poly1305_state>,
    _marker: PhantomData<sodium::crypto_onetimeauth_poly1305_state>,
}

impl Multipart {
    /// Create a new instance of the struct.
    pub fn new(key: &Key) -> Result<Self, AlkaliError> {
        require_init()?;

        let state = unsafe {
            // SAFETY: This call to malloc will allocate the memory required for a
            // `crypto_onetimeauth_state` type, outside of Rust's memory management. The associated
            // memory is always freed in the corresponding `drop` call. We never free the memory in
            // any other place in this struct, and drop can only be called once, so a double-free
            // is not possible. We never give out a pointer to the allocated memory. See the drop
            // implementation for more reasoning on safety.
            let mut state = mem::malloc()?;

            // SAFETY: This function initialises a crypto_onetimeauth_state struct. The first
            // argument should be a pointer to a region of memory sufficient to store such a
            // struct, as defined in C. We pass a pointer to a region of memory sufficient to store
            // the struct as defined in Rust. This definition is is generated via bindgen, and as
            // such, is equivalent to the struct in C, so it is correct to use it as an argument
            // here. The next argument specifies the key to use for the authentication. We have
            // defined the `Key` type to be crypto_onetimeauth_KEYBYTES bytes long, so it is the
            // correct size to use here.
            sodium::crypto_onetimeauth_poly1305_init(
                state.as_mut(),
                key.inner() as *const libc::c_uchar,
            );

            // The state is now initialised correctly.
            state
        };

        Ok(Self {
            state,
            _marker: PhantomData,
        })
    }

    /// Create a new instance of the struct, in the same state as this one.
    pub fn try_clone(&self) -> Result<Self, AlkaliError> {
        let state = unsafe {
            // SAFETY: This call to malloc() will allocate the memory required for a
            // `crypto_onetimeauth_state` type, outside of Rust's memory management. The associated
            // memory is always freed in the corresponding `drop` call. We never free the memory in
            // any other place in this struct, and drop can only be called once, so a double-free
            // is not possible. We never give out a pointer to the allocated memory. See the drop
            // implementation for more reasoning on safety.
            let mut state = mem::malloc()?;

            // SAFETY: We have called `malloc` to allocate sufficient space for one
            // `crypto_onetimeauth_state` struct at each of the two pointers used here, so both are
            // valid for reads/writes of `size_of::<crypto_onetimeauth_state>` bytes. We have just
            // allocated a fresh region of memory for `state`, so it definitely doesn't overlap
            // with `self.state`.
            ptr::copy_nonoverlapping(self.state.as_ptr(), state.as_mut(), 1);

            state
        };

        Ok(Self {
            state,
            _marker: std::marker::PhantomData,
        })
    }

    /// Add message contents to be authenticated.
    pub fn update(&mut self, chunk: &[u8]) {
        unsafe {
            // SAFETY: The first argument to this function is a pointer to a
            // `crypto_onetimeauth_state` struct. The struct must be initialised in order to
            // instantiate the `Multipart` struct, so it is in the correct state to call the
            // crypto_onetimeauth_update function. The next two arguments specify the portion of
            // the message to add to the authentication tag calculation, and its length. We use
            // `chunk.len()` to specify the length, so it is correct for the `chunk` pointer.
            sodium::crypto_onetimeauth_poly1305_update(
                self.state.as_mut(),
                chunk.as_ptr(),
                chunk.len() as libc::c_ulonglong,
            );
        }
    }

    /// Calculate a one-time authentation tag for the specified message and key.
    ///
    /// Equivalent to [`authenticate`] for single-part messages.
    pub fn authenticate(mut self) -> Tag {
        let mut tag = [0u8; TAG_LENGTH];
        unsafe {
            // SAFETY: The first argument to this function is a pointer to a
            // `crypto_onetimeauth_state` struct. The struct must be initialised in order to
            // instantiate the `Multipart` struct, so it is in the correct state to call the
            // crypto_onetimeauth_update function. The next argument specifies the destination to
            // which the authentication tag should be written. We have defined the `Tag` type to be
            // `crypto_onetimeauth_BYTES` bytes long, so it is of adequate size to store the tag.
            sodium::crypto_onetimeauth_poly1305_final(self.state.as_mut(), tag.as_mut_ptr());
        }
        tag
    }

    /// Verify the provided tag is correct for the specified message and key.
    ///
    /// Returns a [`OneTimeAuthError::AuthenticationFailed`] if verification of the one-time
    /// authentication tag failed.
    ///
    /// Equivalent to [`verify`] for single-part messages.
    pub fn verify(mut self, tag: &Tag) -> Result<(), AlkaliError> {
        let mut actual_tag = [0u8; TAG_LENGTH];
        unsafe {
            // SAFETY: The first argument to this function is a pointer to a
            // `crypto_onetimeauth_state` struct. The struct must be initialised in order to
            // instantiate the `Multipart` struct, so it is in the correct state to call the
            // crypto_onetimeauth_update function. The next argument specifies the destination to
            // which the authentication tag should be written. We have defined the `Tag` type to be
            // `crypto_onetimeauth_BYTES` bytes long, so it is of adequate size to store the tag.
            sodium::crypto_onetimeauth_poly1305_final(self.state.as_mut(), actual_tag.as_mut_ptr());
        }

        if crate::util::eq(tag, &actual_tag)? {
            Ok(())
        } else {
            Err(OneTimeAuthError::AuthenticationFailed.into())
        }
    }
}

impl Drop for Multipart {
    fn drop(&mut self) {
        unsafe {
            // SAFETY:
            // * Is a double-free possible in safe code?
            //   * No: We only free in `drop`, which cannot be called manually, and is called
            //     exactly once when the struct is actually dropped. Once the value is dropped,
            //     there's no way to call the method again to cause a double free.
            // * Is a use-after-free possible in safe code?
            //   * No: We only ever free a buffer on drop, and after drop, none of the type's
            //     methods are accessible.
            // * Is a memory leak possible in safe code?
            //   * Yes: If the user uses something like `Box::leak()`, `ManuallyDrop`, or
            //     `std::mem::forget`, the destructor will not be called even though the struct is
            //     dropped. However, it is documented that in these cases heap memory may be
            //     leaked, so this is expected behaviour. In addition, certain signal interrupts or
            //     using panic=abort behaviour will mean the destructor is not called. There's
            //     little we can do about this, but a failure to free is probably reasonable in
            //     such cases. In any other case, `drop` will be called, and the memory freed.
            mem::free(self.state);
        }
    }
}

/// Compute the one-time authentication tag for the given message and key.
///
/// Returns the one-time authentication tag, which is non-secret.
///
/// # Security Considerations
/// A single key must *never* be used to generate an authentication tag for more than one message.
///
/// A common, but dangerous, mistake is to try to verify a tag by generating the tag again yourself
/// via [`authenticate`], and naively comparing the tag you calculate with the other tag. This
/// opens the door to [timing attacks](https://en.wikipedia.org/wiki/Timing_attack). The [`verify`]
/// function uses a constant-time comparison between the tags, and should be used whenever you want
/// to verify a tag, rather than comparing tags yourself.
///
/// The length of an authentication tag generated by this API may be insufficient for the
/// authentication of very long inputs (e.g: large file transfers).
pub fn authenticate(message: &[u8], key: &Key) -> Result<Tag, AlkaliError> {
    require_init()?;

    let mut tag = [0u8; TAG_LENGTH];
    unsafe {
        // SAFETY: The first argument to this function is a pointer to which the calculated
        // authentication tag will be written. We have defined the `tag` buffer to be
        // `crypto_onetimeauth_BYTES` bytes long, so it is of the correct size to store a tag. The
        // next two arguments specify the message to authenticate. We use `message.len()` to
        // specify the length of the message, so it is correct for this pointer. The final argument
        // specifies the key to use for this message's authentication. We define the `Key` type to
        // be `crypto_onetimeauth_KEYBYTES` long, so it is of the expected size for use in this
        // function. The `Key::inner` method simply returns a pointer to the backing memory.
        sodium::crypto_onetimeauth_poly1305(
            tag.as_mut_ptr(),
            message.as_ptr(),
            message.len() as libc::c_ulonglong,
            key.inner() as *const libc::c_uchar,
        );
    }

    Ok(tag)
}

/// Verify that an authentication tag is valid for a given message and key.
///
/// Returns a [`OneTimeAuthError::AuthenticationFailed`] if verification of the authentication tag
/// failed.
pub fn verify(message: &[u8], tag: &Tag, key: &Key) -> Result<(), AlkaliError> {
    require_init()?;

    let verification_result = unsafe {
        // SAFETY: The first argument to this function is the authentication tag to verify. We have
        // defined the `Tag` type to be `crypto_onetimeauth_BYTES` bytes long, so it is of the
        // expected size for use here. The next two arguments specify the message to verify. We use
        // `message.len()` to specify the length of the message, so it is correct for this pointer.
        // The final argument specifies the key to use for this message's verification. We define
        // the `Key` type to be `crypto_onetimeauth_KEYBYTES` long, so it is of the expected size
        // for use in this function. The `Key::inner` method simply returns a pointer to the
        // backing memory.
        sodium::crypto_onetimeauth_poly1305_verify(
            tag.as_ptr(),
            message.as_ptr(),
            message.len() as libc::c_ulonglong,
            key.inner() as *const libc::c_uchar,
        )
    };

    if verification_result == 0 {
        Ok(())
    } else {
        Err(OneTimeAuthError::AuthenticationFailed.into())
    }
}

#[cfg(test)]
mod tests {
    use super::{authenticate, verify, Key, Multipart};
    use crate::random::fill_random;
    use crate::AlkaliError;

    #[test]
    fn key_generation() -> Result<(), AlkaliError> {
        let _key = Key::generate()?;
        Ok(())
    }

    #[test]
    fn auth_and_verify() -> Result<(), AlkaliError> {
        let key_a = Key::generate()?;
        let key_b = Key::generate()?;
        let key_c = Key::generate()?;
        let key_d = Key::generate()?;

        let buf_a = [];
        let mut buf_b = [0; 16];
        let mut buf_c = [0; 1024];
        let mut buf_d = [0; 1 << 20];

        fill_random(&mut buf_b)?;
        fill_random(&mut buf_c)?;
        fill_random(&mut buf_d)?;

        let tag_a = authenticate(&buf_a, &key_a)?;
        let tag_b = authenticate(&buf_b, &key_b)?;
        let tag_c = authenticate(&buf_c, &key_c)?;
        let tag_d = authenticate(&buf_d, &key_d)?;

        verify(&buf_a, &tag_a, &key_a)?;
        verify(&buf_b, &tag_b, &key_b)?;
        verify(&buf_c, &tag_c, &key_c)?;
        verify(&buf_d, &tag_d, &key_d)?;

        fill_random(&mut buf_b)?;
        fill_random(&mut buf_c)?;
        fill_random(&mut buf_d)?;

        assert!(verify(&buf_b, &tag_b, &key_b).is_err());
        assert!(verify(&buf_c, &tag_c, &key_c).is_err());
        assert!(verify(&buf_d, &tag_d, &key_d).is_err());

        Ok(())
    }

    #[test]
    fn single_part_test_vectors() -> Result<(), AlkaliError> {
        let mut key = Key::new_empty()?;
        key.copy_from_slice(&[
            0xee, 0xa6, 0xa7, 0x25, 0x1c, 0x1e, 0x72, 0x91, 0x6d, 0x11, 0xc2, 0xcb, 0x21, 0x4d,
            0x3c, 0x25, 0x25, 0x39, 0x12, 0x1d, 0x8e, 0x23, 0x4e, 0x65, 0x2d, 0x65, 0x1f, 0xa4,
            0xc8, 0xcf, 0xf8, 0x80,
        ]);
        let message = [
            0x8e, 0x99, 0x3b, 0x9f, 0x48, 0x68, 0x12, 0x73, 0xc2, 0x96, 0x50, 0xba, 0x32, 0xfc,
            0x76, 0xce, 0x48, 0x33, 0x2e, 0xa7, 0x16, 0x4d, 0x96, 0xa4, 0x47, 0x6f, 0xb8, 0xc5,
            0x31, 0xa1, 0x18, 0x6a, 0xc0, 0xdf, 0xc1, 0x7c, 0x98, 0xdc, 0xe8, 0x7b, 0x4d, 0xa7,
            0xf0, 0x11, 0xec, 0x48, 0xc9, 0x72, 0x71, 0xd2, 0xc2, 0x0f, 0x9b, 0x92, 0x8f, 0xe2,
            0x27, 0x0d, 0x6f, 0xb8, 0x63, 0xd5, 0x17, 0x38, 0xb4, 0x8e, 0xee, 0xe3, 0x14, 0xa7,
            0xcc, 0x8a, 0xb9, 0x32, 0x16, 0x45, 0x48, 0xe5, 0x26, 0xae, 0x90, 0x22, 0x43, 0x68,
            0x51, 0x7a, 0xcf, 0xea, 0xbd, 0x6b, 0xb3, 0x73, 0x2b, 0xc0, 0xe9, 0xda, 0x99, 0x83,
            0x2b, 0x61, 0xca, 0x01, 0xb6, 0xde, 0x56, 0x24, 0x4a, 0x9e, 0x88, 0xd5, 0xf9, 0xb3,
            0x79, 0x73, 0xf6, 0x22, 0xa4, 0x3d, 0x14, 0xa6, 0x59, 0x9b, 0x1f, 0x65, 0x4c, 0xb4,
            0x5a, 0x74, 0xe3, 0x55, 0xa5,
        ];

        let tag = authenticate(&message, &key)?;
        assert_eq!(
            tag,
            [
                0xf3, 0xff, 0xc7, 0x70, 0x3f, 0x94, 0x00, 0xe5, 0x2a, 0x7d, 0xfb, 0x4b, 0x3d, 0x33,
                0x05, 0xd9,
            ]
        );
        verify(&message, &tag, &key)?;

        Ok(())
    }

    #[test]
    fn multi_part_test_vectors() -> Result<(), AlkaliError> {
        let mut key = Key::new_empty()?;
        key.copy_from_slice(&[
            0xee, 0xa6, 0xa7, 0x25, 0x1c, 0x1e, 0x72, 0x91, 0x6d, 0x11, 0xc2, 0xcb, 0x21, 0x4d,
            0x3c, 0x25, 0x25, 0x39, 0x12, 0x1d, 0x8e, 0x23, 0x4e, 0x65, 0x2d, 0x65, 0x1f, 0xa4,
            0xc8, 0xcf, 0xf8, 0x80,
        ]);

        let mut state = Multipart::new(&key)?;
        state.update(&[
            0x8e, 0x99, 0x3b, 0x9f, 0x48, 0x68, 0x12, 0x73, 0xc2, 0x96, 0x50, 0xba, 0x32, 0xfc,
            0x76, 0xce, 0x48, 0x33, 0x2e, 0xa7, 0x16, 0x4d, 0x96, 0xa4, 0x47, 0x6f, 0xb8, 0xc5,
            0x31, 0xa1, 0x18, 0x6a, 0xc0, 0xdf, 0xc1, 0x7c, 0x98, 0xdc, 0xe8, 0x7b, 0x4d, 0xa7,
            0xf0, 0x11, 0xec, 0x48, 0xc9, 0x72, 0x71, 0xd2, 0xc2, 0x0f, 0x9b, 0x92, 0x8f, 0xe2,
            0x27, 0x0d, 0x6f, 0xb8, 0x63, 0xd5, 0x17, 0x38, 0xb4, 0x8e, 0xee, 0xe3, 0x14, 0xa7,
            0xcc, 0x8a, 0xb9, 0x32, 0x16, 0x45, 0x48, 0xe5, 0x26, 0xae, 0x90, 0x22, 0x43, 0x68,
            0x51, 0x7a, 0xcf, 0xea, 0xbd, 0x6b, 0xb3, 0x73, 0x2b, 0xc0, 0xe9, 0xda, 0x99, 0x83,
            0x2b, 0x61,
        ]);
        state.update(&[]);
        state.update(&[
            0xca, 0x01, 0xb6, 0xde, 0x56, 0x24, 0x4a, 0x9e, 0x88, 0xd5, 0xf9, 0xb3, 0x79, 0x73,
            0xf6, 0x22, 0xa4, 0x3d, 0x14, 0xa6, 0x59, 0x9b, 0x1f, 0x65, 0x4c, 0xb4, 0x5a, 0x74,
            0xe3, 0x55, 0xa5,
        ]);

        let tag = state.authenticate();
        assert_eq!(
            tag,
            [
                0xf3, 0xff, 0xc7, 0x70, 0x3f, 0x94, 0x00, 0xe5, 0x2a, 0x7d, 0xfb, 0x4b, 0x3d, 0x33,
                0x05, 0xd9,
            ]
        );

        let mut state = Multipart::new(&key)?;
        state.update(&[
            0x8e, 0x99, 0x3b, 0x9f, 0x48, 0x68, 0x12, 0x73, 0xc2, 0x96, 0x50, 0xba, 0x32, 0xfc,
            0x76, 0xce, 0x48, 0x33, 0x2e, 0xa7, 0x16, 0x4d, 0x96, 0xa4, 0x47, 0x6f, 0xb8, 0xc5,
            0x31, 0xa1, 0x18, 0x6a, 0xc0, 0xdf, 0xc1, 0x7c, 0x98, 0xdc, 0xe8, 0x7b, 0x4d, 0xa7,
            0xf0, 0x11, 0xec, 0x48, 0xc9, 0x72, 0x71, 0xd2, 0xc2, 0x0f, 0x9b, 0x92, 0x8f, 0xe2,
            0x27, 0x0d, 0x6f, 0xb8, 0x63, 0xd5, 0x17, 0x38, 0xb4, 0x8e, 0xee, 0xe3, 0x14, 0xa7,
            0xcc, 0x8a, 0xb9, 0x32, 0x16, 0x45, 0x48, 0xe5, 0x26, 0xae, 0x90, 0x22, 0x43, 0x68,
            0x51, 0x7a, 0xcf, 0xea, 0xbd, 0x6b, 0xb3, 0x73, 0x2b, 0xc0, 0xe9, 0xda, 0x99, 0x83,
            0x2b, 0x61,
        ]);
        state.update(&[]);
        state.update(&[
            0xca, 0x01, 0xb6, 0xde, 0x56, 0x24, 0x4a, 0x9e, 0x88, 0xd5, 0xf9, 0xb3, 0x79, 0x73,
            0xf6, 0x22, 0xa4, 0x3d, 0x14, 0xa6, 0x59, 0x9b, 0x1f, 0x65, 0x4c, 0xb4, 0x5a, 0x74,
            0xe3, 0x55, 0xa5,
        ]);

        state.verify(&[
            0xf3, 0xff, 0xc7, 0x70, 0x3f, 0x94, 0x00, 0xe5, 0x2a, 0x7d, 0xfb, 0x4b, 0x3d, 0x33,
            0x05, 0xd9,
        ])?;

        Ok(())
    }
}
