//! Generic hash function.
//!
//! This module corresponds to the [`crypto_generichash`
//! API](https://doc.libsodium.org/hashing/generic_hashing) from Sodium.
//!
//! This is a fast [cryptographic hash
//! function](https://en.wikipedia.org/wiki/Cryptographic_hash_function), suitable for use in the
//! standard applications where such a function is necessary. Given an arbitrarily-long input, a
//! cryptographic hash function produces a fixed-length fingerprint (the *digest*). This digest can
//! then be used to identify the original input, as it should be infeasible to find another input
//! which hashes to the same value. See the [`hash` docs](super) for more information on the
//! properties of a cryptographic hash function.
//!
//! This generic hash function can be used in cases such as file integrity checking or generating
//! unique identifiers to index arbitrarily-long data. Please note that the [`hash`](super) module
//! has a number of alternative members which are better suited to certain tasks: For example, the
//! [`hash::pbkdf`](crate::hash::pbkdf) module should be used for password hashing, and the
//! [`hash::short`](crate::hash::short) module is better suited to the construction of hash
//! tables/bloom filters.
//!
//! # Algorithm Details
//! [BLAKE2b](https://www.blake2.net/) is used in this API, a fast, modern hash algorithm optimised
//! for 64-bit platforms.
//!
//! # Security Considerations
//! Generic hash functions *must not* be used for password hashing, they are not sufficiently
//! computationally intensive. Instead, use a [Password-Based Key Derivation
//! Function](https://en.wikipedia.org/wiki/Key_derivation_function#Password_hashing) (PBKDF) such
//! as those available in the [`hash::pbkdf`](crate::hash::pbkdf) module.
//!
//! # Examples
//! Single-part hashing (using [`hash`]):
//!
//! ```rust
//! use alkali::hash::generic;
//!
//! let message = b"Here's some message we wish to hash :)";
//! let hash = generic::hash(message).unwrap();
//! assert_eq!(
//!     hash,
//!     [
//!         0xaa, 0x42, 0xa0, 0xb0, 0xdc, 0x4a, 0x34, 0xf6, 0xb5, 0x4a, 0xa0, 0x33, 0x7b, 0xcb,
//!         0x90, 0x2c, 0xf1, 0x82, 0xfe, 0xb6, 0x6a, 0xe8, 0x9f, 0x05, 0xb3, 0x05, 0xfb, 0xac,
//!         0x9d, 0xad, 0x0d, 0x3b,
//!     ]
//! );
//! ```
//!
//! Keyed hashing (using [`hash_keyed`]):
//!
//! ```rust
//! use alkali::hash::generic;
//!
//! let message = b"We also wish to hash this message ;)";
//!
//! // Generate a random key for the hash
//! let key_a = generic::Key::generate().unwrap();
//! let hash_a = generic::hash_keyed(message, &key_a).unwrap();
//!
//! // We'll now hash the same message with a different key
//! let key_b = generic::Key::generate().unwrap();
//! let hash_b = generic::hash_keyed(message, &key_b).unwrap();
//!
//! // Hashing the same message with a different key should produce a different hash
//! assert_ne!(hash_a, hash_b);
//! ```
//!
//! Multi-part hashing (using [`Multipart`]):
//!
//! ```rust
//! use alkali::hash::generic;
//!
//! let mut output_a = [0u8; generic::DIGEST_LENGTH_DEFAULT];
//! let mut state_a = generic::Multipart::new(&mut output_a).unwrap();
//! state_a.update(b"Here's the first part");
//! state_a.update(b"... And the second!");
//! state_a.calculate_hash();
//!
//! let mut output_b = [255u8; generic::DIGEST_LENGTH_DEFAULT];
//! let mut state_b = generic::Multipart::new(&mut output_b).unwrap();
//! state_b.update(b"Here");
//! state_b.update(b"'s the first ");
//! state_b.update(b"part... And the ");
//! state_b.update(b"second!");
//! state_b.calculate_hash();
//!
//! assert_eq!(output_a, output_b);
//! ```

// TODO: Consider how/if we wish to expose the salt/personalisation parameters of the hash.

use crate::{hardened_buffer, require_init, AlkaliError};
use libsodium_sys as sodium;
use std::mem::MaybeUninit;
use std::ptr;
use thiserror::Error;

/// Error type returned if something went wrong in the generic module.
#[derive(Clone, Copy, Debug, Eq, Error, PartialEq)]
pub enum GenericHashError {
    /// The desired digest length was too short or too long for use with this algorithm.
    ///
    /// The output size of this hash must be at least [`DIGEST_LENGTH_MIN`], and at most
    /// [`DIGEST_LENGTH_MAX`] bytes. A minimum of [`DIGEST_LENGTH_DEFAULT`] should be used if
    /// collisions should be avoided.
    #[error("digest length outside acceptable range")]
    DigestLengthInvalid,

    /// The provided key was too long for use with this algorithm.
    ///
    /// A key for use with this hash must be at most [`KEY_LENGTH_MAX`] bytes. Using a key size of
    /// [`KEY_LENGTH_DEFAULT`] is recommended, and a key shorter than [`KEY_LENGTH_MIN`] should not
    /// be used if it is intended to be secret.
    #[error("key length outsize acceptable range")]
    KeyLengthInvalid,
}

/// The minimum output length of the hash function, in bytes.
///
/// Below the [default output length](DIGEST_LENGTH_DEFAULT), the hash function may not satisfy the
/// expected property of collision resistance, and as such, using a lower output size than
/// [`DIGEST_LENGTH_DEFAULT`] is not recommended.
pub const DIGEST_LENGTH_MIN: usize = sodium::crypto_generichash_blake2b_BYTES_MIN as usize;

/// The default length of the output of the hash function, in bytes.
///
/// This is the default length of the output of the hash function, and is the recommended minimum
/// size for security. At this size, it is computationally infeasible to find two inputs with the
/// same digest. Below the default output length, the hash function may not satisfy the expected
/// property of collision resistance, and as such, using a lower output size than this constant is
/// not recommended.
pub const DIGEST_LENGTH_DEFAULT: usize = sodium::crypto_generichash_blake2b_BYTES as usize;

/// The maximum output length of the hash function, in bytes.
///
/// Below the [default output length](DIGEST_LENGTH_DEFAULT), the hash function may not satisfy the
/// expected property of collision resistance, and as such, using a lower output size than
/// [`DIGEST_LENGTH_DEFAULT`] is not recommended.
pub const DIGEST_LENGTH_MAX: usize = sodium::crypto_generichash_blake2b_BYTES_MAX as usize;

/// The minimum recommended key size for use with the keyed variant of the hash function, in bytes.
///
/// If the key is intended to be a secret value, a key length below this size should not be used.
pub const KEY_LENGTH_MIN: usize = sodium::crypto_generichash_blake2b_KEYBYTES_MIN as usize;

/// The recommended key size for use with the keyed variant of the hash function, in bytes.
///
/// This length should always be sufficient if the key is intended to be a secret value. A key
/// length below [`KEY_LENGTH_MIN`] should not be used.
pub const KEY_LENGTH_DEFAULT: usize = sodium::crypto_generichash_blake2b_KEYBYTES as usize;

/// The maximum key size for use with the keyed variant of the hash function, in bytes.
///
/// A key length below [`KEY_LENGTH_MIN`] should not be used.
pub const KEY_LENGTH_MAX: usize = sodium::crypto_generichash_blake2b_KEYBYTES_MAX as usize;

hardened_buffer! {
    /// Secret key for the keyed variant of the hash function.
    ///
    /// This key is the default (recommended) length, [`KEY_LENGTH_DEFAULT`].
    ///
    /// There are no technical constraints on the contents of a key, but it should be generated
    /// randomly using [`Key::generate`].
    ///
    /// A secret key must not be made public.
    ///
    /// This is a [hardened buffer type](https://docs.rs/alkali#hardened-buffer-types), and will be
    /// zeroed on drop. A number of other security measures are taken to protect its contents.
    Key(KEY_LENGTH_DEFAULT);
}

impl Key {
    /// Generate a new, random key for use with the keyed variant of the hash function.
    pub fn generate() -> Result<Self, AlkaliError> {
        require_init()?;

        let mut key = Self::new_empty()?;
        unsafe {
            // SAFETY: This function expects a pointer to a region of memory sufficient to store a
            // key for BLAKE2b of the default size. We have defined this type based on the
            // crypto_generichash_blake2b_KEYBYTES constant from Sodium, so it definitely has the
            // correct amount of space allocated to store the key. The Key::inner_mut method simply
            // gives a mutable pointer to the backing memory.
            sodium::crypto_generichash_blake2b_keygen(key.inner_mut() as *mut libc::c_uchar);
        }
        Ok(key)
    }
}

/// The digest generated by this hash function, of the default length.
pub type Digest = [u8; DIGEST_LENGTH_DEFAULT];

/// Streaming hash API, for long/multi-part message hashing.
///
/// This can be used to calculate the hash of a message which is too large to fit into memory, or
/// where the message is received in portions.
///
/// # Security Considerations
/// The output size for the hash should be at least [`DIGEST_LENGTH_DEFAULT`] bytes to ensure
/// collision-resistance.
#[derive(Debug)]
pub struct Multipart<'o> {
    output: &'o mut [u8],
    state: sodium::crypto_generichash_blake2b_state,
}

impl<'o> Multipart<'o> {
    /// Create a new instance of the struct.
    ///
    /// `output` should be the location to which the calculated hash will be written. It should be
    /// between [`DIGEST_LENGTH_MIN`] and [`DIGEST_LENGTH_MAX`] bytes. To ensure
    /// collision-resistance, a minimum of [`DIGEST_LENGTH_DEFAULT`] should be used.
    pub fn new(output: &'o mut [u8]) -> Result<Self, AlkaliError> {
        require_init()?;

        if output.len() < DIGEST_LENGTH_MIN || output.len() > DIGEST_LENGTH_MAX {
            return Err(GenericHashError::DigestLengthInvalid.into());
        }

        let mut state_uninit = MaybeUninit::uninit();
        let state = unsafe {
            // SAFETY: This function initialises a `crypto_generichash_blake2b_state` struct. The
            // first argument should be a pointer to a crypto_generichash_blake2b_state struct. We
            // pass a pointer to a region of memory sufficient to store the struct as defined in
            // Rust, rather than C. This definition is generated via bindgen, and as such, is
            // equivalent to the struct in C, so it is correct to use it as an argument for this
            // function. The second and third arguments specify the key to use, and its length. It
            // is specified in the Sodium documentation that using a NULL pointer for the key
            // argument means it is simply ignored, along with the key size, which is the desired
            // behaviour here. The final argument is the desired length of the hash output. We use
            // `output.len()`, which we have ensured is acceptable above.
            sodium::crypto_generichash_blake2b_init(
                state_uninit.as_mut_ptr(),
                ptr::null::<libc::c_uchar>(),
                0,
                output.len(),
            );

            // SAFETY: Following the crypto_generichash_blake2b_init call, the struct is correctly
            // initialised, so it is safe to assume its initialised state.
            state_uninit.assume_init()
        };

        Ok(Self { state, output })
    }

    /// Create a new instance of the struct, using the provided key in the hash calculation.
    ///
    /// `output` should be the location to which the calculated hash will be written. It should be
    /// between [`DIGEST_LENGTH_MIN`] and [`DIGEST_LENGTH_MAX`] bytes. To ensure
    /// collision-resistance, a minimum of [`DIGEST_LENGTH_DEFAULT`] should be used.
    ///
    /// `key` will be used in the calculation of the hash.  The same `(message, key)` combination
    /// will always produce the same hash. A different key will produce a different hash for the
    /// same message.
    ///
    /// This can be used to ensure different applications generate different hashes even when
    /// processing the same data, or as a
    /// [MAC](https://en.wikipedia.org/wiki/Message_authentication_code).
    pub fn new_keyed(output: &'o mut [u8], key: &Key) -> Result<Self, AlkaliError> {
        require_init()?;

        if output.len() < DIGEST_LENGTH_MIN || output.len() > DIGEST_LENGTH_MAX {
            return Err(GenericHashError::DigestLengthInvalid.into());
        }

        let mut state_uninit = MaybeUninit::uninit();
        let state = unsafe {
            // SAFETY: This function initialises a `crypto_generichash_blake2b_state` struct. The
            // first argument should be a pointer to a crypto_generichash_blake2b_state struct. We
            // pass a pointer to a region of memory sufficient to store the struct as defined in
            // Rust, rather than C. This definition is generated via bindgen, and as such, is
            // equivalent to the struct in C, so it is correct to use it as an argument for this
            // function. The second and third arguments specify the key to use, and its length. We
            // have defined the `Key` type to be suitable for use with this function. We use
            // `key.len()` to specify the length, so it is correct here. The final argument is the
            // desired length of the hash output. We use `output.len()`, which we have ensured is
            // acceptable above.
            sodium::crypto_generichash_blake2b_init(
                state_uninit.as_mut_ptr(),
                key.inner() as *const libc::c_uchar,
                key.len(),
                output.len(),
            );

            // SAFETY: Following the crypto_generichash_blake2b_init call, the struct is correctly
            // initialised, so it is safe to assume its initialised state.
            state_uninit.assume_init()
        };

        Ok(Self { state, output })
    }

    /// Add message contents to hash.
    pub fn update(&mut self, chunk: &[u8]) {
        unsafe {
            // SAFETY: The first argument should be a pointer to a
            // crypto_generichash_blake2b_state struct. We pass a pointer to the struct as defined
            // in Rust, rather than C. This definition is generated via bindgen, and as such, is
            // equivalent to the struct in C, so it is correct to use it as an argument for this
            // function. The initialisation of the `Multipart` struct requires a call to the `init`
            // function, so the struct is correctly initialised. The next two arguments specify the
            // portion of the message to write, and its length. We use `chunk.len()` to specify the
            // length of the message, so it is correct here.
            sodium::crypto_generichash_blake2b_update(
                &mut self.state,
                chunk.as_ptr(),
                chunk.len() as libc::c_ulonglong,
            );
        }
    }

    /// Calculate the hash and write it to `self.output`.
    ///
    /// The length of `self.output` is used to determine the length of the output. The whole slice
    /// will be filled.
    pub fn calculate_hash(mut self) {
        unsafe {
            // SAFETY: The first argument should be a pointer to a
            // crypto_generichash_blake2b_state struct. We pass a pointer to the struct as defined
            // in Rust, rather than C. This definition is generated via bindgen, and as such, is
            // equivalent to the struct in C, so it is correct to use it as an argument for this
            // function. The initialisation of the `Multipart` struct requires a call to the `init`
            // function, so the struct is correctly initialised. The next two arguments specify the
            // destination to which the hash should be written. In the initialisation of the
            // `Multipart` struct, we ensure the length of the output is valid for use here.
            sodium::crypto_generichash_blake2b_final(
                &mut self.state,
                self.output.as_mut_ptr(),
                self.output.len(),
            );
        }
    }
}

/// Calculate the hash of the provided message.
///
/// This function returns the hash of the given message, of the default output size, given by
/// [`DIGEST_LENGTH_DEFAULT`]. No key is used. The same message will always produce the same hash.
pub fn hash(message: &[u8]) -> Result<Digest, AlkaliError> {
    let mut digest = [0u8; DIGEST_LENGTH_DEFAULT];
    hash_custom(message, &mut digest)?;
    Ok(digest)
}

/// Calculate the hash of the provided message, outputting a hash of a custom length.
///
/// The [`hash`] function always outputs a hash of the default size, [`DIGEST_LENGTH_DEFAULT`].
/// This function allows a custom digest length to be specified, if this is necessary for your
/// use-case.
///
/// The hash of `message` will be written to `digest`, which must be between [`DIGEST_LENGTH_MIN`]
/// and [`DIGEST_LENGTH_MAX`] bytes. No key is used. The same message will always produce the same
/// hash.
///
/// # Security Considerations
/// An output size less than [`DIGEST_LENGTH_DEFAULT`] is not guaranteed to preserve the expected
/// property of collision resistance. It is recommended to just use [`hash`] unless you have a
/// specific use-case which requires a different output size.
pub fn hash_custom(message: &[u8], digest: &mut [u8]) -> Result<(), AlkaliError> {
    require_init()?;

    if digest.len() < DIGEST_LENGTH_MIN || digest.len() > DIGEST_LENGTH_MAX {
        return Err(GenericHashError::DigestLengthInvalid.into());
    }

    unsafe {
        // SAFETY: The first two parameters to this function specify the destination pointer to
        // which the calculated digest will be written, and the size for this digest. We ensure the
        // `digest` buffer is of appropriate size above, then specify its length using
        // `digest.len()`, so the length provided is correct for the slice. The next two parameters
        // specify the message to hash and its length. We use `message.len()` to specify the size,
        // so it is correct for this slice. The final two parameters specify the key to use in the
        // hash calculation, and its length. It is documented that if the key is set to be a NULL
        // pointer, then Sodium will ignore the key in the hash calculation, which is our desired
        // behaviour here. Therefore, we pass a NULL pointer for the key's location. Sodium will
        // then also ignore the provided key length, so it can be set to any value.
        sodium::crypto_generichash_blake2b(
            digest.as_mut_ptr(),
            digest.len(),
            message.as_ptr(),
            message.len() as libc::c_ulonglong,
            ptr::null::<libc::c_uchar>(),
            0,
        );
    }

    Ok(())
}

/// Calculate the hash of the provided message, using the provided key as part of the hash
/// calculation.
///
/// This function returns the hash of the given message, of the default output size
/// ([`DIGEST_LENGTH_DEFAULT`]), dependent upon the provided `key`. The same `(message, key)`
/// combination will always produce the same hash. A different key will produce a different hash
/// for the same message.
///
/// This can be used to ensure different applications generate different hashes even when
/// processing the same data, or as a
/// [MAC](https://en.wikipedia.org/wiki/Message_authentication_code).
///
/// # Security Considerations
/// With the default key size as used in this function, this function should be secure for use in
/// [message authentication](https://en.wikipedia.org/wiki/Message_authentication_code). However,
/// it is easier to use the [`symmetric::auth`](crate::symmetric::auth) API, which is specifically
/// intended for this purpose.
pub fn hash_keyed(message: &[u8], key: &Key) -> Result<Digest, AlkaliError> {
    let mut digest = [0u8; DIGEST_LENGTH_DEFAULT];
    hash_keyed_custom(message, key.as_ref(), &mut digest)?;
    Ok(digest)
}

/// Calculate the hash of the provided message, using the provided key as part of the hash
/// calculation, outputting a hash of a custom length. **Read the security considerations before
/// use**.
///
/// The [`hash_keyed`] function always outputs a hash of the default size,
/// [`DIGEST_LENGTH_DEFAULT`], and expects a key of the default size, [`KEY_LENGTH_DEFAULT`].
/// This function allows custom digest & key lengths to be specified, if this is necessary for your
/// use-case.
///
/// The hash of `message` will be written to `digest`, which must be between [`DIGEST_LENGTH_MIN`]
/// and [`DIGEST_LENGTH_MAX`] bytes. The key `key` will be used as part of the hash calculation,
/// and must be at most [`KEY_LENGTH_MAX`] bytes. The same `(message, key)` combination will always
/// produce the same hash. A different key will produce a different hash for the same message.
///
/// This can be used to ensure different applications generate different hashes even when
/// processing the same data. For a **sufficiently sized key**, this can also be used as a
/// [MAC](https://en.wikipedia.org/wiki/Message_authentication_code), but **if the key used is too
/// short, this is insecure**.
///
/// # Security Considerations
/// Great care should be taken if the key used for this function is intended to be a secret value,
/// and is not just being used for non-cryptographic purposes. If you are unsure what key size to
/// use, you should be using [`hash_keyed`].
///
/// If the key used is intended to be secret, it must be at least [`KEY_LENGTH_MIN`] bytes, and
/// should ideally be [`KEY_LENGTH_DEFAULT`] bytes or more. Shorter keys are insufficient for
/// cryptographic applications.
///
/// Secret keys must be generated randomly, use the [`random`](crate::random) API to generate
/// random data suitable for this use.
///
/// Keys should immediately be securely erased from memory when they are no longer required for
/// hash calculation. The [`Key`] type from this module does just this, and can be passed as an
/// argument to this function using `key.as_ref()` to obtain a `&[u8]` slice. If you are using a
/// key of a different size, you can use the [zeroize](https://crates.io/crates/zeroize) crate to
/// clear memory very simply, or use the [hard](https://crates.io/crates/hard) crate if you want to
/// make use of the other memory hardening utilities from Sodium.
///
/// For keys of at least [`KEY_LENGTH_MIN`] bytes, this function should be secure for use in
/// [message authentication](https://en.wikipedia.org/wiki/Message_authentication_code). However,
/// it is easier to use the [`symmetric::auth`](crate::symmetric::auth) API, which is specifically
/// intended for this purpose.
///
/// An output size less than [`DIGEST_LENGTH_DEFAULT`] is not guaranteed to preserve the expected
/// property of collision resistance. It is recommended to just use [`hash_keyed`] unless you have
/// a specific use-case which requires a different output size.
pub fn hash_keyed_custom(message: &[u8], key: &[u8], digest: &mut [u8]) -> Result<(), AlkaliError> {
    require_init()?;

    if digest.len() < DIGEST_LENGTH_MIN || digest.len() > DIGEST_LENGTH_MAX {
        return Err(GenericHashError::DigestLengthInvalid.into());
    } else if key.len() > KEY_LENGTH_MAX {
        return Err(GenericHashError::KeyLengthInvalid.into());
    }

    unsafe {
        // SAFETY: The first two parameters to this function specify the destination pointer to
        // which the calculated digest will be written, and the size for this digest. We ensure the
        // `digest` buffer is of appropriate size above, then specify its length using
        // `digest.len()`, so the length provided is correct for the slice. The next two parameters
        // specify the message to hash and its length. We use `message.len()` to specify the size,
        // so it is correct for this slice. The final two parameters specify the key to use in the
        // hash calculation, and its length. We ensure the `key` buffer is of appropriate size
        // above, then specify its length using `key.len()`, so the length provided is correct for
        // the slice.
        sodium::crypto_generichash_blake2b(
            digest.as_mut_ptr(),
            digest.len(),
            message.as_ptr(),
            message.len() as libc::c_ulonglong,
            key.as_ptr(),
            key.len(),
        );
    }

    Ok(())
}

#[cfg(test)]
mod tests {
    use super::{hash, hash_custom, hash_keyed, hash_keyed_custom, Key};
    use crate::AlkaliError;

    #[test]
    fn key_generation() -> Result<(), AlkaliError> {
        let _key = Key::generate()?;
        Ok(())
    }

    #[test]
    fn single_part_vectors() -> Result<(), AlkaliError> {
        let vectors = [
            (
                &[][..],
                [
                    0x0e, 0x57, 0x51, 0xc0, 0x26, 0xe5, 0x43, 0xb2, 0xe8, 0xab, 0x2e, 0xb0, 0x60,
                    0x99, 0xda, 0xa1, 0xd1, 0xe5, 0xdf, 0x47, 0x77, 0x8f, 0x77, 0x87, 0xfa, 0xab,
                    0x45, 0xcd, 0xf1, 0x2f, 0xe3, 0xa8,
                ],
            ),
            (
                &[0x00][..],
                [
                    0x03, 0x17, 0x0a, 0x2e, 0x75, 0x97, 0xb7, 0xb7, 0xe3, 0xd8, 0x4c, 0x05, 0x39,
                    0x1d, 0x13, 0x9a, 0x62, 0xb1, 0x57, 0xe7, 0x87, 0x86, 0xd8, 0xc0, 0x82, 0xf2,
                    0x9d, 0xcf, 0x4c, 0x11, 0x13, 0x14,
                ],
            ),
            (
                &[
                    0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c,
                    0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19,
                    0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
                ][..],
                [
                    0xcb, 0x2f, 0x51, 0x60, 0xfc, 0x1f, 0x7e, 0x05, 0xa5, 0x5e, 0xf4, 0x9d, 0x34,
                    0x0b, 0x48, 0xda, 0x2e, 0x5a, 0x78, 0x09, 0x9d, 0x53, 0x39, 0x33, 0x51, 0xcd,
                    0x57, 0x9d, 0xd4, 0x25, 0x03, 0xd6,
                ],
            ),
            (
                &[
                    0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c,
                    0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19,
                    0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26,
                    0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, 0x30, 0x31, 0x32, 0x33,
                    0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e,
                ][..],
                [
                    0x29, 0xe4, 0x1a, 0x64, 0xfb, 0xdd, 0x2f, 0xd2, 0x76, 0x12, 0x22, 0x86, 0x23,
                    0xc0, 0x70, 0x22, 0x22, 0xbf, 0x36, 0x74, 0x51, 0xe7, 0x32, 0x42, 0x87, 0xf1,
                    0x81, 0xcb, 0x3d, 0xcf, 0x72, 0x37,
                ],
            ),
            (
                &[
                    0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c,
                    0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19,
                    0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26,
                    0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, 0x30, 0x31, 0x32, 0x33,
                    0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f,
                ][..],
                [
                    0x10, 0xd8, 0xe6, 0xd5, 0x34, 0xb0, 0x09, 0x39, 0x84, 0x3f, 0xe9, 0xdc, 0xc4,
                    0xda, 0xe4, 0x8c, 0xdf, 0x00, 0x8f, 0x6b, 0x8b, 0x2b, 0x82, 0xb1, 0x56, 0xf5,
                    0x40, 0x4d, 0x87, 0x48, 0x87, 0xf5,
                ],
            ),
            (
                &[
                    0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c,
                    0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19,
                    0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26,
                    0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, 0x30, 0x31, 0x32, 0x33,
                    0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f, 0x40,
                ][..],
                [
                    0x84, 0xc0, 0x4a, 0xb0, 0x82, 0xc8, 0xae, 0x24, 0x20, 0x65, 0x61, 0xf7, 0x73,
                    0x97, 0x70, 0x4b, 0x62, 0x78, 0x92, 0x08, 0x9a, 0x05, 0x88, 0x7a, 0x2a, 0x19,
                    0x96, 0x47, 0x2b, 0xcf, 0xe1, 0x5d,
                ],
            ),
            (
                &[
                    0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c,
                    0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19,
                    0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26,
                    0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, 0x30, 0x31, 0x32, 0x33,
                    0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f, 0x40,
                    0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d,
                    0x4e, 0x4f, 0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58, 0x59, 0x5a,
                    0x5b, 0x5c, 0x5d, 0x5e, 0x5f, 0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67,
                    0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f, 0x70, 0x71, 0x72, 0x73, 0x74,
                    0x75, 0x76, 0x77, 0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f, 0x80, 0x81,
                    0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e,
                    0x8f, 0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9a, 0x9b,
                    0x9c, 0x9d, 0x9e, 0x9f, 0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 0xa8,
                    0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf, 0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5,
                    0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf, 0xc0, 0xc1, 0xc2,
                    0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,
                    0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 0xd8, 0xd9, 0xda, 0xdb, 0xdc,
                    0xdd, 0xde, 0xdf, 0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9,
                    0xea, 0xeb, 0xec, 0xed, 0xee, 0xef, 0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6,
                    0xf7, 0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe,
                ][..],
                [
                    0x1d, 0x08, 0x50, 0xee, 0x9b, 0xca, 0x0a, 0xbc, 0x96, 0x01, 0xe9, 0xde, 0xab,
                    0xe1, 0x41, 0x8f, 0xed, 0xec, 0x2f, 0xb6, 0xac, 0x41, 0x50, 0xbd, 0x53, 0x02,
                    0xd2, 0x43, 0x0f, 0x9b, 0xe9, 0x43,
                ],
            ),
        ];

        for v in vectors {
            let digest = hash(v.0)?;
            assert_eq!(digest, v.1);
        }

        Ok(())
    }

    #[test]
    fn single_part_custom_vectors() -> Result<(), AlkaliError> {
        let vectors = [
            (
                &[][..],
                [
                    0x78, 0x6a, 0x02, 0xf7, 0x42, 0x01, 0x59, 0x03, 0xc6, 0xc6, 0xfd, 0x85, 0x25,
                    0x52, 0xd2, 0x72, 0x91, 0x2f, 0x47, 0x40, 0xe1, 0x58, 0x47, 0x61, 0x8a, 0x86,
                    0xe2, 0x17, 0xf7, 0x1f, 0x54, 0x19, 0xd2, 0x5e, 0x10, 0x31, 0xaf, 0xee, 0x58,
                    0x53, 0x13, 0x89, 0x64, 0x44, 0x93, 0x4e, 0xb0, 0x4b, 0x90, 0x3a, 0x68, 0x5b,
                    0x14, 0x48, 0xb7, 0x55, 0xd5, 0x6f, 0x70, 0x1a, 0xfe, 0x9b, 0xe2, 0xce,
                ],
            ),
            (
                &[0x00][..],
                [
                    0x2f, 0xa3, 0xf6, 0x86, 0xdf, 0x87, 0x69, 0x95, 0x16, 0x7e, 0x7c, 0x2e, 0x5d,
                    0x74, 0xc4, 0xc7, 0xb6, 0xe4, 0x8f, 0x80, 0x68, 0xfe, 0x0e, 0x44, 0x20, 0x83,
                    0x44, 0xd4, 0x80, 0xf7, 0x90, 0x4c, 0x36, 0x96, 0x3e, 0x44, 0x11, 0x5f, 0xe3,
                    0xeb, 0x2a, 0x3a, 0xc8, 0x69, 0x4c, 0x28, 0xbc, 0xb4, 0xf5, 0xa0, 0xf3, 0x27,
                    0x6f, 0x2e, 0x79, 0x48, 0x7d, 0x82, 0x19, 0x05, 0x7a, 0x50, 0x6e, 0x4b,
                ],
            ),
            (
                &[
                    0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c,
                    0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19,
                    0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
                ][..],
                [
                    0x5c, 0x52, 0x92, 0x0a, 0x72, 0x63, 0xe3, 0x9d, 0x57, 0x92, 0x0c, 0xa0, 0xcb,
                    0x75, 0x2a, 0xc6, 0xd7, 0x9a, 0x04, 0xfe, 0xf8, 0xa7, 0xa2, 0x16, 0xa1, 0xec,
                    0xb7, 0x11, 0x5c, 0xe0, 0x6d, 0x89, 0xfd, 0x7d, 0x73, 0x5b, 0xd6, 0xf4, 0x27,
                    0x25, 0x55, 0xdb, 0xa2, 0x2c, 0x2d, 0x1c, 0x96, 0xe6, 0x35, 0x23, 0x22, 0xc6,
                    0x2c, 0x56, 0x30, 0xfd, 0xe0, 0xf4, 0x77, 0x7a, 0x76, 0xc3, 0xde, 0x2c,
                ],
            ),
            (
                &[
                    0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c,
                    0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19,
                    0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26,
                    0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, 0x30, 0x31, 0x32, 0x33,
                    0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e,
                ][..],
                [
                    0xd1, 0x0b, 0xf9, 0xa1, 0x5b, 0x1c, 0x9f, 0xc8, 0xd4, 0x1f, 0x89, 0xbb, 0x14,
                    0x0b, 0xf0, 0xbe, 0x08, 0xd2, 0xf3, 0x66, 0x61, 0x76, 0xd1, 0x3b, 0xaa, 0xc4,
                    0xd3, 0x81, 0x35, 0x8a, 0xd0, 0x74, 0xc9, 0xd4, 0x74, 0x8c, 0x30, 0x05, 0x20,
                    0xeb, 0x02, 0x6d, 0xae, 0xae, 0xa7, 0xc5, 0xb1, 0x58, 0x89, 0x2f, 0xde, 0x4e,
                    0x8e, 0xc1, 0x7d, 0xc9, 0x98, 0xdc, 0xd5, 0x07, 0xdf, 0x26, 0xeb, 0x63,
                ],
            ),
            (
                &[
                    0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c,
                    0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19,
                    0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26,
                    0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, 0x30, 0x31, 0x32, 0x33,
                    0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f,
                ][..],
                [
                    0x2f, 0xc6, 0xe6, 0x9f, 0xa2, 0x6a, 0x89, 0xa5, 0xed, 0x26, 0x90, 0x92, 0xcb,
                    0x9b, 0x2a, 0x44, 0x9a, 0x44, 0x09, 0xa7, 0xa4, 0x40, 0x11, 0xee, 0xca, 0xd1,
                    0x3d, 0x7c, 0x4b, 0x04, 0x56, 0x60, 0x2d, 0x40, 0x2f, 0xa5, 0x84, 0x4f, 0x1a,
                    0x7a, 0x75, 0x81, 0x36, 0xce, 0x3d, 0x5d, 0x8d, 0x0e, 0x8b, 0x86, 0x92, 0x1f,
                    0xff, 0xf4, 0xf6, 0x92, 0xdd, 0x95, 0xbd, 0xc8, 0xe5, 0xff, 0x00, 0x52,
                ],
            ),
            (
                &[
                    0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c,
                    0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19,
                    0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26,
                    0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, 0x30, 0x31, 0x32, 0x33,
                    0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f, 0x40,
                ][..],
                [
                    0xfc, 0xbe, 0x8b, 0xe7, 0xdc, 0xb4, 0x9a, 0x32, 0xdb, 0xdf, 0x23, 0x94, 0x59,
                    0xe2, 0x63, 0x08, 0xb8, 0x4d, 0xff, 0x1e, 0xa4, 0x80, 0xdf, 0x8d, 0x10, 0x4e,
                    0xef, 0xf3, 0x4b, 0x46, 0xfa, 0xe9, 0x86, 0x27, 0xb4, 0x50, 0xc2, 0x26, 0x7d,
                    0x48, 0xc0, 0x94, 0x6a, 0x69, 0x7c, 0x5b, 0x59, 0x53, 0x14, 0x52, 0xac, 0x04,
                    0x84, 0xf1, 0xc8, 0x4e, 0x3a, 0x33, 0xd0, 0xc3, 0x39, 0xbb, 0x2e, 0x28,
                ],
            ),
            (
                &[
                    0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c,
                    0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19,
                    0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26,
                    0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, 0x30, 0x31, 0x32, 0x33,
                    0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f, 0x40,
                    0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d,
                    0x4e, 0x4f, 0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58, 0x59, 0x5a,
                    0x5b, 0x5c, 0x5d, 0x5e, 0x5f, 0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67,
                    0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f, 0x70, 0x71, 0x72, 0x73, 0x74,
                    0x75, 0x76, 0x77, 0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f, 0x80, 0x81,
                    0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e,
                    0x8f, 0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9a, 0x9b,
                    0x9c, 0x9d, 0x9e, 0x9f, 0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 0xa8,
                    0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf, 0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5,
                    0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf, 0xc0, 0xc1, 0xc2,
                    0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,
                    0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 0xd8, 0xd9, 0xda, 0xdb, 0xdc,
                    0xdd, 0xde, 0xdf, 0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9,
                    0xea, 0xeb, 0xec, 0xed, 0xee, 0xef, 0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6,
                    0xf7, 0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe,
                ][..],
                [
                    0x5b, 0x21, 0xc5, 0xfd, 0x88, 0x68, 0x36, 0x76, 0x12, 0x47, 0x4f, 0xa2, 0xe7,
                    0x0e, 0x9c, 0xfa, 0x22, 0x01, 0xff, 0xee, 0xe8, 0xfa, 0xfa, 0xb5, 0x79, 0x7a,
                    0xd5, 0x8f, 0xef, 0xa1, 0x7c, 0x9b, 0x5b, 0x10, 0x7d, 0xa4, 0xa3, 0xdb, 0x63,
                    0x20, 0xba, 0xaf, 0x2c, 0x86, 0x17, 0xd5, 0xa5, 0x1d, 0xf9, 0x14, 0xae, 0x88,
                    0xda, 0x38, 0x67, 0xc2, 0xd4, 0x1f, 0x0c, 0xc1, 0x4f, 0xa6, 0x79, 0x28,
                ],
            ),
        ];
        let mut output = [0u8; 64];

        for v in vectors {
            hash_custom(v.0, &mut output)?;
            assert_eq!(output, v.1);
        }

        Ok(())
    }

    #[test]
    fn single_part_keyed_vectors() -> Result<(), AlkaliError> {
        let mut key = Key::new_empty()?;
        key.copy_from_slice(&[
            0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d,
            0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b,
            0x1c, 0x1d, 0x1e, 0x1f,
        ]);

        let vectors = [
            (
                &[][..],
                [
                    0x4e, 0x51, 0xe7, 0xa9, 0x13, 0xfc, 0x80, 0x13, 0x7d, 0xa5, 0x28, 0x80, 0xfe,
                    0xcc, 0xa1, 0x75, 0xbf, 0x81, 0xe1, 0x17, 0xd5, 0xc6, 0x81, 0x26, 0xdc, 0x27,
                    0x74, 0x03, 0x35, 0x17, 0xea, 0x0d,
                ],
            ),
            (
                &[0x00][..],
                [
                    0x41, 0xff, 0x93, 0xa4, 0xea, 0xee, 0xbd, 0x3b, 0x78, 0xa9, 0x34, 0x38, 0xa6,
                    0xf6, 0x2a, 0x92, 0xab, 0x59, 0x59, 0xc8, 0x59, 0xe6, 0x82, 0xb7, 0x2c, 0x7d,
                    0xef, 0x40, 0x61, 0x97, 0xca, 0x4d,
                ],
            ),
            (
                &[
                    0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c,
                    0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19,
                    0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
                ][..],
                [
                    0x9e, 0xa4, 0xa7, 0xc5, 0xf7, 0x8b, 0x4c, 0x5b, 0x77, 0x75, 0x84, 0x1e, 0x2a,
                    0x07, 0xdb, 0x51, 0x4c, 0x51, 0x0b, 0x14, 0xf5, 0x00, 0x6d, 0x27, 0xae, 0xe5,
                    0xe2, 0x0f, 0x84, 0x06, 0xbd, 0xd8,
                ],
            ),
            (
                &[
                    0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c,
                    0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19,
                    0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26,
                    0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, 0x30, 0x31, 0x32, 0x33,
                    0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e,
                ][..],
                [
                    0x2f, 0x3d, 0x96, 0xae, 0x02, 0xf0, 0xa3, 0x3f, 0xa3, 0x33, 0x4e, 0xd7, 0x3a,
                    0x04, 0x49, 0x1a, 0xbc, 0x99, 0x36, 0x2d, 0x1e, 0x99, 0xc1, 0x9f, 0xcc, 0x85,
                    0x94, 0x6f, 0x41, 0x7, 0xc5, 0x28,
                ],
            ),
            (
                &[
                    0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c,
                    0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19,
                    0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26,
                    0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, 0x30, 0x31, 0x32, 0x33,
                    0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f,
                ][..],
                [
                    0x7f, 0x46, 0xf0, 0x43, 0xd6, 0x3d, 0xfb, 0x3c, 0x9b, 0x11, 0x02, 0xc8, 0x82,
                    0x4b, 0x63, 0x2c, 0xae, 0x1f, 0xce, 0x45, 0x3e, 0x3c, 0x0e, 0x9d, 0xc9, 0xe7,
                    0x11, 0x3f, 0x3c, 0x1a, 0xcc, 0x36,
                ],
            ),
            (
                &[
                    0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c,
                    0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19,
                    0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26,
                    0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, 0x30, 0x31, 0x32, 0x33,
                    0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f, 0x40,
                ][..],
                [
                    0x22, 0x11, 0x1b, 0xcd, 0x34, 0x28, 0x24, 0xa7, 0x49, 0xda, 0x73, 0xe2, 0x1c,
                    0xa9, 0x88, 0x49, 0x52, 0x14, 0x71, 0x0c, 0x81, 0xbb, 0x4a, 0x05, 0xeb, 0xf3,
                    0xc4, 0x43, 0xe1, 0x7b, 0xe8, 0xc5,
                ],
            ),
            (
                &[
                    0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c,
                    0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19,
                    0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26,
                    0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, 0x30, 0x31, 0x32, 0x33,
                    0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f, 0x40,
                    0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d,
                    0x4e, 0x4f, 0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58, 0x59, 0x5a,
                    0x5b, 0x5c, 0x5d, 0x5e, 0x5f, 0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67,
                    0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f, 0x70, 0x71, 0x72, 0x73, 0x74,
                    0x75, 0x76, 0x77, 0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f, 0x80, 0x81,
                    0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e,
                    0x8f, 0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9a, 0x9b,
                    0x9c, 0x9d, 0x9e, 0x9f, 0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 0xa8,
                    0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf, 0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5,
                    0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf, 0xc0, 0xc1, 0xc2,
                    0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,
                    0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 0xd8, 0xd9, 0xda, 0xdb, 0xdc,
                    0xdd, 0xde, 0xdf, 0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9,
                    0xea, 0xeb, 0xec, 0xed, 0xee, 0xef, 0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6,
                    0xf7, 0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe,
                ][..],
                [
                    0xf6, 0x0f, 0x5c, 0x4a, 0x57, 0x5c, 0x43, 0xdb, 0x6c, 0x93, 0x60, 0x85, 0x15,
                    0x86, 0x6e, 0xa9, 0x98, 0xe0, 0x4b, 0x35, 0x79, 0x2f, 0x5a, 0x92, 0xb2, 0x7a,
                    0xa5, 0x88, 0x0c, 0xd6, 0x72, 0x1e,
                ],
            ),
        ];

        for v in vectors {
            let digest = hash_keyed(v.0, &key)?;
            assert_eq!(digest, v.1);
        }

        Ok(())
    }

    #[test]
    fn single_part_keyed_custom_vectors() -> Result<(), AlkaliError> {
        const KEY: [u8; 64] = [
            0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d,
            0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b,
            0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29,
            0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, 0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
            0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f,
        ];
        let vectors = [
            (
                &[][..],
                [
                    0x10, 0xeb, 0xb6, 0x77, 0x00, 0xb1, 0x86, 0x8e, 0xfb, 0x44, 0x17, 0x98, 0x7a,
                    0xcf, 0x46, 0x90, 0xae, 0x9d, 0x97, 0x2f, 0xb7, 0xa5, 0x90, 0xc2, 0xf0, 0x28,
                    0x71, 0x79, 0x9a, 0xaa, 0x47, 0x86, 0xb5, 0xe9, 0x96, 0xe8, 0xf0, 0xf4, 0xeb,
                    0x98, 0x1f, 0xc2, 0x14, 0xb0, 0x05, 0xf4, 0x2d, 0x2f, 0xf4, 0x23, 0x34, 0x99,
                    0x39, 0x16, 0x53, 0xdf, 0x7a, 0xef, 0xcb, 0xc1, 0x3f, 0xc5, 0x15, 0x68,
                ],
            ),
            (
                &[0x00][..],
                [
                    0x96, 0x1f, 0x6d, 0xd1, 0xe4, 0xdd, 0x30, 0xf6, 0x39, 0x01, 0x69, 0x0c, 0x51,
                    0x2e, 0x78, 0xe4, 0xb4, 0x5e, 0x47, 0x42, 0xed, 0x19, 0x7c, 0x3c, 0x5e, 0x45,
                    0xc5, 0x49, 0xfd, 0x25, 0xf2, 0xe4, 0x18, 0x7b, 0x0b, 0xc9, 0xfe, 0x30, 0x49,
                    0x2b, 0x16, 0xb0, 0xd0, 0xbc, 0x4e, 0xf9, 0xb0, 0xf3, 0x4c, 0x70, 0x03, 0xfa,
                    0xc0, 0x9a, 0x5e, 0xf1, 0x53, 0x2e, 0x69, 0x43, 0x02, 0x34, 0xce, 0xbd,
                ],
            ),
            (
                &[
                    0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c,
                    0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19,
                    0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
                ][..],
                [
                    0x86, 0x22, 0x1f, 0x3a, 0xda, 0x52, 0x03, 0x7b, 0x72, 0x22, 0x4f, 0x10, 0x5d,
                    0x79, 0x99, 0x23, 0x1c, 0x5e, 0x55, 0x34, 0xd0, 0x3d, 0xa9, 0xd9, 0xc0, 0xa1,
                    0x2a, 0xcb, 0x68, 0x46, 0x0c, 0xd3, 0x75, 0xda, 0xf8, 0xe2, 0x43, 0x86, 0x28,
                    0x6f, 0x96, 0x68, 0xf7, 0x23, 0x26, 0xdb, 0xf9, 0x9b, 0xa0, 0x94, 0x39, 0x24,
                    0x37, 0xd3, 0x98, 0xe9, 0x5b, 0xb8, 0x16, 0x1d, 0x71, 0x7f, 0x89, 0x91,
                ],
            ),
            (
                &[
                    0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c,
                    0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19,
                    0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26,
                    0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, 0x30, 0x31, 0x32, 0x33,
                    0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e,
                ][..],
                [
                    0xbd, 0x96, 0x5b, 0xf3, 0x1e, 0x87, 0xd7, 0x03, 0x27, 0x53, 0x6f, 0x2a, 0x34,
                    0x1c, 0xeb, 0xc4, 0x76, 0x8e, 0xca, 0x27, 0x5f, 0xa0, 0x5e, 0xf9, 0x8f, 0x7f,
                    0x1b, 0x71, 0xa0, 0x35, 0x12, 0x98, 0xde, 0x00, 0x6f, 0xba, 0x73, 0xfe, 0x67,
                    0x33, 0xed, 0x01, 0xd7, 0x58, 0x01, 0xb4, 0xa9, 0x28, 0xe5, 0x42, 0x31, 0xb3,
                    0x8e, 0x38, 0xc5, 0x62, 0xb2, 0xe3, 0x3e, 0xa1, 0x28, 0x49, 0x92, 0xfa,
                ],
            ),
            (
                &[
                    0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c,
                    0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19,
                    0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26,
                    0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, 0x30, 0x31, 0x32, 0x33,
                    0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f,
                ][..],
                [
                    0x65, 0x67, 0x6d, 0x80, 0x06, 0x17, 0x97, 0x2f, 0xbd, 0x87, 0xe4, 0xb9, 0x51,
                    0x4e, 0x1c, 0x67, 0x40, 0x2b, 0x7a, 0x33, 0x10, 0x96, 0xd3, 0xbf, 0xac, 0x22,
                    0xf1, 0xab, 0xb9, 0x53, 0x74, 0xab, 0xc9, 0x42, 0xf1, 0x6e, 0x9a, 0xb0, 0xea,
                    0xd3, 0x3b, 0x87, 0xc9, 0x19, 0x68, 0xa6, 0xe5, 0x09, 0xe1, 0x19, 0xff, 0x07,
                    0x78, 0x7b, 0x3e, 0xf4, 0x83, 0xe1, 0xdc, 0xdc, 0xcf, 0x6e, 0x30, 0x22,
                ],
            ),
            (
                &[
                    0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c,
                    0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19,
                    0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26,
                    0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, 0x30, 0x31, 0x32, 0x33,
                    0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f, 0x40,
                ][..],
                [
                    0x93, 0x9f, 0xa1, 0x89, 0x69, 0x9c, 0x5d, 0x2c, 0x81, 0xdd, 0xd1, 0xff, 0xc1,
                    0xfa, 0x20, 0x7c, 0x97, 0x0b, 0x6a, 0x36, 0x85, 0xbb, 0x29, 0xce, 0x1d, 0x3e,
                    0x99, 0xd4, 0x2f, 0x2f, 0x74, 0x42, 0xda, 0x53, 0xe9, 0x5a, 0x72, 0x90, 0x73,
                    0x14, 0xf4, 0x58, 0x83, 0x99, 0xa3, 0xff, 0x5b, 0x0a, 0x92, 0xbe, 0xb3, 0xf6,
                    0xbe, 0x26, 0x94, 0xf9, 0xf8, 0x6e, 0xcf, 0x29, 0x52, 0xd5, 0xb4, 0x1c,
                ],
            ),
            (
                &[
                    0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c,
                    0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19,
                    0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26,
                    0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, 0x30, 0x31, 0x32, 0x33,
                    0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f, 0x40,
                    0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d,
                    0x4e, 0x4f, 0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58, 0x59, 0x5a,
                    0x5b, 0x5c, 0x5d, 0x5e, 0x5f, 0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67,
                    0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f, 0x70, 0x71, 0x72, 0x73, 0x74,
                    0x75, 0x76, 0x77, 0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f, 0x80, 0x81,
                    0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e,
                    0x8f, 0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9a, 0x9b,
                    0x9c, 0x9d, 0x9e, 0x9f, 0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 0xa8,
                    0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf, 0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5,
                    0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf, 0xc0, 0xc1, 0xc2,
                    0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,
                    0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 0xd8, 0xd9, 0xda, 0xdb, 0xdc,
                    0xdd, 0xde, 0xdf, 0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9,
                    0xea, 0xeb, 0xec, 0xed, 0xee, 0xef, 0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6,
                    0xf7, 0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe,
                ][..],
                [
                    0x14, 0x27, 0x09, 0xd6, 0x2e, 0x28, 0xfc, 0xcc, 0xd0, 0xaf, 0x97, 0xfa, 0xd0,
                    0xf8, 0x46, 0x5b, 0x97, 0x1e, 0x82, 0x20, 0x1d, 0xc5, 0x10, 0x70, 0xfa, 0xa0,
                    0x37, 0x2a, 0xa4, 0x3e, 0x92, 0x48, 0x4b, 0xe1, 0xc1, 0xe7, 0x3b, 0xa1, 0x09,
                    0x06, 0xd5, 0xd1, 0x85, 0x3d, 0xb6, 0xa4, 0x10, 0x6e, 0x0a, 0x7b, 0xf9, 0x80,
                    0x0d, 0x37, 0x3d, 0x6d, 0xee, 0x2d, 0x46, 0xd6, 0x2e, 0xf2, 0xa4, 0x61,
                ],
            ),
        ];
        let mut output = [0u8; 64];

        for v in vectors {
            hash_keyed_custom(v.0, &KEY, &mut output)?;
            assert_eq!(output, v.1);
        }

        Ok(())
    }

    // TODO: Multi-part tests
}
