//! Password-based key derivation (password hashing).
//!
//! This module corresponds to the [`crypto_pwhash`
//! API](https://doc.libsodium.org/password_hashing) from Sodium.
//!
//! Many hash algorithms are designed to be fast to compute, while still preserving pre-image and
//! collision resistance. Password hashing is different: A normal user's password is actually not
//! hashed very often (typically only on login), so a very slow hash algorithm doesn't have a major
//! impact on them. However, for an attacker trying to brute force a password given its hash, the
//! slowdown caused by a slow algorithm has a major impact. Therefore, password-based key derivation
//! algorithms, such as those in this module, are intentionally computationally-intensive.
//!
//! Password-based key derivation is used for password storage, in which we store a hash of a
//! password rather than the password itself. When a user wishes to login, we compare the hash of
//! the password they provide with the hash we have stored. The [Wikipedia article on hash
//! functions](https://en.wikipedia.org/wiki/Cryptographic_hash_function#Password_verification)
//! elaborates on this procedure. The [`hash_str`]/[`verify_str`] API is intended for this use. It
//! is also used for key derivation, in which a key is deterministically derived from a low-entropy
//! source (i.e: a password). This key can then be used for other cryptographic applications. The
//! [`derive_key`] API is intended for this use.
//!
//! # Algorithm Details
//! The term for this cryptographic primitve is a [password-based key derivation
//! function](https://en.wikipedia.org/wiki/Key_derivation_function). The default algorithm is
//! [Argon2id](argon2id). Argon2id strikes a balance between resisting GPU-based attacks, and
//! resisting side-channel attacks. [Argon2i](argon2i), which is just intended to be highly
//! resistant to side-channel attacks, and may be less resistant to GPU-based attacks, is also
//! available. [Scrypt](scrypt), an older (and perhaps more widely analysed) algorithm, is also
//! available. Unless you have a specific use-case which requires the use of Argon2i or Scrypt, the
//! default algorithm is probably the best choice.
//!
//! # Choosing Memory and Operations Limits
//! Unlike the other operations exposed by Sodium, the algorithms provided in module are
//! configurable, using the operations and memory limit parameters. Suitable values for these
//! parameters will vary based on application and hardware, but getting them right is important, as
//! they determine the cost of the hash, which corresponds to its brute-force resistance.
//!
//! The [Sodium
//! docs](https://doc.libsodium.org/password_hashing/default_phf#guidelines-for-choosing-the-parameters)
//! provide guidelines for choosing the parameters, copied here:
//!
//! The memory limit directly specifies how much memory the password hash operation will use. Set
//! this to the amount of memory you wish to use for password hashing, in bytes.
//!
//! The operations limit corresponds to the maximum number of operations to perform, and a higher
//! value increases the number of CPU cycles required to compute a hash. Begin by setting this to
//! `3`.
//!
//! If the hash takes too long for your application, reduce the memory limit, keeping the
//! operations limit set to `3`.
//!
//! If the hash is very fast, and you can therefore afford to be more computationally intensive,
//! increase the operations limit until the computation time is no longer acceptable.
//!
//! For online use (website logins), a 1 second computation is likely the acceptable maximum. For
//! interactive desktop applications, a 5 second computation is acceptable if the password is only
//! entered once. For more infrequent use (e.g: restoring encrypted backups), even slower
//! computations can be reasonable.
//!
//! Some baseline values for these parameters are provided for each of the algorithms in this
//! module:
//! * [`OPS_LIMIT_INTERACTIVE`] and [`MEM_LIMIT_INTERACTIVE`]: For interactive, online operations
//! (requires 64 MiB memory)
//! * [`OPS_LIMIT_MODERATE`] and [`MEM_LIMIT_MODERATE`]: More conservative values for online
//! operations (requires 256 MiB memory)
//! * [`OPS_LIMIT_SENSITIVE`] and [`MEM_LIMIT_SENSITIVE`]: For offline operations (requires 1 GiB
//! memory)
//!
//! # Security Considerations
//! It is important to set the operations limit and memory limit to suitable values for your
//! application. See [the section on this subject](#choosing-memory-and-operations-limits).
//!
//! Passwords should ideally be immediately erased from memory when they are no longer required for
//! hash calculation/key derivation. You can use the [zeroize](https://crates.io/crates/zeroize)
//! crate to do this very simply, or use the [hard](https://crates.io/crates/hard) crate if you
//! want to make use of the other memory hardening utilties from Sodium.
//!
//! A common, but dangerous, mistake is to verify the correctness of a password by generating the
//! hash again yourself, and naively comparing the newly calculated hash with the stored hash. This
//! opens the door to [timing attacks](https://en.wikipedia.org/wiki/Timing_attack). The
//! [`hash_str`]/[`verify_str`] API is designed to prevent this: [`verify_str`] uses a
//! constant-time comparison, and is safe to use to verify passwords against a password hash
//! generated with [`hash_str`].
//!
//! # Examples
//! This example demonstrates an example user authentication flow using [`hash_str`] and
//! [`verify_str`]: When a user account is created, we hash the provided password with [`hash_str`],
//! and store this in the database alongside the username. When the user wishes to log in, we
//! retrieve the hash from the database, and verify the password is correct using [`verify_str`].
//! Some example `store_details_in_db` and `retrieve_passwd_hash_from_db` functions are used here,
//! obviously in a real-world application, there would be considerably more boilerplate.
//!
//! ```rust
//! use alkali::hash::pwhash::{
//!     hash_str, verify_str, PasswordHashError, OPS_LIMIT_INTERACTIVE, MEM_LIMIT_INTERACTIVE,
//! };
//! use alkali::AlkaliError;
//! # fn store_details_in_db(_username: &str, _passwd_hash: &str) {}
//!
//! /// Creates a new user account with the specified username and password
//! fn create_user_account(username: &str, passwd: &str) {
//!     let hash = hash_str(passwd, OPS_LIMIT_INTERACTIVE, MEM_LIMIT_INTERACTIVE).unwrap();
//!
//!     store_details_in_db(username, &hash);
//! }
//!
//! # fn retrieve_passwd_hash_from_db(_username: &str) -> &'static str {
//! #     "$argon2id$v=19$m=65536,t=2,p=1$oH2BFYR6JuV1k2IXDvh67w$Xy0iVWdHMbfq2eLN/dRMRcjNcMnkjckbJN\
//! #     v+xSOZ0qc"
//! # }
//! /// Tries to log in a user with the given username and password
//! fn log_in(username: &str, passwd: &str) -> bool {
//!     let hash = retrieve_passwd_hash_from_db(username);
//!
//!     match verify_str(passwd, hash) {
//!         Ok(_) => true,
//!         Err(AlkaliError::PasswordHashError(PasswordHashError::PasswordIncorrect)) => false,
//!         Err(_) => panic!("some other error occurred"),
//!     }
//! }
//! # create_user_account("some_dude", "hunter2");
//! # assert!(log_in("some_dude", "hunter2"));
//! ```
//!
//! Key derivation (using [`derive_key`]):
//!
//! ```rust
//! use alkali::hash::pwhash::{
//!     derive_key, generate_salt, OPS_LIMIT_INTERACTIVE, MEM_LIMIT_INTERACTIVE,
//! };
//!
//! let passwd = b"some data from which a key will be derived";
//! let mut key = [0u8; 32];
//!
//! let salt = generate_salt().unwrap();
//! derive_key(passwd, &salt, OPS_LIMIT_INTERACTIVE, MEM_LIMIT_INTERACTIVE, &mut key).unwrap();
//!
//! // Do something cool with key
//! // Be sure to store the salt, operations limit, and memory limit alongside the key if it will
//! // need to be derived again!
//! ```

// TODO: Implement the Scrypt algorithm

use thiserror::Error;

/// Error type returned if something went wrong in the pwhash module.
#[derive(Clone, Copy, Debug, Eq, Error, PartialEq)]
pub enum PasswordHashError {
    /// The password provided was too short or too long for use with this algorithm.
    ///
    /// Passwords must be at least [`PASSWORD_LENGTH_MIN`] and at most
    /// [`PASSWORD_LENGTH_MAX`](struct@PASSWORD_LENGTH_MAX) bytes.
    #[error("password length outside acceptable range")]
    PasswordLengthInvalid,

    /// The operations limit was smaller than [`OPS_LIMIT_MIN`] or greater than [`OPS_LIMIT_MAX`].
    #[error("operations limit outside acceptable range")]
    OpsLimitInvalid,

    /// The memory limit was smaller than [`MEM_LIMIT_MIN`] or greater than
    /// [`MEM_LIMIT_MAX`](struct@MEM_LIMIT_MAX).
    #[error("memory limit outside acceptable range")]
    MemLimitInvalid,

    /// The requested output is too short or too long for use with this algorithm.
    ///
    /// The derived key size must be at least [`OUTPUT_LENGTH_MIN`] bytes and at most
    /// [`OUTPUT_LENGTH_MAX`](struct@OUTPUT_LENGTH_MAX) bytes.
    #[error("requested output length is too short or too long")]
    OutputLengthInvalid,

    /// The password hash failed.
    ///
    /// This is likely due to memory allocation failing, as the pwhash API is one of the few
    /// portions of Sodium which requires dynamic memory allocation.
    #[error("password hash failed")]
    PasswordHashFailed,

    /// The password verification against the provided hash failed.
    ///
    /// This indicates that the password was incorrect for this hash, or potentially that
    /// calculating the hash failed, although this is unlikely in comparison to the former
    /// possibility.
    #[error("the password was incorrect for this hash")]
    PasswordIncorrect,
}

/// Possible results of [`requires_rehash`].
#[derive(Clone, Copy, Debug, Eq, PartialEq)]
pub enum RehashResult {
    /// The parameters of the hash match, a rehash is not required.
    ParametersMatch,

    /// The hash appears to be in the correct format, but the parameters differ from those
    /// specified.
    ///
    /// You may wish to compute a new hash the next time the user logs in.
    ParametersDiffer,

    /// The hash is not in the correct format.
    ///
    /// You may wish to compute a new hash the next time the user logs in.
    InvalidHash,
}

/// Generates the API for a `pwhash` module based on Argon2i or Argon2id with the given functions &
/// constants from Sodium.
macro_rules! argon_module {
    (
        $opslim_min:expr,       // crypto_pwhash_OPSLIMIT_MIN
        $opslim_int:expr,       // crypto_pwhash_OPSLIMIT_INTERACTIVE
        $opslim_mod:expr,       // crypto_pwhash_OPSLIMIT_MODERATE
        $opslim_sen:expr,       // crypto_pwhash_OPSLIMIT_SENSITIVE
        $opslim_max:expr,       // crypto_pwhash_OPSLIMIT_MAX
        $memlim_min:expr,       // crypto_pwhash_MEMLIMIT_MIN
        $memlim_int:expr,       // crypto_pwhash_MEMLIMIT_INTERACTIVE
        $memlim_mod:expr,       // crypto_pwhash_MEMLIMIT_MODERATE
        $memlim_sen:expr,       // crypto_pwhash_MEMLIMIT_SENSITIVE
        $memlim_max:path,       // crypto_pwhash_memlimit_max
        $pwlen_min:expr,        // crypto_pwhash_PASSWD_MIN
        $pwlen_max:path,        // crypto_pwhash_passwd_max
        $outlen_min:expr,       // crypto_pwhash_BYTES_MIN
        $outlen_max:path,       // crypto_pwhash_bytes_max
        $salt_len:expr,         // crypto_pwhash_SALTBYTES
        $str_len:expr,          // crypto_pwhash_STRBYTES
        $alg_ident:expr,        // crypto_pwhash_ALG_NAME
        $pwhash:path,           // crypto_pwhash
        $pwhash_str:path,       // crypto_pwhash_str
        $pwhash_verify:path,    // crypto_pwhash_str_verify
        $str_needs_rehash:path, // crypto_pwhash_str_needs_rehash
    ) => {
        /// The minimum value for the operations limit.
        pub const OPS_LIMIT_MIN: usize = $opslim_min as usize;

        /// A baseline value for the operations limit for interactive, online use.
        pub const OPS_LIMIT_INTERACTIVE: usize = $opslim_int as usize;

        /// A baseline value for the operations limit intended to be a more conservative choice for
        /// online use.
        pub const OPS_LIMIT_MODERATE: usize = $opslim_mod as usize;

        /// A baseline value for the operations limit for non-interactive/highly sensitive
        /// applications.
        pub const OPS_LIMIT_SENSITIVE: usize = $opslim_sen as usize;

        /// The maximum value for the operations limit.
        pub const OPS_LIMIT_MAX: usize = $opslim_max as usize;

        /// The minimum value for the memory limit.
        pub const MEM_LIMIT_MIN: usize = $memlim_min as usize;

        /// A baseline value for the memory limit for interactive, online use.
        pub const MEM_LIMIT_INTERACTIVE: usize = $memlim_int as usize;

        /// A baseline value for the memory limit intended to be a more conservative choice for
        /// online use.
        pub const MEM_LIMIT_MODERATE: usize = $memlim_mod as usize;

        /// A baseline value for the memory limit for non-interactive/highly sensitive applications.
        pub const MEM_LIMIT_SENSITIVE: usize = $memlim_sen as usize;

        lazy_static::lazy_static! {
            /// The maximum value for the memory limit.
            pub static ref MEM_LIMIT_MAX: usize = unsafe {
                // SAFETY: This function just returns a constant value, and should always be safe
                // to call.
                $memlim_max()
            };
        }

        /// The minimum length for a password to hash.
        pub const PASSWORD_LENGTH_MIN: usize = $pwlen_min as usize;

        lazy_static::lazy_static! {
            /// The maximum length for a password to hash.
            pub static ref PASSWORD_LENGTH_MAX: usize = unsafe {
                // SAFETY: This function just returns a constant value, and should always be safe
                // to call.
                $pwlen_max() as usize
            };
        }

        /// The minimum output size for this algorithm.
        pub const OUTPUT_LENGTH_MIN: usize = $outlen_min as usize;

        lazy_static::lazy_static! {
            /// The maximum output size for this algorithm.
            pub static ref OUTPUT_LENGTH_MAX: usize = unsafe {
                // SAFETY: This function just returns a constant value, and should always be safe
                // to call.
                $outlen_max() as usize
            };
        }

        /// The length of a salt.
        pub const SALT_LENGTH: usize = $salt_len as usize;

        /// The maximum length of the string representation of a password hash.
        pub const STR_LENGTH_MAX: usize = $str_len as usize - 1;

        /// A salt for a password hash, used to introduce non-determinism into the algorithm.
        pub type Salt = [u8; SALT_LENGTH];

        /// Generate a random salt for use with [`derive_key`].
        pub fn generate_salt() -> Result<Salt, $crate::AlkaliError> {
            let mut salt = [0u8; SALT_LENGTH];
            $crate::random::fill_random(&mut salt)?;
            Ok(salt)
        }

        /// Derive a key from a low-entropy input (i.e: a password).
        ///
        /// This function is used when a key, suitable for use with cryptographic algorithms which
        /// use a fixed-size input, must be derived deterministically from a variable-size,
        /// typically low-entropy source (like a password).
        ///
        /// The first argument to this function is the password from which the key is to be
        /// derived. The second argument is a [`Salt`] to use in the hash calculation, which
        /// introduces some non-determinism into the process (the same password will not produce the
        /// same key, unless the same salt is used). This should be generated randomly when the key
        /// is first derived using [`generate_salt`]. The third and fourth arguments are the
        /// operations and memory limits, which determine the computational complexity of the hash.
        /// These values should be chosen specifically for your application, see [Choosing Memory
        /// and Operations Limits](crate::hash::pwhash#choosing-memory-and-operations-limits). The
        /// final argument is the buffer to which the derived key will be written. This can be of
        /// any length between [`OUTPUT_LENGTH_MIN`] and
        /// [`OUTPUT_LENGTH_MAX`](struct@OUTPUT_LENGTH_MAX) bytes.
        ///
        /// If key derivation is successful, the `key` buffer is filled with derived key data. If
        /// the derivation fails, a [`PasswordHashError`](super::PasswordHashError) will be
        /// returned.
        ///
        /// Key derivation is a deterministic operation dependent on the password, salt, operations
        /// limit, and memory limit. These values are **not** stored in the derived key, so they
        /// will need to be stored alongside it. When the same key is derived in the future, these
        /// same parameters must be used. None are secret.
        ///
        /// # Security Concerns
        /// It is important to set the operations limit and memory limit to suitable values for your
        /// application. See [the section on this
        /// subject](crate::hash::pwhash#choosing-memory-and-operations-limits).
        ///
        /// To store a password to verify a user's identity, it's a better idea to use
        /// [`hash_str`], which includes the hash parameters in the generated string, and produces
        /// ASCII output which can easily be stored in any database. The [`verify_str`] function is
        /// also provided to verify a password matches a specific hash.
        pub fn derive_key(
            password: &[u8],
            salt: &Salt,
            ops_limit: usize,
            mem_limit: usize,
            key: &mut [u8],
        ) -> Result<(), $crate::AlkaliError> {
            $crate::require_init()?;

            if password.len() < PASSWORD_LENGTH_MIN || password.len() > *PASSWORD_LENGTH_MAX {
                return Err($crate::hash::pwhash::PasswordHashError::PasswordLengthInvalid.into());
            } else if ops_limit < OPS_LIMIT_MIN || ops_limit > OPS_LIMIT_MAX {
                return Err($crate::hash::pwhash::PasswordHashError::OpsLimitInvalid.into());
            } else if mem_limit < MEM_LIMIT_MIN || mem_limit > *MEM_LIMIT_MAX {
                return Err($crate::hash::pwhash::PasswordHashError::MemLimitInvalid.into());
            } else if key.len() < OUTPUT_LENGTH_MIN || key.len() > *OUTPUT_LENGTH_MAX {
                return Err($crate::hash::pwhash::PasswordHashError::OutputLengthInvalid.into());
            }

            let pwhash_result = unsafe {
                // SAFETY: This function takes 8 parameters. The first two specify a pointer to
                // which a key will be written, and the length of data to write to this pointer. We
                // use `key.len()` to specify the length of the data, which is equal to the length
                // of the buffer. The `key` type itself is a slice of `u8`s, so any data can be
                // written to it safely. The next two arguments are a pointer to a password, and
                // its length. We use `password.len()` to specify the length of the password, so
                // this is the correct length for the pointer. The next argument is a pointer to a
                // salt. We have defined the `Salt` type to be `SALTBYTES` bytes long, the expected
                // length of a salt for this function, so this is valid to use here. The next three
                // parameters are the operations limit, memory limit, and algorithm identifier,
                // which are all just integers. Therefore this function call is safe.
                $pwhash(
                    key.as_mut_ptr(),
                    key.len() as libc::c_ulonglong,
                    password.as_ptr() as *const libc::c_char,
                    password.len() as libc::c_ulonglong,
                    salt.as_ptr(),
                    ops_limit as libc::c_ulonglong,
                    mem_limit,
                    $alg_ident as libc::c_int,
                )
            };

            if pwhash_result == 0 {
                Ok(())
            } else {
                Err($crate::hash::pwhash::PasswordHashError::PasswordHashFailed.into())
            }
        }

        /// Hash a password, for later use to verify a provided password is the same as the
        /// original.
        ///
        /// This function is used when a user's password must be stored, so that we can later
        /// verify their identity. A hash of the password is calculated using this function. We can
        /// later verify a password provided by the user against this hash using the [`verify_str`]
        /// function. So the actual password never needs to be stored.
        ///
        /// The first argument to this function is the password from which the hash will be
        /// calculated. The second and third arguments are the operations and memory limits, which
        /// determine the computational complexity of the hash. These values should be chosen
        /// specifically for your application, see [Choosing Memory and Operations
        /// Limits](crate::hash::pwhash#choosing-memory-and-operations-limits).
        ///
        /// If hashing is successful, a String will be returned containing the hash and the
        /// parameters used to calculate it (including a randomly-generated salt). This hash can
        /// later be used with [`verify_str`] to verify the password.
        ///
        /// # Security Concerns
        /// It is important to set the operations limit and memory limit to suitable values for your
        /// application. See [the section on this
        /// subject](crate::hash::pwhash#choosing-memory-and-operations-limits).
        ///
        /// This function is not suitable for deriving keys for use with other cryptographic
        /// operations, and large parts of its output are predictable. You should instead use
        /// [`derive_key`], which only produces output suitable for use as key-data.
        pub fn hash_str(
            password: &str,
            ops_limit: usize,
            mem_limit: usize,
        ) -> Result<String, $crate::AlkaliError> {
            $crate::require_init()?;

            let mut out = [5u8; STR_LENGTH_MAX + 1];

            if password.len() < PASSWORD_LENGTH_MIN || password.len() > *PASSWORD_LENGTH_MAX {
                return Err($crate::hash::pwhash::PasswordHashError::PasswordLengthInvalid.into());
            } else if ops_limit < OPS_LIMIT_MIN || ops_limit > OPS_LIMIT_MAX {
                return Err($crate::hash::pwhash::PasswordHashError::OpsLimitInvalid.into());
            } else if mem_limit < MEM_LIMIT_MIN || mem_limit > *MEM_LIMIT_MAX {
                return Err($crate::hash::pwhash::PasswordHashError::MemLimitInvalid.into());
            }

            let pwhash_result = unsafe {
                // SAFETY: The first argument to this function is a pointer to which a C-formatted
                // string will be written. This must have enough space to store `STRBYTES` bytes,
                // which we have defined `out` to be. When we use `out`, we will treat it as a
                // C-formatted string. The next two arguments are a pointer to a password to hash,
                // and the length of the password in bytes. We use `password.as_bytes().len()` to
                // specify the length, so it is correct for this pointer. The final two parameters
                // are the operations limit and memory limit, which are both just integers.
                $pwhash_str(
                    out.as_mut_ptr() as *mut libc::c_char,
                    password.as_bytes().as_ptr() as *const libc::c_char,
                    password.as_bytes().len() as libc::c_ulonglong,
                    ops_limit as libc::c_ulonglong,
                    mem_limit,
                )
            };

            if pwhash_result == 0 {
                // We need to take a slice up to & including the first nul byte here, to
                // successfully construct a CString.
                let hash_len = unsafe {
                    // SAFETY: This is a binding to the strlen function from the C standard
                    // library. This function takes a pointer to a C-formatted string (with nul
                    // byte) as an argument, and returns the (inclusive) length to the nul byte.
                    // The $pwhash_str function above was used to fill the contents of this buffer,
                    // which Sodium guarantees with produce a valid C string, including nul byte.
                    // Therefore, it is safe to use strlen to determine the length of the string,
                    // including the nul byte.
                    libc::strlen(out.as_ptr() as *const libc::c_char)
                };
                let output_string = std::ffi::CString::new(&out[..hash_len])
                    .unwrap()
                    .into_string()
                    .unwrap();
                Ok(output_string)
            } else {
                Err($crate::hash::pwhash::PasswordHashError::PasswordHashFailed.into())
            }
        }

        /// Verify `password` matches the provided `hash`.
        ///
        /// The first argument to this function is the password to verify. The second argument is
        /// the hash against whiich the password will be checked, previously calculated using
        /// [`hash_str`].
        ///
        /// This function will return `Ok(())` if `password` is equal to the password used to
        /// calculated `hash`, or a [`PasswordHashError`](super::PasswordHashError) otherwise.
        pub fn verify_str(password: &str, hash: &str) -> Result<(), $crate::AlkaliError> {
            $crate::require_init()?;

            let hash = std::ffi::CString::new(hash).unwrap();
            let hash_ptr = hash.into_raw();

            let verification_result = unsafe {
                // SAFETY: The first argument to this function is a C-formatted string containing
                // the hash against which the password should be verified. We construct the
                // hash_ptr argument by building a CString from a valid Rust string, then calling
                // `into_raw`. The definition of CString in the Rust standard library says that
                // this produces a valid pointer to a C-formatted string. The second and third
                // arguments specify a pointer to a password to verify, and the length of the
                // password. We use `password.as_bytes()` to convert the password to bytes, and
                // use `password.as_bytes().len()` to calculate the length, so this is correct for
                // this pointer.
                $pwhash_verify(
                    hash_ptr,
                    password.as_bytes().as_ptr() as *const libc::c_char,
                    password.as_bytes().len() as libc::c_ulonglong,
                )
            };

            // Make sure we free the hash string's memory
            let _hash = unsafe {
                // SAFETY: This pointer was created using CString::into_raw, so it is safe to
                // create a CString from it.
                std::ffi::CString::from_raw(hash_ptr)
            };

            if verification_result == 0 {
                Ok(())
            } else {
                Err($crate::hash::pwhash::PasswordHashError::PasswordIncorrect.into())
            }
        }

        /// Determine if `hash` is a valid hash string under the given parameters.
        ///
        /// This function is intended to be used if you update the parameters for the hash function
        /// used to store passwords. As users log in, if the password hash needs updating (as
        /// determined using this function), then the hash can be recalculated.
        ///
        /// Returns [`RehashResult::ParametersMatch`](super::RehashResult::ParametersMatch) if the
        /// hash appears to be in the correct format for a hash with the given paramaters. Returns
        /// [`RehashResult::ParametersDiffer`](super::RehashResult::ParametersDiffer) if the hash
        /// is in the correct format for this algorithm, but has different parameters. Returns
        /// [`RehashResult::InvalidHash`](super::RehashResult::InvalidHash) if the hash doesn't
        /// appear to be in the correct format for this algorithm.
        pub fn requires_rehash(
            hash: &str,
            ops_limit: usize,
            mem_limit: usize,
        ) -> Result<$crate::hash::pwhash::RehashResult, $crate::AlkaliError> {
            use $crate::hash::pwhash::RehashResult;

            $crate::require_init()?;

            let hash = std::ffi::CString::new(hash).unwrap();
            let hash_ptr = hash.into_raw();

            let rehash_result = unsafe {
                // SAFETY: The first argument to this function is a C-formatted string containing
                // the hash to check. We define the hash_ptr argument by building a CString from a
                // valid Rust string, then calling `into_raw`. The definition of CString in the Rust
                // standard library says that this produces a valid pointer to a C-formatted string.
                // The remaining two arguments, the operations and memory limits, are just integers.
                $str_needs_rehash(hash_ptr, ops_limit as libc::c_ulonglong, mem_limit)
            };

            // Make sure we free the hash string's memory
            let _hash = unsafe {
                // SAFETY: This pointer was created using CString::into_raw, so it is safe to
                // create a CString from it.
                std::ffi::CString::from_raw(hash_ptr)
            };


            match rehash_result {
                -1 => Ok(RehashResult::InvalidHash),
                0 => Ok(RehashResult::ParametersMatch),
                1 => Ok(RehashResult::ParametersDiffer),
                _ => unreachable!(),
            }
        }
    };
}

/// Generates tests for the [`derive_key`] function of a `pwhash` implementation. Takes test
/// vectors as arguments.
#[allow(unused_macros)]
macro_rules! kdf_tests {
    ( $( {
        pass: $pass:expr,
        salt: $salt:expr,
        ops: $opslim:expr,
        mem: $memlim:expr,
        out: $key:expr,
    }, )* ) => {
        #[test]
        fn key_derivation_test_vectors() -> Result<(), $crate::AlkaliError> {
            $(
                let mut actual_key = vec![0u8; $key.len()];
                super::derive_key(&$pass, &$salt, $opslim, $memlim, &mut actual_key)?;
                assert_eq!(&actual_key, &$key);
            )*

            Ok(())
        }

        #[test]
        fn key_derivation_invalid_params() -> Result<(), $crate::AlkaliError> {
            let password = b"Correct Horse Battery Staple";
            let salt = super::generate_salt()?;
            let mut key = vec![0; super::OUTPUT_LENGTH_MIN];

            assert!(super::derive_key(
                password,
                &salt,
                super::OPS_LIMIT_MIN,
                super::MEM_LIMIT_MIN,
                &mut key,
            ).is_ok());

            let mut key = vec![0; super::OUTPUT_LENGTH_MIN - 1];
            assert!(super::derive_key(
                password,
                &salt,
                super::OPS_LIMIT_MIN,
                super::MEM_LIMIT_MIN,
                &mut key
            )
            .is_err());
            let mut key = vec![0; super::OUTPUT_LENGTH_MIN];

            assert!(super::derive_key(
                password,
                &salt,
                super::OPS_LIMIT_MIN,
                super::MEM_LIMIT_MIN - 1,
                &mut key
            )
            .is_err());

            assert!(super::derive_key(
                password,
                &salt,
                super::OPS_LIMIT_MIN - 1,
                super::MEM_LIMIT_MIN,
                &mut key
            )
            .is_err());

            Ok(())
        }
    }
}

/// Generates some tests for the [`verify_str`] function of a `pwhash` implementation. Takes test
/// vectors as arguments.
#[allow(unused_macros)]
macro_rules! verify_str_valid_tests {
    ( $( {
        pass: $pass:expr,
        hash: $hash:expr,
    }, )* ) => {
        #[test]
        fn verify_str_valid_strings() {
            $(
                assert!(super::verify_str(&$pass, &$hash).is_ok());
            )*
        }
    };
}

/// Generates some tests for the [`verify_str`] function of a `pwhash` implementation. Takes test
/// vectors as arguments.
#[allow(unused_macros)]
macro_rules! verify_str_invalid_tests {
    ( $( {
        pass: $pass:expr,
        hash: $hash:expr,
    }, )* ) => {
        #[test]
        fn verify_str_invalid_strings() {
            $(
                assert!(super::verify_str(&$pass, &$hash).is_err());
            )*
        }
    }
}

/// Generates tests for the [`requires_rehash`] function of a `pwhash` implementation.
#[allow(unused_macros)]
macro_rules! needs_rehash_tests {
    () => {
        #[test]
        fn needs_rehash() -> Result<(), $crate::AlkaliError> {
            use $crate::hash::pwhash::RehashResult;

            const OPS_LIMIT: usize = 3;
            const MEM_LIMIT: usize = 5000000;
            const PASSWORD: &'static str = "Correct Horse Battery Staple";

            let hash = super::hash_str(PASSWORD, OPS_LIMIT, MEM_LIMIT)?;

            assert_eq!(
                super::requires_rehash(&hash, OPS_LIMIT, MEM_LIMIT)?,
                RehashResult::ParametersMatch,
            );

            assert_eq!(
                super::requires_rehash(&hash, OPS_LIMIT, MEM_LIMIT / 2)?,
                RehashResult::ParametersDiffer,
            );
            assert_eq!(
                super::requires_rehash(&hash, OPS_LIMIT - 1, MEM_LIMIT)?,
                RehashResult::ParametersDiffer,
            );
            assert_eq!(
                super::requires_rehash(&hash, OPS_LIMIT, MEM_LIMIT * 2)?,
                RehashResult::ParametersDiffer,
            );
            assert_eq!(
                super::requires_rehash(&hash, OPS_LIMIT + 1, MEM_LIMIT)?,
                RehashResult::ParametersDiffer,
            );

            assert_eq!(
                super::requires_rehash("not valid", OPS_LIMIT, MEM_LIMIT)?,
                RehashResult::InvalidHash,
            );

            Ok(())
        }
    };
}

/// The [Argon2id](https://en.wikipedia.org/wiki/Argon2) password-based key derivation algorithm.
///
/// This algorithm is intended to strike a sensible balance balance between resistance to GPU-based
/// attacks and resisting side-channel analysis.
pub mod argon2id {
    use libsodium_sys as sodium;

    argon_module! {
        sodium::crypto_pwhash_argon2id_OPSLIMIT_MIN,
        sodium::crypto_pwhash_argon2id_OPSLIMIT_INTERACTIVE,
        sodium::crypto_pwhash_argon2id_OPSLIMIT_MODERATE,
        sodium::crypto_pwhash_argon2id_OPSLIMIT_SENSITIVE,
        sodium::crypto_pwhash_argon2id_OPSLIMIT_MAX,
        sodium::crypto_pwhash_argon2id_MEMLIMIT_MIN,
        sodium::crypto_pwhash_argon2id_MEMLIMIT_INTERACTIVE,
        sodium::crypto_pwhash_argon2id_MEMLIMIT_MODERATE,
        sodium::crypto_pwhash_argon2id_MEMLIMIT_SENSITIVE,
        sodium::crypto_pwhash_argon2id_memlimit_max,
        sodium::crypto_pwhash_argon2id_PASSWD_MIN,
        sodium::crypto_pwhash_argon2id_passwd_max,
        sodium::crypto_pwhash_argon2id_BYTES_MIN,
        sodium::crypto_pwhash_argon2id_bytes_max,
        sodium::crypto_pwhash_argon2id_SALTBYTES,
        sodium::crypto_pwhash_argon2id_STRBYTES,
        sodium::crypto_pwhash_ALG_ARGON2ID13,
        sodium::crypto_pwhash_argon2id,
        sodium::crypto_pwhash_argon2id_str,
        sodium::crypto_pwhash_argon2id_str_verify,
        sodium::crypto_pwhash_argon2id_str_needs_rehash,
    }

    #[cfg(test)]
    mod tests {
        kdf_tests! [
            {
                pass: [0xa3, 0x47, 0xae, 0x92, 0xbc, 0xe9, 0xf8, 0x0f, 0x6f, 0x59, 0x5a, 0x44, 0x80,
                       0xfc, 0x9c, 0x2f, 0xe7, 0xe7, 0xd7, 0x14, 0x8d, 0x37, 0x1e, 0x94, 0x87, 0xd7,
                       0x5f, 0x5c, 0x23, 0x00, 0x8f, 0xfa, 0xe0, 0x65, 0x57, 0x7a, 0x92, 0x8f, 0xeb,
                       0xd9, 0xb1, 0x97, 0x3a, 0x5a, 0x95, 0x07, 0x3a, 0xcd, 0xbe, 0xb6, 0xa0, 0x30,
                       0xcf, 0xc0, 0xd7, 0x9c, 0xaa, 0x2d, 0xc5, 0xcd, 0x01, 0x1c, 0xef, 0x02, 0xc0,
                       0x8d, 0xa2, 0x32, 0xd7, 0x6d, 0x52, 0xdf, 0xbc, 0xa3, 0x8c, 0xa8, 0xdc, 0xbd,
                       0x66, 0x5b, 0x17, 0xd1, 0x66, 0x5f, 0x7c, 0xf5, 0xfe, 0x59, 0x77, 0x2e, 0xc9,
                       0x09, 0x73, 0x3b, 0x24, 0xde, 0x97, 0xd6, 0xf5, 0x8d, 0x22, 0x0b, 0x20, 0xc6,
                       0x0d, 0x7c, 0x07, 0xec, 0x1f, 0xd9, 0x3c, 0x52, 0xc3, 0x10, 0x20, 0x30, 0x0c,
                       0x6c, 0x1f, 0xac, 0xd7, 0x79, 0x37, 0xa5, 0x97, 0xc7, 0xa6],
                salt: [0x55, 0x41, 0xfb, 0xc9, 0x95, 0xd5, 0xc1, 0x97, 0xba, 0x29, 0x03, 0x46, 0xd2,
                       0xc5, 0x59, 0xde],
                ops:  5,
                mem:  7256678,
                out:  [0x18, 0xac, 0xec, 0x5d, 0x65, 0x07, 0x73, 0x9f, 0x20, 0x3d, 0x1f, 0x5d, 0x9f,
                       0x1d, 0x86, 0x2f, 0x7c, 0x2c, 0xda, 0xc4, 0xf1, 0x9d, 0x2b, 0xdf, 0xf6, 0x44,
                       0x87, 0xe6, 0x0d, 0x96, 0x9e, 0x3c, 0xed, 0x61, 0x53, 0x37, 0xb9, 0xee, 0xc6,
                       0xac, 0x44, 0x61, 0xc6, 0xca, 0x07, 0xf0, 0x93, 0x97, 0x41, 0xe5, 0x7c, 0x24,
                       0xd0, 0x00, 0x5c, 0x7e, 0xa1, 0x71, 0xa0, 0xee, 0x1e, 0x73, 0x48, 0x24, 0x9d,
                       0x13, 0x5b, 0x38, 0xf2, 0x22, 0xe4, 0xda, 0xd7, 0xb9, 0xa0, 0x33, 0xed, 0x83,
                       0xf5, 0xca, 0x27, 0x27, 0x73, 0x93, 0xe3, 0x16, 0x58, 0x20, 0x33, 0xc7, 0x4a,
                       0xff, 0xe2, 0x56, 0x6a, 0x2b, 0xea, 0x47, 0xf9, 0x1f, 0x0f, 0xd9, 0xfe, 0x49,
                       0xec, 0xe7, 0xe1, 0xf7, 0x9f, 0x3a, 0xd6, 0xe9, 0xb2, 0x3e, 0x02, 0x77, 0xc8,
                       0xec, 0xc4, 0xb3, 0x13, 0x22, 0x57, 0x48, 0xdd, 0x2a, 0x80, 0xf5, 0x67, 0x95,
                       0x34, 0xa0, 0x70, 0x0e, 0x24, 0x6a, 0x79, 0xa4, 0x9b, 0x3f, 0x74, 0xeb, 0x89,
                       0xec, 0x62, 0x05, 0xfe, 0x1e, 0xeb, 0x94, 0x1c, 0x73, 0xb1, 0xfc, 0xf1],
            },
            {
                pass: [0xe1, 0x25, 0xce, 0xe6, 0x1c, 0x8c, 0xb7, 0x77, 0x8d, 0x9e, 0x5a, 0xd0, 0xa6,
                       0xf5, 0xd9, 0x78, 0xce, 0x9f, 0x84, 0xde, 0x21, 0x3a, 0x85, 0x56, 0xd9, 0xff,
                       0xe2, 0x02, 0x02, 0x0a, 0xb4, 0xa6, 0xed, 0x90, 0x74, 0xa4, 0xeb, 0x34, 0x16,
                       0xf9, 0xb1, 0x68, 0xf1, 0x37, 0x51, 0x0f, 0x3a, 0x30, 0xb7, 0x0b, 0x96, 0xcb,
                       0xfa, 0x21, 0x9f, 0xf9, 0x9f, 0x6c, 0x6e, 0xaf, 0xfb, 0x15, 0xc0, 0x6b, 0x60,
                       0xe0, 0x0c, 0xc2, 0x89, 0x02, 0x77, 0xf0, 0xfd, 0x3c, 0x62, 0x21, 0x15, 0x77,
                       0x2f, 0x70, 0x48, 0xad, 0xae, 0xbe, 0xd8, 0x6e],
                salt: [0xf1, 0x19, 0x2d, 0xd5, 0xdc, 0x23, 0x68, 0xb9, 0xcd, 0x42, 0x13, 0x38, 0xb2,
                       0x24, 0x33, 0x45],
                ops:  4,
                mem:  7849083,
                out:  [0x26, 0xba, 0xb5, 0xf1, 0x01, 0x56, 0x0e, 0x48, 0xc7, 0x11, 0xda, 0x4f, 0x05,
                       0xe8, 0x1f, 0x5a, 0x38, 0x02, 0xb7, 0xa9, 0x3d, 0x51, 0x55, 0xb9, 0xca, 0xb1,
                       0x53, 0x06, 0x9c, 0xc4, 0x2b, 0x8e, 0x9f, 0x91, 0x0b, 0xfe, 0xad, 0x74, 0x76,
                       0x52, 0xa0, 0x70, 0x8d, 0x70, 0xe4, 0xde, 0x0b, 0xad, 0xa3, 0x72, 0x18, 0xbd,
                       0x20, 0x3a, 0x12, 0x01, 0xc3, 0x6b, 0x42, 0xf9, 0xa2, 0x69, 0xb6, 0x75, 0xb1,
                       0xf3, 0x0c, 0xfc, 0x36, 0xf3, 0x5a, 0x30, 0x30, 0xe9, 0xc7, 0xf5, 0x7d, 0xfb,
                       0xa0, 0xd3, 0x41, 0xa9, 0x74, 0xc1, 0x88, 0x6f, 0x70, 0x8c, 0x3e, 0x82, 0x97,
                       0xef, 0xbf, 0xe4, 0x11, 0xbb, 0x9d, 0x51, 0x37, 0x52, 0x64, 0xbd, 0x7c, 0x70,
                       0xd5, 0x7a, 0x8a, 0x56, 0xfc, 0x9d, 0xe2, 0xc1, 0xc9, 0x7c, 0x08, 0x77, 0x68,
                       0x03, 0xec, 0x2c, 0xd0, 0x14, 0x0b, 0xba, 0x8e, 0x61, 0xdc, 0x0f, 0x4a, 0xd3,
                       0xd3, 0xd1, 0xa8, 0x9b, 0x4b, 0x71, 0x0a, 0xf8, 0x1b, 0xfe, 0x35, 0xa0, 0xee,
                       0xa1, 0x93, 0xe1, 0x8a, 0x6d, 0xa0, 0xf5, 0xec, 0x05, 0x54, 0x2c, 0x9e, 0xef,
                       0xc4, 0x58, 0x44, 0x58, 0xe1, 0xda, 0x71, 0x56, 0x11, 0xba, 0x09, 0x61, 0x73,
                       0x84, 0x74, 0x8b, 0xd4, 0x3b, 0x9b, 0xf1, 0xf3, 0xa6, 0xdf, 0x4e, 0xcd, 0x09,
                       0x1d, 0x08, 0x75, 0xe0, 0x8d, 0x6e, 0x2f, 0xd8, 0xa5, 0xc7, 0xce, 0x08, 0x90,
                       0x4b, 0x51, 0x60, 0xcd, 0x38, 0x16, 0x7b, 0x76, 0xec, 0x76, 0xef, 0x2d, 0x31,
                       0x00, 0x49, 0x05, 0x5a, 0x56, 0x4d, 0xa2, 0x3d, 0x4e, 0xbd, 0x2b, 0x87, 0xe4,
                       0x21, 0xcc, 0x33, 0xc4, 0x01, 0xe1, 0x2d, 0x5c, 0xd8, 0xd9, 0x36, 0xc9, 0xba,
                       0xf7, 0x5e, 0xbd, 0xfb, 0x55, 0x7d, 0x34, 0x2d, 0x28, 0x58, 0xfc, 0x78, 0x1d,
                       0xa3, 0x18, 0x60],
            },
            {
                pass: [0x92, 0x26, 0x3c, 0xbf, 0x6a, 0xc3, 0x76, 0x49, 0x9f, 0x68, 0xa4, 0x28, 0x9d,
                       0x3b, 0xb5, 0x9e, 0x5a, 0x22, 0x33, 0x5e, 0xba, 0x63, 0xa3, 0x2e, 0x64, 0x10,
                       0x24, 0x91, 0x55, 0xb9, 0x56, 0xb6, 0xa3, 0xb4, 0x8d, 0x4a, 0x44, 0x90, 0x6b,
                       0x18, 0xb8, 0x97, 0x12, 0x73, 0x00, 0xb3, 0x75, 0xb8, 0xf8, 0x34, 0xf1, 0xce,
                       0xff, 0xc7, 0x08, 0x80, 0xa8, 0x85, 0xf4, 0x7c, 0x33, 0x87, 0x67, 0x17, 0xe3,
                       0x92, 0xbe, 0x57, 0xf7, 0xda, 0x3a, 0xe5, 0x8d, 0xa4, 0xfd, 0x1f, 0x43, 0xda,
                       0xa7, 0xe4, 0x4b, 0xb8, 0x2d, 0x37, 0x17, 0xaf, 0x43, 0x19, 0x34, 0x9c, 0x24,
                       0xcd, 0x31, 0xe4, 0x6d, 0x29, 0x58, 0x56, 0xb0, 0x44, 0x1b, 0x6b, 0x28, 0x99,
                       0x92, 0xa1, 0x1c, 0xed, 0x1c, 0xc3, 0xbf, 0x30, 0x11, 0x60, 0x45, 0x90, 0x24,
                       0x4a, 0x3e, 0xb7, 0x37, 0xff, 0x22, 0x11, 0x29, 0x21, 0x5e, 0x4e, 0x43, 0x47,
                       0xf4, 0x91, 0x5d, 0x41, 0x29, 0x2b, 0x51, 0x73, 0xd1, 0x96, 0xeb, 0x9a, 0xdd,
                       0x69, 0x3b, 0xe5, 0x31, 0x9f, 0xda, 0xdc, 0x24, 0x29, 0x06, 0x17, 0x8b, 0xb6,
                       0xc0, 0x28, 0x6c, 0x9b, 0x6c, 0xa6, 0x01, 0x27, 0x46, 0x71, 0x1f, 0x58, 0xc8,
                       0xc3, 0x92, 0x01, 0x6b, 0x2f, 0xdf, 0xc0, 0x9c, 0x64, 0xf0, 0xf6, 0xb6, 0xab,
                       0x7b],
                salt: [0x3b, 0x84, 0x0e, 0x20, 0xe9, 0x55, 0x5e, 0x9f, 0xb0, 0x31, 0xc4, 0xba, 0x1f,
                       0x17, 0x47, 0xce],
                ops:  3,
                mem:  7994791,
                out:  [0x6e, 0xb4, 0x5e, 0x66, 0x85, 0x82, 0xd6, 0x37, 0x88, 0xca, 0x8f, 0x6e, 0x93,
                       0x0c, 0xa6, 0x0b, 0x04, 0x5a, 0x79, 0x5f, 0xca, 0x98, 0x73, 0x44, 0xf9, 0xa7,
                       0xa1, 0x35, 0xaa, 0x3b, 0x51, 0x32, 0xb5, 0x0a, 0x34, 0xa3, 0x86, 0x4c, 0x26,
                       0x58, 0x1f, 0x1f, 0x56, 0xdd, 0x0b, 0xcb, 0xfa, 0xfb, 0xfa, 0x92, 0xcd, 0x9b,
                       0xff, 0x6b, 0x24, 0xa7, 0x34, 0xcf, 0xe8, 0x8f, 0x85, 0x4a, 0xef, 0x4b, 0xda,
                       0x0a, 0x79, 0x83, 0x12, 0x0f, 0x44, 0x93, 0x6e, 0x8f, 0xf3, 0x1d, 0x29, 0x72,
                       0x8a, 0xc0, 0x8c, 0xcc, 0xe6, 0xf3, 0xf9, 0x16, 0xb3, 0xc6, 0x39, 0x62, 0x75,
                       0x5c, 0x23, 0xa1, 0xfa, 0x9b, 0xb4, 0xe8, 0x82, 0x3f, 0xc8, 0x67, 0xbf, 0xd1,
                       0x8f, 0x28, 0x98, 0x0d, 0x94, 0xbc, 0x58, 0x74, 0x42, 0x3a, 0xb7, 0xf9, 0x6c,
                       0xc0, 0xab, 0x78, 0xd8, 0xfa, 0x21, 0xfb, 0xd0, 0x0c, 0xd3, 0xa1, 0xd9, 0x6a,
                       0x73, 0xfa, 0x43, 0x9c, 0xcc, 0x3f, 0xc4, 0xea, 0xb1, 0x59, 0x06, 0x77, 0xb0,
                       0x6c, 0xc7, 0x8b, 0x0f, 0x67, 0x4d, 0xfb, 0x68, 0x0f, 0x23, 0x02, 0x2f, 0xb9,
                       0x02, 0x02, 0x2d, 0xd8, 0x62, 0x08, 0x03, 0x22, 0x9c, 0x6d, 0xdf, 0x79, 0xa8,
                       0x15, 0x6c, 0xcf, 0xce, 0x48, 0xbb, 0xd7, 0x6c, 0x05, 0xab, 0x67, 0x06, 0x34,
                       0xf2, 0x06, 0xe5, 0xb2, 0xe8, 0x96, 0x23, 0x0b, 0xaa, 0x74, 0xa8, 0x56, 0x96,
                       0x4d, 0xbd, 0x85, 0x11, 0xac, 0xb7, 0x1d, 0x75, 0xa1, 0x50, 0x67, 0x66, 0xa1,
                       0x25, 0xd8, 0xce, 0x03, 0x7f, 0x1d, 0xb7, 0x20, 0x86, 0xeb, 0xc3, 0xbc, 0xca,
                       0xef, 0xbd, 0x8c, 0xd9, 0x38, 0x01, 0x67, 0xc2, 0x53, 0x03, 0x86, 0x54, 0x4e,
                       0xbf, 0xbe, 0xad, 0xbe, 0x23, 0x77, 0x84, 0xd1, 0x02, 0xbb, 0x92, 0xa1, 0x0f,
                       0xd2, 0x42],
            },
            {
                pass: [0x4a, 0x85, 0x7e, 0x2e, 0xe8, 0xaa, 0x9b, 0x60, 0x56, 0xf2, 0x42, 0x4e, 0x84,
                       0xd2, 0x4a, 0x72, 0x47, 0x33, 0x78, 0x90, 0x6e, 0xe0, 0x4a, 0x46, 0xcb, 0x05,
                       0x31, 0x15, 0x02, 0xd5, 0x25, 0x0b, 0x82, 0xad, 0x86, 0xb8, 0x3c, 0x8f, 0x20,
                       0xa2, 0x3d, 0xbb, 0x74, 0xf6, 0xda, 0x60, 0xb0, 0xb6, 0xec, 0xff, 0xd6, 0x71,
                       0x34, 0xd4, 0x59, 0x46, 0xac, 0x8e, 0xbf, 0xb3, 0x06, 0x42, 0x94, 0xbc, 0x09,
                       0x7d, 0x43, 0xce, 0xd6, 0x86, 0x42, 0xbf, 0xb8, 0xbb, 0xbd, 0xd0, 0xf5, 0x0b,
                       0x30, 0x11, 0x8f, 0x5e],
                salt: [0x39, 0xd8, 0x2e, 0xef, 0x32, 0x01, 0x0b, 0x8b, 0x79, 0xcc, 0x5b, 0xa8, 0x8e,
                       0xd5, 0x39, 0xfb],
                ops:  3,
                mem:  1432947,
                out:  [0x08, 0xd8, 0xcd, 0x33, 0x0c, 0x57, 0xe1, 0xb4, 0x64, 0x32, 0x41, 0xd0, 0x5b,
                       0xb4, 0x68, 0xba, 0x4e, 0xe4, 0xe9, 0x32, 0xcd, 0x08, 0x58, 0x81, 0x6b, 0xe9,
                       0xef, 0x15, 0x36, 0x0b, 0x27, 0xbb, 0xd0, 0x6a, 0x87, 0x13, 0x0e, 0xe9, 0x22,
                       0x22, 0xbe, 0x26, 0x7a, 0x29, 0xb8, 0x1f, 0x5a, 0xe8, 0xfe, 0x86, 0x13, 0x32,
                       0x4c, 0xfc, 0x48, 0x32, 0xdc, 0x49, 0x38, 0x7f, 0xd0, 0x60, 0x2f, 0x1c, 0x57,
                       0xb4, 0xd0, 0xf3, 0x85, 0x5d, 0xb9, 0x4f, 0xb7, 0xe1, 0x2e, 0xb0, 0x5f, 0x9a,
                       0x48, 0x4a, 0xed, 0x4a, 0x43, 0x07, 0xab, 0xf5, 0x86, 0xcd, 0x3d, 0x55, 0xc8,
                       0x09, 0xbc, 0x08, 0x15, 0x41, 0xe0, 0x0b, 0x68, 0x27, 0x72, 0xfb, 0x20, 0x66,
                       0x50, 0x4f, 0xf9, 0x35, 0xb8, 0xeb, 0xc5, 0x51, 0xa2, 0x08, 0x38, 0x82, 0xf8,
                       0x74, 0xbc, 0x0f, 0xae, 0x68, 0xe5, 0x68, 0x48, 0xae, 0x34, 0xc9, 0x10, 0x97,
                       0xc3, 0xbf, 0x0c, 0xca, 0x8e, 0x75, 0xc0, 0x79, 0x7e, 0xef, 0x3e, 0xfd, 0xe3,
                       0xf7, 0x5e, 0x00, 0x58, 0x15, 0x01, 0x8d, 0xb3, 0xcf, 0x7c, 0x10, 0x9a, 0x81,
                       0x22, 0x64, 0xc4, 0xde, 0x69, 0xdc, 0xb2, 0x23, 0x22, 0xdb, 0xbc, 0xfa, 0x44,
                       0x7f, 0x5b, 0x00, 0xec, 0xd1, 0xb0, 0x4a, 0x7b, 0xe1, 0x56, 0x9c, 0x8e, 0x55,
                       0x6a, 0xdb, 0x7b, 0xba, 0x48, 0xad, 0xf8, 0x1d],
            },
            {
                pass: [0xc7, 0xb0, 0x9a, 0xec, 0x68, 0x0e, 0x7b, 0x42, 0xfe, 0xdd, 0x7f, 0xc7, 0x92,
                       0xe7, 0x8b, 0x2f, 0x6c, 0x1b, 0xea, 0x8f, 0x4a, 0x88, 0x43, 0x20, 0xb6, 0x48,
                       0xf8, 0x1e, 0x8c, 0xf5, 0x15, 0xe8, 0xba, 0x9d, 0xcf, 0xb1, 0x1d, 0x43, 0xc4,
                       0xaa, 0xe1, 0x14, 0xc1, 0x73, 0x4a, 0xa6, 0x9c, 0xa8, 0x2d, 0x44, 0x99, 0x83,
                       0x65, 0xdb, 0x9c, 0x93, 0x74, 0x4f, 0xa2, 0x8b, 0x63, 0xfd, 0x16, 0x00, 0x0e,
                       0x82, 0x61, 0xcb, 0xbe, 0x08, 0x3e, 0x7e, 0x2d, 0xa1, 0xe5, 0xf6, 0x96, 0xbd,
                       0xe0, 0x83, 0x4f, 0xe5, 0x31, 0x46, 0xd7, 0xe0, 0xe3, 0x5e, 0x7d, 0xe9, 0x92,
                       0x0d, 0x04, 0x1f, 0x5a, 0x56, 0x21, 0xaa, 0xbe, 0x02, 0xda, 0x3e, 0x2b, 0x09,
                       0xb4, 0x05, 0xb7, 0x79, 0x37, 0xef, 0xef, 0x31, 0x97, 0xbd, 0x57, 0x72, 0xe4,
                       0x1f, 0xdb, 0x73, 0xfb, 0x52, 0x94, 0x47, 0x8e, 0x45, 0x20, 0x80, 0x63, 0xb5,
                       0xf5, 0x8e, 0x08, 0x9d, 0xbe, 0xb6, 0xd6, 0x34, 0x2a, 0x90, 0x9c, 0x13, 0x07,
                       0xb3, 0xff, 0xf5, 0xfe, 0x2c, 0xf4, 0xda, 0x56, 0xbd, 0xae, 0x50, 0x84,
                       0x8f],
                salt: [0x03, 0x9c, 0x05, 0x6d, 0x93, 0x3b, 0x47, 0x50, 0x32, 0x77, 0x7e, 0xdb, 0xaf,
                       0xfa, 0xc5, 0x0f],
                ops:  3,
                mem:  4886999,
                out:  [0xd6, 0xe9, 0xd6, 0xca, 0xbd, 0x42, 0xfb, 0x9b, 0xa7, 0x16, 0x2f, 0xe9, 0xb8,
                       0xe4, 0x1d, 0x59, 0xd3, 0xc7, 0x03, 0x47, 0x56, 0xcb, 0x46, 0x0c, 0x9a, 0xff,
                       0xe3, 0x93, 0x30, 0x8b, 0xd0, 0x22, 0x5c, 0xe0, 0x37, 0x1f, 0x2e, 0x6c, 0x3c,
                       0xa3, 0x2a, 0xca, 0x20, 0x02, 0xbf, 0x2d, 0x39, 0x09, 0xc6, 0xb6, 0xe7, 0xdf,
                       0xc4, 0xa0, 0x0e, 0x85, 0x0f, 0xf4, 0xf5, 0x70, 0xf8, 0xf7, 0x49, 0xd4, 0xbb,
                       0x6f, 0x00, 0x91, 0xe5, 0x54, 0xbe, 0x67, 0xa9, 0x09, 0x5a, 0xe1, 0xee, 0xfa,
                       0xa1, 0xa9, 0x33, 0x31, 0x6c, 0xbe, 0xc3, 0xc2, 0xfd, 0x4a, 0x14, 0xa5, 0xb6,
                       0x94, 0x1b, 0xda, 0x9b, 0x7e, 0xab, 0xd8, 0x21, 0xd7, 0x9a, 0xbd, 0xe2, 0x47,
                       0x5a, 0x53, 0xaf, 0x1a, 0x85, 0x71, 0xc7, 0xee, 0x46, 0x46, 0x0b, 0xe4, 0x15,
                       0x88, 0x2e, 0x0b, 0x39, 0x3f, 0x48, 0xc1, 0x2f, 0x74, 0x0a, 0x6a, 0x72, 0xcb,
                       0xa9, 0x77, 0x30, 0x00, 0x60, 0x2e, 0x13, 0xb4, 0x0d, 0x3d, 0xfa, 0x6a, 0xc1,
                       0xd4, 0xec, 0x43, 0xa8, 0x38, 0xb7, 0xe3, 0xe1, 0x65, 0xfe, 0xca, 0xd4, 0xb2,
                       0x49, 0x83, 0x89, 0xe6, 0x0a, 0x3f, 0xf9, 0xf0, 0xf8, 0xf4, 0xb9, 0xfc, 0xa1,
                       0x12, 0x6e, 0x64, 0xf4, 0x95, 0x01, 0xe3, 0x86, 0x90],
            },
            {
                pass: [0xb5, 0x40, 0xbe, 0xb0, 0x16, 0xa5, 0x36, 0x65, 0x24, 0xd4, 0x60, 0x51, 0x56,
                       0x49, 0x3f, 0x98, 0x74, 0x51, 0x4a, 0x5a, 0xa5, 0x88, 0x18, 0xcd, 0x0c, 0x6d,
                       0xff, 0xfa, 0xa9, 0xe9, 0x02, 0x05, 0xf1, 0x7b],
                salt: [0x44, 0x07, 0x1f, 0x6d, 0x18, 0x15, 0x61, 0x67, 0x0b, 0xda, 0x72, 0x8d, 0x43,
                       0xfb, 0x79, 0xb4],
                ops:  1,
                mem:  1631659,
                out:  [0x7f, 0xb7, 0x24, 0x09, 0xb0, 0x98, 0x7f, 0x81, 0x90, 0xc3, 0x72, 0x97, 0x10,
                       0xe9, 0x8c, 0x3f, 0x80, 0xc5, 0xa8, 0x72, 0x7d, 0x42, 0x5f, 0xdc, 0xde, 0x7f,
                       0x36, 0x44, 0xd4, 0x67, 0xfe, 0x97, 0x3f, 0x5b, 0x5f, 0xee, 0x68, 0x3b, 0xd3,
                       0xfc, 0xe8, 0x12, 0xcb, 0x9a, 0xe5, 0xe9, 0x92, 0x1a, 0x2d, 0x06, 0xc2, 0xf1,
                       0x90, 0x5e, 0x4e, 0x83, 0x96, 0x92, 0xf2, 0xb9, 0x34, 0xb6, 0x82, 0xf1, 0x1a,
                       0x2f, 0xe2, 0xb9, 0x04, 0x82, 0xea, 0x5d, 0xd2, 0x34, 0x86, 0x35, 0x16, 0xdb,
                       0xa6, 0xf5, 0x2d, 0xc0, 0x70, 0x2d, 0x32, 0x4e, 0xc7, 0x7d, 0x86, 0x0c, 0x2e,
                       0x18, 0x1f, 0x84, 0x47, 0x2b, 0xd7, 0x10, 0x4f, 0xed, 0xce, 0x07, 0x1f, 0xfa,
                       0x93, 0xc5, 0x30, 0x94, 0x94, 0xad, 0x51, 0x62, 0x3d, 0x21, 0x44, 0x47, 0xa7,
                       0xb2, 0xb1, 0x46, 0x2d, 0xc7, 0xd5, 0xd5, 0x5a, 0x1f, 0x6f, 0xd5, 0xb5, 0x4c,
                       0xe0, 0x24, 0x11, 0x8d, 0x86, 0xf0, 0xc6, 0x48, 0x9d, 0x16, 0x54, 0x5a, 0xaa,
                       0x87, 0xb6, 0x68, 0x9d, 0xad, 0x9f, 0x2f, 0xb4, 0x7f, 0xda, 0x98, 0x94, 0xf8,
                       0xe1, 0x2b, 0x87, 0xd9, 0x78, 0xb4, 0x83, 0xcc, 0xd4, 0xcc, 0x5f, 0xd9, 0x59,
                       0x5c, 0xdc, 0x7a, 0x81, 0x84, 0x52, 0xf9, 0x15, 0xce, 0x2f, 0x7d, 0xf9, 0x5e,
                       0xc1, 0x2b, 0x1c, 0x72, 0xe3, 0x78, 0x8d, 0x47, 0x34, 0x41, 0xd8, 0x84, 0xf9,
                       0x74, 0x8e, 0xb1, 0x47, 0x03, 0xc2, 0x1b, 0x45, 0xd8, 0x2f, 0xd6, 0x67, 0xb8,
                       0x5f, 0x5b, 0x2d, 0x98, 0xc1, 0x33, 0x03, 0xb3, 0xfe, 0x76, 0x28, 0x55, 0x31,
                       0xa8, 0x26, 0xb6, 0xfc, 0x0f, 0xe8, 0xe3, 0xdd, 0xde, 0xcf],
            },
            {
                pass: [0xa1, 0x49, 0x75, 0xc2, 0x6c, 0x08, 0x87, 0x55, 0xa8, 0xb7, 0x15, 0xff, 0x25,
                       0x28, 0xd6, 0x47, 0xcd, 0x34, 0x39, 0x87, 0xfc, 0xf4, 0xaa, 0x25, 0xe7, 0x19,
                       0x4a, 0x84, 0x17, 0xfb, 0x2b, 0x4b, 0x3f, 0x72, 0x68, 0xda, 0x9f, 0x31, 0x82,
                       0xb4, 0xcf, 0xb2, 0x2d, 0x13, 0x8b, 0x27, 0x49, 0xd6, 0x73, 0xa4, 0x7e, 0xcc,
                       0x75, 0x25, 0xdd, 0x15, 0xa0, 0xa3, 0xc6, 0x60, 0x46, 0x97, 0x17, 0x84, 0xbb,
                       0x63, 0xd7, 0xea, 0xe2, 0x4c, 0xc8, 0x4f, 0x26, 0x31, 0x71, 0x20, 0x75, 0xa1,
                       0x0e, 0x10, 0xa9, 0x6b, 0x0e, 0x0e, 0xe6, 0x7c, 0x43, 0xe0, 0x1c, 0x42, 0x3c,
                       0xb9, 0xc4, 0x4e, 0x53, 0x71, 0x01, 0x7e, 0x9c, 0x49, 0x69, 0x56, 0xb6, 0x32,
                       0x15, 0x8d, 0xa3, 0xfe, 0x12, 0xad, 0xde, 0xcb, 0x88, 0x91, 0x2e, 0x67, 0x59,
                       0xbc, 0x37, 0xf9, 0xaf, 0x2f, 0x45, 0xaf, 0x72, 0xc5, 0xca, 0xe3, 0xb1, 0x79,
                       0xff, 0xb6, 0x76, 0xa6, 0x97, 0xde, 0x6e, 0xbe, 0x45, 0xcd, 0x4c, 0x16, 0xd4,
                       0xa9, 0xd6, 0x42, 0xd2, 0x9d, 0xdc, 0x01, 0x86, 0xa0, 0xa4, 0x8c, 0xb6, 0xcd,
                       0x62, 0xbf, 0xc3, 0xdd, 0x22, 0x9d, 0x31, 0x3b, 0x30, 0x15, 0x60, 0x97, 0x1e,
                       0x74, 0x0e, 0x2c, 0xf1, 0xf9, 0x9a, 0x9a, 0x09, 0x0a, 0x5b, 0x28, 0x3f, 0x35,
                       0x47, 0x50, 0x57, 0xe9, 0x6d, 0x70, 0x64, 0xe2, 0xe0, 0xfc, 0x81, 0x98, 0x45,
                       0x91, 0x06, 0x8d, 0x55, 0xa3, 0xb4, 0x16, 0x9f, 0x22, 0xcc, 0xcb, 0x07, 0x45,
                       0xa2, 0x68, 0x94, 0x07, 0xea, 0x19, 0x01, 0xa0, 0xa7, 0x66, 0xeb, 0x99],
                salt: [0x3d, 0x96, 0x8b, 0x27, 0x52, 0xb8, 0x83, 0x84, 0x31, 0x16, 0x50, 0x59, 0x31,
                       0x9f, 0x3f, 0xf8],
                ops:  3,
                mem:  1784128,
                out:  [0x4e, 0x70, 0x2b, 0xc5, 0xf8, 0x91, 0xdf, 0x88, 0x4c, 0x6d, 0xda, 0xa2, 0x43,
                       0xaa, 0x84, 0x6c, 0xe3, 0xc0, 0x87, 0xfe, 0x93, 0x0f, 0xef, 0x0f, 0x36, 0xb3,
                       0xc2, 0xbe, 0x34, 0x16, 0x4c, 0xcc, 0x29, 0x5d, 0xb5, 0x09, 0x25, 0x47, 0x43,
                       0xf1, 0x8f, 0x94, 0x71, 0x59, 0xc8, 0x13, 0xbc, 0xd5, 0xdd, 0x8d, 0x94, 0xa3,
                       0xae, 0xc9, 0x3b, 0xbe, 0x57, 0x60, 0x5d, 0x1f, 0xad, 0x1a, 0xef, 0x11, 0x12,
                       0x68, 0x7c, 0x3d, 0x4e, 0xf1, 0xcb, 0x32, 0x9d, 0x21, 0xf1, 0x63, 0x2f, 0x62,
                       0x68, 0x18, 0xd7, 0x66, 0x91, 0x5d, 0x88, 0x6e, 0x8d, 0x81, 0x9e, 0x4b, 0x0b,
                       0x9c, 0x93, 0x07, 0xf4, 0xb6, 0xaf, 0xc0, 0x81, 0xe1, 0x3b, 0x0c, 0xf3, 0x1d,
                       0xb3, 0x82, 0xff, 0x1b, 0xf0, 0x5a, 0x16, 0xaa, 0xc7, 0xaf, 0x69, 0x63, 0x36,
                       0xd7, 0x5e, 0x99, 0xf8, 0x21, 0x63, 0xe0, 0xf3, 0x71, 0xe1, 0xd2, 0x5c, 0x4a,
                       0xdd, 0x80, 0x8e, 0x21, 0x56, 0x97, 0xad, 0x3f, 0x77, 0x9a, 0x51, 0xa4, 0x62,
                       0xf8, 0xbf, 0x52, 0x61, 0x0a, 0xf2, 0x1f, 0xc6, 0x9d, 0xba, 0x6b, 0x07, 0x26,
                       0x06, 0xf2, 0xda, 0xbc, 0xa7, 0xd4, 0xae, 0x1d, 0x91, 0xd9, 0x19],
            },
        ];

        verify_str_valid_tests! [
            {
                pass: "",
                hash: "$argon2id$v=19$m=4882,t=2,p=1$bA81arsiXysd3WbTRzmEOw$Nm8QBM+7RH1DXo9rvp5cwKE\
                       OOOfD2g6JuxlXihoNcpE",
            },
            {
                pass: "^T5H$JYt39n%K*j:W]!1s?vg!:jGi]Ax?..l7[p0v:1jHTpla9;]bUN;?bWyCbtqg ",
                hash: "$argon2id$v=19$m=4096,t=19,p=1$PkEgMTYtYnl0ZXMgc2FsdA$ltB/ue1kPtBMBGfsysMpPi\
                       gE6hiNEKZ9vs8vLNVDQGA",
            },
            {
                pass: "K3S=KyH#)36_?]LxeR8QNKw6X=gFbxai$C%29V*",
                hash: "$argon2id$v=19$m=4096,t=1,p=3$PkEgcHJldHR5IGxvbmcgc2FsdA$HUqx5Z1b/ZypnUrvvJ5\
                       UC2Q+T6Q1WwASK/Kr9dRbGA0",
            },
        ];

        verify_str_invalid_tests! [
            {
                pass: "",
                hash: "$argon2id$v=19$m=4096,t=0,p=1$X1NhbHQAAAAAAAAAAAAAAA$bWh++MKN1OiFHKgIWTLvIi1\
                       iHicmHH7+Fv3K88ifFfI",
            },
            {
                pass: "",
                hash: "$argon2id$v=19$m=2048,t=4,p=1$SWkxaUhpY21ISDcrRnYzSw$Mbg/Eck1kpZir5T9io7C64c\
                       pffdTBaORgyriLQFgQj8",
            },
            {
                pass: "^T5H$JYt39n%K*j:W]!1s?vg!:jGi]Ax?..l7[p0v:1jHTpla9;]bUN;?bWyCbtqg ",
                hash: "$argon2id$v=19$m=4096,t=0,p=1$PkEgMTYtYnl0ZXMgc2FsdA$ltB/ue1kPtBMBGfsysMpPig\
                       E6hiNEKZ9vs8vLNVDQGA",
            },
        ];

        needs_rehash_tests! {}
    }
}

/// The [Argon2i](https://en.wikipedia.org/wiki/Argon2) password-based key derivation algorithm.
///
/// This algorithm is intended to be highly resistant to side-channel attacks, but may be less
/// resistant to GPU-based attacks than [`argon2id`].
pub mod argon2i {
    use libsodium_sys as sodium;

    argon_module! {
        sodium::crypto_pwhash_argon2i_OPSLIMIT_MIN,
        sodium::crypto_pwhash_argon2i_OPSLIMIT_INTERACTIVE,
        sodium::crypto_pwhash_argon2i_OPSLIMIT_MODERATE,
        sodium::crypto_pwhash_argon2i_OPSLIMIT_SENSITIVE,
        sodium::crypto_pwhash_argon2i_OPSLIMIT_MAX,
        sodium::crypto_pwhash_argon2i_MEMLIMIT_MIN,
        sodium::crypto_pwhash_argon2i_MEMLIMIT_INTERACTIVE,
        sodium::crypto_pwhash_argon2i_MEMLIMIT_MODERATE,
        sodium::crypto_pwhash_argon2i_MEMLIMIT_SENSITIVE,
        sodium::crypto_pwhash_argon2i_memlimit_max,
        sodium::crypto_pwhash_argon2i_PASSWD_MIN,
        sodium::crypto_pwhash_argon2i_passwd_max,
        sodium::crypto_pwhash_argon2i_BYTES_MIN,
        sodium::crypto_pwhash_argon2i_bytes_max,
        sodium::crypto_pwhash_argon2i_SALTBYTES,
        sodium::crypto_pwhash_argon2i_STRBYTES,
        sodium::crypto_pwhash_ALG_ARGON2I13,
        sodium::crypto_pwhash_argon2i,
        sodium::crypto_pwhash_argon2i_str,
        sodium::crypto_pwhash_argon2i_str_verify,
        sodium::crypto_pwhash_argon2i_str_needs_rehash,
    }

    #[cfg(test)]
    mod tests {
        kdf_tests! {
            {
                pass: [0xa3, 0x47, 0xae, 0x92, 0xbc, 0xe9, 0xf8, 0x0f, 0x6f, 0x59, 0x5a, 0x44, 0x80,
                       0xfc, 0x9c, 0x2f, 0xe7, 0xe7, 0xd7, 0x14, 0x8d, 0x37, 0x1e, 0x94, 0x87, 0xd7,
                       0x5f, 0x5c, 0x23, 0x00, 0x8f, 0xfa, 0xe0, 0x65, 0x57, 0x7a, 0x92, 0x8f, 0xeb,
                       0xd9, 0xb1, 0x97, 0x3a, 0x5a, 0x95, 0x07, 0x3a, 0xcd, 0xbe, 0xb6, 0xa0, 0x30,
                       0xcf, 0xc0, 0xd7, 0x9c, 0xaa, 0x2d, 0xc5, 0xcd, 0x01, 0x1c, 0xef, 0x02, 0xc0,
                       0x8d, 0xa2, 0x32, 0xd7, 0x6d, 0x52, 0xdf, 0xbc, 0xa3, 0x8c, 0xa8, 0xdc, 0xbd,
                       0x66, 0x5b, 0x17, 0xd1, 0x66, 0x5f, 0x7c, 0xf5, 0xfe, 0x59, 0x77, 0x2e, 0xc9,
                       0x09, 0x73, 0x3b, 0x24, 0xde, 0x97, 0xd6, 0xf5, 0x8d, 0x22, 0x0b, 0x20, 0xc6,
                       0x0d, 0x7c, 0x07, 0xec, 0x1f, 0xd9, 0x3c, 0x52, 0xc3, 0x10, 0x20, 0x30, 0x0c,
                       0x6c, 0x1f, 0xac, 0xd7, 0x79, 0x37, 0xa5, 0x97, 0xc7, 0xa6],
                salt: [0x55, 0x41, 0xfb, 0xc9, 0x95, 0xd5, 0xc1, 0x97, 0xba, 0x29, 0x03, 0x46, 0xd2,
                       0xc5, 0x59, 0xde],
                ops:  5,
                mem:  7256678,
                out:  [0x23, 0xb8, 0x03, 0xc8, 0x4e, 0xaa, 0x25, 0xf4, 0xb4, 0x46, 0x34, 0xcc, 0x1e,
                       0x5e, 0x37, 0x79, 0x2c, 0x53, 0xfc, 0xd9, 0xb1, 0xeb, 0x20, 0xf8, 0x65, 0x32,
                       0x9c, 0x68, 0xe0, 0x9c, 0xbf, 0xa9, 0xf1, 0x96, 0x87, 0x57, 0x90, 0x1b, 0x38,
                       0x3f, 0xce, 0x22, 0x1a, 0xfe, 0x27, 0x71, 0x3f, 0x97, 0x91, 0x4a, 0x04, 0x13,
                       0x95, 0xbb, 0xe1, 0xfb, 0x70, 0xe0, 0x79, 0xe5, 0xbe, 0xd2, 0xc7, 0x14, 0x5b,
                       0x1f, 0x61, 0x54, 0x04, 0x6f, 0x59, 0x58, 0xe9, 0xb1, 0xb2, 0x90, 0x55, 0x45,
                       0x4e, 0x26, 0x4d, 0x1f, 0x22, 0x31, 0xc3, 0x16, 0xf2, 0x6b, 0xe2, 0xe3, 0x73,
                       0x8e, 0x83, 0xa8, 0x03, 0x15, 0xe9, 0xa0, 0x95, 0x1c, 0xe4, 0xb1, 0x37, 0xb5,
                       0x2e, 0x7d, 0x5e, 0xe7, 0xb3, 0x7f, 0x7d, 0x93, 0x6d, 0xce, 0xe5, 0x13, 0x62,
                       0xbc, 0xf7, 0x92, 0x59, 0x5e, 0x3c, 0x89, 0x6a, 0xd5, 0x04, 0x27, 0x34, 0xfc,
                       0x90, 0xc9, 0x2c, 0xae, 0x57, 0x2c, 0xe6, 0x3f, 0xf6, 0x59, 0xa2, 0xf7, 0x97,
                       0x4a, 0x3b, 0xd7, 0x30, 0xd0, 0x4d, 0x52, 0x5d, 0x25, 0x3c, 0xcc, 0x38],
            },
            {
                pass: [0xe1, 0x25, 0xce, 0xe6, 0x1c, 0x8c, 0xb7, 0x77, 0x8d, 0x9e, 0x5a, 0xd0, 0xa6,
                       0xf5, 0xd9, 0x78, 0xce, 0x9f, 0x84, 0xde, 0x21, 0x3a, 0x85, 0x56, 0xd9, 0xff,
                       0xe2, 0x02, 0x02, 0x0a, 0xb4, 0xa6, 0xed, 0x90, 0x74, 0xa4, 0xeb, 0x34, 0x16,
                       0xf9, 0xb1, 0x68, 0xf1, 0x37, 0x51, 0x0f, 0x3a, 0x30, 0xb7, 0x0b, 0x96, 0xcb,
                       0xfa, 0x21, 0x9f, 0xf9, 0x9f, 0x6c, 0x6e, 0xaf, 0xfb, 0x15, 0xc0, 0x6b, 0x60,
                       0xe0, 0x0c, 0xc2, 0x89, 0x02, 0x77, 0xf0, 0xfd, 0x3c, 0x62, 0x21, 0x15, 0x77,
                       0x2f, 0x70, 0x48, 0xad, 0xae, 0xbe, 0xd8, 0x6e],
                salt: [0xf1, 0x19, 0x2d, 0xd5, 0xdc, 0x23, 0x68, 0xb9, 0xcd, 0x42, 0x13, 0x38, 0xb2,
                       0x24, 0x33, 0x45],
                ops:  4,
                mem:  7849083,
                out:  [0x0b, 0xb3, 0x76, 0x9b, 0x06, 0x4b, 0x9c, 0x43, 0xa9, 0x46, 0x04, 0x76, 0xab,
                       0x38, 0xc4, 0xa9, 0xa2, 0x47, 0x0d, 0x55, 0xd4, 0xc9, 0x92, 0xc6, 0xe7, 0x23,
                       0xaf, 0x89, 0x5e, 0x4c, 0x07, 0xc0, 0x9a, 0xf4, 0x1f, 0x22, 0xf9, 0x0e, 0xab,
                       0x58, 0x3a, 0x0c, 0x36, 0x2d, 0x17, 0x7f, 0x46, 0x77, 0xf2, 0x12, 0x48, 0x2f,
                       0xd1, 0x45, 0xbf, 0xb9, 0xac, 0x62, 0x11, 0x63, 0x5e, 0x48, 0x46, 0x11, 0x22,
                       0xbb, 0x49, 0x09, 0x7b, 0x5f, 0xb0, 0x73, 0x9d, 0x2c, 0xd2, 0x2a, 0x39, 0xbf,
                       0x03, 0xd2, 0x68, 0xe7, 0x49, 0x5d, 0x4f, 0xd8, 0xd7, 0x10, 0xaa, 0x15, 0x62,
                       0x02, 0xf0, 0xa0, 0x6e, 0x93, 0x2f, 0xf5, 0x13, 0xe6, 0xe7, 0xc7, 0x6a, 0x4e,
                       0x98, 0xb6, 0xdf, 0x5c, 0xf9, 0x22, 0xf1, 0x24, 0x79, 0x1b, 0x10, 0x76, 0xad,
                       0x90, 0x4e, 0x68, 0x97, 0x27, 0x1f, 0x5d, 0x7d, 0x24, 0xc5, 0x92, 0x9e, 0x2a,
                       0x3b, 0x83, 0x6d, 0x0f, 0x2f, 0x26, 0x97, 0xc2, 0xd7, 0x58, 0xee, 0x79, 0xbf,
                       0x12, 0x64, 0xf3, 0xfa, 0xe6, 0x5f, 0x37, 0x44, 0xe0, 0xf6, 0xd7, 0xd0, 0x7e,
                       0xf6, 0xe8, 0xb3, 0x5b, 0x70, 0xc0, 0xf8, 0x8e, 0x90, 0x36, 0x32, 0x5b, 0xfb,
                       0x24, 0xac, 0x7f, 0x55, 0x03, 0x51, 0x48, 0x6d, 0xa8, 0x7a, 0xef, 0x10, 0xd6,
                       0xb0, 0xcb, 0x77, 0xd1, 0xcf, 0x6e, 0x31, 0xcf, 0x98, 0x39, 0x9c, 0x6f, 0x24,
                       0x1c, 0x60, 0x5c, 0x65, 0x30, 0xdf, 0xfb, 0x47, 0x64, 0x78, 0x4f, 0x6c, 0x0b,
                       0x0b, 0xf6, 0x01, 0xd4, 0xe4, 0x43, 0x1e, 0x8b, 0x18, 0xda, 0xbd, 0xc3, 0x07,
                       0x9c, 0x6e, 0x26, 0x43, 0x02, 0xad, 0xe7, 0x9f, 0x61, 0xcb, 0xd5, 0x49, 0x7c,
                       0x95, 0x48, 0x63, 0x40, 0xbb, 0x89, 0x1a, 0x73, 0x72, 0x23, 0x10, 0x0b, 0xe0,
                       0x42, 0x96, 0x50],
            },
            {
                pass: [0x92, 0x26, 0x3c, 0xbf, 0x6a, 0xc3, 0x76, 0x49, 0x9f, 0x68, 0xa4, 0x28, 0x9d,
                       0x3b, 0xb5, 0x9e, 0x5a, 0x22, 0x33, 0x5e, 0xba, 0x63, 0xa3, 0x2e, 0x64, 0x10,
                       0x24, 0x91, 0x55, 0xb9, 0x56, 0xb6, 0xa3, 0xb4, 0x8d, 0x4a, 0x44, 0x90, 0x6b,
                       0x18, 0xb8, 0x97, 0x12, 0x73, 0x00, 0xb3, 0x75, 0xb8, 0xf8, 0x34, 0xf1, 0xce,
                       0xff, 0xc7, 0x08, 0x80, 0xa8, 0x85, 0xf4, 0x7c, 0x33, 0x87, 0x67, 0x17, 0xe3,
                       0x92, 0xbe, 0x57, 0xf7, 0xda, 0x3a, 0xe5, 0x8d, 0xa4, 0xfd, 0x1f, 0x43, 0xda,
                       0xa7, 0xe4, 0x4b, 0xb8, 0x2d, 0x37, 0x17, 0xaf, 0x43, 0x19, 0x34, 0x9c, 0x24,
                       0xcd, 0x31, 0xe4, 0x6d, 0x29, 0x58, 0x56, 0xb0, 0x44, 0x1b, 0x6b, 0x28, 0x99,
                       0x92, 0xa1, 0x1c, 0xed, 0x1c, 0xc3, 0xbf, 0x30, 0x11, 0x60, 0x45, 0x90, 0x24,
                       0x4a, 0x3e, 0xb7, 0x37, 0xff, 0x22, 0x11, 0x29, 0x21, 0x5e, 0x4e, 0x43, 0x47,
                       0xf4, 0x91, 0x5d, 0x41, 0x29, 0x2b, 0x51, 0x73, 0xd1, 0x96, 0xeb, 0x9a, 0xdd,
                       0x69, 0x3b, 0xe5, 0x31, 0x9f, 0xda, 0xdc, 0x24, 0x29, 0x06, 0x17, 0x8b, 0xb6,
                       0xc0, 0x28, 0x6c, 0x9b, 0x6c, 0xa6, 0x01, 0x27, 0x46, 0x71, 0x1f, 0x58, 0xc8,
                       0xc3, 0x92, 0x01, 0x6b, 0x2f, 0xdf, 0xc0, 0x9c, 0x64, 0xf0, 0xf6, 0xb6, 0xab,
                       0x7b],
                salt: [0x3b, 0x84, 0x0e, 0x20, 0xe9, 0x55, 0x5e, 0x9f, 0xb0, 0x31, 0xc4, 0xba, 0x1f,
                       0x17, 0x47, 0xce],
                ops:  3,
                mem:  7994791,
                out:  [0xe9, 0xaa, 0x07, 0x3b, 0x0b, 0x87, 0x2f, 0x15, 0xc0, 0x83, 0xd1, 0xd7, 0xce,
                       0x52, 0xc0, 0x9f, 0x49, 0x3b, 0x82, 0x7c, 0xa7, 0x8f, 0x13, 0xa0, 0x6c, 0x17,
                       0x21, 0xb4, 0x5b, 0x1e, 0x17, 0xb2, 0x4c, 0x04, 0xe1, 0x9f, 0xe8, 0x69, 0x33,
                       0x31, 0x35, 0x36, 0x01, 0x97, 0xa7, 0xeb, 0x55, 0x99, 0x4f, 0xee, 0x3e, 0x8d,
                       0x96, 0x80, 0xae, 0xdf, 0xdf, 0x76, 0x74, 0xf3, 0xad, 0x7b, 0x84, 0xd5, 0x9d,
                       0x7e, 0xab, 0x03, 0x57, 0x9f, 0xfc, 0x10, 0xc7, 0x09, 0x30, 0x93, 0xbc, 0x48,
                       0xec, 0x84, 0x25, 0x2a, 0xa1, 0xb3, 0x0f, 0x40, 0xf5, 0xe8, 0x38, 0xf1, 0x44,
                       0x3e, 0x15, 0xe2, 0x77, 0x2a, 0x39, 0xf4, 0xe7, 0x74, 0xeb, 0x05, 0x20, 0x97,
                       0xe8, 0x88, 0x1e, 0x94, 0xf1, 0x54, 0x57, 0xb7, 0x79, 0xfa, 0x2a, 0xf2, 0xbb,
                       0xc9, 0xa9, 0x93, 0x68, 0x76, 0x57, 0xc7, 0x70, 0x4a, 0xc8, 0xa3, 0x7c, 0x25,
                       0xc1, 0xdf, 0x42, 0x89, 0xeb, 0x4c, 0x70, 0xda, 0x45, 0xf2, 0xfd, 0x46, 0xbc,
                       0x0f, 0x78, 0x25, 0x97, 0x67, 0xd3, 0xdd, 0x47, 0x8a, 0x7c, 0x36, 0x9c, 0xf8,
                       0x66, 0x75, 0x8b, 0xc3, 0x6d, 0x9b, 0xd8, 0xe2, 0xe3, 0xc9, 0xfb, 0x0c, 0xf7,
                       0xfd, 0x60, 0x73, 0xeb, 0xf6, 0x30, 0xc1, 0xf6, 0x7f, 0xa7, 0xd3, 0x03, 0xc0,
                       0x7d, 0xa4, 0x0b, 0x36, 0x74, 0x9d, 0x15, 0x7e, 0xa3, 0x79, 0x65, 0xfe, 0xf8,
                       0x10, 0xf2, 0xea, 0x05, 0xae, 0x6f, 0xc7, 0xd9, 0x6a, 0x8f, 0x34, 0x70, 0xd7,
                       0x3e, 0x15, 0xb2, 0x2b, 0x42, 0xe8, 0xd6, 0x98, 0x6d, 0xbf, 0xe5, 0x30, 0x32,
                       0x56, 0xb2, 0xb3, 0x56, 0x03, 0x72, 0xc4, 0x45, 0x2f, 0xfb, 0x2a, 0x04, 0xfb,
                       0x7c, 0x66, 0x91, 0x48, 0x9f, 0x70, 0xcb, 0x46, 0x83, 0x1b, 0xe0, 0x67, 0x91,
                       0x17, 0xf7],
            },
            {
                pass: [0x4a, 0x85, 0x7e, 0x2e, 0xe8, 0xaa, 0x9b, 0x60, 0x56, 0xf2, 0x42, 0x4e, 0x84,
                       0xd2, 0x4a, 0x72, 0x47, 0x33, 0x78, 0x90, 0x6e, 0xe0, 0x4a, 0x46, 0xcb, 0x05,
                       0x31, 0x15, 0x02, 0xd5, 0x25, 0x0b, 0x82, 0xad, 0x86, 0xb8, 0x3c, 0x8f, 0x20,
                       0xa2, 0x3d, 0xbb, 0x74, 0xf6, 0xda, 0x60, 0xb0, 0xb6, 0xec, 0xff, 0xd6, 0x71,
                       0x34, 0xd4, 0x59, 0x46, 0xac, 0x8e, 0xbf, 0xb3, 0x06, 0x42, 0x94, 0xbc, 0x09,
                       0x7d, 0x43, 0xce, 0xd6, 0x86, 0x42, 0xbf, 0xb8, 0xbb, 0xbd, 0xd0, 0xf5, 0x0b,
                       0x30, 0x11, 0x8f, 0x5e],
                salt: [0x39, 0xd8, 0x2e, 0xef, 0x32, 0x01, 0x0b, 0x8b, 0x79, 0xcc, 0x5b, 0xa8, 0x8e,
                       0xd5, 0x39, 0xfb],
                ops:  3,
                mem:  1432947,
                out:  [0xc1, 0x21, 0x20, 0x9f, 0x0b, 0xa7, 0x0a, 0xed, 0x93, 0xd4, 0x92, 0x00, 0xe5,
                       0xdc, 0x82, 0xcc, 0xe0, 0x13, 0xce, 0xf2, 0x5e, 0xa3, 0x1e, 0x16, 0x0b, 0xf8,
                       0xdb, 0x3c, 0xf4, 0x48, 0xa5, 0x9d, 0x1a, 0x56, 0xf6, 0xc1, 0x92, 0x59, 0xe1,
                       0x8e, 0xa0, 0x20, 0x55, 0x3c, 0xb7, 0x57, 0x81, 0x76, 0x1d, 0x11, 0x2b, 0x2d,
                       0x94, 0x9a, 0x29, 0x75, 0x84, 0xc6, 0x5e, 0x60, 0xdf, 0x95, 0xad, 0x89, 0xc4,
                       0x10, 0x98, 0x25, 0xa3, 0x17, 0x1d, 0xc6, 0xf2, 0x0b, 0x1f, 0xd6, 0xb0, 0xcd,
                       0xfd, 0x19, 0x48, 0x61, 0xbc, 0x2b, 0x41, 0x42, 0x95, 0xbe, 0xe5, 0xc6, 0xc5,
                       0x26, 0x19, 0xe5, 0x44, 0xab, 0xce, 0x7d, 0x52, 0x06, 0x59, 0xc3, 0xd5, 0x1d,
                       0xe2, 0xc6, 0x0e, 0x89, 0x94, 0x8d, 0x83, 0x06, 0x95, 0xab, 0x38, 0xdc, 0xb7,
                       0x5d, 0xd7, 0xab, 0x06, 0xa4, 0x77, 0x0d, 0xd4, 0xbc, 0x7c, 0x8f, 0x33, 0x55,
                       0x19, 0xe0, 0x4b, 0x03, 0x84, 0x16, 0xb1, 0xa7, 0xdb, 0xd2, 0x5c, 0x02, 0x67,
                       0x86, 0xa8, 0x10, 0x5c, 0x5f, 0xfe, 0x7a, 0x09, 0x31, 0x36, 0x4f, 0x03, 0x76,
                       0xae, 0x57, 0x72, 0xbe, 0x39, 0xb5, 0x1d, 0x91, 0xd3, 0x28, 0x14, 0x64, 0xe0,
                       0xf3, 0xa1, 0x28, 0xe7, 0x15, 0x5a, 0x68, 0xe8, 0x7c, 0xf7, 0x96, 0x26, 0xff,
                       0xca, 0x0b, 0x2a, 0x30, 0x22, 0xfc, 0x84, 0x20],
            },
            {
                pass: [0xc7, 0xb0, 0x9a, 0xec, 0x68, 0x0e, 0x7b, 0x42, 0xfe, 0xdd, 0x7f, 0xc7, 0x92,
                       0xe7, 0x8b, 0x2f, 0x6c, 0x1b, 0xea, 0x8f, 0x4a, 0x88, 0x43, 0x20, 0xb6, 0x48,
                       0xf8, 0x1e, 0x8c, 0xf5, 0x15, 0xe8, 0xba, 0x9d, 0xcf, 0xb1, 0x1d, 0x43, 0xc4,
                       0xaa, 0xe1, 0x14, 0xc1, 0x73, 0x4a, 0xa6, 0x9c, 0xa8, 0x2d, 0x44, 0x99, 0x83,
                       0x65, 0xdb, 0x9c, 0x93, 0x74, 0x4f, 0xa2, 0x8b, 0x63, 0xfd, 0x16, 0x00, 0x0e,
                       0x82, 0x61, 0xcb, 0xbe, 0x08, 0x3e, 0x7e, 0x2d, 0xa1, 0xe5, 0xf6, 0x96, 0xbd,
                       0xe0, 0x83, 0x4f, 0xe5, 0x31, 0x46, 0xd7, 0xe0, 0xe3, 0x5e, 0x7d, 0xe9, 0x92,
                       0x0d, 0x04, 0x1f, 0x5a, 0x56, 0x21, 0xaa, 0xbe, 0x02, 0xda, 0x3e, 0x2b, 0x09,
                       0xb4, 0x05, 0xb7, 0x79, 0x37, 0xef, 0xef, 0x31, 0x97, 0xbd, 0x57, 0x72, 0xe4,
                       0x1f, 0xdb, 0x73, 0xfb, 0x52, 0x94, 0x47, 0x8e, 0x45, 0x20, 0x80, 0x63, 0xb5,
                       0xf5, 0x8e, 0x08, 0x9d, 0xbe, 0xb6, 0xd6, 0x34, 0x2a, 0x90, 0x9c, 0x13, 0x07,
                       0xb3, 0xff, 0xf5, 0xfe, 0x2c, 0xf4, 0xda, 0x56, 0xbd, 0xae, 0x50, 0x84,
                       0x8f],
                salt: [0x03, 0x9c, 0x05, 0x6d, 0x93, 0x3b, 0x47, 0x50, 0x32, 0x77, 0x7e, 0xdb, 0xaf,
                       0xfa, 0xc5, 0x0f],
                ops:  3,
                mem:  4886999,
                out:  [0x91, 0xc3, 0x37, 0xce, 0x89, 0x18, 0xa5, 0x80, 0x5a, 0x59, 0xb0, 0x0b, 0xd1,
                       0x81, 0x9d, 0x3e, 0xb4, 0x35, 0x68, 0x07, 0xcb, 0xd2, 0xa8, 0x0b, 0x27, 0x1c,
                       0x4b, 0x48, 0x2d, 0xce, 0x03, 0xf5, 0xb0, 0x2a, 0xe4, 0xeb, 0x83, 0x1f, 0xf6,
                       0x68, 0xcb, 0xb3, 0x27, 0xb9, 0x3c, 0x30, 0x0b, 0x41, 0xda, 0x48, 0x52, 0xe5,
                       0x54, 0x7b, 0xea, 0x83, 0x42, 0xd5, 0x18, 0xdd, 0x93, 0x11, 0xaa, 0xeb, 0x5f,
                       0x90, 0xec, 0xcf, 0x66, 0xd5, 0x48, 0xf9, 0x27, 0x56, 0x31, 0xf0, 0xb1, 0xfd,
                       0x4b, 0x29, 0x9c, 0xec, 0x5d, 0x2e, 0x86, 0xa5, 0x9e, 0x55, 0xdc, 0x7b, 0x3a,
                       0xfa, 0xb6, 0x20, 0x44, 0x47, 0xb2, 0x1d, 0x1e, 0xf1, 0xda, 0x82, 0x4a, 0xba,
                       0xf3, 0x1a, 0x25, 0xa0, 0xd6, 0x13, 0x5c, 0x4f, 0xe8, 0x1d, 0x34, 0xa0, 0x68,
                       0x16, 0xc8, 0xa6, 0xea, 0xb1, 0x91, 0x41, 0xf5, 0x68, 0x71, 0x08, 0x50, 0x0f,
                       0x37, 0x19, 0xa8, 0x62, 0xaf, 0x8c, 0x5f, 0xee, 0x36, 0xe1, 0x30, 0xc6, 0x99,
                       0x21, 0xe1, 0x1c, 0xe8, 0x3d, 0xfc, 0x72, 0xc5, 0xec, 0x3b, 0x86, 0x2c, 0x1b,
                       0xcc, 0xc5, 0xfd, 0x63, 0xad, 0x57, 0xf4, 0x32, 0xfb, 0xcc, 0xa6, 0xf9, 0xe1,
                       0x8d, 0x5a, 0x59, 0x01, 0x59, 0x50, 0xcd, 0xf0, 0x53],
            },
            {
                pass: [0xa1, 0x49, 0x75, 0xc2, 0x6c, 0x08, 0x87, 0x55, 0xa8, 0xb7, 0x15, 0xff, 0x25,
                       0x28, 0xd6, 0x47, 0xcd, 0x34, 0x39, 0x87, 0xfc, 0xf4, 0xaa, 0x25, 0xe7, 0x19,
                       0x4a, 0x84, 0x17, 0xfb, 0x2b, 0x4b, 0x3f, 0x72, 0x68, 0xda, 0x9f, 0x31, 0x82,
                       0xb4, 0xcf, 0xb2, 0x2d, 0x13, 0x8b, 0x27, 0x49, 0xd6, 0x73, 0xa4, 0x7e, 0xcc,
                       0x75, 0x25, 0xdd, 0x15, 0xa0, 0xa3, 0xc6, 0x60, 0x46, 0x97, 0x17, 0x84, 0xbb,
                       0x63, 0xd7, 0xea, 0xe2, 0x4c, 0xc8, 0x4f, 0x26, 0x31, 0x71, 0x20, 0x75, 0xa1,
                       0x0e, 0x10, 0xa9, 0x6b, 0x0e, 0x0e, 0xe6, 0x7c, 0x43, 0xe0, 0x1c, 0x42, 0x3c,
                       0xb9, 0xc4, 0x4e, 0x53, 0x71, 0x01, 0x7e, 0x9c, 0x49, 0x69, 0x56, 0xb6, 0x32,
                       0x15, 0x8d, 0xa3, 0xfe, 0x12, 0xad, 0xde, 0xcb, 0x88, 0x91, 0x2e, 0x67, 0x59,
                       0xbc, 0x37, 0xf9, 0xaf, 0x2f, 0x45, 0xaf, 0x72, 0xc5, 0xca, 0xe3, 0xb1, 0x79,
                       0xff, 0xb6, 0x76, 0xa6, 0x97, 0xde, 0x6e, 0xbe, 0x45, 0xcd, 0x4c, 0x16, 0xd4,
                       0xa9, 0xd6, 0x42, 0xd2, 0x9d, 0xdc, 0x01, 0x86, 0xa0, 0xa4, 0x8c, 0xb6, 0xcd,
                       0x62, 0xbf, 0xc3, 0xdd, 0x22, 0x9d, 0x31, 0x3b, 0x30, 0x15, 0x60, 0x97, 0x1e,
                       0x74, 0x0e, 0x2c, 0xf1, 0xf9, 0x9a, 0x9a, 0x09, 0x0a, 0x5b, 0x28, 0x3f, 0x35,
                       0x47, 0x50, 0x57, 0xe9, 0x6d, 0x70, 0x64, 0xe2, 0xe0, 0xfc, 0x81, 0x98, 0x45,
                       0x91, 0x06, 0x8d, 0x55, 0xa3, 0xb4, 0x16, 0x9f, 0x22, 0xcc, 0xcb, 0x07, 0x45,
                       0xa2, 0x68, 0x94, 0x07, 0xea, 0x19, 0x01, 0xa0, 0xa7, 0x66, 0xeb, 0x99],
                salt: [0x3d, 0x96, 0x8b, 0x27, 0x52, 0xb8, 0x83, 0x84, 0x31, 0x16, 0x50, 0x59, 0x31,
                       0x9f, 0x3f, 0xf8],
                ops:  3,
                mem:  1784128,
                out:  [0xe9, 0x42, 0x95, 0x1d, 0xfb, 0xc2, 0xd5, 0x08, 0x29, 0x4b, 0x10, 0xf9, 0xe9,
                       0x7b, 0x47, 0xd0, 0xcd, 0x04, 0xe6, 0x68, 0xa0, 0x43, 0xcb, 0x95, 0x67, 0x9c,
                       0xc1, 0x13, 0x9d, 0xf7, 0xc2, 0x7c, 0xd5, 0x43, 0x67, 0x68, 0x87, 0x25, 0xbe,
                       0x9d, 0x06, 0x9f, 0x57, 0x04, 0xc1, 0x22, 0x23, 0xe7, 0xe4, 0xca, 0x18, 0x1f,
                       0xbd, 0x0b, 0xed, 0x18, 0xbb, 0x46, 0x34, 0x79, 0x5e, 0x54, 0x5a, 0x6c, 0x04,
                       0xa7, 0x30, 0x69, 0x33, 0xa4, 0x1a, 0x79, 0x4b, 0xae, 0xdb, 0xb6, 0x28, 0xd4,
                       0x1b, 0xc2, 0x85, 0xe0, 0xb9, 0x08, 0x40, 0x55, 0xae, 0x13, 0x6f, 0x6b, 0x63,
                       0x62, 0x4c, 0x87, 0x4f, 0x5a, 0x1e, 0x1d, 0x8b, 0xe7, 0xb0, 0xb7, 0x22, 0x7a,
                       0x17, 0x1d, 0x2d, 0x7e, 0xd5, 0x78, 0xd8, 0x8b, 0xfd, 0xcf, 0x18, 0x32, 0x31,
                       0x98, 0x96, 0x2d, 0x0d, 0xca, 0xd4, 0x12, 0x6f, 0xd3, 0xf2, 0x1a, 0xde, 0xb1,
                       0xe1, 0x1d, 0x66, 0x25, 0x2e, 0xa0, 0xc5, 0x8c, 0x91, 0x69, 0x6e, 0x91, 0x03,
                       0x1b, 0xfd, 0xcc, 0x2a, 0x9d, 0xc0, 0xe0, 0x28, 0xd1, 0x7b, 0x97, 0x05, 0xba,
                       0x2d, 0x7b, 0xcd, 0xcd, 0x1e, 0x3b, 0xa7, 0x5b, 0x4b, 0x1f, 0xea],
            },
        }

        verify_str_valid_tests! [
            {
                pass: "",
                hash: "$argon2i$v=19$m=4096,t=1,p=1$X1NhbHQAAAAAAAAAAAAAAA$bWh++MKN1OiFHKgIWTLvIi1i\
                       HicmHH7+Fv3K88ifFfI",
            },
            {
                pass: "",
                hash: "$argon2i$v=19$m=2048,t=4,p=1$SWkxaUhpY21ISDcrRnYzSw$Mbg/Eck1kpZir5T9io7C64cp\
                       ffdTBaORgyriLQFgQj8",
            },
            {
                pass: "^T5H$JYt39n%K*j:W]!1s?vg!:jGi]Ax?..l7[p0v:1jHTpla9;]bUN;?bWyCbtqg ",
                hash: "$argon2i$v=19$m=4096,t=3,p=2$X1NhbHQAAAAAAAAAAAAAAA$z/QMiU4lQxGsYNc/+K/bizws\
                       A1P11UG2dj/7+aILJ4I",
            },
            {
                pass: "K3S=KyH#)36_?]LxeR8QNKw6X=gFbxai$C%29V*",
                hash: "$argon2i$v=19$m=4096,t=3,p=1$X1NhbHQAAAAAAAAAAAAAAA$fu2Wsecyt+yPnBvSvYN16oP5\
                       ozRmkp0ixJ1YL19V3Uo",
            },
        ];

        verify_str_invalid_tests! [
            {
                pass: "",
                hash: "",
            },
            {
                pass: "",
                hash: "$argon2i$v=19$m=4096,t=1,p=1$X1NhbHQAAAAAAAAAAAAAAA$bWh++MKN1OiFHKgIWTLvIi1i\
                       HicmHH7+Fv3K88ifFfJ",
            },
            {
                pass: "",
                hash: "$argon2i$v=19$m=2048,t=4,p=1$TWkxaUhpY21ISDcrRnYzSw$Mbg/Eck1kpZir5T9io7C64cp\
                       ffdTBaORgyriLQFgQj8",
            },
            {
                pass: "J3S=KyH#)36_?]LxeR8QNKw6X=gFbxai$C%29V*",
                hash: "$argon2i$v=19$m=4096,t=3,p=1$X1NhbHQAAAAAAAAAAAAAAA$fu2Wsecyt+yPnBvSvYN16oP5\
                       ozRmkp0ixJ1YL19V3Uo",
            },
        ];

        needs_rehash_tests! {}
    }
}

pub use argon2id::*;
