//! The IETF ChaCha20-Poly1305 AEAD construction, as specified in [RFC
//! 8439](https://datatracker.ietf.org/doc/html/rfc8439).
//!
//! This module corresponds to the [`crypto_aead_chacha20poly1305_ietf`
//! API](https://doc.libsodium.org/secret-key_cryptography/aead/chacha20-poly1305/ietf_chacha20-poly1305_construction)
//! from Sodium.
//!
//! # Security Considerations
//! For this algorithm, nonces must *never* be used more than once with the same key. Furthermore,
//! the nonce length for the original ChaCha20 cipher is not sufficient that random nonces can be
//! used without the possibility of collisions, also leading to nonce reuse, so it is unsafe to use
//! a random nonce with this construction. Therefore, careful attention is needed to ensure nonces
//! are only used once. If a key is being reused for multiple messages, it is recommended to
//! increment the nonce for the previous message using [`increment_nonce`] for each message sent.
//! The initial nonce can be any value.
//!
//! In client-server protocols, where both parties are sending messages, use different keys for each
//! direction, or ensure one bit in the nonce is always set in one direction, and always unset in
//! the other, to make sure a nonce is never reused with the same key.
//!
//! For this construction, individual messages can only be up to `64 * (2^32 - 1)` bytes (approx.
//! 256 GiB) long, although a practically unlimited number of messages can be sent. An error will be
//! returned if a message is too long.
//!
//! For this construction, it is possible to efficiently compute multiple keys that would cause a
//! (ciphertext, tag) pair to be verified as authentic. This does not compromise the security of the
//! original message, but if an attacker can force the recipient to use a different key to the key
//! used for encryption, issues could arise. [Sodium's
//! documentation](https://doc.libsodium.org/secret-key_cryptography/aead#robustness) explains how
//! to deal with this issue.
//!
//! If many trusted parties have access to the secret key, there is no way to prove which one of
//! them sent a given message without additional information.
//!
//! This construction exposes the length of the plaintext. If this is undesirable, apply padding to
//! the plaintext prior to encryption via [`util::pad`](crate::util::pad), and remove it following
//! decryption via [`util::unpad`](crate::util::unpad).
//!
//! ## Secret Data
//! * The encryption/decryption key ([`Key`]) must only be known to parties who should be able to
//!   both encrypt and decrypt messages
//!
//! ## Non-Secret Data
//! * MACs ([`MAC`]) are not sensitive
//! * Nonces ([`Nonce`]) are not sensitive
//! * Additional data should not be secret

use super::chacha_aead_module;
use libsodium_sys as sodium;

chacha_aead_module! {
    sodium::crypto_aead_chacha20poly1305_ietf_KEYBYTES,
    sodium::crypto_aead_chacha20poly1305_ietf_ABYTES,
    sodium::crypto_aead_chacha20poly1305_ietf_NPUBBYTES,
    sodium::crypto_aead_chacha20poly1305_ietf_messagebytes_max,
    sodium::crypto_aead_chacha20poly1305_ietf_keygen,
    sodium::crypto_aead_chacha20poly1305_ietf_encrypt,
    sodium::crypto_aead_chacha20poly1305_ietf_decrypt,
    sodium::crypto_aead_chacha20poly1305_ietf_encrypt_detached,
    sodium::crypto_aead_chacha20poly1305_ietf_decrypt_detached,
}

#[cfg(test)]
mod tests {
    use crate::symmetric::aead::aead_tests;

    aead_tests! [
        {
            msg:    [0x4c, 0x61, 0x64, 0x69, 0x65, 0x73, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x47, 0x65,
                     0x6e, 0x74, 0x6c, 0x65, 0x6d, 0x65, 0x6e, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68,
                     0x65, 0x20, 0x63, 0x6c, 0x61, 0x73, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x27, 0x39,
                     0x39, 0x3a, 0x20, 0x49, 0x66, 0x20, 0x49, 0x20, 0x63, 0x6f, 0x75, 0x6c, 0x64,
                     0x20, 0x6f, 0x66, 0x66, 0x65, 0x72, 0x20, 0x79, 0x6f, 0x75, 0x20, 0x6f, 0x6e,
                     0x6c, 0x79, 0x20, 0x6f, 0x6e, 0x65, 0x20, 0x74, 0x69, 0x70, 0x20, 0x66, 0x6f,
                     0x72, 0x20, 0x74, 0x68, 0x65, 0x20, 0x66, 0x75, 0x74, 0x75, 0x72, 0x65, 0x2c,
                     0x20, 0x73, 0x75, 0x6e, 0x73, 0x63, 0x72, 0x65, 0x65, 0x6e, 0x20, 0x77, 0x6f,
                     0x75, 0x6c, 0x64, 0x20, 0x62, 0x65, 0x20, 0x69, 0x74, 0x2e],
            ad:     [0x50, 0x51, 0x52, 0x53, 0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7],
            key:    [0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8a, 0x8b, 0x8c,
                     0x8d, 0x8e, 0x8f, 0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99,
                     0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f],
            nonce:  [0x07, 0x00, 0x00, 0x00, 0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47],
            c:      [0xd3, 0x1a, 0x8d, 0x34, 0x64, 0x8e, 0x60, 0xdb, 0x7b, 0x86, 0xaf, 0xbc, 0x53,
                     0xef, 0x7e, 0xc2, 0xa4, 0xad, 0xed, 0x51, 0x29, 0x6e, 0x08, 0xfe, 0xa9, 0xe2,
                     0xb5, 0xa7, 0x36, 0xee, 0x62, 0xd6, 0x3d, 0xbe, 0xa4, 0x5e, 0x8c, 0xa9, 0x67,
                     0x12, 0x82, 0xfa, 0xfb, 0x69, 0xda, 0x92, 0x72, 0x8b, 0x1a, 0x71, 0xde, 0x0a,
                     0x9e, 0x06, 0x0b, 0x29, 0x05, 0xd6, 0xa5, 0xb6, 0x7e, 0xcd, 0x3b, 0x36, 0x92,
                     0xdd, 0xbd, 0x7f, 0x2d, 0x77, 0x8b, 0x8c, 0x98, 0x03, 0xae, 0xe3, 0x28, 0x09,
                     0x1b, 0x58, 0xfa, 0xb3, 0x24, 0xe4, 0xfa, 0xd6, 0x75, 0x94, 0x55, 0x85, 0x80,
                     0x8b, 0x48, 0x31, 0xd7, 0xbc, 0x3f, 0xf4, 0xde, 0xf0, 0x8e, 0x4b, 0x7a, 0x9d,
                     0xe5, 0x76, 0xd2, 0x65, 0x86, 0xce, 0xc6, 0x4b, 0x61, 0x16],
            mac:    [0x1a, 0xe1, 0x0b, 0x59, 0x4f, 0x09, 0xe2, 0x6a, 0x7e, 0x90, 0x2e, 0xcb, 0xd0,
                     0x60, 0x06, 0x91],
        },
        {
            msg:    [] as [u8; 0],
            ad:     [],
            key:    [0x80, 0xba, 0x31, 0x92, 0xc8, 0x03, 0xce, 0x96, 0x5e, 0xa3, 0x71, 0xd5, 0xff,
                     0x07, 0x3c, 0xf0, 0xf4, 0x3b, 0x6a, 0x2a, 0xb5, 0x76, 0xb2, 0x08, 0x42, 0x6e,
                     0x11, 0x40, 0x9c, 0x09, 0xb9, 0xb0],
            nonce:  [0x4d, 0xa5, 0xbf, 0x8d, 0xfd, 0x58, 0x52, 0xc1, 0xea, 0x12, 0x37, 0x9d],
            c:      [],
            mac:    [0x76, 0xac, 0xb3, 0x42, 0xcf, 0x31, 0x66, 0xa5, 0xb6, 0x3c, 0x0c, 0x0e, 0xa1,
                     0x38, 0x3c, 0x8d],
        },
        {
            msg:    [] as [u8; 0],
            ad:     [0xbd, 0x50, 0x67, 0x64, 0xf2, 0xd2, 0xc4, 0x10],
            key:    [0x7a, 0x4c, 0xd7, 0x59, 0x17, 0x2e, 0x02, 0xeb, 0x20, 0x4d, 0xb2, 0xc3, 0xf5,
                     0xc7, 0x46, 0x22, 0x7d, 0xf5, 0x84, 0xfc, 0x13, 0x45, 0x19, 0x63, 0x91, 0xdb,
                     0xb9, 0x57, 0x7a, 0x25, 0x07, 0x42],
            nonce:  [0xa9, 0x2e, 0xf0, 0xac, 0x99, 0x1d, 0xd5, 0x16, 0xa3, 0xc6, 0xf6, 0x89],
            c:      [],
            mac:    [0x90, 0x6f, 0xa6, 0x28, 0x4b, 0x52, 0xf8, 0x7b, 0x73, 0x59, 0xcb, 0xaa, 0x75,
                     0x63, 0xc7, 0x09],
        },
        {
            msg:    [0x2a],
            ad:     [],
            key:    [0xcc, 0x56, 0xb6, 0x80, 0x55, 0x2e, 0xb7, 0x50, 0x08, 0xf5, 0x48, 0x4b, 0x4c,
                     0xb8, 0x03, 0xfa, 0x50, 0x63, 0xeb, 0xd6, 0xea, 0xb9, 0x1f, 0x6a, 0xb6, 0xae,
                     0xf4, 0x91, 0x6a, 0x76, 0x62, 0x73],
            nonce:  [0x99, 0xe2, 0x3e, 0xc4, 0x89, 0x85, 0xbc, 0xcd, 0xee, 0xab, 0x60, 0xf1],
            c:      [0x3a],
            mac:    [0xca, 0xc2, 0x7d, 0xec, 0x09, 0x68, 0x80, 0x1e, 0x9f, 0x6e, 0xde, 0xd6, 0x9d,
                     0x80, 0x75, 0x22],
        },
        {
            msg:    [0x51],
            ad:     [0x91, 0xca, 0x6c, 0x59, 0x2c, 0xbc, 0xca, 0x53],
            key:    [0x46, 0xf0, 0x25, 0x49, 0x65, 0xf7, 0x69, 0xd5, 0x2b, 0xdb, 0x4a, 0x70, 0xb4,
                     0x43, 0x19, 0x9f, 0x8e, 0xf2, 0x07, 0x52, 0x0d, 0x12, 0x20, 0xc5, 0x5e, 0x4b,
                     0x70, 0xf0, 0xfd, 0xa6, 0x20, 0xee],
            nonce:  [0xab, 0x0d, 0xca, 0x71, 0x6e, 0xe0, 0x51, 0xd2, 0x78, 0x2f, 0x44, 0x03],
            c:      [0xc4],
            mac:    [0x16, 0x83, 0x10, 0xca, 0x45, 0xb1, 0xf7, 0xc6, 0x6c, 0xad, 0x4e, 0x99, 0xe4,
                     0x3f, 0x72, 0xb9],
        },
        {
            msg:    [0x5c, 0x60],
            ad:     [],
            key:    [0x2f, 0x7f, 0x7e, 0x4f, 0x59, 0x2b, 0xb3, 0x89, 0x19, 0x49, 0x89, 0x74, 0x35,
                     0x07, 0xbf, 0x3e, 0xe9, 0xcb, 0xde, 0x17, 0x86, 0xb6, 0x69, 0x5f, 0xe6, 0xc0,
                     0x25, 0xfd, 0x9b, 0xa4, 0xc1, 0x00],
            nonce:  [0x46, 0x1a, 0xf1, 0x22, 0xe9, 0xf2, 0xe0, 0x34, 0x7e, 0x03, 0xf2, 0xdb],
            c:      [0x4d, 0x13],
            mac:    [0x91, 0xe8, 0xb6, 0x1e, 0xfb, 0x39, 0xc1, 0x22, 0x19, 0x54, 0x53, 0x07, 0x7b,
                     0x22, 0xe5, 0xe2],
        },
        {
            msg:    [0xdd, 0xf2],
            ad:     [0x88, 0x36, 0x4f, 0xc8, 0x06, 0x05, 0x18, 0xbf],
            key:    [0xc8, 0x83, 0x3d, 0xce, 0x5e, 0xa9, 0xf2, 0x48, 0xaa, 0x20, 0x30, 0xea, 0xcf,
                     0xe7, 0x2b, 0xff, 0xe6, 0x9a, 0x62, 0x0c, 0xaf, 0x79, 0x33, 0x44, 0xe5, 0x71,
                     0x8f, 0xe0, 0xd7, 0xab, 0x1a, 0x58],
            nonce:  [0x61, 0x54, 0x6b, 0xa5, 0xf1, 0x72, 0x05, 0x90, 0xb6, 0x04, 0x0a, 0xc6],
            c:      [0xb6, 0x0d],
            mac:    [0xea, 0xd0, 0xfd, 0x46, 0x97, 0xec, 0x2e, 0x55, 0x58, 0x23, 0x77, 0x19, 0xd0,
                     0x24, 0x37, 0xa2],
        },
        {
            msg:    [0xab, 0x85, 0xe9, 0xc1, 0x57, 0x17, 0x31],
            ad:     [],
            key:    [0x55, 0x56, 0x81, 0x58, 0xd3, 0xa6, 0x48, 0x3f, 0x1f, 0x70, 0x21, 0xea, 0xb6,
                     0x9b, 0x70, 0x3f, 0x61, 0x42, 0x51, 0xca, 0xdc, 0x1a, 0xf5, 0xd3, 0x4a, 0x37,
                     0x4f, 0xdb, 0xfc, 0x5a, 0xda, 0xc7],
            nonce:  [0x3c, 0x4e, 0x65, 0x4d, 0x66, 0x3f, 0xa4, 0x59, 0x6d, 0xc5, 0x5b, 0xb7],
            c:      [0x5d, 0xfe, 0x34, 0x40, 0xdb, 0xb3, 0xc3],
            mac:    [0xed, 0x7a, 0x43, 0x4e, 0x26, 0x02, 0xd3, 0x94, 0x28, 0x1e, 0x0a, 0xfa, 0x9f,
                     0xb7, 0xaa, 0x42],
        },
        {
            msg:    [0x4e, 0xe5, 0xcd, 0xa2, 0x0d, 0x42, 0x90],
            ad:     [0x84, 0xe4, 0x6b, 0xe8, 0xc0, 0x91, 0x90, 0x53],
            key:    [0xe3, 0xc0, 0x9e, 0x7f, 0xab, 0x1a, 0xef, 0xb5, 0x16, 0xda, 0x6a, 0x33, 0x02,
                     0x2a, 0x1d, 0xd4, 0xeb, 0x27, 0x2c, 0x80, 0xd5, 0x40, 0xc5, 0xda, 0x52, 0xa7,
                     0x30, 0xf3, 0x4d, 0x84, 0x0d, 0x7f],
            nonce:  [0x58, 0x38, 0x93, 0x75, 0xc6, 0x9e, 0xe3, 0x98, 0xde, 0x94, 0x83, 0x96],
            c:      [0x4b, 0xd4, 0x72, 0x12, 0x94, 0x1c, 0xe3],
            mac:    [0x18, 0x5f, 0x14, 0x08, 0xee, 0x7f, 0xbf, 0x18, 0xf5, 0xab, 0xad, 0x6e, 0x22,
                     0x53, 0xa1, 0xba],
        },
        {
            msg:    [0xbe, 0x33, 0x08, 0xf7, 0x2a, 0x2c, 0x6a, 0xed],
            ad:     [],
            key:    [0x51, 0xe4, 0xbf, 0x2b, 0xad, 0x92, 0xb7, 0xaf, 0xf1, 0xa4, 0xbc, 0x05, 0x55,
                     0x0b, 0xa8, 0x1d, 0xf4, 0xb9, 0x6f, 0xab, 0xf4, 0x1c, 0x12, 0xc7, 0xb0, 0x0e,
                     0x60, 0xe4, 0x8d, 0xb7, 0xe1, 0x52],
            nonce:  [0x4f, 0x07, 0xaf, 0xed, 0xfd, 0xc3, 0xb6, 0xc2, 0x36, 0x18, 0x23, 0xd3],
            c:      [0x8e, 0x94, 0x39, 0xa5, 0x6e, 0xee, 0xc8, 0x17],
            mac:    [0xfb, 0xe8, 0xa6, 0xed, 0x8f, 0xab, 0xb1, 0x93, 0x75, 0x39, 0xdd, 0x6c, 0x00,
                     0xe9, 0x00, 0x21],
        },
        {
            msg:    [0xa4, 0xc9, 0xc2, 0x80, 0x1b, 0x71, 0xf7, 0xdf],
            ad:     [0x66, 0xc0, 0xae, 0x70, 0x07, 0x6c, 0xb1, 0x4d],
            key:    [0x11, 0x31, 0xc1, 0x41, 0x85, 0x77, 0xa0, 0x54, 0xde, 0x7a, 0x4a, 0xc5, 0x51,
                     0x95, 0x0f, 0x1a, 0x05, 0x3f, 0x9a, 0xe4, 0x6e, 0x5b, 0x75, 0xfe, 0x4a, 0xbd,
                     0x56, 0x08, 0xd7, 0xcd, 0xda, 0xdd],
            nonce:  [0xb4, 0xea, 0x66, 0x6e, 0xe1, 0x19, 0x56, 0x33, 0x66, 0x48, 0x4a, 0x78],
            c:      [0xb9, 0xb9, 0x10, 0x43, 0x3a, 0xf0, 0x52, 0xb0],
            mac:    [0x45, 0x30, 0xf5, 0x1a, 0xee, 0xe0, 0x24, 0xe0, 0xa4, 0x45, 0xa6, 0x32, 0x8f,
                     0xa6, 0x7a, 0x18],
        },
        {
            msg:    [0x42, 0xba, 0xae, 0x59, 0x78, 0xfe, 0xaf, 0x5c, 0x36, 0x8d, 0x14, 0xe0],
            ad:     [],
            key:    [0x99, 0xb6, 0x2b, 0xd5, 0xaf, 0xbe, 0x3f, 0xb0, 0x15, 0xbd, 0xe9, 0x3f, 0x0a,
                     0xbf, 0x48, 0x39, 0x57, 0xa1, 0xc3, 0xeb, 0x3c, 0xa5, 0x9c, 0xb5, 0x0b, 0x39,
                     0xf7, 0xf8, 0xa9, 0xcc, 0x51, 0xbe],
            nonce:  [0x9a, 0x59, 0xfc, 0xe2, 0x6d, 0xf0, 0x00, 0x5e, 0x07, 0x53, 0x86, 0x56],
            c:      [0xff, 0x7d, 0xc2, 0x03, 0xb2, 0x6c, 0x46, 0x7a, 0x6b, 0x50, 0xdb, 0x33],
            mac:    [0x57, 0x8c, 0x0f, 0x27, 0x58, 0xc2, 0xe1, 0x4e, 0x36, 0xd4, 0xfc, 0x10, 0x6d,
                     0xcb, 0x29, 0xb4],
        },
    ];
}
