//! The [scrypt](https://www.tarsnap.com/scrypt.html) password-based key derivation algorithm.
//!
//! Please note that the `scrypt` API is missing the `MEM_LIMIT_MODERATE` and `OPS_LIMIT_MODERATE`
//! constants included in [`super::argon2id`] and [`super::argon2i`].

use super::pbkdf_module;
use libsodium_sys as sodium;

pbkdf_module! {
    sodium::crypto_pwhash_scryptsalsa208sha256_OPSLIMIT_MIN,
    sodium::crypto_pwhash_scryptsalsa208sha256_OPSLIMIT_INTERACTIVE,
    sodium::crypto_pwhash_scryptsalsa208sha256_OPSLIMIT_SENSITIVE,
    sodium::crypto_pwhash_scryptsalsa208sha256_OPSLIMIT_MAX,
    sodium::crypto_pwhash_scryptsalsa208sha256_MEMLIMIT_MIN,
    sodium::crypto_pwhash_scryptsalsa208sha256_MEMLIMIT_INTERACTIVE,
    sodium::crypto_pwhash_scryptsalsa208sha256_MEMLIMIT_SENSITIVE,
    sodium::crypto_pwhash_scryptsalsa208sha256_memlimit_max,
    sodium::crypto_pwhash_scryptsalsa208sha256_PASSWD_MIN,
    sodium::crypto_pwhash_scryptsalsa208sha256_passwd_max,
    sodium::crypto_pwhash_scryptsalsa208sha256_BYTES_MIN,
    sodium::crypto_pwhash_scryptsalsa208sha256_bytes_max,
    sodium::crypto_pwhash_scryptsalsa208sha256_SALTBYTES,
    sodium::crypto_pwhash_scryptsalsa208sha256_STRBYTES,
    sodium::crypto_pwhash_scryptsalsa208sha256,
    sodium::crypto_pwhash_scryptsalsa208sha256_str,
    sodium::crypto_pwhash_scryptsalsa208sha256_str_verify,
    sodium::crypto_pwhash_scryptsalsa208sha256_str_needs_rehash,
}

#[cfg(test)]
mod tests {
    use super::super::{kdf_tests, verify_password_invalid_tests, verify_password_valid_tests};
    use crate::AlkaliError;

    kdf_tests! {
        {
            pass: [0xa3, 0x47, 0xae, 0x92, 0xbc, 0xe9, 0xf8, 0x0f, 0x6f, 0x59, 0x5a, 0x44, 0x80,
                   0xfc, 0x9c, 0x2f, 0xe7, 0xe7, 0xd7, 0x14, 0x8d, 0x37, 0x1e, 0x94, 0x87, 0xd7,
                   0x5f, 0x5c, 0x23, 0x00, 0x8f, 0xfa, 0xe0, 0x65, 0x57, 0x7a, 0x92, 0x8f, 0xeb,
                   0xd9, 0xb1, 0x97, 0x3a, 0x5a, 0x95, 0x07, 0x3a, 0xcd, 0xbe, 0xb6, 0xa0, 0x30,
                   0xcf, 0xc0, 0xd7, 0x9c, 0xaa, 0x2d, 0xc5, 0xcd, 0x01, 0x1c, 0xef, 0x02, 0xc0,
                   0x8d, 0xa2, 0x32, 0xd7, 0x6d, 0x52, 0xdf, 0xbc, 0xa3, 0x8c, 0xa8, 0xdc, 0xbd,
                   0x66, 0x5b, 0x17, 0xd1, 0x66, 0x5f, 0x7c, 0xf5, 0xfe, 0x59, 0x77, 0x2e, 0xc9,
                   0x09, 0x73, 0x3b, 0x24, 0xde, 0x97, 0xd6, 0xf5, 0x8d, 0x22, 0x0b, 0x20, 0xc6,
                   0x0d, 0x7c, 0x07, 0xec, 0x1f, 0xd9, 0x3c, 0x52, 0xc3, 0x10, 0x20, 0x30, 0x0c,
                   0x6c, 0x1f, 0xac, 0xd7, 0x79, 0x37, 0xa5, 0x97, 0xc7, 0xa6],
            salt: [0x55, 0x41, 0xfb, 0xc9, 0x95, 0xd5, 0xc1, 0x97, 0xba, 0x29, 0x03, 0x46, 0xd2,
                   0xc5, 0x59, 0xde, 0xdf, 0x40, 0x5c, 0xf9, 0x7e, 0x5f, 0x95, 0x48, 0x21, 0x43,
                   0x20, 0x2f, 0x9e, 0x74, 0xf5, 0xc2],
            ops:  32768,
            mem:  16777216,
            out:  [0xd5, 0x49, 0x16, 0x74, 0x80, 0x76, 0xb9, 0xd9, 0xf7, 0x21, 0x98, 0xc8, 0xfb,
                   0xef, 0x56, 0x34, 0x62, 0xdc, 0x8c, 0x70, 0x6e, 0x1a, 0xd3, 0x8a, 0xbd, 0x1f,
                   0xac, 0x57, 0x00, 0x16, 0x72, 0x1a, 0xcd, 0x0a, 0x76, 0x59, 0xab, 0x49, 0xa4,
                   0x72, 0x99, 0xa9, 0x96, 0xb4, 0x35, 0x97, 0x69, 0x0c, 0x0c, 0x94, 0x71, 0x43,
                   0x06, 0x9f, 0x35, 0xd8, 0x3e, 0x60, 0x62, 0x73, 0xdb, 0xf2, 0xd6, 0x22, 0x32,
                   0x13, 0x93, 0x94, 0x9b, 0x8e, 0xd5, 0xa6, 0x83, 0x15, 0x36, 0x2c, 0x4f, 0x84,
                   0x80, 0x43, 0x84, 0xd0, 0x5e, 0x0e, 0x0e, 0x86, 0xbc, 0x00, 0xe3, 0x64, 0x12,
                   0x33, 0xf9, 0xf9, 0x75, 0xab, 0x46, 0xb6, 0x0b, 0xa1, 0x85, 0xc5, 0xe5, 0xfe,
                   0x47, 0xf7, 0x8e, 0xfd, 0x20, 0x7e, 0x69, 0xfd, 0x8f, 0x63, 0x90, 0x73, 0x08,
                   0x28, 0xb9, 0x3b, 0x9b, 0x37, 0x63, 0xea, 0x12, 0x83, 0xca, 0xa0, 0x3b, 0xc3,
                   0x67, 0x26, 0x76, 0x37, 0x15, 0xde, 0x81, 0x19, 0x15, 0x68, 0x1d, 0xd2, 0x14,
                   0x52, 0x4f, 0x5a, 0xd4, 0xdd, 0x38, 0x66, 0x08, 0xca, 0xc6, 0xc7, 0xf2],
        },
        {
            pass: [0xe1, 0x25, 0xce, 0xe6, 0x1c, 0x8c, 0xb7, 0x77, 0x8d, 0x9e, 0x5a, 0xd0, 0xa6,
                   0xf5, 0xd9, 0x78, 0xce, 0x9f, 0x84, 0xde, 0x21, 0x3a, 0x85, 0x56, 0xd9, 0xff,
                   0xe2, 0x02, 0x02, 0x0a, 0xb4, 0xa6, 0xed, 0x90, 0x74, 0xa4, 0xeb, 0x34, 0x16,
                   0xf9, 0xb1, 0x68, 0xf1, 0x37, 0x51, 0x0f, 0x3a, 0x30, 0xb7, 0x0b, 0x96, 0xcb,
                   0xfa, 0x21, 0x9f, 0xf9, 0x9f, 0x6c, 0x6e, 0xaf, 0xfb, 0x15, 0xc0, 0x6b, 0x60,
                   0xe0, 0x0c, 0xc2, 0x89, 0x02, 0x77, 0xf0, 0xfd, 0x3c, 0x62, 0x21, 0x15, 0x77,
                   0x2f, 0x70, 0x48, 0xad, 0xae, 0xbe, 0xd8, 0x6e],
            salt: [0xf1, 0x19, 0x2d, 0xd5, 0xdc, 0x23, 0x68, 0xb9, 0xcd, 0x42, 0x13, 0x38, 0xb2,
                   0x24, 0x33, 0x45, 0x5e, 0xe0, 0xa3, 0x69, 0x9f, 0x93, 0x79, 0xa0, 0x8b, 0x96,
                   0x50, 0xea, 0x2c, 0x12, 0x6f, 0x0d],
            ops:  535778,
            mem:  16777216,
            out:  [0xd6, 0x62, 0xfb, 0xc6, 0x25, 0xad, 0xef, 0x2c, 0xd2, 0x7a, 0xfa, 0x89, 0x9e,
                   0xcc, 0x9a, 0x4d, 0xd9, 0x07, 0x36, 0x2d, 0x77, 0x16, 0xe9, 0xa1, 0x7e, 0x72,
                   0xd3, 0x74, 0xdb, 0xd8, 0x84, 0x53, 0x10, 0xe3, 0xdc, 0x56, 0x97, 0xc9, 0x66,
                   0x7e, 0xc7, 0x81, 0x1e, 0x2b, 0xdc, 0x15, 0xbf, 0xc0, 0x0c, 0xae, 0x05, 0xc4,
                   0x54, 0x3d, 0x85, 0x9f, 0xfe, 0x8c, 0xe4, 0x94, 0xd1, 0xd2, 0x28, 0xb0, 0xbb,
                   0xa3, 0xe3, 0xa3, 0xc5, 0x0b, 0x26, 0x96, 0x2b, 0x15, 0x7f, 0x05, 0x6d, 0x44,
                   0xd5, 0xbb, 0xde, 0x6d, 0xf2, 0x4b, 0xd8, 0x96, 0xf0, 0x72, 0x6f, 0x41, 0xa1,
                   0xc3, 0xe2, 0xc2, 0xdd, 0x66, 0x3c, 0x4b, 0x45, 0x5b, 0x48, 0x78, 0x21, 0x90,
                   0xd2, 0x2b, 0xf3, 0xf7, 0xae, 0x37, 0x29, 0x92, 0xea, 0xd6, 0xd6, 0x9d, 0x0b,
                   0x1d, 0xd2, 0xf9, 0x4e, 0x64, 0xe5, 0x6d, 0xfc, 0x30, 0x51, 0xb0, 0x75, 0xcc,
                   0x95, 0xe4, 0x7c, 0xcd, 0xbb, 0xa3, 0x61, 0xb5, 0xfc, 0x53, 0x09, 0xd3, 0x43,
                   0xa1, 0xdb, 0xa7, 0xb1, 0xec, 0xb3, 0x3b, 0x30, 0x20, 0x15, 0x96, 0xee, 0x5f,
                   0xf4, 0x21, 0xf1, 0x2b, 0xb5, 0xc7, 0x72, 0x86, 0x44, 0x37, 0x46, 0xd7, 0xbd,
                   0xb1, 0x06, 0x00, 0x8c, 0x07, 0x3c, 0x05, 0x47, 0xc1, 0xf0, 0x66, 0xa6, 0x70,
                   0x02, 0xef, 0xd0, 0xfe, 0xcf, 0x66, 0xe7, 0xa6, 0x4c, 0x94, 0x51, 0x6d, 0x6e,
                   0xec, 0xca, 0x00, 0xb4, 0xed, 0x85, 0x7e, 0x17, 0xcd, 0x1d, 0xaf, 0x6b, 0x35,
                   0x70, 0x74, 0x02, 0x2a, 0x03, 0xb8, 0x6f, 0x25, 0xa4, 0x65, 0x5c, 0xb4, 0xa3,
                   0x74, 0x13, 0x31, 0x32, 0x9c, 0xa9, 0xfd, 0x73, 0x32, 0x14, 0x1a, 0xe0, 0x26,
                   0x77, 0xd3, 0x0b, 0x5c, 0x26, 0x18, 0xfc, 0xdc, 0x01, 0x65, 0x53, 0x3c, 0xa2,
                   0x2a, 0xdb, 0xdb],
        },
        {
            pass: [0x92, 0x26, 0x3c, 0xbf, 0x6a, 0xc3, 0x76, 0x49, 0x9f, 0x68, 0xa4, 0x28, 0x9d,
                   0x3b, 0xb5, 0x9e, 0x5a, 0x22, 0x33, 0x5e, 0xba, 0x63, 0xa3, 0x2e, 0x64, 0x10,
                   0x24, 0x91, 0x55, 0xb9, 0x56, 0xb6, 0xa3, 0xb4, 0x8d, 0x4a, 0x44, 0x90, 0x6b,
                   0x18, 0xb8, 0x97, 0x12, 0x73, 0x00, 0xb3, 0x75, 0xb8, 0xf8, 0x34, 0xf1, 0xce,
                   0xff, 0xc7, 0x08, 0x80, 0xa8, 0x85, 0xf4, 0x7c, 0x33, 0x87, 0x67, 0x17, 0xe3,
                   0x92, 0xbe, 0x57, 0xf7, 0xda, 0x3a, 0xe5, 0x8d, 0xa4, 0xfd, 0x1f, 0x43, 0xda,
                   0xa7, 0xe4, 0x4b, 0xb8, 0x2d, 0x37, 0x17, 0xaf, 0x43, 0x19, 0x34, 0x9c, 0x24,
                   0xcd, 0x31, 0xe4, 0x6d, 0x29, 0x58, 0x56, 0xb0, 0x44, 0x1b, 0x6b, 0x28, 0x99,
                   0x92, 0xa1, 0x1c, 0xed, 0x1c, 0xc3, 0xbf, 0x30, 0x11, 0x60, 0x45, 0x90, 0x24,
                   0x4a, 0x3e, 0xb7, 0x37, 0xff, 0x22, 0x11, 0x29, 0x21, 0x5e, 0x4e, 0x43, 0x47,
                   0xf4, 0x91, 0x5d, 0x41, 0x29, 0x2b, 0x51, 0x73, 0xd1, 0x96, 0xeb, 0x9a, 0xdd,
                   0x69, 0x3b, 0xe5, 0x31, 0x9f, 0xda, 0xdc, 0x24, 0x29, 0x06, 0x17, 0x8b, 0xb6,
                   0xc0, 0x28, 0x6c, 0x9b, 0x6c, 0xa6, 0x01, 0x27, 0x46, 0x71, 0x1f, 0x58, 0xc8,
                   0xc3, 0x92, 0x01, 0x6b, 0x2f, 0xdf, 0xc0, 0x9c, 0x64, 0xf0, 0xf6, 0xb6, 0xab,
                   0x7b],
            salt: [0x3b, 0x84, 0x0e, 0x20, 0xe9, 0x55, 0x5e, 0x9f, 0xb0, 0x31, 0xc4, 0xba, 0x1f,
                   0x17, 0x47, 0xce, 0x25, 0xcc, 0x1d, 0x0f, 0xf6, 0x64, 0xbe, 0x67, 0x6b, 0x9b,
                   0x4a, 0x90, 0x64, 0x1f, 0xf1, 0x94],
            ops:  311757,
            mem:  16777216,
            out:  [0x50, 0x38, 0xc0, 0x48, 0x35, 0xce, 0x37, 0x0d, 0x8c, 0x87, 0x16, 0x58, 0xcc,
                   0xad, 0x62, 0xb7, 0x0a, 0xef, 0xe6, 0xdd, 0xcb, 0x65, 0x36, 0x59, 0xc4, 0x27,
                   0x09, 0x76, 0xf0, 0x95, 0xb1, 0xf4, 0x5a, 0xbc, 0x19, 0x1a, 0xc4, 0x61, 0xcd,
                   0x2e, 0xe5, 0xfc, 0x62, 0xb3, 0x35, 0x12, 0x11, 0x7f, 0x40, 0x54, 0x69, 0x84,
                   0x61, 0x9c, 0x04, 0x59, 0xd4, 0xdc, 0xf5, 0x7d, 0x1c, 0xe8, 0xf4, 0xbb, 0xb7,
                   0x89, 0xdf, 0x3e, 0x68, 0xf0, 0x35, 0xf9, 0xed, 0x78, 0x8a, 0x80, 0x8e, 0xf9,
                   0x3a, 0xc0, 0xeb, 0x5f, 0x9b, 0xf3, 0x5c, 0xdb, 0x1c, 0x7f, 0xc1, 0x30, 0x97,
                   0x09, 0xbd, 0xbb, 0x12, 0x07, 0xe4, 0x48, 0xd5, 0x67, 0xd3, 0xef, 0x7b, 0x6b,
                   0x2a, 0x4b, 0x0d, 0xc8, 0xb5, 0xde, 0xf7, 0xc4, 0x7d, 0xea, 0xfe, 0xbf, 0x6e,
                   0x98, 0xe6, 0xae, 0x1c, 0xae, 0xc7, 0x9a, 0x09, 0x26, 0x42, 0xe3, 0x68, 0x13,
                   0x4d, 0x33, 0x2e, 0xd6, 0xed, 0xdc, 0x3a, 0xf0, 0xc9, 0x43, 0x20, 0x9b, 0x06,
                   0xfb, 0x36, 0xc5, 0xd7, 0xc3, 0x82, 0x17, 0x04, 0x8a, 0x8b, 0x1b, 0x1c, 0x96,
                   0xef, 0xb6, 0xce, 0x0a, 0xd1, 0x5d, 0x1b, 0x93, 0x99, 0x67, 0x05, 0x31, 0x4f,
                   0xf4, 0x34, 0xf2, 0xe8, 0x8f, 0xbc, 0x0a, 0x53, 0x94, 0xbc, 0x67, 0x92, 0x7d,
                   0x38, 0x23, 0x61, 0xba, 0x40, 0xd4, 0x3e, 0x99, 0x37, 0x0b, 0xfa, 0xa4, 0xae,
                   0x20, 0xff, 0x4e, 0x41, 0xd9, 0xba, 0x56, 0xea, 0x55, 0x86, 0x70, 0x6a, 0x63,
                   0x7f, 0x5d, 0x1e, 0x65, 0xd5, 0x2d, 0xd6, 0x72, 0x1b, 0xd1, 0xe7, 0x18, 0x0f,
                   0xee, 0x51, 0x2f, 0x80, 0x48, 0x0e, 0x48, 0x26, 0x16, 0x32, 0x60, 0xac, 0x35,
                   0xf6, 0xf1, 0xca, 0x6c, 0x6f, 0xd7, 0x29, 0xc5, 0x40, 0x9d, 0x42, 0x0d, 0xc3,
                   0x71, 0x9d],
        },
        {
            pass: [0x4a, 0x85, 0x7e, 0x2e, 0xe8, 0xaa, 0x9b, 0x60, 0x56, 0xf2, 0x42, 0x4e, 0x84,
                   0xd2, 0x4a, 0x72, 0x47, 0x33, 0x78, 0x90, 0x6e, 0xe0, 0x4a, 0x46, 0xcb, 0x05,
                   0x31, 0x15, 0x02, 0xd5, 0x25, 0x0b, 0x82, 0xad, 0x86, 0xb8, 0x3c, 0x8f, 0x20,
                   0xa2, 0x3d, 0xbb, 0x74, 0xf6, 0xda, 0x60, 0xb0, 0xb6, 0xec, 0xff, 0xd6, 0x71,
                   0x34, 0xd4, 0x59, 0x46, 0xac, 0x8e, 0xbf, 0xb3, 0x06, 0x42, 0x94, 0xbc, 0x09,
                   0x7d, 0x43, 0xce, 0xd6, 0x86, 0x42, 0xbf, 0xb8, 0xbb, 0xbd, 0xd0, 0xf5, 0x0b,
                   0x30, 0x11, 0x8f, 0x5e],
            salt: [0x39, 0xd8, 0x2e, 0xef, 0x32, 0x01, 0x0b, 0x8b, 0x79, 0xcc, 0x5b, 0xa8, 0x8e,
                   0xd5, 0x39, 0xfb, 0xab, 0xa7, 0x41, 0x10, 0x0f, 0x2e, 0xdb, 0xec, 0xa7, 0xcc,
                   0x17, 0x1f, 0xfe, 0xab, 0xf2, 0x58],
            ops:  758010,
            mem:  16777216,
            out:  [0xa7, 0x57, 0xcc, 0xa7, 0xdd, 0x90, 0xf4, 0x86, 0xe0, 0x38, 0x28, 0xa9, 0x78,
                   0xe5, 0xb7, 0xfd, 0x3e, 0x85, 0x71, 0x70, 0x2a, 0xf9, 0xa0, 0x9d, 0xef, 0x4d,
                   0xda, 0xa7, 0x1a, 0x3b, 0x01, 0x27, 0x0c, 0x95, 0x90, 0x6f, 0x40, 0x87, 0xf9,
                   0xef, 0x0e, 0x2c, 0x6a, 0xa5, 0x39, 0xa4, 0xd5, 0x3e, 0xc6, 0xc6, 0xd5, 0x26,
                   0x4f, 0x63, 0x13, 0xab, 0xf5, 0x32, 0xc8, 0xd7, 0xfd, 0x70, 0xd4, 0x5c, 0x2a,
                   0x30, 0x14, 0x63, 0x31, 0x2b, 0xd5, 0x8c, 0xfa, 0x76, 0x6e, 0x93, 0xd2, 0xc5,
                   0xb9, 0xa9, 0xb6, 0xa7, 0xd4, 0x80, 0x67, 0x76, 0xcd, 0xd0, 0x9c, 0xbb, 0xa4,
                   0x4f, 0xe8, 0x0f, 0xe4, 0xbe, 0x77, 0x52, 0x0b, 0xfd, 0xae, 0x13, 0xb5, 0x52,
                   0xc2, 0x95, 0x0e, 0xb2, 0xf8, 0x6c, 0xb4, 0xc7, 0xab, 0x9d, 0xfc, 0x0d, 0x6c,
                   0x09, 0xa1, 0xd7, 0xdc, 0xb3, 0x5c, 0x5c, 0x61, 0x1e, 0x1f, 0x4b, 0x9f, 0x67,
                   0x00, 0xc6, 0xcc, 0x89, 0x67, 0x88, 0x6d, 0x05, 0xe7, 0x4f, 0x56, 0x44, 0x4a,
                   0x9d, 0x63, 0x6c, 0x5d, 0x03, 0xd1, 0x33, 0x20, 0x57, 0x2f, 0x58, 0x0c, 0x97,
                   0xc3, 0x9b, 0x11, 0x16, 0x17, 0x6e, 0xe5, 0xde, 0xd8, 0x0a, 0xe4, 0x8a, 0xb3,
                   0xbf, 0x0a, 0x51, 0xd2, 0x43, 0x8c, 0x98, 0xc1, 0x49, 0xe2, 0x96, 0xea, 0xcb,
                   0x02, 0x1f, 0xc0, 0x1d, 0xdd, 0xb1, 0x2e, 0x45],
        },
        {
            pass: [0x18, 0x45, 0xe3, 0x75, 0x47, 0x95, 0x37, 0xe9, 0xdd, 0x4f, 0x44, 0x86, 0xd5,
                   0xc9, 0x1a, 0xc7, 0x27, 0x75, 0xd6, 0x66, 0x05, 0xee, 0xb1, 0x1a, 0x78, 0x7b,
                   0x78, 0xa7, 0x74, 0x5f, 0x1f, 0xd0, 0x05, 0x2d, 0x52, 0x6c, 0x67, 0x23, 0x5d,
                   0xba, 0xe1, 0xb2, 0xa4, 0xd5, 0x75, 0xa7, 0x4c, 0xb5, 0x51, 0xc8, 0xe9, 0x09,
                   0x6c, 0x59, 0x3a, 0x49, 0x7a, 0xee, 0x74, 0xba, 0x30, 0x47, 0xd9, 0x11, 0x35,
                   0x8e, 0xde, 0x57, 0xbc, 0x27, 0xc9, 0xea, 0x18, 0x29, 0x82, 0x43, 0x48, 0xda,
                   0xaa, 0xb6, 0x06, 0x21, 0x7c, 0xc9, 0x31, 0xdc, 0xb6, 0x62, 0x77, 0x87, 0xbd,
                   0x6e, 0x4e, 0x58, 0x54, 0xf0, 0xe8],
            salt: [0x3e, 0xe9, 0x1a, 0x80, 0x5a, 0xa6, 0x2c, 0xfb, 0xe8, 0xdc, 0xe2, 0x9a, 0x2d,
                   0x9a, 0x44, 0x37, 0x3a, 0x50, 0x06, 0xf4, 0xa4, 0xce, 0x24, 0x02, 0x2a, 0xca,
                   0x9c, 0xec, 0xb2, 0x9d, 0x14, 0x73],
            ops:  233177,
            mem:  16777216,
            out:  [0x82, 0x76, 0x5c, 0x04, 0x0c, 0x58, 0xc1, 0x81, 0x0f, 0x8c, 0x05, 0x3e, 0xf5,
                   0xc2, 0x48, 0x55, 0x62, 0x99, 0x38, 0x54, 0x76, 0xbd, 0xe4, 0x4b, 0xdd, 0x91,
                   0xa0, 0xd9, 0xa2, 0x39, 0xf2, 0x4e, 0x9b, 0x17, 0x17, 0xfd, 0x8b, 0x23, 0x20,
                   0x9f, 0xfa, 0x45, 0xb7, 0xaa, 0x79, 0x37, 0x29, 0x6c, 0x60, 0x1b, 0x79, 0xe7,
                   0x7d, 0xa9, 0x9e, 0x8d, 0x2f, 0xda, 0x0e, 0xa4, 0x45, 0x9b, 0xe2, 0xd0, 0x90,
                   0x0f, 0x5b, 0xc5, 0xa2, 0x69, 0xb5, 0x48, 0x8d, 0x87, 0x3d, 0x46, 0x32, 0xd1,
                   0xba, 0xf7, 0x59, 0x65, 0xe5, 0x09, 0xee, 0x24, 0xb1, 0x25, 0x01, 0xa9, 0xce,
                   0x3b, 0xbb, 0xd8, 0xb7, 0xd7, 0x59, 0x98, 0x7d, 0x54, 0x5a, 0x1c, 0x22, 0x1a,
                   0x36, 0x31, 0x95, 0xe5, 0x80, 0x2d, 0x76, 0x8b, 0x3b, 0x9e, 0x00, 0xeb, 0xe5,
                   0xac, 0x0e, 0xd8, 0xad, 0x23, 0x62, 0xc1, 0xc4, 0x15, 0x7b, 0x91, 0x0a, 0x40,
                   0xf9, 0x4a, 0xdf, 0x25, 0x61, 0xa2, 0xb0, 0xd3, 0xe6, 0x5d, 0xbb, 0x06, 0xf2,
                   0x44, 0xe5, 0xac, 0x44, 0xd3, 0x62, 0x10, 0x3d, 0xf5, 0x4c, 0x9b, 0x91, 0x75,
                   0x77, 0x7b, 0x3d, 0xb1, 0xcd, 0xad, 0xb0, 0x3e, 0x97, 0x7a, 0xb8, 0xa7, 0x9b,
                   0xaf, 0x1e, 0x1e, 0x18, 0xec, 0x9f, 0x5d, 0x0f, 0x25, 0xc4, 0x87, 0xdd, 0xc5,
                   0x3d, 0x7e, 0x81, 0x91, 0x0f, 0x83, 0x57, 0x6b, 0x44, 0xe9, 0xca, 0xee, 0xce,
                   0x26, 0xe2, 0xeb, 0x37, 0x65, 0x69, 0xad, 0x3a, 0x8c, 0xdc, 0xcb, 0xde, 0x8b,
                   0xc3, 0x55, 0x21, 0x0e],
        },
        {
            pass: [0xc7, 0xb0, 0x9a, 0xec, 0x68, 0x0e, 0x7b, 0x42, 0xfe, 0xdd, 0x7f, 0xc7, 0x92,
                   0xe7, 0x8b, 0x2f, 0x6c, 0x1b, 0xea, 0x8f, 0x4a, 0x88, 0x43, 0x20, 0xb6, 0x48,
                   0xf8, 0x1e, 0x8c, 0xf5, 0x15, 0xe8, 0xba, 0x9d, 0xcf, 0xb1, 0x1d, 0x43, 0xc4,
                   0xaa, 0xe1, 0x14, 0xc1, 0x73, 0x4a, 0xa6, 0x9c, 0xa8, 0x2d, 0x44, 0x99, 0x83,
                   0x65, 0xdb, 0x9c, 0x93, 0x74, 0x4f, 0xa2, 0x8b, 0x63, 0xfd, 0x16, 0x00, 0x0e,
                   0x82, 0x61, 0xcb, 0xbe, 0x08, 0x3e, 0x7e, 0x2d, 0xa1, 0xe5, 0xf6, 0x96, 0xbd,
                   0xe0, 0x83, 0x4f, 0xe5, 0x31, 0x46, 0xd7, 0xe0, 0xe3, 0x5e, 0x7d, 0xe9, 0x92,
                   0x0d, 0x04, 0x1f, 0x5a, 0x56, 0x21, 0xaa, 0xbe, 0x02, 0xda, 0x3e, 0x2b, 0x09,
                   0xb4, 0x05, 0xb7, 0x79, 0x37, 0xef, 0xef, 0x31, 0x97, 0xbd, 0x57, 0x72, 0xe4,
                   0x1f, 0xdb, 0x73, 0xfb, 0x52, 0x94, 0x47, 0x8e, 0x45, 0x20, 0x80, 0x63, 0xb5,
                   0xf5, 0x8e, 0x08, 0x9d, 0xbe, 0xb6, 0xd6, 0x34, 0x2a, 0x90, 0x9c, 0x13, 0x07,
                   0xb3, 0xff, 0xf5, 0xfe, 0x2c, 0xf4, 0xda, 0x56, 0xbd, 0xae, 0x50, 0x84, 0x8f],
            salt: [0x03, 0x9c, 0x05, 0x6d, 0x93, 0x3b, 0x47, 0x50, 0x32, 0x77, 0x7e, 0xdb, 0xaf,
                   0xfa, 0xc5, 0x0f, 0x14, 0x3f, 0x64, 0xc1, 0x23, 0x32, 0x9e, 0xd9, 0xcf, 0x59,
                   0xe3, 0xb6, 0x5d, 0x3f, 0x43, 0xb6],
            ops:  234753,
            mem:  16777216,
            out:  [0xca, 0x92, 0x16, 0xd4, 0x12, 0x7e, 0x2e, 0x4a, 0x6e, 0xe3, 0x58, 0x4b, 0x49,
                   0xbe, 0x10, 0x62, 0x17, 0xbb, 0x61, 0xcc, 0x80, 0x70, 0x16, 0xd4, 0x6d, 0x0c,
                   0xfb, 0xb1, 0xfd, 0x72, 0x2e, 0x2b, 0xba, 0xc3, 0x35, 0x41, 0x38, 0x6b, 0xdf,
                   0xea, 0xc4, 0x1a, 0x29, 0x9e, 0xad, 0x22, 0x79, 0x09, 0x93, 0xfc, 0xaa, 0x8e,
                   0x1d, 0x23, 0xbd, 0x1c, 0x84, 0x26, 0xaf, 0xa5, 0xff, 0x4c, 0x08, 0xe7, 0x31,
                   0xdc, 0x47, 0x6e, 0xf8, 0x34, 0xf1, 0x42, 0xc3, 0x2d, 0xfb, 0x2c, 0x1b, 0xe1,
                   0x2b, 0x99, 0x78, 0x80, 0x2e, 0x63, 0xb2, 0xcd, 0x6f, 0x22, 0x6b, 0x1a, 0x8d,
                   0xf5, 0x9f, 0x0c, 0x79, 0x15, 0x4d, 0x7e, 0xf4, 0x29, 0x6a, 0x68, 0xec, 0x65,
                   0x45, 0x38, 0xd9, 0x87, 0x10, 0x4f, 0x9a, 0x11, 0xac, 0xa1, 0xb7, 0xc8, 0x3a,
                   0xb2, 0xed, 0x8f, 0xd6, 0x9d, 0xa6, 0xb8, 0x8f, 0x0b, 0xcb, 0xd2, 0x7d, 0x3f,
                   0xea, 0x01, 0x32, 0x9c, 0xec, 0xf1, 0x0c, 0x57, 0xec, 0x3b, 0xa1, 0x63, 0xd5,
                   0x7b, 0x38, 0x80, 0x1b, 0xd6, 0xc3, 0xb3, 0x1c, 0xe5, 0x27, 0xb3, 0x37, 0x17,
                   0xbb, 0x56, 0xa4, 0x6f, 0x78, 0xfb, 0x96, 0xbe, 0x9f, 0x24, 0x24, 0xa2, 0x1b,
                   0x32, 0x84, 0x23, 0x23, 0x88, 0xcb, 0xba, 0x6a, 0x74],
        },
        {
            pass: [0x8f, 0x3a, 0x06, 0xe2, 0xfd, 0x87, 0x11, 0x35, 0x0a, 0x51, 0x7b, 0xb1, 0x2e,
                   0x31, 0xf3, 0xd3, 0x42, 0x3e, 0x8d, 0xc0, 0xbb, 0x14, 0xaa, 0xc8, 0x24, 0x0f,
                   0xca, 0x09, 0x95, 0x93, 0x8d, 0x59, 0xbb, 0x37, 0xbd, 0x0a, 0x7d, 0xfc, 0x9c,
                   0x9c, 0xc0, 0x70, 0x56, 0x84, 0xb4, 0x66, 0x12, 0xe8, 0xc8, 0xb1, 0xd6, 0x65,
                   0x5f, 0xb0, 0xf9, 0x88, 0x75, 0x62, 0xbb, 0x98, 0x99, 0x79, 0x1a, 0x02, 0x50,
                   0xd1, 0x32, 0x0f, 0x94, 0x5e, 0xda, 0x48, 0xcd, 0xc2, 0x0c, 0x23, 0x3f, 0x40,
                   0xa5, 0xbb, 0x0a, 0x7e, 0x3a, 0xc5, 0xad, 0x72, 0x50, 0xce, 0x68, 0x4f, 0x68,
                   0xfc, 0x0b, 0x8c, 0x96, 0x33, 0xbf, 0xd7, 0x5a, 0xad, 0x11, 0x65, 0x25, 0xaf,
                   0x7b, 0xdc, 0xdb, 0xbd, 0xb4, 0xe0, 0x0a, 0xb1, 0x63, 0xfd, 0x4d, 0xf0, 0x8f,
                   0x24, 0x3f, 0x12, 0x55, 0x7e],
            salt: [0x90, 0x63, 0x1f, 0x68, 0x6a, 0x8c, 0x3d, 0xbc, 0x07, 0x03, 0xff, 0xa3, 0x53,
                   0xbc, 0x1f, 0xdf, 0x35, 0x77, 0x45, 0x68, 0xac, 0x62, 0x40, 0x6f, 0x98, 0xa1,
                   0x3e, 0xd8, 0xf4, 0x75, 0x95, 0xfd],
            ops:  695191,
            mem:  16777216,
            out:  [0x64, 0x29, 0xae, 0xe4, 0x20, 0x8c, 0x89, 0x84, 0x2d, 0x3c, 0x84, 0xb9, 0x9f,
                   0xd1, 0xc2, 0xeb, 0x96, 0xed, 0xb6, 0x09, 0xd7, 0xdd, 0x4a, 0xa7, 0xa3, 0x57,
                   0x33, 0xea, 0xa6, 0x94, 0x95, 0x3e, 0xba, 0x57, 0x01, 0xaa, 0x24, 0xbd, 0xeb,
                   0x9c, 0x42, 0xfa, 0x70, 0xc8, 0x75, 0x41, 0xe7, 0x5d, 0x24, 0x7f, 0xc5, 0xc4,
                   0x49, 0x20, 0x5c],
        },
        {
            pass: [0xb5, 0x40, 0xbe, 0xb0, 0x16, 0xa5, 0x36, 0x65, 0x24, 0xd4, 0x60, 0x51, 0x56,
                   0x49, 0x3f, 0x98, 0x74, 0x51, 0x4a, 0x5a, 0xa5, 0x88, 0x18, 0xcd, 0x0c, 0x6d,
                   0xff, 0xfa, 0xa9, 0xe9, 0x02, 0x05, 0xf1, 0x7b],
            salt: [0x44, 0x07, 0x1f, 0x6d, 0x18, 0x15, 0x61, 0x67, 0x0b, 0xda, 0x72, 0x8d, 0x43,
                   0xfb, 0x79, 0xb4, 0x43, 0xbb, 0x80, 0x5a, 0xfd, 0xeb, 0xaf, 0x98, 0x62, 0x2b,
                   0x51, 0x65, 0xe0, 0x1b, 0x15, 0xfb],
            ops:  78652,
            mem:  16777216,
            out:  [0xd7, 0xb1, 0xef, 0x46, 0x4b, 0xe0, 0x3c, 0xe9, 0x05, 0x0b, 0x51, 0x08, 0xe2,
                   0x5f, 0x0b, 0x8e, 0x82, 0x12, 0x99, 0x98, 0x6f, 0xe0, 0xff, 0x89, 0xe1, 0x7f,
                   0xba, 0xe6, 0x5b, 0xa9, 0xfa, 0xd1, 0x67, 0xfb, 0xd2, 0x65, 0x86, 0x6a, 0xc0,
                   0x3e, 0xfc, 0x86, 0xab, 0x0b, 0x50, 0xd4, 0x6d, 0x67, 0x40, 0xa5, 0x9a, 0xdf,
                   0x59, 0x49, 0xb4, 0x4f, 0x7f, 0x9f, 0x3a, 0xc3, 0xf3, 0xd4, 0xcc, 0x9f, 0x12,
                   0x89, 0x66, 0xdb, 0x90, 0x99, 0xde, 0xb1, 0xb6, 0xb7, 0x85, 0x05, 0x24, 0x2b,
                   0x24, 0x01, 0xa1, 0x93, 0x82, 0x04, 0x08, 0xeb, 0x07, 0x80, 0xb2, 0x71, 0x62,
                   0xeb, 0xaf, 0xb7, 0xc5, 0x05, 0xb0, 0xe7, 0xc3, 0x2c, 0xe6, 0x6c, 0x6e, 0xfc,
                   0x0b, 0xe4, 0x87, 0x00, 0x8c, 0x12, 0x01, 0x45, 0x46, 0x80, 0x49, 0x8a, 0x2f,
                   0xc0, 0x6e, 0x00, 0xb4, 0x54, 0xe0, 0xb2, 0x09, 0x33, 0x90, 0x6b, 0xbb, 0x0e,
                   0x43, 0xb3, 0x99, 0xb9, 0xee, 0x46, 0xd8, 0x82, 0xf1, 0x07, 0xdf, 0x1e, 0xbd,
                   0xd1, 0xe7, 0xcd, 0x86, 0x7c, 0x9c, 0xdb, 0xa6, 0x01, 0x5b, 0x7e, 0x80, 0x06,
                   0x4a, 0xe8, 0xb3, 0x41, 0x7d, 0x96, 0x95, 0x24, 0xbe, 0xc0, 0x46, 0xe7, 0x82,
                   0xa1, 0x3b, 0x12, 0x5f, 0x05, 0x8c, 0xd3, 0x6b, 0x5d, 0x1a, 0xe6, 0x58, 0x86,
                   0xae, 0x7c, 0xaa, 0xb4, 0x5a, 0x6d, 0x98, 0x65, 0x1a, 0xda, 0x43, 0x5b, 0x8e,
                   0xe1, 0x1d, 0x5c, 0x12, 0x24, 0x23, 0x2f, 0x5f, 0x51, 0x5d, 0xf9, 0x74, 0x13,
                   0x8d, 0xd6, 0xcf, 0x34, 0x7b, 0x73, 0x04, 0x81, 0xd4, 0xb0, 0x73, 0xaf, 0x8f,
                   0xf0, 0x39, 0x4f, 0xe9, 0xf0, 0xb8, 0xcd, 0xfd, 0x99, 0xf5],
        },
        {
            pass: [0xa1, 0x49, 0x75, 0xc2, 0x6c, 0x08, 0x87, 0x55, 0xa8, 0xb7, 0x15, 0xff, 0x25,
                   0x28, 0xd6, 0x47, 0xcd, 0x34, 0x39, 0x87, 0xfc, 0xf4, 0xaa, 0x25, 0xe7, 0x19,
                   0x4a, 0x84, 0x17, 0xfb, 0x2b, 0x4b, 0x3f, 0x72, 0x68, 0xda, 0x9f, 0x31, 0x82,
                   0xb4, 0xcf, 0xb2, 0x2d, 0x13, 0x8b, 0x27, 0x49, 0xd6, 0x73, 0xa4, 0x7e, 0xcc,
                   0x75, 0x25, 0xdd, 0x15, 0xa0, 0xa3, 0xc6, 0x60, 0x46, 0x97, 0x17, 0x84, 0xbb,
                   0x63, 0xd7, 0xea, 0xe2, 0x4c, 0xc8, 0x4f, 0x26, 0x31, 0x71, 0x20, 0x75, 0xa1,
                   0x0e, 0x10, 0xa9, 0x6b, 0x0e, 0x0e, 0xe6, 0x7c, 0x43, 0xe0, 0x1c, 0x42, 0x3c,
                   0xb9, 0xc4, 0x4e, 0x53, 0x71, 0x01, 0x7e, 0x9c, 0x49, 0x69, 0x56, 0xb6, 0x32,
                   0x15, 0x8d, 0xa3, 0xfe, 0x12, 0xad, 0xde, 0xcb, 0x88, 0x91, 0x2e, 0x67, 0x59,
                   0xbc, 0x37, 0xf9, 0xaf, 0x2f, 0x45, 0xaf, 0x72, 0xc5, 0xca, 0xe3, 0xb1, 0x79,
                   0xff, 0xb6, 0x76, 0xa6, 0x97, 0xde, 0x6e, 0xbe, 0x45, 0xcd, 0x4c, 0x16, 0xd4,
                   0xa9, 0xd6, 0x42, 0xd2, 0x9d, 0xdc, 0x01, 0x86, 0xa0, 0xa4, 0x8c, 0xb6, 0xcd,
                   0x62, 0xbf, 0xc3, 0xdd, 0x22, 0x9d, 0x31, 0x3b, 0x30, 0x15, 0x60, 0x97, 0x1e,
                   0x74, 0x0e, 0x2c, 0xf1, 0xf9, 0x9a, 0x9a, 0x09, 0x0a, 0x5b, 0x28, 0x3f, 0x35,
                   0x47, 0x50, 0x57, 0xe9, 0x6d, 0x70, 0x64, 0xe2, 0xe0, 0xfc, 0x81, 0x98, 0x45,
                   0x91, 0x06, 0x8d, 0x55, 0xa3, 0xb4, 0x16, 0x9f, 0x22, 0xcc, 0xcb, 0x07, 0x45,
                   0xa2, 0x68, 0x94, 0x07, 0xea, 0x19, 0x01, 0xa0, 0xa7, 0x66, 0xeb, 0x99],
            salt: [0x3d, 0x96, 0x8b, 0x27, 0x52, 0xb8, 0x83, 0x84, 0x31, 0x16, 0x50, 0x59, 0x31,
                   0x9f, 0x3f, 0xf8, 0x91, 0x0b, 0x7b, 0x8e, 0xcb, 0x54, 0xea, 0x01, 0xd3, 0xf5,
                   0x47, 0x69, 0xe9, 0xd9, 0x8d, 0xaf],
            ops:  717248,
            mem:  16777216,
            out:  [0x2d, 0xe0, 0xba, 0x72, 0x83, 0x3e, 0xf0, 0x85, 0x8d, 0x80, 0x07, 0x2c, 0x7a,
                   0xb9, 0x91, 0xcb, 0xc2, 0x57, 0x1b, 0x94, 0x43, 0x57, 0xed, 0x2b, 0x8d, 0x3e,
                   0xbd, 0x3f, 0x60, 0x94, 0xc6, 0x53, 0x3a, 0x3d, 0x9d, 0xb4, 0x21, 0x6f, 0x78,
                   0x13, 0xcc, 0x54, 0x50, 0xee, 0x9f, 0xfc, 0x1f, 0xd5, 0x64, 0xe9, 0x37, 0xad,
                   0x70, 0x5b, 0xbd, 0x5d, 0xc0, 0x3b, 0xbf, 0x7a, 0x6b, 0x7e, 0x24, 0xa9, 0xd0,
                   0x80, 0x9b, 0x5a, 0xa5, 0x19, 0xe9, 0x1b, 0xa6, 0x5c, 0x27, 0x1b, 0xac, 0x3c,
                   0x31, 0x65, 0x61, 0x6b, 0x8e, 0xb1, 0x79, 0x76, 0x11, 0x1a, 0x87, 0x38, 0xf3,
                   0xff, 0x5d, 0x41, 0xe1, 0xef, 0x6f, 0xa7, 0x26, 0x88, 0x56, 0x20, 0x15, 0xd4,
                   0x80, 0xa6, 0x5a, 0xa7, 0x91, 0x8b, 0x74, 0xb0, 0xda, 0x5a, 0xea, 0xd2, 0xc3,
                   0x45, 0x3e, 0x8b, 0x84, 0x24, 0x99, 0xc4, 0xb3, 0x3f, 0xc2, 0x82, 0xba, 0x1e,
                   0x21, 0x22, 0x15, 0x50, 0x08, 0xb7, 0x5d, 0x44, 0xd2, 0xec, 0xf2, 0xaa, 0x1c,
                   0xe0, 0x21, 0x69, 0x35, 0xad, 0x41, 0x84, 0x32, 0x8e, 0xfe, 0xf1, 0x6c, 0xf6,
                   0xfc, 0x4c, 0x1d, 0x05, 0x9e, 0xb8, 0x85, 0x5a, 0x1e, 0x61, 0x9e],
        },
    }

    verify_password_valid_tests! [
        {
            pass: "^T5H$JYt39n%K*j:W]!1s?vg!:jGi]Ax?..l7[p0v:1jHTpla9;]bUN;?bWyCbtqg nrDFal+Jxl\
                   3,2`#^tFSu%v_+7iYse8-cCkNf!tD=KrW)",
            hash: "$7$B6....1....75gBMAGwfFWZqBdyF3WdTQnWdUsuTiWjG1fF9c1jiSD$tc8RoB3.Em3/zNgMLW\
                    o2u00oGIoTyJv4fl3Fl8Tix72",
        },
        {
            pass: "bl72h6#y<':MFRZ>B IA1=NRkCKS%W8`1I.2uQxJN0g)N N aTt^4K!Iw5r H6;crDsv^a55j9ts\
                   k'/GqweZn;cdk6+F_St6:#*=?ZCD_lw>.",
            hash: "$7$A6....3....Iahc6qM0.UQJHVgE4h9oa1/4OWlWLm9CCtfguvz6bQD$QnXCo3M7nIqtry2WKs\
                   UZ5gQ.mY0wAlJu.WUhtE8vF66",
        },
        {
            pass: "Py >e.5b+tLo@rL`dC2k@eJ&4eVl!W=JJ4+k&mAt@gt',FS1JjqKW3aq21:]^kna`mde7kVkN5Nr\
                   pKUptu)@4*b&?BE_sJMG1=&@`3GBCV]Wg7xwgo7x3El",
            hash: "$7$96..../....f6bEusKt79kK4wdYN0ki2nw4bJQ7P3rN6k3BSigsK/D$Dsvuw7vXj5xijmrb/N\
                   OhdgoyK/OiSIYv88cEtl9Cik7",
        },
        {
            pass: "2vj;Um]FKOL27oam(:Uo8+UmSTvb1FD*h?jk_,S=;RDgF-$Fjk?]9yvfxe@fN^!NN(Cuml?+2Raa",
            hash: "$7$86....I....7XwIxLtCx4VphmFeUa6OGuGJrFaIaYzDiLNu/tyUPhD$U3q5GCEqCWxMwh.YQH\
                   DJrlg7FIZgViv9pcXE3h1vg61",
        },
        {
            pass: "CT=[9uUoGav,J`kU+348tA50ue#sL:ABZ3QgF+r[#vh:tTOiL>s8tv%,Jeo]jH/_4^i(*jD-_ku[\
                   9Ko[=86 06V",
            hash: "$7$A6....2....R3.bjH6YS9wz9z8Jsj.3weGQ3J80ZZElGw2oVux1TP6$i5u6lFzXDHaIgYEICi\
                   nLD6WNaovbiXP8SnLrDRdKgA9",
        },
        {
            pass: "J#wNn`hDgOpTHNI.w^1a70%f,.9V_m038H_JIJQln`vdWnn/rmILR?9H5g(+`;@H(2VosN9Fgk[W\
                   EjaBr'yB9Q19-imNa04[Mk5kvGcSn-TV",
            hash: "$7$B6....1....Dj1y.4mF1J9XmT/6IDskYdCLaPFJTq9xcCwXQ1DpT92$92/hYfZLRq1nTLyIz.\
                   uc/dC6wLqwnsoqpkadrCXusm6",
        },
        {
            pass: "j4BS38Asa;p)[K+9TY!3YDj<LK-`nLVXQw9%*QfM",
            hash: "$7$B6....1....5Ods8mojVwXJq4AywF/uI9BdMSiJ/zT8hQP/4cB68VC$nk4ExHNXJ802froj51\
                   /1wJTrSZvTIyyK7PecOxRRaz0",
        },
        {
            pass: "M.R>Qw+!qJb]>pP :_.9`dxM9k [eR7Y!yL-3)sNs[R,j_/^ TH=5ny'15>6UXWcQW^6D%XCsO[v\
                   N[%ReA-`tV1vW(Nt*0KVK#]45P_A",
            hash: "$7$B6....1....D/eyk8N5y6Z8YVQEsw521cTx.9zzLuK7YDs1KMMh.o4$alfW8ZbsUWnXc.vqon\
                   2zoljVk24Tt1.IsCuo2KurvS2",
        },
        {
            pass: "K3S=KyH#)36_?]LxeR8QNKw6X=gFb'ai$C%29V* tyh^Wo$TN-#Q4qkmtTCf0LLb.^E$0uykkP",
            hash: "$7$B6....1....CuBuU97xgAage8whp/JNKobo0TFbsORGVbfcQIefyP8$aqalP.XofGViB8EPLO\
                   NqHma8vs1xc9uTIMYh9CgE.S8",
        },
        {
            pass: "Y0!?iQa9M%5ekffW(`",
            hash: "$7$A6....1....TrXs5Zk6s8sWHpQgWDIXTR8kUU3s6Jc3s.DtdS8M2i4$a4ik5hGDN7foMuHOW.\
                   cp.CtX01UyCeO0.JAG.AHPpx5",
        },
    ];

    verify_password_invalid_tests! [
        {
            pass: "Y0!?iQa9M%5ekffW(`",
            hash: "$7$A6....1....$TrXs5Zk6s8sWHpQgWDIXTR8kUU3s6Jc3s.DtdS8M2i4a4ik5hGDN7foMuHOW.\
                   cp.CtX01UyCeO0.JAG.AHPpx5",
        },
        {
            pass: "Y0!?iQa9M%5ekffW(`",
            hash: "$7$.6....1....TrXs5Zk6s8sWHpQgWDIXTR8kUU3s6Jc3s.DtdS8M2i4$a4ik5hGDN7foMuHOW.\
                   cp.CtX01UyCeO0.JAG.AHPpx5",
        },
        {
            pass: "Y0!?iQa9M%5ekffW(`",
            hash: "$7$A.....1....TrXs5Zk6s8sWHpQgWDIXTR8kUU3s6Jc3s.DtdS8M2i4$a4ik5hGDN7foMuHOW.\
                   cp.CtX01UyCeO0.JAG.AHPpx5",
        },
        {
            pass: "Y0!?iQa9M%5ekffW(`",
            hash: "$7$A6.........TrXs5Zk6s8sWHpQgWDIXTR8kUU3s6Jc3s.DtdS8M2i4$a4ik5hGDN7foMuHOW.\
                   cp.CtX01UyCeO0.JAG.AHPpx5",
        },
        {
            pass: "Y0!?iQa9M%5ekffW(`",
            hash: "$7$A6....1....TrXs5Zk6s8sWHpQgWDIXTR8kUU3s6Jc3s.DtdS8M2i44269$a4ik5hGDN7foMu\
                   HOW.cp.CtX01UyCeO0.JAG.AH",
        },
        {
            pass: "Y0!?iQa9M%5ekffW(`",
            hash: "$7$A6....1....TrXs5Zk6s8sWHpQgWDIXTR8kUU3s6Jc3s.DtdS8M2i4$a4ik5hGDN7foMuHOW.\
                   cp.CtX01UyCeO0.JAG.AHPpx54269",
        },
        {
            pass: "Y0!?iQa9M%5ekffW(`",
            hash: "$7^A6....1....TrXs5Zk6s8sWHpQgWDIXTR8kUU3s6Jc3s.DtdS8M2i4$a4ik5hGDN7foMuHOW.\
                   cp.CtX01UyCeO0.JAG.AHPpx5",
        },
        {
            pass: "Y0!?iQa9M%5ekffW(`",
            hash: "$7$!6....1....TrXs5Zk6s8sWHpQgWDIXTR8kUU3s6Jc3s.DtdS8M2i4$a4ik5hGDN7foMuHOW.\
                   cp.CtX01UyCeO0.JAG.AHPpx5",
        },
        {
            pass: "Y0!?iQa9M%5ekffW(`",
            hash: "$7$A!....1....TrXs5Zk6s8sWHpQgWDIXTR8kUU3s6Jc3s.DtdS8M2i4$a4ik5hGDN7foMuHOW.\
                   cp.CtX01UyCeO0.JAG.AHPpx5",
        },
        {
            pass: "Y0!?iQa9M%5ekffW(`",
            hash: "$7$A6....!....TrXs5Zk6s8sWHpQgWDIXTR8kUU3s6Jc3s.DtdS8M2i4$a4ik5hGDN7foMuHOW.\
                   cp.CtX01UyCeO0.JAG.AHPpx5",
        },
        {
            pass: "",
            hash: "$7$A6....1....TrXs5Zk6s8sWHpQgWDIXTR8kUU3s6Jc3s.DtdS8M2i4$a4ik5hGDN7foMuHOW.\
                   cp.CtX01UyCeO0.JAG.AHPpx5",
        },
        {
            pass: "Y0!?iQa9M%5ekffW(`",
            hash: "$7fA6....1....TrXs5Zk6s8sWHpQgWDIXTR8kUU3s6Jc3s.DtdS8M2i4#a4ik5hGDN7foMuHOW.\
                   cp.CtX01UyCeO0.JAG.AHPpx5",
        },
        {
            pass: "Y0!?iQa9M%5ekffW(`",
            hash: "$7$AX....1....TrXs5Zk6s8sWHpQgWDIXTR8kUU3s6Jc3s.DtdS8M2i4$a4ik5hGDN7foMuHOW.\
                   cp.CtX01UyCeO0.JAG.AHPpx5",
        },
        {
            pass: "Y0!?iQa9M%5ekffW(`",
            hash: "$7$A6....1!...TrXs5Zk6s8sWHpQgWDIXTR8kUU3s6Jc3s.DtdS8M2i4$a4ik5hGDN7foMuHOW.\
                   cp.CtX01UyCeO0.JAG.AHPpx5",
        },
        {
            pass: "Y0!?iQa9M%5ekffW(`",
            hash: "$7$A6....1",
        },
        {
            pass: "Y0!?iQa9M%5ekffW(`",
            hash: "$7$",
        },
        {
            pass: "Y0!?iQa9M%5ekffW(`",
            hash: "",
        },
        {
            pass: "Y0!?iQa9M%5ekffW(`",
            hash: "$7$A6....1....TrXs5Zk6s8sWHpQgWDIXTR8kUU3s6Jc3s.DtdS8M2i4$",
        },
        {
            pass: "test",
            hash: "$7$.6..../.....lgPchkGHqbeONR/xtuXyjCrt9kUSg6NlKFQO0OSxo/$.DbajbPYH9T7sg3fOt\
                   cgxvJzzfIgJBIxMkeQ8b24YQ.",
        },
        {
            pass: "test",
            hash: "$7$z6..../.....lgPchkGHqbeONR/xtuXyjCrt9kUSg6NlKFQO0OSxo/$.DbajbPYH9T7sg3fOt\
                   cgxvJzzfIgJBIxMkeQ8b24YQ.",
        },
        {
            pass: "test",
            hash: "$7$8zzzzzzzzzz.lgPchkGHqbeONR/xtuXyjCrt9kUSg6NlKFQO0OSxo/$.DbajbPYH9T7sg3fOt\
                   cgxvJzzfIgJBIxMkeQ8b24YQ.",
        },
        {
            pass: "test",
            hash: "$7$8.....zzzzz.lgPchkGHqbeONR/xtuXyjCrt9kUSg6NlKFQO0OSxo/$.DbajbPYH9T7sg3fOt\
                   cgxvJzzfIgJBIxMkeQ8b24YQ.",
        },
        {
            pass: "test",
            hash: "$7$86..../..../lgPchkGHqbeONR/xtuXyjCrt9kUSg6NlKFQO0OSxo/$.DbajbPYH9T7sg3fOt\
                   cgxvJzzfIgJBIxMkeQ8b24YQ.",
        },
    ];

    #[test]
    fn needs_rehash() -> Result<(), AlkaliError> {
        use crate::hash::pbkdf::RehashResult;

        const OPS_LIMIT: usize = super::OPS_LIMIT_INTERACTIVE;
        const MEM_LIMIT: usize = super::MEM_LIMIT_INTERACTIVE;
        const PASSWORD: &'static str = "Correct Horse Battery Staple";

        let hash = super::store_password(PASSWORD, OPS_LIMIT, MEM_LIMIT)?;

        assert_eq!(
            super::requires_rehash(&hash, OPS_LIMIT, MEM_LIMIT)?,
            RehashResult::ParametersMatch,
        );

        assert_eq!(
            super::requires_rehash(&hash, OPS_LIMIT, MEM_LIMIT / 2)?,
            RehashResult::ParametersDiffer,
        );
        assert_eq!(
            super::requires_rehash(&hash, OPS_LIMIT / 2, MEM_LIMIT)?,
            RehashResult::ParametersDiffer,
        );
        assert_eq!(
            super::requires_rehash(&hash, OPS_LIMIT * 2, MEM_LIMIT)?,
            RehashResult::ParametersDiffer,
        );

        assert_eq!(
            super::requires_rehash("not valid", OPS_LIMIT, MEM_LIMIT)?,
            RehashResult::InvalidHash,
        );

        Ok(())
    }
}
