//! Generic hash function.
//!
//! This module corresponds to the [`crypto_generichash`
//! API](https://doc.libsodium.org/hashing/generic_hashing) from Sodium.
//!
//! This is a fast [cryptographic hash
//! function](https://en.wikipedia.org/wiki/Cryptographic_hash_function), suitable for use in the
//! standard applications where such a function is necessary. Given an arbitrarily-long input, a
//! cryptographic hash function produces a fixed-length fingerprint (the *digest*). This digest can
//! then be used to uniquely identify the original input, as it should be infeasible to find another
//! input which hashes to the same value. See the [`hash` docs](super) for more information on the
//! properties of a cryptographic hash function.
//!
//! This generic hash function can be used in cases such as file integrity checking or generating
//! unique identifiers to index arbitrarily-long data. Please note that the [`hash`](super) module
//! has a number of alternative members which are better suited to certain tasks: For example, the
//! [`hash::pbkdf`](crate::hash::pbkdf) module should be used for password hashing, and the
//! [`hash::short`](crate::hash::short) module is better suited to the construction of hash
//! tables/bloom filters.
//!
//! # Algorithm Details
//! [BLAKE2b](https://www.blake2.net/) is used in this API, a fast, modern hash algorithm optimised
//! for 64-bit platforms.
//!
//! # Security Considerations
//! Generic hash functions *must not* be used for password hashing, they are not sufficiently
//! computationally intensive. Instead, use a [Password-Based Key Derivation
//! Function](https://en.wikipedia.org/wiki/Key_derivation_function#Password_hashing) (PBKDF) such
//! as those available in the [`hash::pbkdf`](crate::hash::pbkdf) module.
//!
//! The output size of this hash function is user-configurable, and must take a value between
//! [`DIGEST_LENGTH_MIN`] and [`DIGEST_LENGTH_MAX`]. However, output lengths below
//! [`DIGEST_LENGTH_DEFAULT`] may not preserve the expected property of collision resistance, and if
//! such a digest size is used, this function should not be treated as a generic cryptographic hash
//! function. Only use an output length below [`DIGEST_LENGTH_DEFAULT`] if doing so is absolutely
//! necessary for your use case.
//!
//! This hash function supports a keyed variant. A key can be used in order to produce different
//! hashes for different application domains, or (for a sufficient key size) to turn this hash
//! function into a [MAC](https://en.wikipedia.org/wiki/Message_authentication_code). If a key is to
//! be used as a secret value, it must be at least [`KEY_LENGTH_MIN`] bytes. Keys below this length
//! are not of sufficient size to be thought of as cryptographic secrets. A key size of
//! [`KEY_LENGTH_DEFAULT`] is recommended for secret key values.
//!
//! # Examples
//! Single-part hashing (using [`hash`]):
//!
//! ```rust
//! use alkali::hash::generic;
//!
//! let message = b"Here's some message we wish to hash :)";
//! // `None` here indicates we don't want to use a key.
//! let hash = generic::hash(message, None).unwrap();
//! // The `hash` function produces an output of length `DIGEST_LENGTH_DEFAULT`.
//! assert_eq!(
//!     hash,
//!     [
//!         0xaa, 0x42, 0xa0, 0xb0, 0xdc, 0x4a, 0x34, 0xf6, 0xb5, 0x4a, 0xa0, 0x33, 0x7b, 0xcb,
//!         0x90, 0x2c, 0xf1, 0x82, 0xfe, 0xb6, 0x6a, 0xe8, 0x9f, 0x05, 0xb3, 0x05, 0xfb, 0xac,
//!         0x9d, 0xad, 0x0d, 0x3b,
//!     ]
//! );
//! ```
//!
//! Keyed hashing (using [`hash`]):
//!
//! ```rust
//! use alkali::hash::generic;
//!
//! let message = b"We also wish to hash this message ;)";
//!
//! // Generate a random key for the hash.
//! let key_a = generic::Key::generate().unwrap();
//! let hash_a = generic::hash(message, Some(key_a.as_slice())).unwrap();
//!
//! // We'll now hash the same message with a different key.
//! let key_b = generic::Key::generate().unwrap();
//! let hash_b = generic::hash(message, Some(key_b.as_slice())).unwrap();
//!
//! // Hashing the same message with a different key should produce a different hash.
//! assert_ne!(hash_a, hash_b);
//! ```
//!
//! Multi-part hashing (using [`Multipart`]):
//!
//! ```rust
//! use alkali::hash::generic;
//!
//! // We need to specify the desired output length at initialisation.
//! let mut state_a = generic::Multipart::new(generic::DIGEST_LENGTH_DEFAULT).unwrap();
//! // Add some data to be hashed.
//! state_a.update(b"Here's the first part");
//! state_a.update(b"... And the second!");
//! // `output` must be the same length specified in `Multipart::new`.
//! let mut output_a = [0u8; generic::DIGEST_LENGTH_DEFAULT];
//! // `Multipart::calculate` calculates the hash of the data added to the Multipart state.
//! state_a.calculate(&mut output_a).unwrap();
//!
//! let mut state_b = generic::Multipart::new(generic::DIGEST_LENGTH_DEFAULT).unwrap();
//! // We don't need to specify the message in the same layout as we did when we first calculated
//! // the hash: The message contents will be concatenated.
//! state_b.update(b"Here");
//! state_b.update(b"'s the first ");
//! state_b.update(b"part... And the ");
//! state_b.update(b"second!");
//! let mut output_b = [0u8; generic::DIGEST_LENGTH_DEFAULT];
//! // `Multipart::compare` compares the hash of the data added to the Multipart state with some
//! // other hash, in constant time.
//! assert!(state_b.compare(&output_a).unwrap());
//! ```

// TODO: Consider how/if we wish to expose the salt/personalisation parameters of the hash.

use thiserror::Error;

/// Error type returned if something went wrong in the generic module.
#[derive(Clone, Copy, Debug, Eq, Error, PartialEq)]
pub enum GenericHashError {
    /// The desired digest length was too short or too long for use with this algorithm.
    ///
    /// The output size of this hash must be at least [`DIGEST_LENGTH_MIN`], and at most
    /// [`DIGEST_LENGTH_MAX`] bytes. A minimum of [`DIGEST_LENGTH_DEFAULT`] should be used if
    /// collisions should be avoided, which is generally a necessary property for a cryptographic
    /// hash function to have any utility.
    #[error("digest length outside acceptable range")]
    DigestLengthInvalid,

    /// The output buffer size for [`Multipart::calculate`] differs from the output length
    /// specified in [`Multipart::new`] or [`Multipart::new_keyed`].
    #[error("output length changed from length specified on instantiation")]
    OutputLengthChanged,

    /// The provided key was too long for use with this algorithm.
    ///
    /// A key for use with this hash must be at most [`KEY_LENGTH_MAX`] bytes. Using a key size of
    /// [`KEY_LENGTH_DEFAULT`] is recommended, and a key shorter than [`KEY_LENGTH_MIN`] should not
    /// be used if it is intended to be a secret value.
    #[error("key length outsize acceptable range")]
    KeyLengthInvalid,
}

/// The [BLAKE2b](https://www.blake2.net) hash function.
pub mod blake2b {
    use super::GenericHashError;
    use crate::{assert_not_err, mem, require_init, unexpected_err, AlkaliError};
    use libsodium_sys as sodium;
    use std::marker::PhantomData;
    use std::ptr;

    /// The minimum output length of the hash function, in bytes.
    ///
    /// Below the [default output length](DIGEST_LENGTH_DEFAULT), the hash function may not satisfy
    /// the expected property of collision resistance, and as such, you should not use a smaller
    /// output length than [`DIGEST_LENGTH_DEFAULT`], unless doing so is absolutely necessary for
    /// your use case.
    pub const DIGEST_LENGTH_MIN: usize = sodium::crypto_generichash_blake2b_BYTES_MIN as usize;

    /// The default length of the output of the hash function, in bytes.
    ///
    /// This is the default length of the output of the hash function, and is the recommended
    /// minimum size for cryptographic use. At this size, it is computationally infeasible to find
    /// two inputs with the same digest. Below this default output length, the hash function may not
    /// satisfy the expected property of collision resistance, and as such, should not be thought of
    /// as a cryptographic hash function.
    pub const DIGEST_LENGTH_DEFAULT: usize = sodium::crypto_generichash_blake2b_BYTES as usize;

    /// The maximum output length of the hash function, in bytes.
    ///
    /// Below the [default output length](DIGEST_LENGTH_DEFAULT), the hash function may not satisfy
    /// the expected property of collision resistance, and as such, using a lower output size than
    /// [`DIGEST_LENGTH_DEFAULT`] is not recommended.
    pub const DIGEST_LENGTH_MAX: usize = sodium::crypto_generichash_blake2b_BYTES_MAX as usize;

    /// The minimum recommended key size for use with the keyed variant of the hash function, in
    /// bytes.
    ///
    /// If the key is intended to be a secret value, a key length below this size must not be used.
    /// If the key is just being used to produce different hashes in different application domains,
    /// and is non-secret, this limit can be ignored.
    pub const KEY_LENGTH_MIN: usize = sodium::crypto_generichash_blake2b_KEYBYTES_MIN as usize;

    /// The recommended key size for use with the keyed variant of the hash function, in bytes.
    ///
    /// This length should always be sufficient if the key is intended to be a secret value. A key
    /// length below [`KEY_LENGTH_MIN`] must not be used if the key is intended to be secret.
    pub const KEY_LENGTH_DEFAULT: usize = sodium::crypto_generichash_blake2b_KEYBYTES as usize;

    /// The maximum key size for use with the keyed variant of the hash function, in bytes.
    ///
    /// A key length below [`KEY_LENGTH_MIN`] must not be used if the key is intended to be secret..
    pub const KEY_LENGTH_MAX: usize = sodium::crypto_generichash_blake2b_KEYBYTES_MAX as usize;

    mem::hardened_buffer! {
        /// Secret key for the keyed variant of the hash function.
        ///
        /// Keys can be used to turn this hash function into a
        /// [MAC](https://en.wikipedia.org/wiki/Message_authentication_code), or simply to cause the
        /// hash function to produce different output values in different application domains.
        ///
        /// This key is the default (recommended) length, [`KEY_LENGTH_DEFAULT`], but other key
        /// sizes can be used. This type is simply provided as a helper for a common use case. If
        /// the key used with the hash function is intended to be a secret value, the key must be at
        /// least [`KEY_LENGTH_MIN`] bytes.
        ///
        /// There are no *technical* constraints on the contents of a key, but if it is intended to
        /// be secret it should be indistinguishable from random noise. A random key can be securely
        /// generated using [`Key::generate`].
        ///
        /// A secret key must not be made public.
        ///
        /// This is a [hardened buffer type](https://docs.rs/alkali#hardened-buffer-types), and will
        /// be zeroed on drop. A number of other security measures are taken to protect its
        /// contents. This type in particular can be thought of as roughly equivalent to a `[u8;
        /// KEY_LENGTH_DEFAULT]`, and implements [`std::ops::Deref`], so it can be used like it is
        /// an `&[u8]`. This struct uses heap memory while in scope, allocated using Sodium's
        /// [secure memory management](https://doc.libsodium.org/memory_management).
        pub Key(KEY_LENGTH_DEFAULT);
    }

    impl Key {
        /// Generate a new, random key for use with the keyed variant of the hash function.
        pub fn generate() -> Result<Self, AlkaliError> {
            require_init()?;

            let mut key = Self::new_empty()?;
            unsafe {
                // SAFETY: The argument to this function should be a pointer to a region of memory
                // sufficient to store a key of the default size for this algorithm. We have defined
                // the `Key` type to allocate `crypto_generichash_KEYBYTES`, which is the default
                // key size for this algorithm, so it is valid for writes of the required length.
                // The `Key::inner_mut` method simply returns a mutable pointer to its backing
                // memory.
                sodium::crypto_generichash_blake2b_keygen(key.inner_mut() as *mut libc::c_uchar);
            }
            Ok(key)
        }
    }

    /// Stores a digest ("fingerprint") of a message calculated using this hash function, of the
    /// default length.
    pub type Digest = [u8; DIGEST_LENGTH_DEFAULT];

    /// Streaming hash API, for long/multi-part message hashing.
    ///
    /// This can be used to calculate the hash of a message which is too large to fit into memory,
    /// or where the message is received in portions.
    ///
    /// This struct uses heap memory while in scope, allocated using Sodium's [secure memory
    /// utilities](https://doc.libsodium.org/memory_management).
    ///
    /// # Security Considerations
    /// The output size for the hash must be at least [`DIGEST_LENGTH_DEFAULT`] bytes to preserve
    /// collision-resistance, which is an expected property of a cryptographic hash function. Only
    /// use an output length below this value if doing so is absolutely necessary for your use case,
    /// or if collision resistance is not a necessary property.
    #[derive(Debug)]
    pub struct Multipart {
        output_len: usize,
        state: ptr::NonNull<sodium::crypto_generichash_blake2b_state>,
        _marker: PhantomData<sodium::crypto_generichash_blake2b_state>,
    }

    impl Multipart {
        /// Initialises a `crypto_generichash_state` struct.
        ///
        /// The caller is responsible for ensuring that `key_ptr` and `key_len` are valid for use
        /// with `crypto_generichash_init`, as the second and third parameters. `key_ptr` should be
        /// a pointer from which the key will be read, and `key_len` should be the length of the key
        /// to read. `key_ptr` can be null to use the non-keyed variant.
        unsafe fn init_impl(
            output_len: usize,
            key_ptr: *const libc::c_uchar,
            key_len: usize,
        ) -> Result<Self, AlkaliError> {
            require_init()?;

            if !(DIGEST_LENGTH_MIN..=DIGEST_LENGTH_MAX).contains(&output_len) {
                return Err(GenericHashError::DigestLengthInvalid.into());
            }

            // SAFETY: This call to `malloc` will allocate the memory required for a
            // `crypto_generichash_state` type, outside of Rust's memory management. The
            // associated memory is always freed in the corresponding `drop` call for the
            // Multipart struct, unless initialisation fails, in which case it is freed before
            // `Multipart::new` returns, and not used again. We never free the memory in any
            // other place in this struct, and drop can only be called once, so a double-free is
            // not possible. We never expose a pointer to the allocated memory directly. See the
            // drop implementation for more reasoning on safety.
            let mut state = mem::malloc()?;

            // SAFETY: This function initialises a `crypto_generichash_state` struct. The first
            // argument should be a pointer to a region of memory sufficient to store the
            // struct, allocated above. The type of `state` is a `NonNull` pointer, and the
            // unsafe block will return early if allocation failed, so this pointer is valid to
            // use here. The next two arguments specify a pointer to the key to use, and the
            // length of the key to read from this pointer. It is the caller's responsibility to
            // ensure these arguments are valid. The final argument specifies the output length
            // of the hash function, which must be between `crypto_generichash_BYTES_MIN` and
            // `crypto_generichash_BYTES_MAX`. We verify this condition above, and return early
            // with an error if the value of `output_len` is invalid. Sodium's documentation
            // states that after this function has been called, if the return value indicates
            // success, then the memory pointed to by `state` is correctly initialised, and is a
            // valid representation of a `crypto_generichash_state` struct which can be used
            // with other functions from Sodium.
            let init_result = sodium::crypto_generichash_blake2b_init(
                state.as_mut(),
                key_ptr,
                key_len,
                output_len,
            );

            // This return value is not possible in the current implementation of
            // `crypto_generichash_init` in Sodium, but could be in the future.
            if init_result != 0 {
                // SAFETY: The memory we free here was allocated previously in this function
                // using Sodium's allocator, and has not yet been freed, so it is valid to free
                // it here. The `unexpected_err` macro below will always panic, so this function
                // will not return, and an instance of `Self` is never initialised, preventing a
                // double-free or use-after-free.
                mem::free(state);
                unexpected_err!("crypto_generichash_blake2b_init");
            }

            Ok(Self {
                output_len,
                state,
                _marker: PhantomData,
            })
        }

        /// Create a new instance of the struct.
        ///
        /// `output_len` should be the desired output length of the hash function. It must be
        /// between [`DIGEST_LENGTH_MIN`] and [`DIGEST_LENGTH_MAX`] bytes.
        ///
        /// # Security Considerations
        /// The output size for the hash must be at least [`DIGEST_LENGTH_DEFAULT`] bytes to
        /// preserve collision-resistance, which is an expected property of a cryptographic hash
        /// function. Only use an output length below this value if doing so is absolutely necessary
        /// for your use case, or if collision resistance is not a necessary property.
        pub fn new(output_len: usize) -> Result<Self, AlkaliError> {
            unsafe {
                // SAFETY: The `Self::init_impl` function requires that we ensure that `key_ptr` and
                // `key_len` are a valid key pointer and length for the `crypto_generichash_init`
                // function. It is documented that if `key_ptr` is simply set to a null pointer,
                // then Sodium will ignore the key, and the unkeyed variant of BLAKE2b will be used.
                // In this case, the `key_len` parameter is also ignored.
                Self::init_impl(output_len, ptr::null(), 0)
            }
        }

        /// Create a new instance of the struct, using a key as part of the hash calculation.
        ///
        /// `output_len` should be the desired output length of the hash function. It must be
        /// between [`DIGEST_LENGTH_MIN`] and [`DIGEST_LENGTH_MAX`] bytes.
        ///
        /// `key` should be the key to use for the hash calculation. For a given message `M`, and
        /// different keys `K1` and `K2`, `BLAKE2b(M, K1)` should be different to `BLAKE2b(M, K2)`.
        /// `key` may be any length up to [`KEY_LENGTH_MAX`] bytes, but see the security
        /// considerations for more information on key types. It is recommended to use a [`Key`]
        /// struct to store a key for this algorithm. The [`Key`] type can be used for this
        /// argument.
        ///
        /// # Security Considerations
        /// The output size for the hash must be at least [`DIGEST_LENGTH_DEFAULT`] bytes to
        /// preserve collision-resistance, which is an expected property of a cryptographic hash
        /// function. Only use an output length below this value if doing so is absolutely necessary
        /// for your use case, or if collision resistance is not a necessary property.
        ///
        /// If the key is intended to be a secret value, it must be at least [`KEY_LENGTH_MIN`]
        /// bytes. It is recommended that secret keys be at least [`KEY_LENGTH_DEFAULT`] bytes, and
        /// the [`Key`] type is this length for this reason. If you want to use a secret key of a
        /// custom length, you may wish to use the [`crate::mem`] module to define a custom key type
        /// stored in protected memory. Keys shorter than [`KEY_LENGTH_MIN`] are not suitable for
        /// secret values, but may be useful to produce different hashes across different
        /// application domains.
        pub fn new_keyed(output_len: usize, key: &[u8]) -> Result<Self, AlkaliError> {
            if key.len() > KEY_LENGTH_MAX {
                return Err(GenericHashError::KeyLengthInvalid.into());
            }

            unsafe {
                // SAFETY: The `Self::init_impl` function requires that we ensure that `key_ptr` and
                // `key_len` are a valid key pointer and length for the `crypto_generichash_init`
                // function. We use `key.len()` to specify the length of the key to read from `key`,
                // so `key` is definitely valid for reads of this length. We verify above that the
                // key length is within acceptable bounds.
                Self::init_impl(output_len, key.as_ptr(), key.len())
            }
        }

        /// Try to clone this Multipart state.
        ///
        /// This function initialises a new instance of this struct, in the same state as the
        /// current one, so any data written to be added to the hash calculation in the current
        /// struct will be included in the hash calculation in the new struct. If a key was provided
        /// for this struct, it will also be used for the hash calculation in the new struct.
        pub fn try_clone(&self) -> Result<Self, AlkaliError> {
            // We do not use `require_init` here, as it must be called to initialise a `Multipart`
            // struct.

            let state = unsafe {
                // SAFETY: This call to `malloc` will allocate the memory required for a
                // `crypto_generichash_state` type, outside of Rust's memory management. The
                // associated memory is always freed in the corresponding `drop` call for the
                // Multipart struct. We never free the memory in any other place in this struct, and
                // drop can only be called once, so a double-free is not possible. We never expose a
                // pointer to the allocated memory directly. See the drop implementation for more
                // reasoning on safety.
                let mut state = mem::malloc()?;

                // SAFETY: We have called `malloc` to allocate sufficient space for one
                // `crypto_generichash_state` struct at each of the two pointers used here:
                // `self.state` was allocated in a call to `Self::init_impl`, and `state` was
                // allocated above, so both are valid for reads/writes of
                // `size_of::<crypto_generichash_state>` bytes. We have just allocated a fresh
                // region of memory for `state`, so it definitely doesn't overlap with `self.state`.
                // To initialise an instance of `Self`, `self.state` must be a valid representation
                // of a `crypto_generichash_state`. No methods within `Self` would cause
                // `self.state` to point to an invalid representation of a
                // `crypto_generichash_state` struct. Therefore, after the copy, `state` must also
                // point to a valid representation of a `crypto_generichash_state` struct, and can
                // be used with the multipart hash functions from Sodium.
                ptr::copy_nonoverlapping(self.state.as_ptr(), state.as_mut(), 1);

                state
            };

            Ok(Self {
                output_len: self.output_len,
                state,
                _marker: PhantomData,
            })
        }

        /// Get the output length specified when instantiating this struct.
        ///
        /// The output buffer provided to [`Self::calculate`] must be this length.
        pub fn get_output_len(&self) -> usize {
            self.output_len
        }

        /// Add message contents to hash.
        pub fn update(&mut self, chunk: &[u8]) {
            // We do not use `require_init` here, as it must be called to initialise a `Multipart`
            // struct.

            let update_result = unsafe {
                // SAFETY: The first argument to this function should be a pointer to a
                // `crypto_generichash_state` struct. `self.state` must be correctly initialised,
                // and point to a valid representation of a `crypto_generichash_state` struct to
                // instantiate the `Multipart` struct, so it is in the correct state to use with
                // this function. The next two arguments specify the portion of the message to add
                // to the hash calculation, and its length. We use `chunk.len()` to specify the
                // length of the message to read from `chunk`, so `chunk` is definitely valid for
                // reads of the specified length.
                sodium::crypto_generichash_blake2b_update(
                    self.state.as_mut(),
                    chunk.as_ptr(),
                    chunk.len() as libc::c_ulonglong,
                )
            };

            assert_not_err!(update_result, "crypto_generichash_blake2b_update");
        }

        /// Finalise the hash state, writing the hash to `output`.
        ///
        /// This method is not marked as public since we want to use the type system to ensure the
        /// `Multipart` struct cannot be used again after being finalised.
        fn finalise(&mut self, output: &mut [u8]) -> Result<(), AlkaliError> {
            // We do not use `require_init` here, as it must be called to initialise a `Multipart`
            // struct.

            if output.len() != self.output_len {
                return Err(GenericHashError::OutputLengthChanged.into());
            }

            let finalise_result = unsafe {
                // SAFETY: The first argument to this function should be a pointer to a
                // `crypto_generichash_state` struct. `self.state` must be correctly initialised,
                // and point to a valid representation of a `crypto_generichash_state` struct to
                // instantiate the `Multipart` struct, so it is in the correct state to use with
                // this function. The second and third arguments specify the destination to which
                // the hash should be written, and the length of the hash. We verify above that the
                // output length specified here is the same as the output length specified when
                // creating the struct. We then use `self.output_len` to specify the number of bytes
                // to write to `output`, which we have verified above is equal to `output.len()`, so
                // `output` is definitely valid for writes of the specified length.
                sodium::crypto_generichash_blake2b_final(
                    self.state.as_mut(),
                    output.as_mut_ptr(),
                    self.output_len,
                )
            };
            assert_not_err!(finalise_result, "crypto_generichash_blake2b_final");

            Ok(())
        }

        /// Calculate the hash for the specified message and write it to `output`.
        ///
        /// `output` must be the same length specified when instantiating the struct with
        /// [`Multipart::new`] or [`Multipart::new_keyed`], otherwise an error will be returned.
        pub fn calculate(mut self, output: &mut [u8]) -> Result<(), AlkaliError> {
            self.finalise(output)
        }

        /// Calculate the hash for the specified message and return it as a `Vec<u8>`.
        ///
        /// The returned vector will be the same length specified when instantiating the struct with
        /// [`Multipart::new`] or [`Multipart::new_keyed`].
        pub fn calculate_to_vec(mut self) -> Result<Vec<u8>, AlkaliError> {
            let mut output = vec![0u8; self.output_len];
            self.finalise(&mut output)?;
            Ok(output)
        }

        /// Compare the hash of the specified message to another hash, returning true if the message
        /// hashes to the given value, and false otherwise.
        ///
        /// `digest` is the hash against which the message will be compared. It must be the same
        /// length specified when instantiating the struct with [`Multipart::new`] or
        /// [`Multipart::new_keyed`], otherwise an error will be returned.
        ///
        /// This comparison runs in constant time for a given digest length.
        pub fn compare(mut self, digest: &[u8]) -> Result<bool, AlkaliError> {
            if digest.len() != self.output_len {
                return Err(GenericHashError::OutputLengthChanged.into());
            }

            let mut output = vec![0u8; self.output_len];
            self.finalise(&mut output)?;
            mem::eq(digest, &output)
        }
    }

    impl Drop for Multipart {
        fn drop(&mut self) {
            // We do not use `require_init` here, as it must be called to initialise a `Multipart`
            // struct.

            unsafe {
                // SAFETY:
                // * Is a double-free possible in safe code?
                //   * No: We only free in `drop`, which cannot be called manually, and is called
                //     exactly once when the struct is actually dropped. Once the value is dropped,
                //     there's no way to call the method again to cause a double free.
                // * Is a use-after-free possible in safe code?
                //   * No: We only ever free a buffer on drop, and after drop, none of the type's
                //     methods are accessible.
                // * Is a memory leak possible in safe code?
                //   * Yes: If the user uses something like `Box::leak()`, `ManuallyDrop`, or
                //     `std::mem::forget`, the destructor will not be called even though the struct
                //     is dropped. However, it is documented that in these cases heap memory may be
                //     leaked, so this is expected behaviour. In addition, certain signal interrupts
                //     or using panic=abort behaviour will mean the destructor is not called.
                //     There's little we can do about this, but a failure to free is probably
                //     reasonable in such cases. In any other case, `drop` will be called, and the
                //     memory freed.
                // `self.state` was allocated in the `Multipart` constructor using Sodium's
                // allocator, so it is correct to free it using Sodium's allocator.
                mem::free(self.state);
            }
        }
    }

    /// Calculate the hash of the provided message.
    ///
    /// This function returns the hash of `message`, of the default output size (equal to
    /// [`DIGEST_LENGTH_DEFAULT`]).
    ///
    /// A `key` may optionally be provided. If no key is provided, the unkeyed variant of BLAKE2b
    /// will be used, and the same message will always produce the same hash. If a key is provided,
    /// it will be used as part of the hash calculation. The same `(message, key)` pair will always
    /// produce the same hash, but a message hashed with a different keys will likely produce a
    /// different hash. The `key` may be of any length up to [`KEY_LENGTH_MAX`], but please see the
    /// security considerations section below.
    ///
    /// # Security Considerations
    /// If a key is used in the hash calculation, the key length must be chosen carefully. If the
    /// key is not a secret value, and is just being used to produce different hashes in different
    /// contexts, the key can be of any length. However, if the key is intended to be secret
    /// (thereby turning the hash into a MAC), it must be at least [`KEY_LENGTH_MIN`] bytes long.
    /// Shorter keys are insufficient to be cryptographic secrets. It is recommended to use a key of
    /// at least [`KEY_LENGTH_DEFAULT`] bytes, and the [`Key`] type exists for this purpose.
    pub fn hash(message: &[u8], key: Option<&[u8]>) -> Result<Digest, AlkaliError> {
        let mut digest = [0u8; DIGEST_LENGTH_DEFAULT];
        hash_custom(message, key, &mut digest)?;
        Ok(digest)
    }

    /// Calculate the hash of the provided message, outputting a digest of a custom length.
    ///
    /// The [`hash`] function will always output a hash of the default size,
    /// [`DIGEST_LENGTH_DEFAULT`]. This function allows the user to specify a custom output size, if
    /// this is necessary for your use case.
    ///
    /// The hash of `message` will be written to `digest`, which must be between
    /// [`DIGEST_LENGTH_MIN`] and [`DIGEST_LENGTH_MAX`] bytes.
    ///
    /// A `key` may optionally be provided. If no key is provided, the unkeyed variant of BLAKE2b
    /// will be used, and the same message will always produce the same hash. If a key is provided,
    /// it will be used as part of the hash calculation. The same `(message, key)` pair will always
    /// produce the same hash, but a message hashed with a different keys will likely produce a
    /// different hash. The `key` may be of any length up to [`KEY_LENGTH_MAX`], but please see the
    /// security considerations section below.
    ///
    /// # Security Considerations
    /// The output size of this function is user-configurable. However, digest lengths below
    /// [`DIGEST_LENGTH_DEFAULT`] may not preserve the expected property of collision resistance. If
    /// such a digest size is used, the function should no longer be treated as a cryptographic hash
    /// function. Only use an output length below [`DIGEST_LENGTH_DEFAULT`] if doing so is
    /// absolutely necessary for your use-case.
    ///
    /// If a key is used in the hash calculation, the key length must be chosen carefully. If the
    /// key is not a secret value, and is just being used to produce different hashes in different
    /// contexts, the key can be of any length. However, if the key is intended to be secret
    /// (thereby turning the hash into a MAC), it must be at least [`KEY_LENGTH_MIN`] bytes long.
    /// Shorter keys are insufficient to be cryptographic secrets. It is recommended to use a key of
    /// at least [`KEY_LENGTH_DEFAULT`] bytes, and the [`Key`] type exists for this purpose.
    pub fn hash_custom(
        message: &[u8],
        key: Option<&[u8]>,
        digest: &mut [u8],
    ) -> Result<(), AlkaliError> {
        require_init()?;

        if digest.len() < DIGEST_LENGTH_MIN || digest.len() > DIGEST_LENGTH_MAX {
            return Err(GenericHashError::DigestLengthInvalid.into());
        }

        let (key_ptr, key_len) = if let Some(key) = key {
            if key.len() > KEY_LENGTH_MAX {
                return Err(GenericHashError::KeyLengthInvalid.into());
            }
            (key.as_ptr(), key.len())
        } else {
            (ptr::null(), 0)
        };

        let hash_result = unsafe {
            // SAFETY: The first two arguments to this function specify the destination to which the
            // calculated hash will be written, and the length of the hash to write. The length must
            // be between `DIGEST_LENGTH_MIN` and `DIGEST_LENGTH_MAX`, which we verified above. We
            // use `digest.len()` to specify the length to write to `digest`, so the pointer is
            // definitely valid for writes of this length. The next two arguments specify a pointer
            // to the message to hash and its length. We use `message.len()` to specify the length
            // to read from `message`, so the pointer is definitely valid for reads of this length.
            // The last two arguments specify a pointer to the key and its length. If a key was
            // provided, we use `key.len()` to specify the length to read from `key`, so the
            // `key_ptr` is valid for reads of this length. If a key was not provided, we use a null
            // pointer to specify this. It is documented that in such a case, Sodium will simply
            // ignore the key & length.
            sodium::crypto_generichash_blake2b(
                digest.as_mut_ptr(),
                digest.len(),
                message.as_ptr(),
                message.len() as libc::c_ulonglong,
                key_ptr,
                key_len,
            )
        };
        assert_not_err!(hash_result, "crypto_generichash_blake2b");

        Ok(())
    }

    /// Calculate the hash of the provided message, returning a digest of a custom length stored in
    /// a vector.
    ///
    /// The [`hash`] function will always output a hash of the default size,
    /// [`DIGEST_LENGTH_DEFAULT`]. This function allows the user to specify a custom output size, if
    /// this is necessary for your use case.
    ///
    /// A `key` may optionally be provided. If no key is provided, the unkeyed variant of BLAKE2b
    /// will be used, and the same message will always produce the same hash. If a key is provided,
    /// it will be used as part of the hash calculation. The same `(message, key)` pair will always
    /// produce the same hash, but a message hashed with a different keys will likely produce a
    /// different hash. The `key` may be of any length up to [`KEY_LENGTH_MAX`], but please see the
    /// security considerations section below.
    ///
    /// The hash of `message` will be `digest_len` bytes, which must be between
    /// [`DIGEST_LENGTH_MIN`] and [`DIGEST_LENGTH_MAX`] bytes. This hash will be returned as a vector.
    ///
    /// # Security Considerations
    /// The output size of this function is user-configurable. However, digest lengths below
    /// [`DIGEST_LENGTH_DEFAULT`] may not preserve the expected property of collision resistance. If
    /// such a digest size is used, the function should no longer be treated as a cryptographic hash
    /// function. Only use an output length below [`DIGEST_LENGTH_DEFAULT`] if doing so is
    /// absolutely necessary for your use-case.
    ///
    /// If a key is used in the hash calculation, the key length must be chosen carefully. If the
    /// key is not a secret value, and is just being used to produce different hashes in different
    /// contexts, the key can be of any length. However, if the key is intended to be secret
    /// (thereby turning the hash into a MAC), it must be at least [`KEY_LENGTH_MIN`] bytes long.
    /// Shorter keys are insufficient to be cryptographic secrets. It is recommended to use a key of
    /// at least [`KEY_LENGTH_DEFAULT`] bytes, and the [`Key`] type exists for this purpose.
    pub fn hash_custom_to_vec(
        message: &[u8],
        key: Option<&[u8]>,
        digest_len: usize,
    ) -> Result<Vec<u8>, AlkaliError> {
        let mut digest = vec![0; digest_len];
        hash_custom(message, key, &mut digest)?;
        Ok(digest)
    }

    #[cfg(test)]
    mod tests {
        use super::{hash, hash_custom, Key, Multipart, DIGEST_LENGTH_DEFAULT};
        use crate::{random, AlkaliError};

        #[test]
        fn key_generation() -> Result<(), AlkaliError> {
            let _key = Key::generate()?;
            Ok(())
        }

        #[test]
        fn single_part_vectors() -> Result<(), AlkaliError> {
            let vectors = [
                (
                    &[][..],
                    [
                        0x0e, 0x57, 0x51, 0xc0, 0x26, 0xe5, 0x43, 0xb2, 0xe8, 0xab, 0x2e, 0xb0,
                        0x60, 0x99, 0xda, 0xa1, 0xd1, 0xe5, 0xdf, 0x47, 0x77, 0x8f, 0x77, 0x87,
                        0xfa, 0xab, 0x45, 0xcd, 0xf1, 0x2f, 0xe3, 0xa8,
                    ],
                ),
                (
                    &[0x00][..],
                    [
                        0x03, 0x17, 0x0a, 0x2e, 0x75, 0x97, 0xb7, 0xb7, 0xe3, 0xd8, 0x4c, 0x05,
                        0x39, 0x1d, 0x13, 0x9a, 0x62, 0xb1, 0x57, 0xe7, 0x87, 0x86, 0xd8, 0xc0,
                        0x82, 0xf2, 0x9d, 0xcf, 0x4c, 0x11, 0x13, 0x14,
                    ],
                ),
                (
                    &[
                        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b,
                        0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
                        0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
                    ][..],
                    [
                        0xcb, 0x2f, 0x51, 0x60, 0xfc, 0x1f, 0x7e, 0x05, 0xa5, 0x5e, 0xf4, 0x9d,
                        0x34, 0x0b, 0x48, 0xda, 0x2e, 0x5a, 0x78, 0x09, 0x9d, 0x53, 0x39, 0x33,
                        0x51, 0xcd, 0x57, 0x9d, 0xd4, 0x25, 0x03, 0xd6,
                    ],
                ),
                (
                    &[
                        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b,
                        0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
                        0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23,
                        0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
                        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b,
                        0x3c, 0x3d, 0x3e,
                    ][..],
                    [
                        0x29, 0xe4, 0x1a, 0x64, 0xfb, 0xdd, 0x2f, 0xd2, 0x76, 0x12, 0x22, 0x86,
                        0x23, 0xc0, 0x70, 0x22, 0x22, 0xbf, 0x36, 0x74, 0x51, 0xe7, 0x32, 0x42,
                        0x87, 0xf1, 0x81, 0xcb, 0x3d, 0xcf, 0x72, 0x37,
                    ],
                ),
                (
                    &[
                        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b,
                        0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
                        0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23,
                        0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
                        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b,
                        0x3c, 0x3d, 0x3e, 0x3f,
                    ][..],
                    [
                        0x10, 0xd8, 0xe6, 0xd5, 0x34, 0xb0, 0x09, 0x39, 0x84, 0x3f, 0xe9, 0xdc,
                        0xc4, 0xda, 0xe4, 0x8c, 0xdf, 0x00, 0x8f, 0x6b, 0x8b, 0x2b, 0x82, 0xb1,
                        0x56, 0xf5, 0x40, 0x4d, 0x87, 0x48, 0x87, 0xf5,
                    ],
                ),
                (
                    &[
                        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b,
                        0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
                        0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23,
                        0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
                        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b,
                        0x3c, 0x3d, 0x3e, 0x3f, 0x40,
                    ][..],
                    [
                        0x84, 0xc0, 0x4a, 0xb0, 0x82, 0xc8, 0xae, 0x24, 0x20, 0x65, 0x61, 0xf7,
                        0x73, 0x97, 0x70, 0x4b, 0x62, 0x78, 0x92, 0x08, 0x9a, 0x05, 0x88, 0x7a,
                        0x2a, 0x19, 0x96, 0x47, 0x2b, 0xcf, 0xe1, 0x5d,
                    ],
                ),
                (
                    &[
                        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b,
                        0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
                        0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23,
                        0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
                        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b,
                        0x3c, 0x3d, 0x3e, 0x3f, 0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,
                        0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f, 0x50, 0x51, 0x52, 0x53,
                        0x54, 0x55, 0x56, 0x57, 0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f,
                        0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a, 0x6b,
                        0x6c, 0x6d, 0x6e, 0x6f, 0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77,
                        0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f, 0x80, 0x81, 0x82, 0x83,
                        0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
                        0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9a, 0x9b,
                        0x9c, 0x9d, 0x9e, 0x9f, 0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7,
                        0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf, 0xb0, 0xb1, 0xb2, 0xb3,
                        0xb4, 0xb5, 0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,
                        0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xcb,
                        0xcc, 0xcd, 0xce, 0xcf, 0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7,
                        0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf, 0xe0, 0xe1, 0xe2, 0xe3,
                        0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef,
                        0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9, 0xfa, 0xfb,
                        0xfc, 0xfd, 0xfe,
                    ][..],
                    [
                        0x1d, 0x08, 0x50, 0xee, 0x9b, 0xca, 0x0a, 0xbc, 0x96, 0x01, 0xe9, 0xde,
                        0xab, 0xe1, 0x41, 0x8f, 0xed, 0xec, 0x2f, 0xb6, 0xac, 0x41, 0x50, 0xbd,
                        0x53, 0x02, 0xd2, 0x43, 0x0f, 0x9b, 0xe9, 0x43,
                    ],
                ),
            ];

            for v in vectors {
                let digest = hash(v.0, None)?;
                assert_eq!(digest, v.1);
            }

            Ok(())
        }

        #[test]
        fn single_part_custom_vectors() -> Result<(), AlkaliError> {
            let vectors = [
                (
                    &[][..],
                    [
                        0x78, 0x6a, 0x02, 0xf7, 0x42, 0x01, 0x59, 0x03, 0xc6, 0xc6, 0xfd, 0x85,
                        0x25, 0x52, 0xd2, 0x72, 0x91, 0x2f, 0x47, 0x40, 0xe1, 0x58, 0x47, 0x61,
                        0x8a, 0x86, 0xe2, 0x17, 0xf7, 0x1f, 0x54, 0x19, 0xd2, 0x5e, 0x10, 0x31,
                        0xaf, 0xee, 0x58, 0x53, 0x13, 0x89, 0x64, 0x44, 0x93, 0x4e, 0xb0, 0x4b,
                        0x90, 0x3a, 0x68, 0x5b, 0x14, 0x48, 0xb7, 0x55, 0xd5, 0x6f, 0x70, 0x1a,
                        0xfe, 0x9b, 0xe2, 0xce,
                    ],
                ),
                (
                    &[0x00][..],
                    [
                        0x2f, 0xa3, 0xf6, 0x86, 0xdf, 0x87, 0x69, 0x95, 0x16, 0x7e, 0x7c, 0x2e,
                        0x5d, 0x74, 0xc4, 0xc7, 0xb6, 0xe4, 0x8f, 0x80, 0x68, 0xfe, 0x0e, 0x44,
                        0x20, 0x83, 0x44, 0xd4, 0x80, 0xf7, 0x90, 0x4c, 0x36, 0x96, 0x3e, 0x44,
                        0x11, 0x5f, 0xe3, 0xeb, 0x2a, 0x3a, 0xc8, 0x69, 0x4c, 0x28, 0xbc, 0xb4,
                        0xf5, 0xa0, 0xf3, 0x27, 0x6f, 0x2e, 0x79, 0x48, 0x7d, 0x82, 0x19, 0x05,
                        0x7a, 0x50, 0x6e, 0x4b,
                    ],
                ),
                (
                    &[
                        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b,
                        0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
                        0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
                    ][..],
                    [
                        0x5c, 0x52, 0x92, 0x0a, 0x72, 0x63, 0xe3, 0x9d, 0x57, 0x92, 0x0c, 0xa0,
                        0xcb, 0x75, 0x2a, 0xc6, 0xd7, 0x9a, 0x04, 0xfe, 0xf8, 0xa7, 0xa2, 0x16,
                        0xa1, 0xec, 0xb7, 0x11, 0x5c, 0xe0, 0x6d, 0x89, 0xfd, 0x7d, 0x73, 0x5b,
                        0xd6, 0xf4, 0x27, 0x25, 0x55, 0xdb, 0xa2, 0x2c, 0x2d, 0x1c, 0x96, 0xe6,
                        0x35, 0x23, 0x22, 0xc6, 0x2c, 0x56, 0x30, 0xfd, 0xe0, 0xf4, 0x77, 0x7a,
                        0x76, 0xc3, 0xde, 0x2c,
                    ],
                ),
                (
                    &[
                        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b,
                        0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
                        0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23,
                        0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
                        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b,
                        0x3c, 0x3d, 0x3e,
                    ][..],
                    [
                        0xd1, 0x0b, 0xf9, 0xa1, 0x5b, 0x1c, 0x9f, 0xc8, 0xd4, 0x1f, 0x89, 0xbb,
                        0x14, 0x0b, 0xf0, 0xbe, 0x08, 0xd2, 0xf3, 0x66, 0x61, 0x76, 0xd1, 0x3b,
                        0xaa, 0xc4, 0xd3, 0x81, 0x35, 0x8a, 0xd0, 0x74, 0xc9, 0xd4, 0x74, 0x8c,
                        0x30, 0x05, 0x20, 0xeb, 0x02, 0x6d, 0xae, 0xae, 0xa7, 0xc5, 0xb1, 0x58,
                        0x89, 0x2f, 0xde, 0x4e, 0x8e, 0xc1, 0x7d, 0xc9, 0x98, 0xdc, 0xd5, 0x07,
                        0xdf, 0x26, 0xeb, 0x63,
                    ],
                ),
                (
                    &[
                        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b,
                        0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
                        0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23,
                        0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
                        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b,
                        0x3c, 0x3d, 0x3e, 0x3f,
                    ][..],
                    [
                        0x2f, 0xc6, 0xe6, 0x9f, 0xa2, 0x6a, 0x89, 0xa5, 0xed, 0x26, 0x90, 0x92,
                        0xcb, 0x9b, 0x2a, 0x44, 0x9a, 0x44, 0x09, 0xa7, 0xa4, 0x40, 0x11, 0xee,
                        0xca, 0xd1, 0x3d, 0x7c, 0x4b, 0x04, 0x56, 0x60, 0x2d, 0x40, 0x2f, 0xa5,
                        0x84, 0x4f, 0x1a, 0x7a, 0x75, 0x81, 0x36, 0xce, 0x3d, 0x5d, 0x8d, 0x0e,
                        0x8b, 0x86, 0x92, 0x1f, 0xff, 0xf4, 0xf6, 0x92, 0xdd, 0x95, 0xbd, 0xc8,
                        0xe5, 0xff, 0x00, 0x52,
                    ],
                ),
                (
                    &[
                        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b,
                        0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
                        0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23,
                        0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
                        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b,
                        0x3c, 0x3d, 0x3e, 0x3f, 0x40,
                    ][..],
                    [
                        0xfc, 0xbe, 0x8b, 0xe7, 0xdc, 0xb4, 0x9a, 0x32, 0xdb, 0xdf, 0x23, 0x94,
                        0x59, 0xe2, 0x63, 0x08, 0xb8, 0x4d, 0xff, 0x1e, 0xa4, 0x80, 0xdf, 0x8d,
                        0x10, 0x4e, 0xef, 0xf3, 0x4b, 0x46, 0xfa, 0xe9, 0x86, 0x27, 0xb4, 0x50,
                        0xc2, 0x26, 0x7d, 0x48, 0xc0, 0x94, 0x6a, 0x69, 0x7c, 0x5b, 0x59, 0x53,
                        0x14, 0x52, 0xac, 0x04, 0x84, 0xf1, 0xc8, 0x4e, 0x3a, 0x33, 0xd0, 0xc3,
                        0x39, 0xbb, 0x2e, 0x28,
                    ],
                ),
                (
                    &[
                        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b,
                        0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
                        0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23,
                        0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
                        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b,
                        0x3c, 0x3d, 0x3e, 0x3f, 0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,
                        0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f, 0x50, 0x51, 0x52, 0x53,
                        0x54, 0x55, 0x56, 0x57, 0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f,
                        0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a, 0x6b,
                        0x6c, 0x6d, 0x6e, 0x6f, 0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77,
                        0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f, 0x80, 0x81, 0x82, 0x83,
                        0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
                        0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9a, 0x9b,
                        0x9c, 0x9d, 0x9e, 0x9f, 0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7,
                        0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf, 0xb0, 0xb1, 0xb2, 0xb3,
                        0xb4, 0xb5, 0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,
                        0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xcb,
                        0xcc, 0xcd, 0xce, 0xcf, 0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7,
                        0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf, 0xe0, 0xe1, 0xe2, 0xe3,
                        0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef,
                        0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9, 0xfa, 0xfb,
                        0xfc, 0xfd, 0xfe,
                    ][..],
                    [
                        0x5b, 0x21, 0xc5, 0xfd, 0x88, 0x68, 0x36, 0x76, 0x12, 0x47, 0x4f, 0xa2,
                        0xe7, 0x0e, 0x9c, 0xfa, 0x22, 0x01, 0xff, 0xee, 0xe8, 0xfa, 0xfa, 0xb5,
                        0x79, 0x7a, 0xd5, 0x8f, 0xef, 0xa1, 0x7c, 0x9b, 0x5b, 0x10, 0x7d, 0xa4,
                        0xa3, 0xdb, 0x63, 0x20, 0xba, 0xaf, 0x2c, 0x86, 0x17, 0xd5, 0xa5, 0x1d,
                        0xf9, 0x14, 0xae, 0x88, 0xda, 0x38, 0x67, 0xc2, 0xd4, 0x1f, 0x0c, 0xc1,
                        0x4f, 0xa6, 0x79, 0x28,
                    ],
                ),
            ];
            let mut output = [0u8; 64];

            for v in vectors {
                hash_custom(v.0, None, &mut output)?;
                assert_eq!(output, v.1);
            }

            Ok(())
        }

        #[test]
        fn single_part_keyed_vectors() -> Result<(), AlkaliError> {
            let mut key = Key::new_empty()?;
            key.copy_from_slice(&[
                0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d,
                0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b,
                0x1c, 0x1d, 0x1e, 0x1f,
            ]);

            let vectors = [
                (
                    &[][..],
                    [
                        0x4e, 0x51, 0xe7, 0xa9, 0x13, 0xfc, 0x80, 0x13, 0x7d, 0xa5, 0x28, 0x80,
                        0xfe, 0xcc, 0xa1, 0x75, 0xbf, 0x81, 0xe1, 0x17, 0xd5, 0xc6, 0x81, 0x26,
                        0xdc, 0x27, 0x74, 0x03, 0x35, 0x17, 0xea, 0x0d,
                    ],
                ),
                (
                    &[0x00][..],
                    [
                        0x41, 0xff, 0x93, 0xa4, 0xea, 0xee, 0xbd, 0x3b, 0x78, 0xa9, 0x34, 0x38,
                        0xa6, 0xf6, 0x2a, 0x92, 0xab, 0x59, 0x59, 0xc8, 0x59, 0xe6, 0x82, 0xb7,
                        0x2c, 0x7d, 0xef, 0x40, 0x61, 0x97, 0xca, 0x4d,
                    ],
                ),
                (
                    &[
                        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b,
                        0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
                        0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
                    ][..],
                    [
                        0x9e, 0xa4, 0xa7, 0xc5, 0xf7, 0x8b, 0x4c, 0x5b, 0x77, 0x75, 0x84, 0x1e,
                        0x2a, 0x07, 0xdb, 0x51, 0x4c, 0x51, 0x0b, 0x14, 0xf5, 0x00, 0x6d, 0x27,
                        0xae, 0xe5, 0xe2, 0x0f, 0x84, 0x06, 0xbd, 0xd8,
                    ],
                ),
                (
                    &[
                        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b,
                        0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
                        0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23,
                        0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
                        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b,
                        0x3c, 0x3d, 0x3e,
                    ][..],
                    [
                        0x2f, 0x3d, 0x96, 0xae, 0x02, 0xf0, 0xa3, 0x3f, 0xa3, 0x33, 0x4e, 0xd7,
                        0x3a, 0x04, 0x49, 0x1a, 0xbc, 0x99, 0x36, 0x2d, 0x1e, 0x99, 0xc1, 0x9f,
                        0xcc, 0x85, 0x94, 0x6f, 0x41, 0x7, 0xc5, 0x28,
                    ],
                ),
                (
                    &[
                        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b,
                        0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
                        0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23,
                        0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
                        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b,
                        0x3c, 0x3d, 0x3e, 0x3f,
                    ][..],
                    [
                        0x7f, 0x46, 0xf0, 0x43, 0xd6, 0x3d, 0xfb, 0x3c, 0x9b, 0x11, 0x02, 0xc8,
                        0x82, 0x4b, 0x63, 0x2c, 0xae, 0x1f, 0xce, 0x45, 0x3e, 0x3c, 0x0e, 0x9d,
                        0xc9, 0xe7, 0x11, 0x3f, 0x3c, 0x1a, 0xcc, 0x36,
                    ],
                ),
                (
                    &[
                        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b,
                        0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
                        0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23,
                        0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
                        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b,
                        0x3c, 0x3d, 0x3e, 0x3f, 0x40,
                    ][..],
                    [
                        0x22, 0x11, 0x1b, 0xcd, 0x34, 0x28, 0x24, 0xa7, 0x49, 0xda, 0x73, 0xe2,
                        0x1c, 0xa9, 0x88, 0x49, 0x52, 0x14, 0x71, 0x0c, 0x81, 0xbb, 0x4a, 0x05,
                        0xeb, 0xf3, 0xc4, 0x43, 0xe1, 0x7b, 0xe8, 0xc5,
                    ],
                ),
                (
                    &[
                        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b,
                        0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
                        0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23,
                        0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
                        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b,
                        0x3c, 0x3d, 0x3e, 0x3f, 0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,
                        0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f, 0x50, 0x51, 0x52, 0x53,
                        0x54, 0x55, 0x56, 0x57, 0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f,
                        0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a, 0x6b,
                        0x6c, 0x6d, 0x6e, 0x6f, 0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77,
                        0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f, 0x80, 0x81, 0x82, 0x83,
                        0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
                        0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9a, 0x9b,
                        0x9c, 0x9d, 0x9e, 0x9f, 0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7,
                        0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf, 0xb0, 0xb1, 0xb2, 0xb3,
                        0xb4, 0xb5, 0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,
                        0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xcb,
                        0xcc, 0xcd, 0xce, 0xcf, 0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7,
                        0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf, 0xe0, 0xe1, 0xe2, 0xe3,
                        0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef,
                        0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9, 0xfa, 0xfb,
                        0xfc, 0xfd, 0xfe,
                    ][..],
                    [
                        0xf6, 0x0f, 0x5c, 0x4a, 0x57, 0x5c, 0x43, 0xdb, 0x6c, 0x93, 0x60, 0x85,
                        0x15, 0x86, 0x6e, 0xa9, 0x98, 0xe0, 0x4b, 0x35, 0x79, 0x2f, 0x5a, 0x92,
                        0xb2, 0x7a, 0xa5, 0x88, 0x0c, 0xd6, 0x72, 0x1e,
                    ],
                ),
            ];

            for v in vectors {
                let digest = hash(v.0, Some(key.as_slice()))?;
                assert_eq!(digest, v.1);
            }

            Ok(())
        }

        #[test]
        fn single_part_keyed_custom_vectors() -> Result<(), AlkaliError> {
            let key = [
                0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d,
                0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b,
                0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29,
                0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, 0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
                0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f,
            ];
            let vectors = [
                (
                    &[][..],
                    [
                        0x10, 0xeb, 0xb6, 0x77, 0x00, 0xb1, 0x86, 0x8e, 0xfb, 0x44, 0x17, 0x98,
                        0x7a, 0xcf, 0x46, 0x90, 0xae, 0x9d, 0x97, 0x2f, 0xb7, 0xa5, 0x90, 0xc2,
                        0xf0, 0x28, 0x71, 0x79, 0x9a, 0xaa, 0x47, 0x86, 0xb5, 0xe9, 0x96, 0xe8,
                        0xf0, 0xf4, 0xeb, 0x98, 0x1f, 0xc2, 0x14, 0xb0, 0x05, 0xf4, 0x2d, 0x2f,
                        0xf4, 0x23, 0x34, 0x99, 0x39, 0x16, 0x53, 0xdf, 0x7a, 0xef, 0xcb, 0xc1,
                        0x3f, 0xc5, 0x15, 0x68,
                    ],
                ),
                (
                    &[0x00][..],
                    [
                        0x96, 0x1f, 0x6d, 0xd1, 0xe4, 0xdd, 0x30, 0xf6, 0x39, 0x01, 0x69, 0x0c,
                        0x51, 0x2e, 0x78, 0xe4, 0xb4, 0x5e, 0x47, 0x42, 0xed, 0x19, 0x7c, 0x3c,
                        0x5e, 0x45, 0xc5, 0x49, 0xfd, 0x25, 0xf2, 0xe4, 0x18, 0x7b, 0x0b, 0xc9,
                        0xfe, 0x30, 0x49, 0x2b, 0x16, 0xb0, 0xd0, 0xbc, 0x4e, 0xf9, 0xb0, 0xf3,
                        0x4c, 0x70, 0x03, 0xfa, 0xc0, 0x9a, 0x5e, 0xf1, 0x53, 0x2e, 0x69, 0x43,
                        0x02, 0x34, 0xce, 0xbd,
                    ],
                ),
                (
                    &[
                        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b,
                        0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
                        0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
                    ][..],
                    [
                        0x86, 0x22, 0x1f, 0x3a, 0xda, 0x52, 0x03, 0x7b, 0x72, 0x22, 0x4f, 0x10,
                        0x5d, 0x79, 0x99, 0x23, 0x1c, 0x5e, 0x55, 0x34, 0xd0, 0x3d, 0xa9, 0xd9,
                        0xc0, 0xa1, 0x2a, 0xcb, 0x68, 0x46, 0x0c, 0xd3, 0x75, 0xda, 0xf8, 0xe2,
                        0x43, 0x86, 0x28, 0x6f, 0x96, 0x68, 0xf7, 0x23, 0x26, 0xdb, 0xf9, 0x9b,
                        0xa0, 0x94, 0x39, 0x24, 0x37, 0xd3, 0x98, 0xe9, 0x5b, 0xb8, 0x16, 0x1d,
                        0x71, 0x7f, 0x89, 0x91,
                    ],
                ),
                (
                    &[
                        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b,
                        0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
                        0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23,
                        0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
                        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b,
                        0x3c, 0x3d, 0x3e,
                    ][..],
                    [
                        0xbd, 0x96, 0x5b, 0xf3, 0x1e, 0x87, 0xd7, 0x03, 0x27, 0x53, 0x6f, 0x2a,
                        0x34, 0x1c, 0xeb, 0xc4, 0x76, 0x8e, 0xca, 0x27, 0x5f, 0xa0, 0x5e, 0xf9,
                        0x8f, 0x7f, 0x1b, 0x71, 0xa0, 0x35, 0x12, 0x98, 0xde, 0x00, 0x6f, 0xba,
                        0x73, 0xfe, 0x67, 0x33, 0xed, 0x01, 0xd7, 0x58, 0x01, 0xb4, 0xa9, 0x28,
                        0xe5, 0x42, 0x31, 0xb3, 0x8e, 0x38, 0xc5, 0x62, 0xb2, 0xe3, 0x3e, 0xa1,
                        0x28, 0x49, 0x92, 0xfa,
                    ],
                ),
                (
                    &[
                        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b,
                        0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
                        0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23,
                        0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
                        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b,
                        0x3c, 0x3d, 0x3e, 0x3f,
                    ][..],
                    [
                        0x65, 0x67, 0x6d, 0x80, 0x06, 0x17, 0x97, 0x2f, 0xbd, 0x87, 0xe4, 0xb9,
                        0x51, 0x4e, 0x1c, 0x67, 0x40, 0x2b, 0x7a, 0x33, 0x10, 0x96, 0xd3, 0xbf,
                        0xac, 0x22, 0xf1, 0xab, 0xb9, 0x53, 0x74, 0xab, 0xc9, 0x42, 0xf1, 0x6e,
                        0x9a, 0xb0, 0xea, 0xd3, 0x3b, 0x87, 0xc9, 0x19, 0x68, 0xa6, 0xe5, 0x09,
                        0xe1, 0x19, 0xff, 0x07, 0x78, 0x7b, 0x3e, 0xf4, 0x83, 0xe1, 0xdc, 0xdc,
                        0xcf, 0x6e, 0x30, 0x22,
                    ],
                ),
                (
                    &[
                        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b,
                        0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
                        0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23,
                        0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
                        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b,
                        0x3c, 0x3d, 0x3e, 0x3f, 0x40,
                    ][..],
                    [
                        0x93, 0x9f, 0xa1, 0x89, 0x69, 0x9c, 0x5d, 0x2c, 0x81, 0xdd, 0xd1, 0xff,
                        0xc1, 0xfa, 0x20, 0x7c, 0x97, 0x0b, 0x6a, 0x36, 0x85, 0xbb, 0x29, 0xce,
                        0x1d, 0x3e, 0x99, 0xd4, 0x2f, 0x2f, 0x74, 0x42, 0xda, 0x53, 0xe9, 0x5a,
                        0x72, 0x90, 0x73, 0x14, 0xf4, 0x58, 0x83, 0x99, 0xa3, 0xff, 0x5b, 0x0a,
                        0x92, 0xbe, 0xb3, 0xf6, 0xbe, 0x26, 0x94, 0xf9, 0xf8, 0x6e, 0xcf, 0x29,
                        0x52, 0xd5, 0xb4, 0x1c,
                    ],
                ),
                (
                    &[
                        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b,
                        0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
                        0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23,
                        0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
                        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b,
                        0x3c, 0x3d, 0x3e, 0x3f, 0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,
                        0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f, 0x50, 0x51, 0x52, 0x53,
                        0x54, 0x55, 0x56, 0x57, 0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f,
                        0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a, 0x6b,
                        0x6c, 0x6d, 0x6e, 0x6f, 0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77,
                        0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f, 0x80, 0x81, 0x82, 0x83,
                        0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
                        0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9a, 0x9b,
                        0x9c, 0x9d, 0x9e, 0x9f, 0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7,
                        0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf, 0xb0, 0xb1, 0xb2, 0xb3,
                        0xb4, 0xb5, 0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,
                        0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xcb,
                        0xcc, 0xcd, 0xce, 0xcf, 0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7,
                        0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf, 0xe0, 0xe1, 0xe2, 0xe3,
                        0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef,
                        0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9, 0xfa, 0xfb,
                        0xfc, 0xfd, 0xfe,
                    ][..],
                    [
                        0x14, 0x27, 0x09, 0xd6, 0x2e, 0x28, 0xfc, 0xcc, 0xd0, 0xaf, 0x97, 0xfa,
                        0xd0, 0xf8, 0x46, 0x5b, 0x97, 0x1e, 0x82, 0x20, 0x1d, 0xc5, 0x10, 0x70,
                        0xfa, 0xa0, 0x37, 0x2a, 0xa4, 0x3e, 0x92, 0x48, 0x4b, 0xe1, 0xc1, 0xe7,
                        0x3b, 0xa1, 0x09, 0x06, 0xd5, 0xd1, 0x85, 0x3d, 0xb6, 0xa4, 0x10, 0x6e,
                        0x0a, 0x7b, 0xf9, 0x80, 0x0d, 0x37, 0x3d, 0x6d, 0xee, 0x2d, 0x46, 0xd6,
                        0x2e, 0xf2, 0xa4, 0x61,
                    ],
                ),
            ];
            let mut output = [0u8; 64];

            for v in vectors {
                hash_custom(v.0, Some(&key), &mut output)?;
                assert_eq!(output, v.1);
            }

            Ok(())
        }

        #[test]
        fn multi_part_vectors() -> Result<(), AlkaliError> {
            let vectors = [
                (
                    &[][..],
                    [
                        0x0e, 0x57, 0x51, 0xc0, 0x26, 0xe5, 0x43, 0xb2, 0xe8, 0xab, 0x2e, 0xb0,
                        0x60, 0x99, 0xda, 0xa1, 0xd1, 0xe5, 0xdf, 0x47, 0x77, 0x8f, 0x77, 0x87,
                        0xfa, 0xab, 0x45, 0xcd, 0xf1, 0x2f, 0xe3, 0xa8,
                    ],
                ),
                (
                    &[0x00][..],
                    [
                        0x03, 0x17, 0x0a, 0x2e, 0x75, 0x97, 0xb7, 0xb7, 0xe3, 0xd8, 0x4c, 0x05,
                        0x39, 0x1d, 0x13, 0x9a, 0x62, 0xb1, 0x57, 0xe7, 0x87, 0x86, 0xd8, 0xc0,
                        0x82, 0xf2, 0x9d, 0xcf, 0x4c, 0x11, 0x13, 0x14,
                    ],
                ),
                (
                    &[
                        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b,
                        0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
                        0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
                    ][..],
                    [
                        0xcb, 0x2f, 0x51, 0x60, 0xfc, 0x1f, 0x7e, 0x05, 0xa5, 0x5e, 0xf4, 0x9d,
                        0x34, 0x0b, 0x48, 0xda, 0x2e, 0x5a, 0x78, 0x09, 0x9d, 0x53, 0x39, 0x33,
                        0x51, 0xcd, 0x57, 0x9d, 0xd4, 0x25, 0x03, 0xd6,
                    ],
                ),
                (
                    &[
                        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b,
                        0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
                        0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23,
                        0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
                        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b,
                        0x3c, 0x3d, 0x3e,
                    ][..],
                    [
                        0x29, 0xe4, 0x1a, 0x64, 0xfb, 0xdd, 0x2f, 0xd2, 0x76, 0x12, 0x22, 0x86,
                        0x23, 0xc0, 0x70, 0x22, 0x22, 0xbf, 0x36, 0x74, 0x51, 0xe7, 0x32, 0x42,
                        0x87, 0xf1, 0x81, 0xcb, 0x3d, 0xcf, 0x72, 0x37,
                    ],
                ),
                (
                    &[
                        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b,
                        0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
                        0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23,
                        0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
                        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b,
                        0x3c, 0x3d, 0x3e, 0x3f,
                    ][..],
                    [
                        0x10, 0xd8, 0xe6, 0xd5, 0x34, 0xb0, 0x09, 0x39, 0x84, 0x3f, 0xe9, 0xdc,
                        0xc4, 0xda, 0xe4, 0x8c, 0xdf, 0x00, 0x8f, 0x6b, 0x8b, 0x2b, 0x82, 0xb1,
                        0x56, 0xf5, 0x40, 0x4d, 0x87, 0x48, 0x87, 0xf5,
                    ],
                ),
                (
                    &[
                        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b,
                        0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
                        0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23,
                        0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
                        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b,
                        0x3c, 0x3d, 0x3e, 0x3f, 0x40,
                    ][..],
                    [
                        0x84, 0xc0, 0x4a, 0xb0, 0x82, 0xc8, 0xae, 0x24, 0x20, 0x65, 0x61, 0xf7,
                        0x73, 0x97, 0x70, 0x4b, 0x62, 0x78, 0x92, 0x08, 0x9a, 0x05, 0x88, 0x7a,
                        0x2a, 0x19, 0x96, 0x47, 0x2b, 0xcf, 0xe1, 0x5d,
                    ],
                ),
                (
                    &[
                        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b,
                        0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
                        0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23,
                        0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
                        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b,
                        0x3c, 0x3d, 0x3e, 0x3f, 0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,
                        0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f, 0x50, 0x51, 0x52, 0x53,
                        0x54, 0x55, 0x56, 0x57, 0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f,
                        0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a, 0x6b,
                        0x6c, 0x6d, 0x6e, 0x6f, 0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77,
                        0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f, 0x80, 0x81, 0x82, 0x83,
                        0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
                        0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9a, 0x9b,
                        0x9c, 0x9d, 0x9e, 0x9f, 0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7,
                        0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf, 0xb0, 0xb1, 0xb2, 0xb3,
                        0xb4, 0xb5, 0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,
                        0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xcb,
                        0xcc, 0xcd, 0xce, 0xcf, 0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7,
                        0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf, 0xe0, 0xe1, 0xe2, 0xe3,
                        0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef,
                        0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9, 0xfa, 0xfb,
                        0xfc, 0xfd, 0xfe,
                    ][..],
                    [
                        0x1d, 0x08, 0x50, 0xee, 0x9b, 0xca, 0x0a, 0xbc, 0x96, 0x01, 0xe9, 0xde,
                        0xab, 0xe1, 0x41, 0x8f, 0xed, 0xec, 0x2f, 0xb6, 0xac, 0x41, 0x50, 0xbd,
                        0x53, 0x02, 0xd2, 0x43, 0x0f, 0x9b, 0xe9, 0x43,
                    ],
                ),
            ];

            for v in vectors {
                let mut state = Multipart::new(DIGEST_LENGTH_DEFAULT)?;
                state.update(&v.0);
                let state_b = state.try_clone()?;
                assert!(state_b.compare(&v.1)?);
                let mut digest = [0u8; DIGEST_LENGTH_DEFAULT];
                state.calculate(&mut digest)?;
                assert_eq!(digest, v.1);

                for _ in 0..1000 {
                    let mut state = Multipart::new(DIGEST_LENGTH_DEFAULT)?;
                    let boundary = random::random_u32_in_range(0, v.0.len() as u32)? as usize;
                    state.update(&v.0[..boundary]);
                    state.update(&v.0[boundary..]);
                    let state_b = state.try_clone()?;
                    assert!(state_b.compare(&v.1)?);
                    let mut digest = [0u8; DIGEST_LENGTH_DEFAULT];
                    state.calculate(&mut digest)?;
                    assert_eq!(digest, v.1);

                    let mut state = Multipart::new(DIGEST_LENGTH_DEFAULT)?;
                    let boundary_a = random::random_u32_in_range(0, v.0.len() as u32)? as usize;
                    let boundary_b =
                        random::random_u32_in_range(boundary_a as u32, v.0.len() as u32)? as usize;
                    state.update(&v.0[..boundary_a]);
                    state.update(&v.0[boundary_a..boundary_b]);
                    state.update(&v.0[boundary_b..]);
                    let state_b = state.try_clone()?;
                    assert!(state_b.compare(&v.1)?);
                    let mut digest = [0u8; DIGEST_LENGTH_DEFAULT];
                    state.calculate(&mut digest)?;
                    assert_eq!(digest, v.1);
                }
            }

            Ok(())
        }

        #[test]
        fn multi_part_keyed_vectors() -> Result<(), AlkaliError> {
            let key = [
                0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d,
                0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b,
                0x1c, 0x1d, 0x1e, 0x1f,
            ];

            let vectors = [
                (
                    &[][..],
                    [
                        0x4e, 0x51, 0xe7, 0xa9, 0x13, 0xfc, 0x80, 0x13, 0x7d, 0xa5, 0x28, 0x80,
                        0xfe, 0xcc, 0xa1, 0x75, 0xbf, 0x81, 0xe1, 0x17, 0xd5, 0xc6, 0x81, 0x26,
                        0xdc, 0x27, 0x74, 0x03, 0x35, 0x17, 0xea, 0x0d,
                    ],
                ),
                (
                    &[0x00][..],
                    [
                        0x41, 0xff, 0x93, 0xa4, 0xea, 0xee, 0xbd, 0x3b, 0x78, 0xa9, 0x34, 0x38,
                        0xa6, 0xf6, 0x2a, 0x92, 0xab, 0x59, 0x59, 0xc8, 0x59, 0xe6, 0x82, 0xb7,
                        0x2c, 0x7d, 0xef, 0x40, 0x61, 0x97, 0xca, 0x4d,
                    ],
                ),
                (
                    &[
                        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b,
                        0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
                        0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
                    ][..],
                    [
                        0x9e, 0xa4, 0xa7, 0xc5, 0xf7, 0x8b, 0x4c, 0x5b, 0x77, 0x75, 0x84, 0x1e,
                        0x2a, 0x07, 0xdb, 0x51, 0x4c, 0x51, 0x0b, 0x14, 0xf5, 0x00, 0x6d, 0x27,
                        0xae, 0xe5, 0xe2, 0x0f, 0x84, 0x06, 0xbd, 0xd8,
                    ],
                ),
                (
                    &[
                        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b,
                        0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
                        0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23,
                        0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
                        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b,
                        0x3c, 0x3d, 0x3e,
                    ][..],
                    [
                        0x2f, 0x3d, 0x96, 0xae, 0x02, 0xf0, 0xa3, 0x3f, 0xa3, 0x33, 0x4e, 0xd7,
                        0x3a, 0x04, 0x49, 0x1a, 0xbc, 0x99, 0x36, 0x2d, 0x1e, 0x99, 0xc1, 0x9f,
                        0xcc, 0x85, 0x94, 0x6f, 0x41, 0x7, 0xc5, 0x28,
                    ],
                ),
                (
                    &[
                        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b,
                        0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
                        0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23,
                        0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
                        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b,
                        0x3c, 0x3d, 0x3e, 0x3f,
                    ][..],
                    [
                        0x7f, 0x46, 0xf0, 0x43, 0xd6, 0x3d, 0xfb, 0x3c, 0x9b, 0x11, 0x02, 0xc8,
                        0x82, 0x4b, 0x63, 0x2c, 0xae, 0x1f, 0xce, 0x45, 0x3e, 0x3c, 0x0e, 0x9d,
                        0xc9, 0xe7, 0x11, 0x3f, 0x3c, 0x1a, 0xcc, 0x36,
                    ],
                ),
                (
                    &[
                        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b,
                        0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
                        0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23,
                        0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
                        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b,
                        0x3c, 0x3d, 0x3e, 0x3f, 0x40,
                    ][..],
                    [
                        0x22, 0x11, 0x1b, 0xcd, 0x34, 0x28, 0x24, 0xa7, 0x49, 0xda, 0x73, 0xe2,
                        0x1c, 0xa9, 0x88, 0x49, 0x52, 0x14, 0x71, 0x0c, 0x81, 0xbb, 0x4a, 0x05,
                        0xeb, 0xf3, 0xc4, 0x43, 0xe1, 0x7b, 0xe8, 0xc5,
                    ],
                ),
                (
                    &[
                        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b,
                        0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
                        0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23,
                        0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
                        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b,
                        0x3c, 0x3d, 0x3e, 0x3f, 0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,
                        0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f, 0x50, 0x51, 0x52, 0x53,
                        0x54, 0x55, 0x56, 0x57, 0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f,
                        0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a, 0x6b,
                        0x6c, 0x6d, 0x6e, 0x6f, 0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77,
                        0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f, 0x80, 0x81, 0x82, 0x83,
                        0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
                        0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9a, 0x9b,
                        0x9c, 0x9d, 0x9e, 0x9f, 0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7,
                        0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf, 0xb0, 0xb1, 0xb2, 0xb3,
                        0xb4, 0xb5, 0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,
                        0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xcb,
                        0xcc, 0xcd, 0xce, 0xcf, 0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7,
                        0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf, 0xe0, 0xe1, 0xe2, 0xe3,
                        0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef,
                        0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9, 0xfa, 0xfb,
                        0xfc, 0xfd, 0xfe,
                    ][..],
                    [
                        0xf6, 0x0f, 0x5c, 0x4a, 0x57, 0x5c, 0x43, 0xdb, 0x6c, 0x93, 0x60, 0x85,
                        0x15, 0x86, 0x6e, 0xa9, 0x98, 0xe0, 0x4b, 0x35, 0x79, 0x2f, 0x5a, 0x92,
                        0xb2, 0x7a, 0xa5, 0x88, 0x0c, 0xd6, 0x72, 0x1e,
                    ],
                ),
            ];

            for v in vectors {
                let mut state = Multipart::new_keyed(DIGEST_LENGTH_DEFAULT, &key)?;
                state.update(&v.0);
                let state_b = state.try_clone()?;
                assert!(state_b.compare(&v.1)?);
                let mut digest = [0u8; DIGEST_LENGTH_DEFAULT];
                state.calculate(&mut digest)?;
                assert_eq!(digest, v.1);

                for _ in 0..1000 {
                    let mut state = Multipart::new_keyed(DIGEST_LENGTH_DEFAULT, &key)?;
                    let boundary = random::random_u32_in_range(0, v.0.len() as u32)? as usize;
                    state.update(&v.0[..boundary]);
                    state.update(&v.0[boundary..]);
                    let state_b = state.try_clone()?;
                    assert!(state_b.compare(&v.1)?);
                    let mut digest = [0u8; DIGEST_LENGTH_DEFAULT];
                    state.calculate(&mut digest)?;
                    assert_eq!(digest, v.1);

                    let mut state = Multipart::new_keyed(DIGEST_LENGTH_DEFAULT, &key)?;
                    let boundary_a = random::random_u32_in_range(0, v.0.len() as u32)? as usize;
                    let boundary_b =
                        random::random_u32_in_range(boundary_a as u32, v.0.len() as u32)? as usize;
                    state.update(&v.0[..boundary_a]);
                    state.update(&v.0[boundary_a..boundary_b]);
                    state.update(&v.0[boundary_b..]);
                    let state_b = state.try_clone()?;
                    assert!(state_b.compare(&v.1)?);
                    let mut digest = [0u8; DIGEST_LENGTH_DEFAULT];
                    state.calculate(&mut digest)?;
                    assert_eq!(digest, v.1);
                }
            }

            Ok(())
        }
    }
}

pub use blake2b::*;
