use std::fmt::Debug;

use tokio::sync::{mpsc, oneshot};

use crate::message;

/// Remote to a [`crate::ResponseManager`] task. can send responses and await requests that will be matched by ids ([`u64`])
#[derive(Debug, Clone)]
pub struct ResponseManagerRemote<R>
where
	R: Send + Debug,
{
	sender: mpsc::UnboundedSender<message::Message<R>>,
}

impl<R> ResponseManagerRemote<R>
where
	R: Send + Debug,
{
	/// Constructor, can also be generated by calling [`crate::ResponseManagerHandle`]::remote ot clonning an existing one.
	pub fn new(sender: mpsc::UnboundedSender<message::Message<R>>) -> Self {
		Self { sender }
	}

	/// Waits for a response of the same id to be received and returns it.
	pub async fn request(&self, request_id: u64) -> R {
		let (resolve, receive) = oneshot::channel();
		let signal = message::Message::Request(message::Request {
			request_id,
			resolve,
		});
		self.sender.send(signal).unwrap();
		receive.await.unwrap()
	}

	/// Send the response to the task.
	pub async fn receive(&self, request_id: u64, signal: R) {
		let signal = message::Message::Receive(message::Receive { request_id, signal });
		self.sender.send(signal).unwrap();
	}
}
