use std::io::{Read, Write};

#[derive(serde_derive::Serialize, serde_derive::Deserialize, Default)]
pub struct Config {
    pub default: Profile,
    pub profiles: Option<Vec<Profile>>,
}

#[derive(serde_derive::Serialize, serde_derive::Deserialize, Default)]
pub struct Profile {
    pub geometry_x: i32,
    pub geometry_y: i32,
    pub geometry_w: i32,
    pub geometry_h: i32,
    //
    pub decorated: bool,
    pub sticky: bool,
    pub above: bool,
    pub below: bool,
    //
    pub transparent: bool,
    pub opaque_back: bool,
}

pub struct ConfigFile {
    pub path: std::path::PathBuf,
    pub err: Result<(), Box<dyn std::error::Error>>,
    pub conf: Config,
}

impl ConfigFile {
    pub fn is_ok(&self) -> bool {
        self.err.is_ok()
    }
    pub fn is_err(&self) -> bool {
        self.err.is_err()
    }
    pub fn load_from_config_file<P: AsRef<std::path::Path>>(a_path: P) -> ConfigFile {
        let path_buf = a_path.as_ref().to_path_buf();
        match ConfigFile::load_from_file(&path_buf) {
            Ok(c_conf) => ConfigFile {
                path: path_buf,
                err: Ok(()),
                conf: c_conf,
            },
            Err(c_err) => ConfigFile {
                path: path_buf,
                err: Err(c_err),
                conf: Config::default(),
            },
        }
    }
    fn load_from_file<P: AsRef<std::path::Path>>(
        a_path: P,
    ) -> Result<Config, Box<dyn std::error::Error>> {
        let mut s = String::new();
        let mut f = std::fs::File::open(a_path)?;
        let _sz = f.read_to_string(&mut s)?;
        let conf: Config = toml::from_str(&s)?;
        Ok(conf)
    }
    pub fn save_to_config_file(&self) -> Result<(), Box<dyn std::error::Error>> {
        self.save_to_file(&self.path)
    }
    fn save_to_file<P: AsRef<std::path::Path>>(
        &self,
        path: P,
    ) -> Result<(), Box<dyn std::error::Error>> {
        let mut f = std::fs::OpenOptions::new()
            .write(true)
            .create(true)
            .truncate(true)
            .open(path)?;
        let s = toml::to_string_pretty(&self.conf)?;
        let header = "# Autogenerated config file\n";
        f.write_all(header.as_bytes())?;
        f.write_all(s.as_bytes())?;
        Ok(())
    }
}
