//! The GERG2008 equation of state.

const RGERG: f64 = 8.314_472;
pub const NC_GERG: usize = 21;
const MAXFLDS: usize = 21;
const MAXMDL: usize = 10;
const MAXTRMM: usize = 12;
const MAXTRMP: usize = 24;
const EPSILON: f64 = 1.0e-15;

// Molar masses [g/mol]
const MMI_GERG: [f64; MAXFLDS + 1] = [
    0.0, 16.042_46,  // Methane
    28.013_4,   // Nitrogen
    44.009_5,   // Carbon dioxide
    30.069_04,  // Ethane
    44.095_62,  // Propane
    58.122_2,   // Isobutane
    58.122_2,   // n-Butane
    72.148_78,  // Isopentane
    72.148_78,  // n-Pentane
    86.175_36,  // Hexane
    100.201_94, // Heptane
    114.228_52, // Octane
    128.255_1,  // Nonane
    142.281_68, // Decane
    2.015_88,   // Hydrogen
    31.998_8,   // Oxygen
    28.010_1,   // Carbon monoxide
    18.015_28,  // Water
    34.080_88,  // Hydrogen sulfide
    4.002_602,  // Helium
    39.948,     // Argon
];

const KPOL: [usize; MAXFLDS + 1] = [
    0, 6, 6, 4, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 5, 6, 6, 7, 6, 4, 6,
];

const KEXP: [usize; MAXFLDS + 1] = [
    0, 18, 18, 18, 18, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 9, 6, 6, 9, 6, 8, 6,
];

const KPOLIJ: [usize; MAXMDL + 1] = [0, 2, 5, 2, 3, 2, 3, 4, 6, 6, 10];

const KEXPIJ: [usize; MAXMDL + 1] = [0, 10, 4, 7, 3, 4, 3, 0, 6, 6, 0];

// Critical densities [mol/l]
const DC: [f64; MAXFLDS + 1] = [
    0.0,
    10.139_342_719,
    11.183_9,
    10.624_978_698,
    6.870_854_54,
    5.000_043_088,
    3.860_142_94,
    3.920_016_792,
    3.271,
    3.215_577_588,
    2.705_877_875,
    2.315_324_434,
    2.056_404_127,
    1.81,
    1.64,
    14.94,
    13.63,
    10.85,
    17.873_716_09,
    10.19,
    17.399,
    13.407_429_659,
];

// Critical temperatures [K]
const TC: [f64; MAXFLDS + 1] = [
    0.0, 190.564, 126.192, 304.128_2, 305.322, 369.825, 407.817, 425.125, 460.35, 469.7, 507.82,
    540.13, 569.32, 594.55, 617.7, 33.19, 154.595, 132.86, 647.096, 373.1, 5.195_3, 150.687,
];

const NOIK: [[f64; MAXTRMP + 1]; MAXFLDS + 1] = [
    [
        0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
        0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    ],
    [
        0.0,
        0.573_357_042_391_62,
        -1.676_068_752_373,
        0.234_052_918_349_16,
        -0.219_473_763_434_41,
        0.016_369_201_404_128,
        0.015_004_406_389_28,
        0.098_990_489_492_918,
        0.583_827_709_290_55,
        -0.747_868_675_603_9,
        0.300_333_028_579_74,
        0.209_855_438_065_68,
        -0.018_590_151_133_061,
        -0.157_825_583_390_49,
        0.127_167_352_207_91,
        -0.032_019_743_894_346,
        -0.068_049_729_364_536,
        0.024_291_412_853_736,
        0.005_144_045_163_944_4,
        -0.019_084_949_733_532,
        0.005_522_967_724_129_1,
        -0.004_419_739_297_608_5,
        0.040_061_416_708_429,
        -0.033_752_085_907_575,
        -0.002_512_765_821_335_7,
    ],
    [
        0.0,
        0.598_897_118_012_01,
        -1.694_155_748_073_1,
        0.245_797_361_917_18,
        -0.237_224_567_551_75,
        0.017_954_918_715_141,
        0.014_592_875_720_215,
        0.100_080_659_362_06,
        0.731_571_153_855_3,
        -0.883_722_723_363_66,
        0.318_876_602_467_08,
        0.207_664_917_287_99,
        -0.019_379_315_454_158,
        -0.169_366_415_549_83,
        0.135_468_460_417_01,
        -0.033_066_712_095_307,
        -0.060_690_817_018_557,
        0.012_797_548_292_871,
        0.005_874_366_410_729_9,
        -0.018_451_951_971_969,
        0.004_722_662_204_247_2,
        -0.005_202_407_968_059_9,
        0.043_563_505_956_635,
        -0.036_251_690_750_939,
        -0.002_897_402_686_654_3,
    ],
    [
        0.0,
        0.526_465_648_046_53,
        -1.499_572_504_259_2,
        0.273_297_867_337_82,
        0.129_495_000_227_86,
        0.154_040_883_418_41,
        -0.581_869_509_468_14,
        -0.180_224_948_382_96,
        -0.095_389_904_072_812,
        -0.008_048_681_931_767_9,
        -0.035_547_751_273_09,
        -0.280_790_148_824_05,
        -0.082_435_890_081_677,
        0.010_832_427_979_006,
        -0.006_707_399_316_109_7,
        -0.004_682_790_760_052_4,
        -0.028_359_911_832_177,
        0.019_500_174_744_098,
        -0.216_091_375_071_66,
        0.437_727_949_269_72,
        -0.221_307_901_135_93,
        0.015_190_189_957_331,
        -0.015_380_948_953_3,
        0.0,
        0.0,
    ],
    [
        0.0,
        0.635_967_804_507_14,
        -1.737_798_178_545_9,
        0.289_140_609_262_72,
        -0.337_142_768_456_94,
        0.022_405_964_699_561,
        0.015_715_424_886_913,
        0.114_506_342_537_45,
        1.061_204_937_974_5,
        -1.285_522_443_942_3,
        0.394_146_307_776_52,
        0.313_909_246_820_41,
        -0.021_592_277_117_247,
        -0.217_236_665_649_05,
        -0.289_995_744_394_89,
        0.423_211_730_257_3,
        0.046_434_100_259_26,
        -0.131_383_983_297_41,
        0.011_492_850_364_368,
        -0.033_387_688_429_909,
        0.015_183_171_583_644,
        -0.004_761_080_564_765_7,
        0.046_917_166_277_885,
        -0.039_401_755_804_649,
        -0.003_256_995_624_761_1,
    ],
    [
        0.0,
        1.040_397_310_735_8,
        -2.831_840_408_140_3,
        0.843_938_096_062_94,
        -0.076_559_591_850_023,
        0.094_697_373_057_28,
        0.000_247_964_754_970_06,
        0.277_437_604_228_7,
        -0.043_846_000_648_377,
        -0.269_910_647_843_5,
        -0.069_313_413_089_86,
        -0.029_632_145_981_653,
        0.014_040_126_751_38,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
    ],
    [
        0.0,
        1.042_933_158_91,
        -2.818_427_254_889_2,
        0.861_762_323_978_5,
        -0.106_136_194_524_87,
        0.098_615_749_302_134,
        0.000_239_482_086_823_22,
        0.303_300_048_569_5,
        -0.041_598_156_135_099,
        -0.299_919_374_700_58,
        -0.080_369_342_764_109,
        -0.029_761_373_251_151,
        0.013_059_630_303_14,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
    ],
    [
        0.0,
        1.062_627_741_145_5,
        -2.862_095_182_835,
        0.887_382_334_037_77,
        -0.125_705_811_553_45,
        0.102_863_087_081_06,
        0.000_253_580_406_026_54,
        0.323_252_002_339_82,
        -0.037_950_761_057_432,
        -0.325_348_020_144_52,
        -0.079_050_969_051_011,
        -0.020_636_720_547_775,
        0.005_705_380_933_475,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
    ],
    [
        0.0,
        1.096_3,
        -3.040_2,
        1.031_7,
        -0.154_1,
        0.115_35,
        0.000_298_09,
        0.395_71,
        -0.045_881,
        -0.358_04,
        -0.101_07,
        -0.035_484,
        0.018_156,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
    ],
    [
        0.0,
        1.096_864_309_800_1,
        -2.998_888_829_806_1,
        0.995_168_867_992_12,
        -0.161_707_085_585_39,
        0.113_344_600_727_75,
        0.000_267_605_951_507_48,
        0.409_798_819_869_31,
        -0.040_876_423_083_075,
        -0.381_694_824_694_47,
        -0.109_319_568_439_93,
        -0.032_073_223_327_99,
        0.016_877_016_216_975,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
    ],
    [
        0.0,
        1.055_323_801_366_1,
        -2.612_061_589_062_9,
        0.766_138_829_672_6,
        -0.297_703_206_224_59,
        0.118_799_077_333_58,
        0.000_279_228_610_626_17,
        0.463_475_898_441_05,
        0.011_433_196_980_297,
        -0.482_569_687_381_31,
        -0.093_750_558_924_659,
        -0.006_727_324_715_599_4,
        -0.005_114_158_358_542_8,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
    ],
    [
        0.0,
        1.054_374_764_526_2,
        -2.650_068_150_614_4,
        0.817_300_478_275_43,
        -0.304_513_912_534_28,
        0.122_538_687_108,
        0.000_272_664_727_439_28,
        0.498_658_256_816_7,
        -0.000_714_328_150_841_76,
        -0.542_368_955_254_5,
        -0.138_018_216_107_56,
        -0.006_159_528_738_001_1,
        0.000_486_025_103_930_22,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
    ],
    [
        0.0,
        1.072_254_487_563_3,
        -2.463_295_117_200_3,
        0.653_866_740_549_28,
        -0.363_249_740_856_28,
        0.127_132_696_267_6,
        0.000_307_135_727_779_3,
        0.526_568_569_875_4,
        0.019_362_862_857_653,
        -0.589_394_268_491_55,
        -0.140_699_639_919_34,
        -0.007_896_633_050_003_6,
        0.003_303_659_796_810_9,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
    ],
    [
        0.0,
        1.115_1,
        -2.702,
        0.834_16,
        -0.388_28,
        0.137_6,
        0.000_281_85,
        0.620_37,
        0.015_847,
        -0.617_26,
        -0.150_43,
        -0.012_982,
        0.004_432_5,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
    ],
    [
        0.0,
        1.046_1,
        -2.480_7,
        0.743_72,
        -0.525_79,
        0.153_15,
        0.000_328_65,
        0.841_78,
        0.055_424,
        -0.735_55,
        -0.185_07,
        -0.020_775,
        0.012_335,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
    ],
    [
        0.0,
        5.357_992_845_125_2,
        -6.205_025_253_059_5,
        0.138_302_413_270_86,
        -0.071_397_954_896_129,
        0.015_474_053_959_733,
        -0.149_768_064_057_71,
        -0.026_368_723_988_451,
        0.056_681_303_156_066,
        -0.060_063_958_030_436,
        -0.450_439_420_271_3,
        0.424_788_402_445,
        -0.021_997_640_827_139,
        -0.010_499_521_374_53,
        -0.002_895_590_286_681_6,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
    ],
    [
        0.0,
        0.888_782_863_697_01,
        -2.487_943_331_214_8,
        0.597_501_907_758_86,
        0.009_650_181_706_188_1,
        0.071_970_428_712_77,
        0.000_223_374_430_001_95,
        0.185_586_863_914_74,
        -0.038_129_368_035_76,
        -0.153_522_453_830_06,
        -0.026_726_814_910_919,
        -0.025_675_298_677_127,
        0.009_571_430_212_366_8,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
    ],
    [
        0.0,
        0.905_54,
        -2.451_5,
        0.531_49,
        0.024_173,
        0.072_156,
        0.000_188_18,
        0.194_05,
        -0.043_268,
        -0.127_78,
        -0.027_896,
        -0.034_154,
        0.016_329,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
    ],
    [
        0.0,
        0.827_284_087_495_86,
        -1.860_222_041_658_4,
        -1.119_900_961_374_4,
        0.156_357_539_760_56,
        0.873_758_448_590_25,
        -0.366_744_037_157_31,
        0.053_987_893_432_436,
        1.095_769_021_449_9,
        0.053_213_037_828_563,
        0.013_050_533_930_825,
        -0.410_795_204_344_76,
        0.146_374_433_441_2,
        -0.055_726_838_623_719,
        -0.011_201_774_143_8,
        -0.006_606_275_806_809_9,
        0.004_691_852_200_453_8,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
    ],
    [
        0.0,
        0.876_41,
        -2.036_7,
        0.216_34,
        -0.050_199,
        0.066_994,
        0.000_190_76,
        0.202_27,
        -0.004_534_8,
        -0.222_3,
        -0.034_714,
        -0.014_885,
        0.007_415_4,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
    ],
    [
        0.0,
        -0.455_790_240_067_37,
        1.251_639_075_492_5,
        -1.543_823_165_062_1,
        0.020_467_489_707_221,
        -0.344_762_123_807_81,
        -0.020_858_459_512_787,
        0.016_227_414_711_778,
        -0.057_471_818_200_892,
        0.019_462_416_430_715,
        -0.033_295_680_123_02,
        -0.010_863_577_372_367,
        -0.022_173_365_245_954,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
    ],
    [
        0.0,
        0.850_957_148_039_69,
        -2.400_322_294_348,
        0.541_278_414_764_66,
        0.016_919_770_692_538,
        0.068_825_965_019_035,
        0.000_214_280_328_153_38,
        0.174_298_953_219_92,
        -0.033_654_495_604_194,
        -0.135_267_998_576_91,
        -0.016_387_350_791_552,
        -0.024_987_666_851_475,
        0.008_876_920_481_570_9,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
    ],
];

const FIJ: [[f64; MAXFLDS + 1]; MAXFLDS + 1] = [
    [
        0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
        0.0, 0.0, 0.0, 0.0,
    ],
    [
        0.0,
        0.0,
        1.0,
        1.0,
        1.0,
        1.0,
        0.771_035_405_688,
        1.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        1.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
    ],
    [
        0.0, 1.0, 0.0, 1.0, 1.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
        0.0, 0.0, 0.0, 0.0,
    ],
    [
        0.0, 1.0, 1.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
        0.0, 0.0, 0.0, 0.0,
    ],
    [
        0.0,
        1.0,
        1.0,
        0.0,
        0.0,
        0.130_424_765_15,
        0.260_632_376_098,
        0.281_570_073_085,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
    ],
    [
        0.0,
        1.0,
        0.0,
        0.0,
        0.130_424_765_15,
        0.0,
        -0.055_160_977_102_4,
        0.031_257_260_048_9,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
    ],
    [
        0.0,
        0.771_035_405_688,
        0.0,
        0.0,
        0.260_632_376_098,
        -0.055_160_977_102_4,
        0.0,
        -0.055_124_029_300_9,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
    ],
    [
        0.0,
        1.0,
        0.0,
        0.0,
        0.281_570_073_085,
        0.031_257_260_048_9,
        -0.055_124_029_300_9,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
    ],
    [
        0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
        0.0, 0.0, 0.0, 0.0,
    ],
    [
        0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
        0.0, 0.0, 0.0, 0.0,
    ],
    [
        0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
        0.0, 0.0, 0.0, 0.0,
    ],
    [
        0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
        0.0, 0.0, 0.0, 0.0,
    ],
    [
        0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
        0.0, 0.0, 0.0, 0.0,
    ],
    [
        0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
        0.0, 0.0, 0.0, 0.0,
    ],
    [
        0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
        0.0, 0.0, 0.0, 0.0,
    ],
    [
        0.0, 1.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
        0.0, 0.0, 0.0, 0.0,
    ],
    [
        0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
        0.0, 0.0, 0.0, 0.0,
    ],
    [
        0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
        0.0, 0.0, 0.0, 0.0,
    ],
    [
        0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
        0.0, 0.0, 0.0, 0.0,
    ],
    [
        0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
        0.0, 0.0, 0.0, 0.0,
    ],
    [
        0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
        0.0, 0.0, 0.0, 0.0,
    ],
    [
        0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
        0.0, 0.0, 0.0, 0.0,
    ],
];

const MNUMB: [[usize; MAXFLDS + 1]; MAXFLDS + 1] = [
    [
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ],
    [
        0, 0, 3, 4, 1, 2, 10, 10, 0, 0, 0, 0, 0, 0, 0, 7, 0, 0, 0, 0, 0, 0,
    ],
    [
        0, 0, 0, 5, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ],
    [
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ],
    [
        0, 0, 0, 0, 0, 10, 10, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ],
    [
        0, 0, 0, 0, 0, 0, 10, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ],
    [
        0, 0, 0, 0, 0, 0, 0, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ],
    [
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ],
    [
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ],
    [
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ],
    [
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ],
    [
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ],
    [
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ],
    [
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ],
    [
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ],
    [
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ],
    [
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ],
    [
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ],
    [
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ],
    [
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ],
    [
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ],
    [
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ],
];

const TH0I: [[f64; 7 + 1]; MAXFLDS + 1] = [
    [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],
    [0.0, 0.0, 0.0, 0.0, 820.659, 178.41, 1062.82, 1090.53],
    [0.0, 0.0, 0.0, 0.0, 662.738, 680.562, 1740.06, 0.0],
    [0.0, 0.0, 0.0, 0.0, 919.306, 865.07, 483.553, 341.109],
    [0.0, 0.0, 0.0, 0.0, 559.314, 223.284, 1031.38, 1071.29],
    [0.0, 0.0, 0.0, 0.0, 479.856, 200.893, 955.312, 1027.29],
    [0.0, 0.0, 0.0, 0.0, 438.27, 198.018, 1905.02, 893.765],
    [0.0, 0.0, 0.0, 0.0, 468.27, 183.636, 1914.1, 903.185],
    [0.0, 0.0, 0.0, 0.0, 292.503, 910.237, 1919.37, 0.0],
    [0.0, 0.0, 0.0, 0.0, 178.67, 840.538, 1774.25, 0.0],
    [0.0, 0.0, 0.0, 0.0, 182.326, 859.207, 1826.59, 0.0],
    [0.0, 0.0, 0.0, 0.0, 169.789, 836.195, 1760.46, 0.0],
    [0.0, 0.0, 0.0, 0.0, 158.922, 815.064, 1693.07, 0.0],
    [0.0, 0.0, 0.0, 0.0, 156.854, 814.882, 1693.79, 0.0],
    [0.0, 0.0, 0.0, 0.0, 164.947, 836.264, 1750.24, 0.0],
    [0.0, 0.0, 0.0, 0.0, 228.734, 326.843, 1651.71, 1671.69],
    [0.0, 0.0, 0.0, 0.0, 2235.71, 1116.69, 0.0, 0.0],
    [0.0, 0.0, 0.0, 0.0, 1550.45, 704.525, 0.0, 0.0],
    [0.0, 0.0, 0.0, 0.0, 268.795, 1141.41, 2507.37, 0.0],
    [0.0, 0.0, 0.0, 0.0, 1833.63, 847.181, 0.0, 0.0],
    [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],
    [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],
];

const GVIJ: [[f64; MAXFLDS + 1]; MAXFLDS + 1] = [
    [
        0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
        0.0, 0.0, 0.0, 0.0,
    ],
    [
        0.0,
        0.098_625_722_368_187_76,
        0.095_133_493_812_036_16,
        0.096_578_065_843_442_47,
        0.121_067_150_981_627_4,
        0.149_629_560_902_829_6,
        0.177_142_878_531_499_7,
        0.168_549_571_767_053_5,
        0.246_023_759_456_518_3,
        0.197_246_344_040_166_7,
        0.206_928_570_333_252,
        0.251_286_096_120_934_1,
        0.270_120_969_792_858_5,
        0.302_011_423_603_743_2,
        0.332_721_926_904_087_9,
        0.083_288_329_374_180_65,
        0.085_374_935_369_851_47,
        0.095_686_188_154_656_49,
        0.120_849_666_248_266_4,
        0.103_620_728_234_878_7,
        0.067_170_435_977_880_04,
        0.090_333_316_493_750_66,
    ],
    [
        0.0,
        0.0,
        0.089_414_247_266_159_39,
        0.093_976_878_290_626_52,
        0.117_552_694_273_574_4,
        0.144_708_115_708_046_5,
        0.173_161_981_855_694_2,
        0.180_587_404_870_933_2,
        0.203_132_699_819_361_4,
        0.191_849_542_767_128,
        0.236_247_631_178_835_2,
        0.305_713_007_440_63,
        0.278_279_818_587_633_2,
        0.280_241_889_768_157_6,
        0.271_560_058_994_166_1,
        0.073_244_004_764_540_85,
        0.080_851_216_033_617_49,
        0.091_572_113_755_441,
        0.078_142_014_899_848_3,
        0.107_222_008_773_53,
        0.065_347_633_141_440_51,
        0.082_298_305_773_323_84,
    ],
    [
        0.0,
        0.0,
        0.0,
        0.094_117_835_755_118_8,
        0.122_154_704_865_701_8,
        0.146_706_031_228_471_7,
        0.189_768_243_875_887_7,
        0.231_887_380_214_763_9,
        0.212_840_930_480_359_3,
        0.198_634_807_894_710_2,
        0.171_534_431_724_811_6,
        0.311_475_305_284_069_2,
        0.270_827_762_158_326,
        0.252_442_496_313_592_6,
        0.327_183_634_074_859_9,
        0.083_128_424_899_598_18,
        0.083_312_516_354_425_94,
        0.093_138_458_258_295_96,
        0.107_420_354_855_897_8,
        0.089_237_461_600_278_98,
        0.054_357_340_421_292_35,
        0.087_151_255_876_559_04,
    ],
    [
        0.0,
        0.0,
        0.0,
        0.0,
        0.145_542_303_970_824_6,
        0.171_439_365_674_31,
        0.198_181_803_990_479_8,
        0.196_268_445_120_878_2,
        0.225_629_605_147_272_2,
        0.222_215_611_620_290_8,
        0.281_233_001_051_548_7,
        0.278_503_903_183_360_8,
        0.305_134_780_963_871_4,
        0.349_014_245_970_574_2,
        0.354_589_973_363_468_3,
        0.103_586_791_293_512_1,
        0.105_370_452_957_861_6,
        0.140_361_310_558_919_3,
        0.093_725_416_701_403_04,
        0.125_355_370_238_732_5,
        0.094_796_404_930_836_16,
        0.106_143_269_913_884_8,
    ],
    [
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.199_998_276_494_852_5,
        0.228_347_099_212_782_2,
        0.227_126_142_952_769_1,
        0.259_064_005_174_434_3,
        0.267_951_554_304_686_6,
        0.292_148_006_852_644_2,
        0.325_226_634_163_791_7,
        0.355_365_011_999_465_4,
        0.416_261_828_552_551_7,
        0.381_009_711_454_573_6,
        0.130_588_950_813_983_3,
        0.126_292_961_559_656_3,
        0.154_249_960_990_131,
        0.114_453_829_026_461_5,
        0.135_458_363_103_943_5,
        0.114_337_326_622_726_6,
        0.127_164_837_789_916,
    ],
    [
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.259_057_764_322_064_2,
        0.256_954_583_745_802_2,
        0.282_387_335_133_217_1,
        0.285_021_942_414_396_4,
        0.314_311_796_510_656_7,
        0.345_481_338_946_24,
        0.372_671_751_642_764_8,
        0.405_771_976_030_558_7,
        0.434_406_931_006_615_5,
        0.162_996_084_277_148_5,
        0.147_245_086_809_324_3,
        0.175_611_831_444_615_2,
        0.132_630_106_157_242_4,
        0.166_039_699_365_105_2,
        0.133_979_466_599_228_5,
        0.148_210_795_747_834_4,
    ],
    [
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.255_100_948_047_163_4,
        0.280_408_927_214_094_1,
        0.287_244_915_596_137_9,
        0.319_613_410_520_719_6,
        0.342_222_953_982_071_1,
        0.375_168_395_630_833_1,
        0.403_793_567_987_235_9,
        0.408_721_807_026_147_1,
        0.173_480_435_902_425_4,
        0.145_884_809_258_596_8,
        0.173_633_423_313_345_9,
        0.160_739_068_756_283_4,
        0.154_208_454_333_291_1,
        0.132_702_294_121_291_5,
        0.178_363_100_030_283_5,
    ],
    [
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.305_716_906_144_909_8,
        0.308_351_513_337_592,
        0.337_641_367_610_972_4,
        0.368_810_910_098_109_8,
        0.396_001_322_433_529_5,
        0.429_101_546_917_779_1,
        0.457_736_501_832_258_1,
        0.186_325_655_154_291_1,
        0.162_902_280_326_008_8,
        0.198_941_402_307_542_1,
        0.147_250_852_777_724_3,
        0.152_715_353_033_351_2,
        0.148_697_474_158_183,
        0.163_935_214_060_532_6,
    ],
    [
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.310_986_120_730_482,
        0.340_275_974_886_206_8,
        0.371_445_517_256_474_1,
        0.419_274_119_431_907_3,
        0.434_911_763_500_412_4,
        0.451_010_855_699_546_8,
        0.188_960_262_522_072_1,
        0.164_630_126_146_966_4,
        0.201_576_009_695_613_8,
        0.142_417_076_615_553_9,
        0.195_661_444_099_798_2,
        0.150_323_533_772_074_7,
        0.165_670_343_303_836,
    ],
    [
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.369_565_828_982_581_1,
        0.400_529_147_739_814_7,
        0.427_925_783_916_199_7,
        0.464_359_738_330_806_9,
        0.486_903_116_002_920_9,
        0.220_511_785_735_017_1,
        0.183_375_669_131_334_3,
        0.230_865_863_767_620_4,
        0.194_744_057_122_789_9,
        0.206_936_010_017_032_7,
        0.167_986_329_953_974_7,
        0.184_493_338_652_349,
    ],
    [
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.431_904_913_762_940_9,
        0.461_661_186_411_043_5,
        0.490_395_127_131_934_9,
        0.517_263_608_143_118_8,
        0.227_285_009_452_853_1,
        0.202_536_768_767_111_3,
        0.262_035_406_199_415_3,
        0.184_397_379_698_649_6,
        0.203_279_959_282_160_6,
        0.186_077_668_227_651_2,
        0.203_730_920_493_201_2,
    ],
    [
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.486_285_738_717_543_4,
        0.519_385_963_043_583_5,
        0.547_088_518_050_974_6,
        0.276_610_071_574_800_6,
        0.218_717_081_818_728_1,
        0.289_225_818_575_585_7,
        0.119_662_445_397_562_3,
        0.242_755_971_577_449_2,
        0.201_380_340_777_504_2,
        0.219_973_952_305_339_9,
    ],
    [
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.552_486_187_845_303_9,
        0.581_121_142_522_951_9,
        0.309_710_296_061_872_4,
        0.237_860_200_593_475_5,
        0.322_326_043_212_524_8,
        0.217_635_940_194_718_2,
        0.285_082_944_972_181_5,
        0.219_512_218_090_115_5,
        0.239_189_308_865_779_5,
    ],
    [
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.609_756_097_560_975_6,
        0.468_138_795_864_401_7,
        0.254_005_211_926_358_7,
        0.238_801_692_180_450_1,
        0.128_401_541_832_529_9,
        0.320_533_488_499_416_8,
        0.234_825_032_463_424_7,
        0.255_393_746_785_770_9,
    ],
    [
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.066_934_404_283_801_87,
        0.070_101_798_504_477_41,
        0.088_456_010_569_906_05,
        0.061_277_192_964_608_01,
        0.081_542_719_064_971_29,
        0.062_084_420_794_103_17,
        0.070_690_964_258_949_23,
    ],
    [
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.073_367_571_533_382_25,
        0.082_409_741_309_062_03,
        0.073_465_961_850_719_96,
        0.085_152_093_135_497_74,
        0.065_098_083_417_719_75,
        0.073_505_545_272_719_9,
    ],
    [
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.092_165_898_617_511_52,
        0.072_557_337_299_267_25,
        0.083_382_082_821_044_81,
        0.073_460_599_707_612_89,
        0.096_333_233_634_713_09,
    ],
    [
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.055_948_074_533_839_15,
        0.076_191_321_570_660_86,
        0.056_707_896_819_804_31,
        0.067_348_513_016_338_36,
    ],
    [
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.098_135_426_889_106_97,
        0.076_001_723_841_027_73,
        0.085_822_706_464_593_79,
    ],
    [
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.057_474_567_503_879_53,
        0.065_658_908_140_010_21,
    ],
    [
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.074_585_511_573_333_55,
    ],
];

const BTIJ: [[f64; MAXFLDS + 1]; MAXFLDS + 1] = [
    [
        0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
        0.0, 0.0, 0.0, 0.0,
    ],
    [
        0.0,
        1.0,
        0.996_201_274_447_368_8,
        1.045_760_847_547_76,
        0.992_686_437_173_633_9,
        0.979_467_106_104_893_1,
        0.961_018_981_461_851_5,
        0.988_383_751_673_508_1,
        1.0,
        0.984_317_025_862_625_6,
        0.964_019_205_395_971_3,
        0.955_372_393_883_277_8,
        0.916_756_048_962_226_3,
        0.898_167_074_243_799_4,
        0.879_427_245_310_619_3,
        1.0,
        1.0,
        0.974_981_928_491_239_8,
        1.130_679_010_535_891,
        1.022_303_050_787_581,
        1.0,
        0.981_990_387_032_227_1,
    ],
    [
        0.0,
        0.0,
        1.0,
        1.011_823_803_472_132,
        1.015_401_706_807_559,
        1.005_361_826_090_574,
        0.985_787_123_569_696_8,
        0.989_060_550_658_074_7,
        1.0,
        1.0,
        1.0,
        1.0,
        1.0,
        0.914_661_652_382_302_6,
        0.917_638_404_749_195_8,
        0.895_170_183_650_429_7,
        0.994_389_070_790_166_9,
        1.0,
        1.0,
        1.009_406_750_298_678,
        0.480_067_125_512_625_3,
        0.998_140_551_192_044_7,
    ],
    [
        0.0,
        0.0,
        0.0,
        1.0,
        1.027_934_702_719_095,
        1.068_371_416_575_41,
        1.047_224_395_384_351,
        1.036_672_193_386_368,
        1.038_729_823_111_436,
        1.054_730_632_930_632,
        1.0,
        1.023_752_023_121_105,
        1.060_263_102_417_008,
        1.015_436_354_877_504,
        1.040_458_732_428_864,
        0.887_967_349_904_837_9,
        1.0,
        1.0,
        0.994_751_315_306_562,
        1.032_326_273_851_984,
        0.590_404_182_284_416_9,
        0.993_037_886_124_456_8,
    ],
    [
        0.0,
        0.0,
        0.0,
        0.0,
        1.0,
        0.992_413_830_325_693_5,
        1.0,
        0.998_261_863_936_346_9,
        1.0,
        0.997_379_610_862_591,
        1.0,
        1.0,
        0.968_390_363_957_066,
        1.0,
        0.942_681_881_175_999_7,
        0.870_432_705_556_168_5,
        1.0,
        1.0,
        1.0,
        0.980_490_799_868_601_2,
        1.0,
        1.0,
    ],
    [
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        1.0,
        0.996_028_546_959_240_8,
        1.000_620_674_279_263,
        0.988_760_609_705_580_7,
        0.992_980_404_108_328_4,
        1.0,
        1.0,
        1.0,
        1.0,
        0.970_877_638_725_300_3,
        1.0,
        1.0,
        1.0,
        1.0,
        0.985_202_264_070_485_1,
        1.0,
        1.0,
    ],
    [
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        1.0,
        0.999_844_924_012_606_6,
        1.0,
        1.0,
        1.0,
        1.0,
        1.0,
        1.0,
        1.0,
        1.0,
        1.0,
        1.0,
        1.0,
        0.949_748_770_607_100_3,
        1.0,
        1.0,
    ],
    [
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        1.0,
        1.0,
        1.0,
        1.0,
        1.0,
        1.0,
        1.0,
        0.987_416_608_471_285,
        1.0,
        1.0,
        1.0,
        1.0,
        0.972_122_812_569_449_1,
        1.0,
        1.0,
    ],
    [
        0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
        1.0, 1.0, 1.0, 1.0,
    ],
    [
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        1.0,
        1.0,
        1.0,
        1.0,
        1.0,
        1.0,
        1.0,
        1.0,
        1.0,
        1.0,
        0.925_456_796_568_235_9,
        1.0,
        1.0,
    ],
    [
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        1.0,
        1.0,
        1.0,
        1.0,
        0.995_287_584_833_820_2,
        1.0,
        1.0,
        1.0,
        1.0,
        0.971_981_286_692_378_7,
        1.0,
        1.0,
    ],
    [
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        1.0,
        1.0,
        1.0,
        1.0,
        1.0,
        1.0,
        1.0,
        1.0,
        0.977_997_214_742_632,
        1.0,
        1.0,
    ],
    [
        0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
        1.0, 1.0, 1.0, 1.0,
    ],
    [
        0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.0, 1.0, 1.0, 1.0, 1.0,
        1.0, 1.0, 1.0, 1.0,
    ],
    [
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        1.0,
        1.133_837_562_661_612,
        1.0,
        1.101_648_891_523_215,
        0.804_900_135_122_903_8,
        0.946_906_898_054_336_6,
        1.0,
        1.0,
    ],
    [
        0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.0, 1.0, 1.0,
        1.0, 1.0, 1.0, 1.0,
    ],
    [
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        1.0,
        1.0,
        1.0,
        1.0,
        1.0,
        1.000_046_206_533_749,
    ],
    [
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        1.0,
        1.0,
        1.051_725_596_885_534,
        1.0,
        1.0,
    ],
    [
        0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
        1.0, 1.0, 1.0, 1.0,
    ],
    [
        0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
        0.0, 1.0, 1.0, 1.0,
    ],
    [
        0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
        0.0, 0.0, 1.0, 1.0,
    ],
    [
        0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
        0.0, 0.0, 0.0, 1.0,
    ],
];

const GTIJ: [[f64; MAXFLDS + 1]; MAXFLDS + 1] = [
    [
        0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
        0.0, 0.0, 0.0, 0.0,
    ],
    [
        0.0,
        190.564,
        151.570_147_988_173_6,
        240.196_046_603_165_1,
        252.274_820_131_493_7,
        288.652_309_907_548_5,
        317.318_455_699_613_3,
        331.530_711_111_505_1,
        352.135_440_823_683_6,
        370.784_127_000_142_7,
        406.402_115_624_185_6,
        432.701_006_110_287_2,
        457.053_959_702_641_6,
        487.604_671_752_71,
        504.567_599_178_678_1,
        107.573_991_329_273_3,
        163.057_867_108_428_7,
        155.146_178_058_228_4,
        289.687_738_412_709_7,
        259.129_371_659_019_4,
        99.421_923_036_735_12,
        166.218_143_769_241_7,
    ],
    [
        0.0,
        0.0,
        126.192,
        218.273_425_689_361,
        217.306_965_184_814_6,
        260.203_846_144_132_5,
        289.308_783_977_365_2,
        300.578_923_555_517_3,
        333.039_684_807_440_5,
        345.475_676_075_968_7,
        372.784_596_732_709,
        397.088_429_483_295_8,
        464.582_525_668_831_6,
        458.205_126_519_860_2,
        487.333_566_973_837_8,
        71.989_241_554_759_73,
        138.606_445_425_442_1,
        128.631_783_260_774_5,
        276.845_963_076_097_1,
        209.320_455_367_382_9,
        26.111_410_211_921_85,
        136.395_527_287_525_4,
    ],
    [
        0.0,
        0.0,
        0.0,
        304.128_2,
        278.349_613_161_909_6,
        315.023_370_975_912,
        335.162_841_549_422_2,
        333.705_391_360_057_9,
        366.560_473_210_836_7,
        380.091_785_835_649_9,
        408.190_755_691_022_4,
        429.019_537_786_016_4,
        460.364_447_848_114_4,
        488.775_432_616_089_3,
        506.441_515_868_393_9,
        168.796_522_287_274_4,
        216.833_344_020_240_6,
        201.013_613_101_202_7,
        343.104_175_002_480_4,
        316.934_343_085_291_7,
        97.964_609_025_244_19,
        224.202_340_996_615_1,
    ],
    [
        0.0,
        0.0,
        0.0,
        0.0,
        305.322,
        339.683_277_970_995_1,
        364.612_318_395_696_9,
        372.503_001_190_061_1,
        382.835_999_844_791_7,
        403.411_145_251_601_4,
        430.058_259_476_988_1,
        460.728_368_874_730_6,
        479.470_433_993_767_8,
        449.935_999_966_161_9,
        521.659_808_690_4,
        178.633_426_235_147,
        217.258_497_164_092_5,
        215.350_144_480_998_3,
        444.491_445_263_010_6,
        301.699_447_408_447,
        39.827_620_900_576_02,
        214.494_886_218_762_8,
    ],
    [
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        369.825,
        389.619_876_908_913_5,
        399.567_380_174_618_8,
        411.628_460_380_588_1,
        418.781_654_109_911_3,
        444.483_251_724_503_6,
        469.304_642_456_162_8,
        488.082_364_018_104_1,
        520.481_221_712_897_1,
        537.302_377_459_437_8,
        255.727_865_724_955,
        239.108_962_347_712_9,
        265.456_928_259_797_4,
        293.684_121_732_059_2,
        333.979_489_823_783_4,
        43.833_227_379_466_36,
        236.067_405_151_579_5,
    ],
    [
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        407.817,
        416.945_345_945_802_8,
        434.083_500_063_419_4,
        438.758_500_146_207_3,
        457.818_499_864_190_2,
        473.973_500_206_221_4,
        488.568_499_849_367,
        501.183_500_125_529_2,
        512.758_499_810_912,
        220.503_500_017_977_6,
        251.090_559_589_563_2,
        270.338_500_015_977_1,
        513.708_817_747_953_4,
        356.245_912_039_186_2,
        46.029_682_381_046_25,
        247.896_591_906_786_8,
    ],
    [
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        425.125,
        442.737_500_122_883_7,
        447.814_938_721_707_9,
        468.891_096_111_637,
        489.388_473_029_385_5,
        508.291_184_946_544_3,
        509.837_499_809_891_4,
        548.148_240_839_282_5,
        298.062_680_480_104_1,
        256.363_412_707_429_9,
        278.992_499_897_548_7,
        322.834_311_071_452_8,
        363.677_088_246_845_7,
        46.996_296_795_598_7,
        315.122_450_578_332_2,
    ],
    [
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        460.35,
        465.024_999_778_386,
        484.085_000_164_041_5,
        500.240_000_165_009_2,
        514.834_999_877_052_8,
        527.449_999_975_204_8,
        539.024_999_907_433,
        246.770_000_019_091,
        266.772_952_620_763_2,
        296.605_000_068_418_2,
        545.793_590_654_928_8,
        407.245_442_229_103,
        48.904_563_744_092_43,
        263.379_498_917_436_6,
    ],
    [
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        469.7,
        488.759_999_762_539_1,
        504.915_000_190_436_1,
        525.608_893_801_269,
        583.103_991_171_831_1,
        565.053_287_389_948,
        251.444_999_986_778_6,
        269.468_498_158_875_7,
        301.280_000_016_606_9,
        246.802_048_210_432_4,
        386.232_949_986_275_4,
        49.398_708_586_358_81,
        266.040_756_088_235_5,
    ],
    [
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        507.82,
        523.601_663_232_348_9,
        538.569_999_908_750_8,
        579.900_747_724_270_4,
        574.920_574_338_087_7,
        392.227_147_980_458_8,
        280.189_994_289_589_2,
        320.339_999_906_665_5,
        326.566_210_866_854_2,
        410.287_900_604_373_7,
        51.364_163_051_684_19,
        276.625_870_699_036_4,
    ],
    [
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        540.13,
        553.863_707_361_212_5,
        567.338_864_171_355_6,
        578.902_362_673_500_6,
        424.321_091_901_073_1,
        288.966_083_390_421_4,
        336.495_000_092_165_9,
        591.198_750_404_633_4,
        449.918_898_068_286_9,
        52.972_987_352_045_76,
        285.290_324_599_345_6,
    ],
    [
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        569.32,
        581.935_000_162_335_3,
        597.278_410_109_460_9,
        301.254_999_962_992_2,
        296.671_578_348_853_6,
        351.090_000_045_770_4,
        401.853_825_430_948_4,
        460.883_165_238_219_5,
        54.385_551_353_277_6,
        292.897_802_723_065_9,
    ],
    [
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        594.55,
        606.124_999_839_865_3,
        313.870_000_006_970_3,
        303.173_971_920_414,
        363.705_000_059_025_9,
        620.266_819_038_387_7,
        511.752_131_742_767_5,
        55.577_563_953_451_57,
        299.317_483_368_412_7,
    ],
    [
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        617.7,
        577.229_986_231_689_6,
        309.019_953_239_269,
        542.300_428_206_455_3,
        419.971_857_684_438_9,
        515.588_884_486_030_2,
        56.649_243_684_271_72,
        305.089_101_575_261_1,
    ],
    [
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        33.19,
        71.631_055_066_919_12,
        91.473_186_945_338_13,
        146.550_729_237_353_2,
        111.279_778_037_161_8,
        13.131_336_832_173_6,
        70.719_880_726_709_36,
    ],
    [
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        154.595,
        143.316_055_276_441_4,
        305.144_073_078_044_1,
        240.165_348_249_908_8,
        28.340_208_247_294_16,
        151.171_394_957_527_4,
    ],
    [
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        132.86,
        293.211_825_409_549_2,
        233.523_371_243_601_1,
        26.272_562_836_541_09,
        135.014_868_008_322_6,
    ],
    [
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        647.096,
        462.163_871_373_824,
        57.981_530_238_516_46,
        334.416_846_789_014,
    ],
    [
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        373.1,
        44.026_883_037_526_05,
        237.110_353_422_198_7,
    ],
    [
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        5.195_3,
        27.979_709_989_562_07,
    ],
    [
        0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
        0.0, 0.0, 0.0, 150.687,
    ],
];

/// Implements the GERG2008 equation of state described in
/// AGA Report No. 8, Part 2, First Edition, April 2017.
#[derive(Default)]
pub struct Gerg2008 {
    /// Temperature in K
    pub t: f64,
    /// Pressure in kPa
    pub p: f64,
    /// Molar concentration in mol/l
    pub d: f64,
    /// Compressibility factor
    pub z: f64,
    /// Molar mass in g/mol
    pub mm: f64,
    /// First derivative of pressure with respect
    /// to density at constant temperature in kPa/(mol/l)
    pub dp_dd: f64,
    /// Second derivative of pressure with respect
    /// to density at constant temperature in kPa/(mol/l)^2
    pub d2p_dd2: f64,
    /// Second derivative of pressure with respect to
    /// temperature and density in kPa/(mol/l)/K (currently not calculated)
    pub d2p_dtd: f64,
    /// First derivative of pressure with respect to
    /// temperature at constant density in kPa/K
    pub dp_dt: f64,
    /// Internal energy in J/mol
    pub u: f64,
    /// Enthalpy in J/mol
    pub h: f64,
    /// Entropy in J/(mol-K)
    pub s: f64,
    /// Isochoric heat capacity in J/(mol-K)
    pub cv: f64,
    /// Isobaric heat capacity in J/(mol-K)
    pub cp: f64,
    /// Speed of sound in m/s
    pub w: f64,
    /// Gibbs energy in J/mol
    pub g: f64,
    /// Joule-Thomson coefficient in K/kPa
    pub jt: f64,
    /// Isentropic Exponent
    pub kappa: f64,
    /// Composition in mole fractions
    pub x: [f64; NC_GERG + 1],

    drold: f64,
    trold: f64,
    told: f64,
    trold2: f64,
    xold: [f64; NC_GERG + 1],
    a: f64,
    a0: [f64; 3],
    ar: [[f64; 4]; 4],
    dpddsave: f64,
    taup: [[f64; MAXTRMP + 1]; MAXFLDS + 1],
    taupijk: [[f64; MAXTRMM + 1]; MAXFLDS + 1],

    coik: [[usize; MAXTRMP + 1]; MAXFLDS + 1],
    doik: [[usize; MAXTRMP + 1]; MAXFLDS + 1],
    toik: [[f64; MAXTRMP + 1]; MAXFLDS + 1],
    dijk: [[usize; MAXTRMM + 1]; MAXTRMM + 1],
    tijk: [[f64; MAXTRMM + 1]; MAXMDL + 1],
    cijk: [[f64; MAXTRMM + 1]; MAXMDL + 1],
    eijk: [[f64; MAXTRMM + 1]; MAXMDL + 1],
    gijk: [[f64; MAXTRMM + 1]; MAXMDL + 1],
    nijk: [[f64; MAXTRMM + 1]; MAXMDL + 1],
    n0i: [[f64; 7 + 1]; MAXFLDS + 1],
    bvij: [[f64; MAXFLDS + 1]; MAXFLDS + 1],
    gvij: [[f64; MAXFLDS + 1]; MAXFLDS + 1],
    btij: [[f64; MAXFLDS + 1]; MAXFLDS + 1],
    gtij: [[f64; MAXFLDS + 1]; MAXFLDS + 1],
}

impl Gerg2008 {
    pub fn new() -> Self {
        Default::default()
    }

    pub fn setup(&mut self) {
        const RS: f64 = 8.31451;
        const RSR: f64 = RS / RGERG;
        let mut vc3 = [0.0; MAXFLDS + 1];
        let mut tc2 = [0.0; MAXFLDS + 1];
        let mut bijk = [[0.0; MAXTRMM + 1]; MAXMDL + 1];

        for i in 1..=MAXFLDS {
            vc3[i] = 1.0 / DC[i].powf(1.0 / 3.0) / 2.0;
            tc2[i] = TC[i].sqrt();
            self.coik[i][1] = 0;
            self.doik[i][1] = 1;
            self.toik[i][1] = 0.25;
            self.coik[i][2] = 0;
            self.doik[i][2] = 1;
            self.toik[i][2] = 1.125;
            self.coik[i][3] = 0;
            self.doik[i][3] = 1;
            self.toik[i][3] = 1.5;
            self.coik[i][4] = 0;
            self.doik[i][4] = 2;
            self.toik[i][4] = 1.375;
            self.coik[i][5] = 0;
            self.doik[i][5] = 3;
            self.toik[i][5] = 0.25;
            self.coik[i][6] = 0;
            self.doik[i][6] = 7;
            self.toik[i][6] = 0.875;
            self.coik[i][7] = 1;
            self.doik[i][7] = 2;
            self.toik[i][7] = 0.625;
            self.coik[i][8] = 1;
            self.doik[i][8] = 5;
            self.toik[i][8] = 1.75;
            self.coik[i][9] = 2;
            self.doik[i][9] = 1;
            self.toik[i][9] = 3.625;
            self.coik[i][10] = 2;
            self.doik[i][10] = 4;
            self.toik[i][10] = 3.625;
            self.coik[i][11] = 3;
            self.doik[i][11] = 3;
            self.toik[i][11] = 14.5;
            self.coik[i][12] = 3;
            self.doik[i][12] = 4;
            self.toik[i][12] = 12.0;
        }

        for i in 1..=4 {
            if i != 3 {
                self.coik[i][1] = 0;
                self.doik[i][1] = 1;
                self.toik[i][1] = 0.125;
                self.coik[i][2] = 0;
                self.doik[i][2] = 1;
                self.toik[i][2] = 1.125;
                self.coik[i][3] = 0;
                self.doik[i][3] = 2;
                self.toik[i][3] = 0.375;
                self.coik[i][4] = 0;
                self.doik[i][4] = 2;
                self.toik[i][4] = 1.125;
                self.coik[i][5] = 0;
                self.doik[i][5] = 4;
                self.toik[i][5] = 0.625;
                self.coik[i][6] = 0;
                self.doik[i][6] = 4;
                self.toik[i][6] = 1.5;
                self.coik[i][7] = 1;
                self.doik[i][7] = 1;
                self.toik[i][7] = 0.625;
                self.coik[i][8] = 1;
                self.doik[i][8] = 1;
                self.toik[i][8] = 2.625;
                self.coik[i][9] = 1;
                self.doik[i][9] = 1;
                self.toik[i][9] = 2.75;
                self.coik[i][10] = 1;
                self.doik[i][10] = 2;
                self.toik[i][10] = 2.125;
                self.coik[i][11] = 1;
                self.doik[i][11] = 3;
                self.toik[i][11] = 2.0;
                self.coik[i][12] = 1;
                self.doik[i][12] = 6;
                self.toik[i][12] = 1.75;
                self.coik[i][13] = 2;
                self.doik[i][13] = 2;
                self.toik[i][13] = 4.5;
                self.coik[i][14] = 2;
                self.doik[i][14] = 3;
                self.toik[i][14] = 4.75;
                self.coik[i][15] = 2;
                self.doik[i][15] = 3;
                self.toik[i][15] = 5.0;
                self.coik[i][16] = 2;
                self.doik[i][16] = 4;
                self.toik[i][16] = 4.0;
                self.coik[i][17] = 2;
                self.doik[i][17] = 4;
                self.toik[i][17] = 4.5;
                self.coik[i][18] = 3;
                self.doik[i][18] = 2;
                self.toik[i][18] = 7.5;
                self.coik[i][19] = 3;
                self.doik[i][19] = 3;
                self.toik[i][19] = 14.0;
                self.coik[i][20] = 3;
                self.doik[i][20] = 4;
                self.toik[i][20] = 11.5;
                self.coik[i][21] = 6;
                self.doik[i][21] = 5;
                self.toik[i][21] = 26.0;
                self.coik[i][22] = 6;
                self.doik[i][22] = 6;
                self.toik[i][22] = 28.0;
                self.coik[i][23] = 6;
                self.doik[i][23] = 6;
                self.toik[i][23] = 30.0;
                self.coik[i][24] = 6;
                self.doik[i][24] = 7;
                self.toik[i][24] = 16.0;
            }
        }

        self.coik[3][1] = 0;
        self.doik[3][1] = 1;
        self.toik[3][1] = 0.0;
        self.coik[3][2] = 0;
        self.doik[3][2] = 1;
        self.toik[3][2] = 1.25;
        self.coik[3][3] = 0;
        self.doik[3][3] = 2;
        self.toik[3][3] = 1.625;
        self.coik[3][4] = 0;
        self.doik[3][4] = 3;
        self.toik[3][4] = 0.375;
        self.coik[3][5] = 1;
        self.doik[3][5] = 3;
        self.toik[3][5] = 0.375;
        self.coik[3][6] = 1;
        self.doik[3][6] = 3;
        self.toik[3][6] = 1.375;
        self.coik[3][7] = 1;
        self.doik[3][7] = 4;
        self.toik[3][7] = 1.125;
        self.coik[3][8] = 1;
        self.doik[3][8] = 5;
        self.toik[3][8] = 1.375;
        self.coik[3][9] = 1;
        self.doik[3][9] = 6;
        self.toik[3][9] = 0.125;
        self.coik[3][10] = 1;
        self.doik[3][10] = 6;
        self.toik[3][10] = 1.625;
        self.coik[3][11] = 2;
        self.doik[3][11] = 1;
        self.toik[3][11] = 3.75;
        self.coik[3][12] = 2;
        self.doik[3][12] = 4;
        self.toik[3][12] = 3.5;
        self.coik[3][13] = 3;
        self.doik[3][13] = 1;
        self.toik[3][13] = 7.5;
        self.coik[3][14] = 3;
        self.doik[3][14] = 1;
        self.toik[3][14] = 8.0;
        self.coik[3][15] = 3;
        self.doik[3][15] = 3;
        self.toik[3][15] = 6.0;
        self.coik[3][16] = 3;
        self.doik[3][16] = 3;
        self.toik[3][16] = 16.0;
        self.coik[3][17] = 3;
        self.doik[3][17] = 4;
        self.toik[3][17] = 11.0;
        self.coik[3][18] = 5;
        self.doik[3][18] = 5;
        self.toik[3][18] = 24.0;
        self.coik[3][19] = 5;
        self.doik[3][19] = 5;
        self.toik[3][19] = 26.0;
        self.coik[3][20] = 5;
        self.doik[3][20] = 5;
        self.toik[3][20] = 28.0;
        self.coik[3][21] = 6;
        self.doik[3][21] = 5;
        self.toik[3][21] = 24.0;
        self.coik[3][22] = 6;
        self.doik[3][22] = 5;
        self.toik[3][22] = 26.0;
        // Hydrogen
        self.coik[15][1] = 0;
        self.doik[15][1] = 1;
        self.toik[15][1] = 0.5;
        self.coik[15][2] = 0;
        self.doik[15][2] = 1;
        self.toik[15][2] = 0.625;
        self.coik[15][3] = 0;
        self.doik[15][3] = 2;
        self.toik[15][3] = 0.375;
        self.coik[15][4] = 0;
        self.doik[15][4] = 2;
        self.toik[15][4] = 0.625;
        self.coik[15][5] = 0;
        self.doik[15][5] = 4;
        self.toik[15][5] = 1.125;
        self.coik[15][6] = 1;
        self.doik[15][6] = 1;
        self.toik[15][6] = 2.625;
        self.coik[15][7] = 1;
        self.doik[15][7] = 5;
        self.toik[15][7] = 0.0;
        self.coik[15][8] = 1;
        self.doik[15][8] = 5;
        self.toik[15][8] = 0.25;
        self.coik[15][9] = 1;
        self.doik[15][9] = 5;
        self.toik[15][9] = 1.375;
        self.coik[15][10] = 2;
        self.doik[15][10] = 1;
        self.toik[15][10] = 4.0;
        self.coik[15][11] = 2;
        self.doik[15][11] = 1;
        self.toik[15][11] = 4.25;
        self.coik[15][12] = 3;
        self.doik[15][12] = 2;
        self.toik[15][12] = 5.0;
        self.coik[15][13] = 3;
        self.doik[15][13] = 5;
        self.toik[15][13] = 8.0;
        self.coik[15][14] = 5;
        self.doik[15][14] = 1;
        self.toik[15][14] = 8.0;
        // Water
        self.coik[18][1] = 0;
        self.doik[18][1] = 1;
        self.toik[18][1] = 0.5;
        self.coik[18][2] = 0;
        self.doik[18][2] = 1;
        self.toik[18][2] = 1.25;
        self.coik[18][3] = 0;
        self.doik[18][3] = 1;
        self.toik[18][3] = 1.875;
        self.coik[18][4] = 0;
        self.doik[18][4] = 2;
        self.toik[18][4] = 0.125;
        self.coik[18][5] = 0;
        self.doik[18][5] = 2;
        self.toik[18][5] = 1.5;
        self.coik[18][6] = 0;
        self.doik[18][6] = 3;
        self.toik[18][6] = 1.0;
        self.coik[18][7] = 0;
        self.doik[18][7] = 4;
        self.toik[18][7] = 0.75;
        self.coik[18][8] = 1;
        self.doik[18][8] = 1;
        self.toik[18][8] = 1.5;
        self.coik[18][9] = 1;
        self.doik[18][9] = 5;
        self.toik[18][9] = 0.625;
        self.coik[18][10] = 1;
        self.doik[18][10] = 5;
        self.toik[18][10] = 2.625;
        self.coik[18][11] = 2;
        self.doik[18][11] = 1;
        self.toik[18][11] = 5.0;
        self.coik[18][12] = 2;
        self.doik[18][12] = 2;
        self.toik[18][12] = 4.0;
        self.coik[18][13] = 2;
        self.doik[18][13] = 4;
        self.toik[18][13] = 4.5;
        self.coik[18][14] = 3;
        self.doik[18][14] = 4;
        self.toik[18][14] = 3.0;
        self.coik[18][15] = 5;
        self.doik[18][15] = 1;
        self.toik[18][15] = 4.0;
        self.coik[18][16] = 5;
        self.doik[18][16] = 1;
        self.toik[18][16] = 6.0;
        // Helium
        self.coik[20][1] = 0;
        self.doik[20][1] = 1;
        self.toik[20][1] = 0.0;
        self.coik[20][2] = 0;
        self.doik[20][2] = 1;
        self.toik[20][2] = 0.125;
        self.coik[20][3] = 0;
        self.doik[20][3] = 1;
        self.toik[20][3] = 0.75;
        self.coik[20][4] = 0;
        self.doik[20][4] = 4;
        self.toik[20][4] = 1.0;
        self.coik[20][5] = 1;
        self.doik[20][5] = 1;
        self.toik[20][5] = 0.75;
        self.coik[20][6] = 1;
        self.doik[20][6] = 3;
        self.toik[20][6] = 2.625;
        self.coik[20][7] = 1;
        self.doik[20][7] = 5;
        self.toik[20][7] = 0.125;
        self.coik[20][8] = 1;
        self.doik[20][8] = 5;
        self.toik[20][8] = 1.25;
        self.coik[20][9] = 1;
        self.doik[20][9] = 5;
        self.toik[20][9] = 2.0;
        self.coik[20][10] = 2;
        self.doik[20][10] = 2;
        self.toik[20][10] = 1.0;
        self.coik[20][11] = 3;
        self.doik[20][11] = 1;
        self.toik[20][11] = 4.5;
        self.coik[20][12] = 3;
        self.doik[20][12] = 2;
        self.toik[20][12] = 5.0;

        // Exponents in mixture equations
        // Methane-Nitrogen
        self.dijk[3][1] = 1;
        self.tijk[3][1] = 0.0;
        self.cijk[3][1] = 0.0;
        self.eijk[3][1] = 0.0;
        bijk[3][1] = 0.0;
        self.gijk[3][1] = 0.0;
        self.nijk[3][1] = -9.803_898_551_733_5E-03;
        self.dijk[3][2] = 4;
        self.tijk[3][2] = 1.85;
        self.cijk[3][2] = 0.0;
        self.eijk[3][2] = 0.0;
        bijk[3][2] = 0.0;
        self.gijk[3][2] = 0.0;
        self.nijk[3][2] = 4.248_727_014_300_5E-04;
        self.dijk[3][3] = 1;
        self.tijk[3][3] = 7.85;
        self.cijk[3][3] = 1.0;
        self.eijk[3][3] = 0.5;
        bijk[3][3] = 1.0;
        self.gijk[3][3] = 0.5;
        self.nijk[3][3] = -0.034_800_214_576_142;
        self.dijk[3][4] = 2;
        self.tijk[3][4] = 5.4;
        self.cijk[3][4] = 1.0;
        self.eijk[3][4] = 0.5;
        bijk[3][4] = 1.0;
        self.gijk[3][4] = 0.5;
        self.nijk[3][4] = -0.133_338_130_138_96;
        self.dijk[3][5] = 2;
        self.tijk[3][5] = 0.0;
        self.cijk[3][5] = 0.25;
        self.eijk[3][5] = 0.5;
        bijk[3][5] = 2.5;
        self.gijk[3][5] = 0.5;
        self.nijk[3][5] = -0.011_993_694_974_627;
        self.dijk[3][6] = 2;
        self.tijk[3][6] = 0.75;
        self.cijk[3][6] = 0.0;
        self.eijk[3][6] = 0.5;
        bijk[3][6] = 3.0;
        self.gijk[3][6] = 0.5;
        self.nijk[3][6] = 0.069_243_379_775_168;
        self.dijk[3][7] = 2;
        self.tijk[3][7] = 2.8;
        self.cijk[3][7] = 0.0;
        self.eijk[3][7] = 0.5;
        bijk[3][7] = 3.0;
        self.gijk[3][7] = 0.5;
        self.nijk[3][7] = -0.310_225_081_482_49;
        self.dijk[3][8] = 2;
        self.tijk[3][8] = 4.45;
        self.cijk[3][8] = 0.0;
        self.eijk[3][8] = 0.5;
        bijk[3][8] = 3.0;
        self.gijk[3][8] = 0.5;
        self.nijk[3][8] = 0.244_954_917_532_26;
        self.dijk[3][9] = 3;
        self.tijk[3][9] = 4.25;
        self.cijk[3][9] = 0.0;
        self.eijk[3][9] = 0.5;
        bijk[3][9] = 3.0;
        self.gijk[3][9] = 0.5;
        self.nijk[3][9] = 0.223_698_167_169_81;
        self.dijk[4][1] = 1;
        self.tijk[4][1] = 2.6;
        self.cijk[4][1] = 0.0;
        self.eijk[4][1] = 0.0;
        bijk[4][1] = 0.0;
        self.gijk[4][1] = 0.0;
        self.nijk[4][1] = -0.108_593_873_549_42;
        self.dijk[4][2] = 2;
        self.tijk[4][2] = 1.95;
        self.cijk[4][2] = 0.0;
        self.eijk[4][2] = 0.0;
        bijk[4][2] = 0.0;
        self.gijk[4][2] = 0.0;
        self.nijk[4][2] = 0.080_228_576_727_389;
        self.dijk[4][3] = 3;
        self.tijk[4][3] = 0.0;
        self.cijk[4][3] = 0.0;
        self.eijk[4][3] = 0.0;
        bijk[4][3] = 0.0;
        self.gijk[4][3] = 0.0;
        self.nijk[4][3] = -9.330_398_511_571_7E-03;
        self.dijk[4][4] = 1;
        self.tijk[4][4] = 3.95;
        self.cijk[4][4] = 1.0;
        self.eijk[4][4] = 0.5;
        bijk[4][4] = 1.0;
        self.gijk[4][4] = 0.5;
        self.nijk[4][4] = 0.040_989_274_005_848;
        self.dijk[4][5] = 2;
        self.tijk[4][5] = 7.95;
        self.cijk[4][5] = 0.5;
        self.eijk[4][5] = 0.5;
        bijk[4][5] = 2.0;
        self.gijk[4][5] = 0.5;
        self.nijk[4][5] = -0.243_380_197_724_94;
        self.dijk[4][6] = 3;
        self.tijk[4][6] = 8.0;
        self.cijk[4][6] = 0.0;
        self.eijk[4][6] = 0.5;
        bijk[4][6] = 3.0;
        self.gijk[4][6] = 0.5;
        self.nijk[4][6] = 0.238_553_472_811_24;
        self.dijk[1][1] = 3;
        self.tijk[1][1] = 0.65;
        self.cijk[1][1] = 0.0;
        self.eijk[1][1] = 0.0;
        bijk[1][1] = 0.0;
        self.gijk[1][1] = 0.0;
        self.nijk[1][1] = -8.092_605_029_874_6E-04;
        self.dijk[1][2] = 4;
        self.tijk[1][2] = 1.55;
        self.cijk[1][2] = 0.0;
        self.eijk[1][2] = 0.0;
        bijk[1][2] = 0.0;
        self.gijk[1][2] = 0.0;
        self.nijk[1][2] = -7.538_192_508_005_9E-04;
        self.dijk[1][3] = 1;
        self.tijk[1][3] = 3.1;
        self.cijk[1][3] = 1.0;
        self.eijk[1][3] = 0.5;
        bijk[1][3] = 1.0;
        self.gijk[1][3] = 0.5;
        self.nijk[1][3] = -0.041_618_768_891_219;
        self.dijk[1][4] = 2;
        self.tijk[1][4] = 5.9;
        self.cijk[1][4] = 1.0;
        self.eijk[1][4] = 0.5;
        bijk[1][4] = 1.0;
        self.gijk[1][4] = 0.5;
        self.nijk[1][4] = -0.234_521_736_815_69;
        self.dijk[1][5] = 2;
        self.tijk[1][5] = 7.05;
        self.cijk[1][5] = 1.0;
        self.eijk[1][5] = 0.5;
        bijk[1][5] = 1.0;
        self.gijk[1][5] = 0.5;
        self.nijk[1][5] = 0.140_038_405_845_86;
        self.dijk[1][6] = 2;
        self.tijk[1][6] = 3.35;
        self.cijk[1][6] = 0.875;
        self.eijk[1][6] = 0.5;
        bijk[1][6] = 1.25;
        self.gijk[1][6] = 0.5;
        self.nijk[1][6] = 0.063_281_744_807_738;
        self.dijk[1][7] = 2;
        self.tijk[1][7] = 1.2;
        self.cijk[1][7] = 0.75;
        self.eijk[1][7] = 0.5;
        bijk[1][7] = 1.5;
        self.gijk[1][7] = 0.5;
        self.nijk[1][7] = -0.034_660_425_848_809;
        self.dijk[1][8] = 2;
        self.tijk[1][8] = 5.8;
        self.cijk[1][8] = 0.5;
        self.eijk[1][8] = 0.5;
        bijk[1][8] = 2.0;
        self.gijk[1][8] = 0.5;
        self.nijk[1][8] = -0.239_187_473_342_51;
        self.dijk[1][9] = 2;
        self.tijk[1][9] = 2.7;
        self.cijk[1][9] = 0.0;
        self.eijk[1][9] = 0.5;
        bijk[1][9] = 3.0;
        self.gijk[1][9] = 0.5;
        self.nijk[1][9] = 1.985_525_506_689_1E-03;
        self.dijk[1][10] = 3;
        self.tijk[1][10] = 0.45;
        self.cijk[1][10] = 0.0;
        self.eijk[1][10] = 0.5;
        bijk[1][10] = 3.0;
        self.gijk[1][10] = 0.5;
        self.nijk[1][10] = 6.177_774_617_155_5;
        self.dijk[1][11] = 3;
        self.tijk[1][11] = 0.55;
        self.cijk[1][11] = 0.0;
        self.eijk[1][11] = 0.5;
        bijk[1][11] = 3.0;
        self.gijk[1][11] = 0.5;
        self.nijk[1][11] = -6.957_535_827_110_5;
        self.dijk[1][12] = 3;
        self.tijk[1][12] = 1.95;
        self.cijk[1][12] = 0.0;
        self.eijk[1][12] = 0.5;
        bijk[1][12] = 3.0;
        self.gijk[1][12] = 0.5;
        self.nijk[1][12] = 1.063_018_530_638_8;
        self.dijk[2][1] = 3;
        self.tijk[2][1] = 1.85;
        self.cijk[2][1] = 0.0;
        self.eijk[2][1] = 0.0;
        bijk[2][1] = 0.0;
        self.gijk[2][1] = 0.0;
        self.nijk[2][1] = 0.013_746_429_958_576;
        self.dijk[2][2] = 3;
        self.tijk[2][2] = 3.95;
        self.cijk[2][2] = 0.0;
        self.eijk[2][2] = 0.0;
        bijk[2][2] = 0.0;
        self.gijk[2][2] = 0.0;
        self.nijk[2][2] = -7.442_501_212_955_2E-03;
        self.dijk[2][3] = 4;
        self.tijk[2][3] = 0.0;
        self.cijk[2][3] = 0.0;
        self.eijk[2][3] = 0.0;
        bijk[2][3] = 0.0;
        self.gijk[2][3] = 0.0;
        self.nijk[2][3] = -4.551_660_021_368_5E-03;
        self.dijk[2][4] = 4;
        self.tijk[2][4] = 1.85;
        self.cijk[2][4] = 0.0;
        self.eijk[2][4] = 0.0;
        bijk[2][4] = 0.0;
        self.gijk[2][4] = 0.0;
        self.nijk[2][4] = -5.454_660_335_023_7E-03;
        self.dijk[2][5] = 4;
        self.tijk[2][5] = 3.85;
        self.cijk[2][5] = 0.0;
        self.eijk[2][5] = 0.0;
        bijk[2][5] = 0.0;
        self.gijk[2][5] = 0.0;
        self.nijk[2][5] = 2.368_201_682_447_1E-03;
        self.dijk[2][6] = 1;
        self.tijk[2][6] = 5.25;
        self.cijk[2][6] = 0.25;
        self.eijk[2][6] = 0.5;
        bijk[2][6] = 0.75;
        self.gijk[2][6] = 0.5;
        self.nijk[2][6] = 0.180_077_637_214_38;
        self.dijk[2][7] = 1;
        self.tijk[2][7] = 3.85;
        self.cijk[2][7] = 0.25;
        self.eijk[2][7] = 0.5;
        bijk[2][7] = 1.0;
        self.gijk[2][7] = 0.5;
        self.nijk[2][7] = -0.447_739_429_324_86;
        self.dijk[2][8] = 1;
        self.tijk[2][8] = 0.2;
        self.cijk[2][8] = 0.0;
        self.eijk[2][8] = 0.5;
        bijk[2][8] = 2.0;
        self.gijk[2][8] = 0.5;
        self.nijk[2][8] = 0.019_327_374_888_2;
        self.dijk[2][9] = 2;
        self.tijk[2][9] = 6.5;
        self.cijk[2][9] = 0.0;
        self.eijk[2][9] = 0.5;
        bijk[2][9] = 3.0;
        self.gijk[2][9] = 0.5;
        self.nijk[2][9] = -0.306_321_978_046_24;
        self.dijk[5][1] = 2;
        self.tijk[5][1] = 1.85;
        self.cijk[5][1] = 0.0;
        self.eijk[5][1] = 0.0;
        bijk[5][1] = 0.0;
        self.gijk[5][1] = 0.0;
        self.nijk[5][1] = 0.286_616_250_283_99;
        self.dijk[5][2] = 3;
        self.tijk[5][2] = 1.4;
        self.cijk[5][2] = 0.0;
        self.eijk[5][2] = 0.0;
        bijk[5][2] = 0.0;
        self.gijk[5][2] = 0.0;
        self.nijk[5][2] = -0.109_198_338_612_47;
        self.dijk[5][3] = 1;
        self.tijk[5][3] = 3.2;
        self.cijk[5][3] = 0.25;
        self.eijk[5][3] = 0.5;
        bijk[5][3] = 0.75;
        self.gijk[5][3] = 0.5;
        self.nijk[5][3] = -1.137_403_208_227;
        self.dijk[5][4] = 1;
        self.tijk[5][4] = 2.5;
        self.cijk[5][4] = 0.25;
        self.eijk[5][4] = 0.5;
        bijk[5][4] = 1.0;
        self.gijk[5][4] = 0.5;
        self.nijk[5][4] = 0.765_805_442_373_58;
        self.dijk[5][5] = 1;
        self.tijk[5][5] = 8.0;
        self.cijk[5][5] = 0.0;
        self.eijk[5][5] = 0.5;
        bijk[5][5] = 2.0;
        self.gijk[5][5] = 0.5;
        self.nijk[5][5] = 4.263_800_092_681_9E-03;
        self.dijk[5][6] = 2;
        self.tijk[5][6] = 3.75;
        self.cijk[5][6] = 0.0;
        self.eijk[5][6] = 0.5;
        bijk[5][6] = 3.0;
        self.gijk[5][6] = 0.5;
        self.nijk[5][6] = 0.176_735_382_045_34;
        self.dijk[6][1] = 2;
        self.tijk[6][1] = 0.0;
        self.cijk[6][1] = 0.0;
        self.eijk[6][1] = 0.0;
        bijk[6][1] = 0.0;
        self.gijk[6][1] = 0.0;
        self.nijk[6][1] = -0.473_765_181_266_08;
        self.dijk[6][2] = 2;
        self.tijk[6][2] = 0.05;
        self.cijk[6][2] = 0.0;
        self.eijk[6][2] = 0.0;
        bijk[6][2] = 0.0;
        self.gijk[6][2] = 0.0;
        self.nijk[6][2] = 0.489_611_934_610_01;
        self.dijk[6][3] = 3;
        self.tijk[6][3] = 0.0;
        self.cijk[6][3] = 0.0;
        self.eijk[6][3] = 0.0;
        bijk[6][3] = 0.0;
        self.gijk[6][3] = 0.0;
        self.nijk[6][3] = -5.701_106_209_053_5E-03;
        self.dijk[6][4] = 1;
        self.tijk[6][4] = 3.65;
        self.cijk[6][4] = 1.0;
        self.eijk[6][4] = 0.5;
        bijk[6][4] = 1.0;
        self.gijk[6][4] = 0.5;
        self.nijk[6][4] = -0.199_668_200_413_2;
        self.dijk[6][5] = 2;
        self.tijk[6][5] = 4.9;
        self.cijk[6][5] = 1.0;
        self.eijk[6][5] = 0.5;
        bijk[6][5] = 1.0;
        self.gijk[6][5] = 0.5;
        self.nijk[6][5] = -0.694_111_031_017_23;
        self.dijk[6][6] = 2;
        self.tijk[6][6] = 4.45;
        self.cijk[6][6] = 0.875;
        self.eijk[6][6] = 0.5;
        bijk[6][6] = 1.25;
        self.gijk[6][6] = 0.5;
        self.nijk[6][6] = 0.692_261_927_390_21;
        self.dijk[7][1] = 1;
        self.tijk[7][1] = 2.0;
        self.cijk[7][1] = 0.0;
        self.eijk[7][1] = 0.0;
        bijk[7][1] = 0.0;
        self.gijk[7][1] = 0.0;
        self.nijk[7][1] = -0.251_571_349_719_34;
        self.dijk[7][2] = 3;
        self.tijk[7][2] = -1.0;
        self.cijk[7][2] = 0.0;
        self.eijk[7][2] = 0.0;
        bijk[7][2] = 0.0;
        self.gijk[7][2] = 0.0;
        self.nijk[7][2] = -6.220_384_111_198_3E-03;
        self.dijk[7][3] = 3;
        self.tijk[7][3] = 1.75;
        self.cijk[7][3] = 0.0;
        self.eijk[7][3] = 0.0;
        bijk[7][3] = 0.0;
        self.gijk[7][3] = 0.0;
        self.nijk[7][3] = 0.088_850_315_184_396;
        self.dijk[7][4] = 4;
        self.tijk[7][4] = 1.4;
        self.cijk[7][4] = 0.0;
        self.eijk[7][4] = 0.0;
        bijk[7][4] = 0.0;
        self.gijk[7][4] = 0.0;
        self.nijk[7][4] = -0.035_592_212_573_239;
        self.dijk[10][1] = 1;
        self.tijk[10][1] = 1.0;
        self.cijk[10][1] = 0.0;
        self.eijk[10][1] = 0.0;
        bijk[10][1] = 0.0;
        self.gijk[10][1] = 0.0;
        self.nijk[10][1] = 2.557_477_684_411_8;
        self.dijk[10][2] = 1;
        self.tijk[10][2] = 1.55;
        self.cijk[10][2] = 0.0;
        self.eijk[10][2] = 0.0;
        bijk[10][2] = 0.0;
        self.gijk[10][2] = 0.0;
        self.nijk[10][2] = -7.984_635_713_635_3;
        self.dijk[10][3] = 1;
        self.tijk[10][3] = 1.7;
        self.cijk[10][3] = 0.0;
        self.eijk[10][3] = 0.0;
        bijk[10][3] = 0.0;
        self.gijk[10][3] = 0.0;
        self.nijk[10][3] = 4.785_913_146_580_6;
        self.dijk[10][4] = 2;
        self.tijk[10][4] = 0.25;
        self.cijk[10][4] = 0.0;
        self.eijk[10][4] = 0.0;
        bijk[10][4] = 0.0;
        self.gijk[10][4] = 0.0;
        self.nijk[10][4] = -0.732_653_923_695_87;
        self.dijk[10][5] = 2;
        self.tijk[10][5] = 1.35;
        self.cijk[10][5] = 0.0;
        self.eijk[10][5] = 0.0;
        bijk[10][5] = 0.0;
        self.gijk[10][5] = 0.0;
        self.nijk[10][5] = 1.380_547_134_531_2;
        self.dijk[10][6] = 3;
        self.tijk[10][6] = 0.0;
        self.cijk[10][6] = 0.0;
        self.eijk[10][6] = 0.0;
        bijk[10][6] = 0.0;
        self.gijk[10][6] = 0.0;
        self.nijk[10][6] = 0.283_496_034_763_65;
        self.dijk[10][7] = 3;
        self.tijk[10][7] = 1.25;
        self.cijk[10][7] = 0.0;
        self.eijk[10][7] = 0.0;
        bijk[10][7] = 0.0;
        self.gijk[10][7] = 0.0;
        self.nijk[10][7] = -0.490_873_859_404_25;
        self.dijk[10][8] = 4;
        self.tijk[10][8] = 0.0;
        self.cijk[10][8] = 0.0;
        self.eijk[10][8] = 0.0;
        bijk[10][8] = 0.0;
        self.gijk[10][8] = 0.0;
        self.nijk[10][8] = -0.102_918_889_214_47;
        self.dijk[10][9] = 4;
        self.tijk[10][9] = 0.7;
        self.cijk[10][9] = 0.0;
        self.eijk[10][9] = 0.0;
        bijk[10][9] = 0.0;
        self.gijk[10][9] = 0.0;
        self.nijk[10][9] = 0.118_363_146_819_68;
        self.dijk[10][10] = 4;
        self.tijk[10][10] = 5.4;
        self.cijk[10][10] = 0.0;
        self.eijk[10][10] = 0.0;
        bijk[10][10] = 0.0;
        self.gijk[10][10] = 0.0;
        self.nijk[10][10] = 5.552_738_572_194_3E-05;

        self.n0i[1][3] = 4.000_88;
        self.n0i[1][4] = 0.763_15;
        self.n0i[1][5] = 0.004_6;
        self.n0i[1][6] = 8.744_320;
        self.n0i[1][7] = -4.469_21;
        self.n0i[1][1] = 29.838_433_97;
        self.n0i[1][2] = -15_999.691_51;
        self.n0i[2][3] = 3.500_31;
        self.n0i[2][4] = 0.137_320;
        self.n0i[2][5] = -0.146_6;
        self.n0i[2][6] = 0.900_66;
        self.n0i[2][7] = 0.0;
        self.n0i[2][1] = 17.567_707_85;
        self.n0i[2][2] = -2_801.729_072;
        self.n0i[3][3] = 3.500_02;
        self.n0i[3][4] = 2.044_52;
        self.n0i[3][5] = -1.060_44;
        self.n0i[3][6] = 2.033_66;
        self.n0i[3][7] = 0.013_93;
        self.n0i[3][1] = 20.658_446_96;
        self.n0i[3][2] = -4_902.171_516;
        self.n0i[4][3] = 4.002_63;
        self.n0i[4][4] = 4.339_39;
        self.n0i[4][5] = 1.237_22;
        self.n0i[4][6] = 13.197_4;
        self.n0i[4][7] = -6.019_89;
        self.n0i[4][1] = 36.730_059_38;
        self.n0i[4][2] = -23_639.653_01;
        self.n0i[5][3] = 4.029_39;
        self.n0i[5][4] = 6.605_69;
        self.n0i[5][5] = 3.197;
        self.n0i[5][6] = 19.192_1;
        self.n0i[5][7] = -8.372_67;
        self.n0i[5][1] = 44.709_096_19;
        self.n0i[5][2] = -31_236.635_51;
        self.n0i[6][3] = 4.067_14;
        self.n0i[6][4] = 8.975_75;
        self.n0i[6][5] = 5.251_56;
        self.n0i[6][6] = 25.142_3;
        self.n0i[6][7] = 16.138_8;
        self.n0i[6][1] = 34.301_803_49;
        self.n0i[6][2] = -38_525.502_76;
        self.n0i[7][3] = 4.339_44;
        self.n0i[7][4] = 9.448_93;
        self.n0i[7][5] = 6.894_06;
        self.n0i[7][6] = 24.461_8;
        self.n0i[7][7] = 14.782_4;
        self.n0i[7][1] = 36.532_377_83;
        self.n0i[7][2] = -38_957.809_33;
        self.n0i[8][3] = 4.0;
        self.n0i[8][4] = 11.761_8;
        self.n0i[8][5] = 20.110_1;
        self.n0i[8][6] = 33.168_8;
        self.n0i[8][7] = 0.0;
        self.n0i[8][1] = 43.172_186_26;
        self.n0i[8][2] = -51_198.309_46;
        self.n0i[9][3] = 4.0;
        self.n0i[9][4] = 8.950_43;
        self.n0i[9][5] = 21.836;
        self.n0i[9][6] = 33.403_2;
        self.n0i[9][7] = 0.0;
        self.n0i[9][1] = 42.678_370_89;
        self.n0i[9][2] = -45_215.83;
        self.n0i[10][3] = 4.0;
        self.n0i[10][4] = 11.697_7;
        self.n0i[10][5] = 26.814_2;
        self.n0i[10][6] = 38.616_4;
        self.n0i[10][7] = 0.0;
        self.n0i[10][1] = 46.997_171_88;
        self.n0i[10][2] = -52_746.833_18;
        self.n0i[11][3] = 4.0;
        self.n0i[11][4] = 13.726_6;
        self.n0i[11][5] = 30.470_7;
        self.n0i[11][6] = 43.556_1;
        self.n0i[11][7] = 0.0;
        self.n0i[11][1] = 52.076_316_31;
        self.n0i[11][2] = -57_104.810_56;
        self.n0i[12][3] = 4.0;
        self.n0i[12][4] = 15.686_5;
        self.n0i[12][5] = 33.802_9;
        self.n0i[12][6] = 48.173_1;
        self.n0i[12][7] = 0.0;
        self.n0i[12][1] = 57.258_309_34;
        self.n0i[12][2] = -60_546.763_85;
        self.n0i[13][3] = 4.0;
        self.n0i[13][4] = 18.024_1;
        self.n0i[13][5] = 38.123_5;
        self.n0i[13][6] = 53.341_5;
        self.n0i[13][7] = 0.0;
        self.n0i[13][1] = 62.096_469_01;
        self.n0i[13][2] = -66_600.128_37;
        self.n0i[14][3] = 4.0;
        self.n0i[14][4] = 21.006_9;
        self.n0i[14][5] = 43.493_1;
        self.n0i[14][6] = 58.365_7;
        self.n0i[14][7] = 0.0;
        self.n0i[14][1] = 65.939_091_54;
        self.n0i[14][2] = -74_131.454_83;
        self.n0i[15][3] = 2.479_06;
        self.n0i[15][4] = 0.958_06;
        self.n0i[15][5] = 0.454_44;
        self.n0i[15][6] = 1.560_39;
        self.n0i[15][7] = -1.375_6;
        self.n0i[15][1] = 13.075_202_88;
        self.n0i[15][2] = -5_836.943_696;
        self.n0i[16][3] = 3.501_46;
        self.n0i[16][4] = 1.075_58;
        self.n0i[16][5] = 1.013_34;
        self.n0i[16][6] = 0.0;
        self.n0i[16][7] = 0.0;
        self.n0i[16][1] = 16.801_717_3;
        self.n0i[16][2] = -2_318.322_69;
        self.n0i[17][3] = 3.500_55;
        self.n0i[17][4] = 1.028_65;
        self.n0i[17][5] = 0.004_93;
        self.n0i[17][6] = 0.0;
        self.n0i[17][7] = 0.0;
        self.n0i[17][1] = 17.457_868_99;
        self.n0i[17][2] = -2_635.244_116;
        self.n0i[18][3] = 4.003_92;
        self.n0i[18][4] = 0.010_59;
        self.n0i[18][5] = 0.987_63;
        self.n0i[18][6] = 3.069_04;
        self.n0i[18][7] = 0.0;
        self.n0i[18][1] = 21.578_827_05;
        self.n0i[18][2] = -7_766.733_078;
        self.n0i[19][3] = 4.0;
        self.n0i[19][4] = 3.119_42;
        self.n0i[19][5] = 1.002_43;
        self.n0i[19][6] = 0.0;
        self.n0i[19][7] = 0.0;
        self.n0i[19][1] = 21.583_094_4;
        self.n0i[19][2] = -6_069.035_869;
        self.n0i[20][3] = 2.5;
        self.n0i[20][4] = 0.0;
        self.n0i[20][5] = 0.0;
        self.n0i[20][6] = 0.0;
        self.n0i[20][7] = 0.0;
        self.n0i[20][1] = 10.046_395_07;
        self.n0i[20][2] = -745.375;
        self.n0i[21][3] = 2.5;
        self.n0i[21][4] = 0.0;
        self.n0i[21][5] = 0.0;
        self.n0i[21][6] = 0.0;
        self.n0i[21][7] = 0.0;
        self.n0i[21][1] = 10.046_395_07;
        self.n0i[21][2] = -745.375;

        self.bvij[1][2] = 0.998_721_377;
        self.gvij[1][2] = 1.013_950_311;
        self.btij[1][2] = 0.998_098_83;
        self.gtij[1][2] = 0.979_273_013; // CH4-N2
        self.bvij[1][3] = 0.999_518_072;
        self.gvij[1][3] = 1.002_806_594;
        self.btij[1][3] = 1.022_624_49;
        self.gtij[1][3] = 0.975_665_369; // CH4-CO2
        self.bvij[1][4] = 0.997_547_866;
        self.gvij[1][4] = 1.006_617_867;
        self.btij[1][4] = 0.996_336_508;
        self.gtij[1][4] = 1.049_707_697; // CH4-C2H6
        self.bvij[1][5] = 1.004_827_07;
        self.gvij[1][5] = 1.038_470_657;
        self.btij[1][5] = 0.989_680_305;
        self.gtij[1][5] = 1.098_655_531; // CH4-C3H8
        self.bvij[1][6] = 1.011_240_388;
        self.gvij[1][6] = 1.054_319_053;
        self.btij[1][6] = 0.980_315_756;
        self.gtij[1][6] = 1.161_117_729; // CH4-i-C4H10
        self.bvij[1][7] = 0.979_105_972;
        self.gvij[1][7] = 1.045_375_122;
        self.btij[1][7] = 0.994_174_91;
        self.gtij[1][7] = 1.171_607_691; // CH4-C4H10
        self.bvij[1][8] = 1.0;
        self.gvij[1][8] = 1.343_685_343;
        self.btij[1][8] = 1.0;
        self.gtij[1][8] = 1.188_899_743; // CH4-i-C5H12
        self.bvij[1][9] = 0.948_330_12;
        self.gvij[1][9] = 1.124_508_039;
        self.btij[1][9] = 0.992_127_525;
        self.gtij[1][9] = 1.249_173_968; // CH4-C5H12
        self.bvij[1][10] = 0.958_015_294;
        self.gvij[1][10] = 1.052_643_846;
        self.btij[1][10] = 0.981_844_797;
        self.gtij[1][10] = 1.330_570_181; // CH4-C6H14
        self.bvij[1][11] = 0.962_050_831;
        self.gvij[1][11] = 1.156_655_935;
        self.btij[1][11] = 0.977_431_529;
        self.gtij[1][11] = 1.379_850_328; // CH4-C7H16
        self.bvij[1][12] = 0.994_740_603;
        self.gvij[1][12] = 1.116_549_372;
        self.btij[1][12] = 0.957_473_785;
        self.gtij[1][12] = 1.449_245_409; // CH4-C8H18
        self.bvij[1][13] = 1.002_852_287;
        self.gvij[1][13] = 1.141_895_355;
        self.btij[1][13] = 0.947_716_769;
        self.gtij[1][13] = 1.528_532_478; // CH4-C9H20
        self.bvij[1][14] = 1.033_086_292;
        self.gvij[1][14] = 1.146_089_637;
        self.btij[1][14] = 0.937_777_823;
        self.gtij[1][14] = 1.568_231_489; // CH4-C10H22
        self.bvij[1][15] = 1.0;
        self.gvij[1][15] = 1.018_702_573;
        self.btij[1][15] = 1.0;
        self.gtij[1][15] = 1.352_643_115; // CH4-H2
        self.bvij[1][16] = 1.0;
        self.gvij[1][16] = 1.0;
        self.btij[1][16] = 1.0;
        self.gtij[1][16] = 0.95; // CH4-O2
        self.bvij[1][17] = 0.997_340_772;
        self.gvij[1][17] = 1.006_102_927;
        self.btij[1][17] = 0.987_411_732;
        self.gtij[1][17] = 0.987_473_033; // CH4-CO
        self.bvij[1][18] = 1.012_783_169;
        self.gvij[1][18] = 1.585_018_334;
        self.btij[1][18] = 1.063_333_913;
        self.gtij[1][18] = 0.775_810_513; // CH4-H2O
        self.bvij[1][19] = 1.012_599_087;
        self.gvij[1][19] = 1.040_161_207;
        self.btij[1][19] = 1.011_090_031;
        self.gtij[1][19] = 0.961_155_729; // CH4-H2S
        self.bvij[1][20] = 1.0;
        self.gvij[1][20] = 0.881_405_683;
        self.btij[1][20] = 1.0;
        self.gtij[1][20] = 3.159_776_855; // CH4-He
        self.bvij[1][21] = 1.034_630_259;
        self.gvij[1][21] = 1.014_678_542;
        self.btij[1][21] = 0.990_954_281;
        self.gtij[1][21] = 0.989_843_388; // CH4-Ar
        self.bvij[2][3] = 0.977_794_634;
        self.gvij[2][3] = 1.047_578_256;
        self.btij[2][3] = 1.005_894_529;
        self.gtij[2][3] = 1.107_654_104; // N2-CO2
        self.bvij[2][4] = 0.978_880_168;
        self.gvij[2][4] = 1.042_352_891;
        self.btij[2][4] = 1.007_671_428;
        self.gtij[2][4] = 1.098_650_964; // N2-C2H6
        self.bvij[2][5] = 0.974_424_681;
        self.gvij[2][5] = 1.081_025_408;
        self.btij[2][5] = 1.002_677_329;
        self.gtij[2][5] = 1.201_264_026; // N2-C3H8
        self.bvij[2][6] = 0.986_415_83;
        self.gvij[2][6] = 1.100_576_129;
        self.btij[2][6] = 0.992_868_13;
        self.gtij[2][6] = 1.284_462_634; // N2-i-C4H10
        self.bvij[2][7] = 0.996_082_61;
        self.gvij[2][7] = 1.146_949_309;
        self.btij[2][7] = 0.994_515_234;
        self.gtij[2][7] = 1.304_886_838; // N2-C4H10
        self.bvij[2][8] = 1.0;
        self.gvij[2][8] = 1.154_135_439;
        self.btij[2][8] = 1.0;
        self.gtij[2][8] = 1.381_770_77; // N2-i-C5H12
        self.bvij[2][9] = 1.0;
        self.gvij[2][9] = 1.078_877_166;
        self.btij[2][9] = 1.0;
        self.gtij[2][9] = 1.419_029_041; // N2-C5H12
        self.bvij[2][10] = 1.0;
        self.gvij[2][10] = 1.195_952_177;
        self.btij[2][10] = 1.0;
        self.gtij[2][10] = 1.472_607_971; // N2-C6H14
        self.bvij[2][11] = 1.0;
        self.gvij[2][11] = 1.404_554_09;
        self.btij[2][11] = 1.0;
        self.gtij[2][11] = 1.520_975_334; // N2-C7H16
        self.bvij[2][12] = 1.0;
        self.gvij[2][12] = 1.186_067_025;
        self.btij[2][12] = 1.0;
        self.gtij[2][12] = 1.733_280_051; // N2-C8H18
        self.bvij[2][13] = 1.0;
        self.gvij[2][13] = 1.100_405_929;
        self.btij[2][13] = 0.956_379_45;
        self.gtij[2][13] = 1.749_119_996; // N2-C9H20
        self.bvij[2][14] = 1.0;
        self.gvij[2][14] = 1.0;
        self.btij[2][14] = 0.957_934_447;
        self.gtij[2][14] = 1.822_157_123; // N2-C10H22
        self.bvij[2][15] = 0.972_532_065;
        self.gvij[2][15] = 0.970_115_357;
        self.btij[2][15] = 0.946_134_337;
        self.gtij[2][15] = 1.175_696_583; // N2-H2
        self.bvij[2][16] = 0.999_521_77;
        self.gvij[2][16] = 0.997_082_328;
        self.btij[2][16] = 0.997_190_589;
        self.gtij[2][16] = 0.995_157_044; // N2-O2
        self.bvij[2][17] = 1.0;
        self.gvij[2][17] = 1.008_690_943;
        self.btij[2][17] = 1.0;
        self.gtij[2][17] = 0.993_425_388; // N2-CO
        self.bvij[2][18] = 1.0;
        self.gvij[2][18] = 1.094_749_685;
        self.btij[2][18] = 1.0;
        self.gtij[2][18] = 0.968_808_467; // N2-H2O
        self.bvij[2][19] = 0.910_394_249;
        self.gvij[2][19] = 1.256_844_157;
        self.btij[2][19] = 1.004_692_366;
        self.gtij[2][19] = 0.960_174_2; // N2-H2S
        self.bvij[2][20] = 0.969_501_055;
        self.gvij[2][20] = 0.932_629_867;
        self.btij[2][20] = 0.692_868_765;
        self.gtij[2][20] = 1.471_831_58; // N2-He
        self.bvij[2][21] = 1.004_166_412;
        self.gvij[2][21] = 1.002_212_182;
        self.btij[2][21] = 0.999_069_843;
        self.gtij[2][21] = 0.990_034_831; // N2-Ar
        self.bvij[3][4] = 1.002_525_718;
        self.gvij[3][4] = 1.032_876_701;
        self.btij[3][4] = 1.013_871_147;
        self.gtij[3][4] = 0.900_949_53; // CO2-C2H6
        self.bvij[3][5] = 0.996_898_004;
        self.gvij[3][5] = 1.047_596_298;
        self.btij[3][5] = 1.033_620_538;
        self.gtij[3][5] = 0.908_772_477; // CO2-C3H8
        self.bvij[3][6] = 1.076_551_882;
        self.gvij[3][6] = 1.081_909_003;
        self.btij[3][6] = 1.023_339_824;
        self.gtij[3][6] = 0.929_982_936; // CO2-i-C4H10
        self.bvij[3][7] = 1.174_760_923;
        self.gvij[3][7] = 1.222_437_324;
        self.btij[3][7] = 1.018_171_004;
        self.gtij[3][7] = 0.911_498_231; // CO2-C4H10
        self.bvij[3][8] = 1.060_793_104;
        self.gvij[3][8] = 1.116_793_198;
        self.btij[3][8] = 1.019_180_957;
        self.gtij[3][8] = 0.961_218_039; // CO2-i-C5H12
        self.bvij[3][9] = 1.024_311_498;
        self.gvij[3][9] = 1.068_406_078;
        self.btij[3][9] = 1.027_000_795;
        self.gtij[3][9] = 0.979_217_302; // CO2-C5H12
        self.bvij[3][10] = 1.0;
        self.gvij[3][10] = 0.851_343_711;
        self.btij[3][10] = 1.0;
        self.gtij[3][10] = 1.038_675_574; // CO2-C6H14
        self.bvij[3][11] = 1.205_469_976;
        self.gvij[3][11] = 1.164_585_914;
        self.btij[3][11] = 1.011_806_317;
        self.gtij[3][11] = 1.046_169_823; // CO2-C7H16
        self.bvij[3][12] = 1.026_169_373;
        self.gvij[3][12] = 1.104_043_935;
        self.btij[3][12] = 1.029_690_78;
        self.gtij[3][12] = 1.074_455_386; // CO2-C8H18
        self.bvij[3][13] = 1.0;
        self.gvij[3][13] = 0.973_386_152;
        self.btij[3][13] = 1.007_688_62;
        self.gtij[3][13] = 1.140_671_202; // CO2-C9H20
        self.bvij[3][14] = 1.000_151_132;
        self.gvij[3][14] = 1.183_394_668;
        self.btij[3][14] = 1.020_028_79;
        self.gtij[3][14] = 1.145_512_213; // CO2-C10H22
        self.bvij[3][15] = 0.904_142_159;
        self.gvij[3][15] = 1.152_792_55;
        self.btij[3][15] = 0.942_320_195;
        self.gtij[3][15] = 1.782_924_792; // CO2-H2
        self.bvij[3][16] = 1.0;
        self.gvij[3][16] = 1.0;
        self.btij[3][16] = 1.0;
        self.gtij[3][16] = 1.0; // CO2-O2
        self.bvij[3][17] = 1.0;
        self.gvij[3][17] = 1.0;
        self.btij[3][17] = 1.0;
        self.gtij[3][17] = 1.0; // CO2-CO
        self.bvij[3][18] = 0.949_055_959;
        self.gvij[3][18] = 1.542_328_793;
        self.btij[3][18] = 0.997_372_205;
        self.gtij[3][18] = 0.775_453_996; // CO2-H2O
        self.bvij[3][19] = 0.906_630_564;
        self.gvij[3][19] = 1.024_085_837;
        self.btij[3][19] = 1.016_034_583;
        self.gtij[3][19] = 0.926_018_88; // CO2-H2S
        self.bvij[3][20] = 0.846_647_561;
        self.gvij[3][20] = 0.864_141_549;
        self.btij[3][20] = 0.768_377_63;
        self.gtij[3][20] = 3.207_456_948; // CO2-He
        self.bvij[3][21] = 1.008_392_428;
        self.gvij[3][21] = 1.029_205_465;
        self.btij[3][21] = 0.996_512_863;
        self.gtij[3][21] = 1.050_971_635; // CO2-Ar
        self.bvij[4][5] = 0.997_607_277;
        self.gvij[4][5] = 1.003_034_72;
        self.btij[4][5] = 0.996_199_694;
        self.gtij[4][5] = 1.014_730_19; // C2H6-C3H8
        self.bvij[4][6] = 1.0;
        self.gvij[4][6] = 1.006_616_886;
        self.btij[4][6] = 1.0;
        self.gtij[4][6] = 1.033_283_811; // C2H6-i-C4H10
        self.bvij[4][7] = 0.999_157_205;
        self.gvij[4][7] = 1.006_179_146;
        self.btij[4][7] = 0.999_130_554;
        self.gtij[4][7] = 1.034_832_749; // C2H6-C4H10
        self.bvij[4][8] = 1.0;
        self.gvij[4][8] = 1.045_439_935;
        self.btij[4][8] = 1.0;
        self.gtij[4][8] = 1.021_150_247; // C2H6-i-C5H12
        self.bvij[4][9] = 0.993_851_009;
        self.gvij[4][9] = 1.026_085_655;
        self.btij[4][9] = 0.998_688_946;
        self.gtij[4][9] = 1.066_665_676; // C2H6-C5H12
        self.bvij[4][10] = 1.0;
        self.gvij[4][10] = 1.169_701_102;
        self.btij[4][10] = 1.0;
        self.gtij[4][10] = 1.092_177_796; // C2H6-C6H14
        self.bvij[4][11] = 1.0;
        self.gvij[4][11] = 1.057_666_085;
        self.btij[4][11] = 1.0;
        self.gtij[4][11] = 1.134_532_014; // C2H6-C7H16
        self.bvij[4][12] = 1.007_469_726;
        self.gvij[4][12] = 1.071_917_985;
        self.btij[4][12] = 0.984_068_272;
        self.gtij[4][12] = 1.168_636_194; // C2H6-C8H18
        self.bvij[4][13] = 1.0;
        self.gvij[4][13] = 1.143_534_73;
        self.btij[4][13] = 1.0;
        self.gtij[4][13] = 1.056_033_03; // C2H6-C9H20
        self.bvij[4][14] = 0.995_676_258;
        self.gvij[4][14] = 1.098_361_281;
        self.btij[4][14] = 0.970_918_061;
        self.gtij[4][14] = 1.237_191_558; // C2H6-C10H22
        self.bvij[4][15] = 0.925_367_171;
        self.gvij[4][15] = 1.106_072_04;
        self.btij[4][15] = 0.932_969_831;
        self.gtij[4][15] = 1.902_008_495; // C2H6-H2
        self.bvij[4][16] = 1.0;
        self.gvij[4][16] = 1.0;
        self.btij[4][16] = 1.0;
        self.gtij[4][16] = 1.0; // C2H6-O2
        self.bvij[4][17] = 1.0;
        self.gvij[4][17] = 1.201_417_898;
        self.btij[4][17] = 1.0;
        self.gtij[4][17] = 1.069_224_728; // C2H6-CO
        self.bvij[4][18] = 1.0;
        self.gvij[4][18] = 1.0;
        self.btij[4][18] = 1.0;
        self.gtij[4][18] = 1.0; // C2H6-H2O
        self.bvij[4][19] = 1.010_817_909;
        self.gvij[4][19] = 1.030_988_277;
        self.btij[4][19] = 0.990_197_354;
        self.gtij[4][19] = 0.902_736_66; // C2H6-H2S
        self.bvij[4][20] = 1.0;
        self.gvij[4][20] = 1.0;
        self.btij[4][20] = 1.0;
        self.gtij[4][20] = 1.0; // C2H6-He
        self.bvij[4][21] = 1.0;
        self.gvij[4][21] = 1.0;
        self.btij[4][21] = 1.0;
        self.gtij[4][21] = 1.0; // C2H6-Ar
        self.bvij[5][6] = 0.999_243_146;
        self.gvij[5][6] = 1.001_156_119;
        self.btij[5][6] = 0.998_012_298;
        self.gtij[5][6] = 1.005_250_774; // C3H8-i-C4H10
        self.bvij[5][7] = 0.999_795_868;
        self.gvij[5][7] = 1.003_264_179;
        self.btij[5][7] = 1.000_310_289;
        self.gtij[5][7] = 1.007_392_782; // C3H8-C4H10
        self.bvij[5][8] = 1.040_459_289;
        self.gvij[5][8] = 0.999_432_118;
        self.btij[5][8] = 0.994_364_425;
        self.gtij[5][8] = 1.003_269_5; // C3H8-i-C5H12
        self.bvij[5][9] = 1.044_919_431;
        self.gvij[5][9] = 1.019_921_513;
        self.btij[5][9] = 0.996_484_021;
        self.gtij[5][9] = 1.008_344_412; // C3H8-C5H12
        self.bvij[5][10] = 1.0;
        self.gvij[5][10] = 1.057_872_566;
        self.btij[5][10] = 1.0;
        self.gtij[5][10] = 1.025_657_518; // C3H8-C6H14
        self.bvij[5][11] = 1.0;
        self.gvij[5][11] = 1.079_648_053;
        self.btij[5][11] = 1.0;
        self.gtij[5][11] = 1.050_044_169; // C3H8-C7H16
        self.bvij[5][12] = 1.0;
        self.gvij[5][12] = 1.102_764_612;
        self.btij[5][12] = 1.0;
        self.gtij[5][12] = 1.063_694_129; // C3H8-C8H18
        self.bvij[5][13] = 1.0;
        self.gvij[5][13] = 1.199_769_134;
        self.btij[5][13] = 1.0;
        self.gtij[5][13] = 1.109_973_833; // C3H8-C9H20
        self.bvij[5][14] = 0.984_104_227;
        self.gvij[5][14] = 1.053_040_574;
        self.btij[5][14] = 0.985_331_233;
        self.gtij[5][14] = 1.140_905_252; // C3H8-C10H22
        self.bvij[5][15] = 1.0;
        self.gvij[5][15] = 1.074_006_11;
        self.btij[5][15] = 1.0;
        self.gtij[5][15] = 2.308_215_191; // C3H8-H2
        self.bvij[5][16] = 1.0;
        self.gvij[5][16] = 1.0;
        self.btij[5][16] = 1.0;
        self.gtij[5][16] = 1.0; // C3H8-O2
        self.bvij[5][17] = 1.0;
        self.gvij[5][17] = 1.108_143_673;
        self.btij[5][17] = 1.0;
        self.gtij[5][17] = 1.197_564_208; // C3H8-CO
        self.bvij[5][18] = 1.0;
        self.gvij[5][18] = 1.011_759_763;
        self.btij[5][18] = 1.0;
        self.gtij[5][18] = 0.600_340_961; // C3H8-H2O
        self.bvij[5][19] = 0.936_811_219;
        self.gvij[5][19] = 1.010_593_999;
        self.btij[5][19] = 0.992_573_556;
        self.gtij[5][19] = 0.905_829_247; // C3H8-H2S
        self.bvij[5][20] = 1.0;
        self.gvij[5][20] = 1.0;
        self.btij[5][20] = 1.0;
        self.gtij[5][20] = 1.0; // C3H8-He
        self.bvij[5][21] = 1.0;
        self.gvij[5][21] = 1.0;
        self.btij[5][21] = 1.0;
        self.gtij[5][21] = 1.0; // C3H8-Ar

        // The beta values for isobutane+butane are the reciprocal values of
        // those in the GERG-2008 publication because the order was reversed in this work.
        self.bvij[6][7] = 0.999_120_311;
        self.gvij[6][7] = 1.000_414_44;
        self.btij[6][7] = 0.999_922_459;
        self.gtij[6][7] = 1.001_432_824; // C4H10-i-C4H10
        self.bvij[6][8] = 1.0;
        self.gvij[6][8] = 1.002_284_353;
        self.btij[6][8] = 1.0;
        self.gtij[6][8] = 1.001_835_788; // i-C4H10-i-C5H1
        self.bvij[6][9] = 1.0;
        self.gvij[6][9] = 1.002_779_804;
        self.btij[6][9] = 1.0;
        self.gtij[6][9] = 1.002_495_889; // i-C4H10-C5H12
        self.bvij[6][10] = 1.0;
        self.gvij[6][10] = 1.010_493_989;
        self.btij[6][10] = 1.0;
        self.gtij[6][10] = 1.006_018_054; // i-C4H10-C6H14
        self.bvij[6][11] = 1.0;
        self.gvij[6][11] = 1.021_668_316;
        self.btij[6][11] = 1.0;
        self.gtij[6][11] = 1.009_885_76; // i-C4H10-C7H16
        self.bvij[6][12] = 1.0;
        self.gvij[6][12] = 1.032_807_063;
        self.btij[6][12] = 1.0;
        self.gtij[6][12] = 1.013_945_424; // i-C4H10-C8H18
        self.bvij[6][13] = 1.0;
        self.gvij[6][13] = 1.047_298_475;
        self.btij[6][13] = 1.0;
        self.gtij[6][13] = 1.017_817_492; // i-C4H10-C9H20
        self.bvij[6][14] = 1.0;
        self.gvij[6][14] = 1.060_243_344;
        self.btij[6][14] = 1.0;
        self.gtij[6][14] = 1.021_624_748; // i-C4H10-C10H22
        self.bvij[6][15] = 1.0;
        self.gvij[6][15] = 1.147_595_688;
        self.btij[6][15] = 1.0;
        self.gtij[6][15] = 1.895_305_393; // i-C4H10-H2
        self.bvij[6][16] = 1.0;
        self.gvij[6][16] = 1.0;
        self.btij[6][16] = 1.0;
        self.gtij[6][16] = 1.0; // i-C4H10-O2
        self.bvij[6][17] = 1.0;
        self.gvij[6][17] = 1.087_272_232;
        self.btij[6][17] = 1.0;
        self.gtij[6][17] = 1.161_390_082; // i-C4H10-CO
        self.bvij[6][18] = 1.0;
        self.gvij[6][18] = 1.0;
        self.btij[6][18] = 1.0;
        self.gtij[6][18] = 1.0; // i-C4H10-H2O
        self.bvij[6][19] = 1.012_994_431;
        self.gvij[6][19] = 0.988_591_117;
        self.btij[6][19] = 0.974_550_548;
        self.gtij[6][19] = 0.937_130_844; // i-C4H10-H2S
        self.bvij[6][20] = 1.0;
        self.gvij[6][20] = 1.0;
        self.btij[6][20] = 1.0;
        self.gtij[6][20] = 1.0; // i-C4H10-He
        self.bvij[6][21] = 1.0;
        self.gvij[6][21] = 1.0;
        self.btij[6][21] = 1.0;
        self.gtij[6][21] = 1.0; // i-C4H10-Ar
        self.bvij[7][8] = 1.0;
        self.gvij[7][8] = 1.002_728_434;
        self.btij[7][8] = 1.0;
        self.gtij[7][8] = 1.000_792_201; // C4H10-i-C5H12
        self.bvij[7][9] = 1.0;
        self.gvij[7][9] = 1.018_159_65;
        self.btij[7][9] = 1.0;
        self.gtij[7][9] = 1.002_143_640; // C4H10-C5H12
        self.bvij[7][10] = 1.0;
        self.gvij[7][10] = 1.034_995_284;
        self.btij[7][10] = 1.0;
        self.gtij[7][10] = 1.009_157_06; // C4H10-C6H14
        self.bvij[7][11] = 1.0;
        self.gvij[7][11] = 1.019_174_227;
        self.btij[7][11] = 1.0;
        self.gtij[7][11] = 1.021_283_378; // C4H10-C7H16
        self.bvij[7][12] = 1.0;
        self.gvij[7][12] = 1.046_905_515;
        self.btij[7][12] = 1.0;
        self.gtij[7][12] = 1.033_180_106; // C4H10-C8H18
        self.bvij[7][13] = 1.0;
        self.gvij[7][13] = 1.049_219_137;
        self.btij[7][13] = 1.0;
        self.gtij[7][13] = 1.014_096_448; // C4H10-C9H20
        self.bvij[7][14] = 0.976_951_968;
        self.gvij[7][14] = 1.027_845_529;
        self.btij[7][14] = 0.993_688_386;
        self.gtij[7][14] = 1.076_466_918; // C4H10-C10H22
        self.bvij[7][15] = 1.0;
        self.gvij[7][15] = 1.232_939_523;
        self.btij[7][15] = 1.0;
        self.gtij[7][15] = 2.509_259_945; // C4H10-H2
        self.bvij[7][16] = 1.0;
        self.gvij[7][16] = 1.0;
        self.btij[7][16] = 1.0;
        self.gtij[7][16] = 1.0; // C4H10-O2
        self.bvij[7][17] = 1.0;
        self.gvij[7][17] = 1.084_740_904;
        self.btij[7][17] = 1.0;
        self.gtij[7][17] = 1.173_916_162; // C4H10-CO
        self.bvij[7][18] = 1.0;
        self.gvij[7][18] = 1.223_638_763;
        self.btij[7][18] = 1.0;
        self.gtij[7][18] = 0.615_512_682; // C4H10-H2O
        self.bvij[7][19] = 0.908_113_163;
        self.gvij[7][19] = 1.033_366_041;
        self.btij[7][19] = 0.985_962_886;
        self.gtij[7][19] = 0.926_156_602; // C4H10-H2S
        self.bvij[7][20] = 1.0;
        self.gvij[7][20] = 1.0;
        self.btij[7][20] = 1.0;
        self.gtij[7][20] = 1.0; // C4H10-He
        self.bvij[7][21] = 1.0;
        self.gvij[7][21] = 1.214_638_734;
        self.btij[7][21] = 1.0;
        self.gtij[7][21] = 1.245_039_498; // C4H10-Ar
        self.bvij[8][9] = 1.0;
        self.gvij[8][9] = 1.000_024_335;
        self.btij[8][9] = 1.0;
        self.gtij[8][9] = 1.000_050_537; // C5H12-i-C5H12
        self.bvij[8][10] = 1.0;
        self.gvij[8][10] = 1.002_995_876;
        self.btij[8][10] = 1.0;
        self.gtij[8][10] = 1.001_204_174; // i-C5H12-C6H14
        self.bvij[8][11] = 1.0;
        self.gvij[8][11] = 1.009_928_206;
        self.btij[8][11] = 1.0;
        self.gtij[8][11] = 1.003_194_615; // i-C5H12-C7H16
        self.bvij[8][12] = 1.0;
        self.gvij[8][12] = 1.017_880_545;
        self.btij[8][12] = 1.0;
        self.gtij[8][12] = 1.005_647_48; // i-C5H12-C8H18
        self.bvij[8][13] = 1.0;
        self.gvij[8][13] = 1.028_994_325;
        self.btij[8][13] = 1.0;
        self.gtij[8][13] = 1.008_191_499; // i-C5H12-C9H20
        self.bvij[8][14] = 1.0;
        self.gvij[8][14] = 1.039_372_957;
        self.btij[8][14] = 1.0;
        self.gtij[8][14] = 1.010_825_138; // i-C5H12-C10H22
        self.bvij[8][15] = 1.0;
        self.gvij[8][15] = 1.184_340_443;
        self.btij[8][15] = 1.0;
        self.gtij[8][15] = 1.996_386_669; // i-C5H12-H2
        self.bvij[8][16] = 1.0;
        self.gvij[8][16] = 1.0;
        self.btij[8][16] = 1.0;
        self.gtij[8][16] = 1.0; // i-C5H12-O2
        self.bvij[8][17] = 1.0;
        self.gvij[8][17] = 1.116_694_577;
        self.btij[8][17] = 1.0;
        self.gtij[8][17] = 1.199_326_059; // i-C5H12-CO
        self.bvij[8][18] = 1.0;
        self.gvij[8][18] = 1.0;
        self.btij[8][18] = 1.0;
        self.gtij[8][18] = 1.0; // i-C5H12-H2O
        self.bvij[8][19] = 1.0;
        self.gvij[8][19] = 0.835_763_343;
        self.btij[8][19] = 1.0;
        self.gtij[8][19] = 0.982_651_529; // i-C5H12-H2S
        self.bvij[8][20] = 1.0;
        self.gvij[8][20] = 1.0;
        self.btij[8][20] = 1.0;
        self.gtij[8][20] = 1.0; // i-C5H12-He
        self.bvij[8][21] = 1.0;
        self.gvij[8][21] = 1.0;
        self.btij[8][21] = 1.0;
        self.gtij[8][21] = 1.0; // i-C5H12-Ar
        self.bvij[9][10] = 1.0;
        self.gvij[9][10] = 1.002_480_637;
        self.btij[9][10] = 1.0;
        self.gtij[9][10] = 1.000_761_237; // C5H12-C6H14
        self.bvij[9][11] = 1.0;
        self.gvij[9][11] = 1.008_972_412;
        self.btij[9][11] = 1.0;
        self.gtij[9][11] = 1.002_441_051; // C5H12-C7H16
        self.bvij[9][12] = 1.0;
        self.gvij[9][12] = 1.069_223_964;
        self.btij[9][12] = 1.0;
        self.gtij[9][12] = 1.016_422_347; // C5H12-C8H18
        self.bvij[9][13] = 1.0;
        self.gvij[9][13] = 1.034_910_633;
        self.btij[9][13] = 1.0;
        self.gtij[9][13] = 1.103_421_755; // C5H12-C9H20
        self.bvij[9][14] = 1.0;
        self.gvij[9][14] = 1.016_370_338;
        self.btij[9][14] = 1.0;
        self.gtij[9][14] = 1.049_035_838; // C5H12-C10H22
        self.bvij[9][15] = 1.0;
        self.gvij[9][15] = 1.188_334_783;
        self.btij[9][15] = 1.0;
        self.gtij[9][15] = 2.013_859_174; // C5H12-H2
        self.bvij[9][16] = 1.0;
        self.gvij[9][16] = 1.0;
        self.btij[9][16] = 1.0;
        self.gtij[9][16] = 1.0; // C5H12-O2
        self.bvij[9][17] = 1.0;
        self.gvij[9][17] = 1.119_954_454;
        self.btij[9][17] = 1.0;
        self.gtij[9][17] = 1.206_043_295; // C5H12-CO
        self.bvij[9][18] = 1.0;
        self.gvij[9][18] = 0.956_677_31;
        self.btij[9][18] = 1.0;
        self.gtij[9][18] = 0.447_666_011; // C5H12-H2O
        self.bvij[9][19] = 0.984_613_203;
        self.gvij[9][19] = 1.076_539_234;
        self.btij[9][19] = 0.962_006_651;
        self.gtij[9][19] = 0.959_065_662; // C5H12-H2S
        self.bvij[9][20] = 1.0;
        self.gvij[9][20] = 1.0;
        self.btij[9][20] = 1.0;
        self.gtij[9][20] = 1.0; // C5H12-He
        self.bvij[9][21] = 1.0;
        self.gvij[9][21] = 1.0;
        self.btij[9][21] = 1.0;
        self.gtij[9][21] = 1.0; // C5H12-Ar
        self.bvij[10][11] = 1.0;
        self.gvij[10][11] = 1.001_508_227;
        self.btij[10][11] = 1.0;
        self.gtij[10][11] = 0.999_762_786; // C6H14-C7H16
        self.bvij[10][12] = 1.0;
        self.gvij[10][12] = 1.006_268_954;
        self.btij[10][12] = 1.0;
        self.gtij[10][12] = 1.001_633_952; // C6H14-C8H18
        self.bvij[10][13] = 1.0;
        self.gvij[10][13] = 1.020_761_68;
        self.btij[10][13] = 1.0;
        self.gtij[10][13] = 1.055_369_591; // C6H14-C9H20
        self.bvij[10][14] = 1.001_516_371;
        self.gvij[10][14] = 1.013_511_439;
        self.btij[10][14] = 0.997_641_01;
        self.gtij[10][14] = 1.028_939_539; // C6H14-C10H22
        self.bvij[10][15] = 1.0;
        self.gvij[10][15] = 1.243_461_678;
        self.btij[10][15] = 1.0;
        self.gtij[10][15] = 3.021_197_546; // C6H14-H2
        self.bvij[10][16] = 1.0;
        self.gvij[10][16] = 1.0;
        self.btij[10][16] = 1.0;
        self.gtij[10][16] = 1.0; // C6H14-O2
        self.bvij[10][17] = 1.0;
        self.gvij[10][17] = 1.155_145_836;
        self.btij[10][17] = 1.0;
        self.gtij[10][17] = 1.233_272_781; // C6H14-CO
        self.bvij[10][18] = 1.0;
        self.gvij[10][18] = 1.170_217_596;
        self.btij[10][18] = 1.0;
        self.gtij[10][18] = 0.569_681_333; // C6H14-H2O
        self.bvij[10][19] = 0.754_473_958;
        self.gvij[10][19] = 1.339_283_552;
        self.btij[10][19] = 0.985_891_113;
        self.gtij[10][19] = 0.956_075_596; // C6H14-H2S
        self.bvij[10][20] = 1.0;
        self.gvij[10][20] = 1.0;
        self.btij[10][20] = 1.0;
        self.gtij[10][20] = 1.0; // C6H14-He
        self.bvij[10][21] = 1.0;
        self.gvij[10][21] = 1.0;
        self.btij[10][21] = 1.0;
        self.gtij[10][21] = 1.0; // C6H14-Ar
        self.bvij[11][12] = 1.0;
        self.gvij[11][12] = 1.006_767_176;
        self.btij[11][12] = 1.0;
        self.gtij[11][12] = 0.998_793_111; // C7H16-C8H18
        self.bvij[11][13] = 1.0;
        self.gvij[11][13] = 1.001_370_076;
        self.btij[11][13] = 1.0;
        self.gtij[11][13] = 1.001_150_096; // C7H16-C9H20
        self.bvij[11][14] = 1.0;
        self.gvij[11][14] = 1.002_972_346;
        self.btij[11][14] = 1.0;
        self.gtij[11][14] = 1.002_229_938; // C7H16-C10H22
        self.bvij[11][15] = 1.0;
        self.gvij[11][15] = 1.159_131_722;
        self.btij[11][15] = 1.0;
        self.gtij[11][15] = 3.169_143_057; // C7H16-H2
        self.bvij[11][16] = 1.0;
        self.gvij[11][16] = 1.0;
        self.btij[11][16] = 1.0;
        self.gtij[11][16] = 1.0; // C7H16-O2
        self.bvij[11][17] = 1.0;
        self.gvij[11][17] = 1.190_354_273;
        self.btij[11][17] = 1.0;
        self.gtij[11][17] = 1.256_123_503; // C7H16-CO
        self.bvij[11][18] = 1.0;
        self.gvij[11][18] = 1.0;
        self.btij[11][18] = 1.0;
        self.gtij[11][18] = 1.0; // C7H16-H2O
        self.bvij[11][19] = 0.828_967_164;
        self.gvij[11][19] = 1.087_956_749;
        self.btij[11][19] = 0.988_937_417;
        self.gtij[11][19] = 1.013_453_092; // C7H16-H2S
        self.bvij[11][20] = 1.0;
        self.gvij[11][20] = 1.0;
        self.btij[11][20] = 1.0;
        self.gtij[11][20] = 1.0; // C7H16-He
        self.bvij[11][21] = 1.0;
        self.gvij[11][21] = 1.0;
        self.btij[11][21] = 1.0;
        self.gtij[11][21] = 1.0; // C7H16-Ar
        self.bvij[12][13] = 1.0;
        self.gvij[12][13] = 1.001_357_085;
        self.btij[12][13] = 1.0;
        self.gtij[12][13] = 1.000_235_044; // C8H18-C9H20
        self.bvij[12][14] = 1.0;
        self.gvij[12][14] = 1.002_553_544;
        self.btij[12][14] = 1.0;
        self.gtij[12][14] = 1.007_186_267; // C8H18-C10H22
        self.bvij[12][15] = 1.0;
        self.gvij[12][15] = 1.305_249_405;
        self.btij[12][15] = 1.0;
        self.gtij[12][15] = 2.191_555_216; // C8H18-H2
        self.bvij[12][16] = 1.0;
        self.gvij[12][16] = 1.0;
        self.btij[12][16] = 1.0;
        self.gtij[12][16] = 1.0; // C8H18-O2
        self.bvij[12][17] = 1.0;
        self.gvij[12][17] = 1.219_206_702;
        self.btij[12][17] = 1.0;
        self.gtij[12][17] = 1.276_565_536; // C8H18-CO
        self.bvij[12][18] = 1.0;
        self.gvij[12][18] = 0.599_484_191;
        self.btij[12][18] = 1.0;
        self.gtij[12][18] = 0.662_072_469; // C8H18-H2O
        self.bvij[12][19] = 1.0;
        self.gvij[12][19] = 1.0;
        self.btij[12][19] = 1.0;
        self.gtij[12][19] = 1.0; // C8H18-H2S
        self.bvij[12][20] = 1.0;
        self.gvij[12][20] = 1.0;
        self.btij[12][20] = 1.0;
        self.gtij[12][20] = 1.0; // C8H18-He
        self.bvij[12][21] = 1.0;
        self.gvij[12][21] = 1.0;
        self.btij[12][21] = 1.0;
        self.gtij[12][21] = 1.0; // C8H18-Ar
        self.bvij[13][14] = 1.0;
        self.gvij[13][14] = 1.000_810_52;
        self.btij[13][14] = 1.0;
        self.gtij[13][14] = 1.000_182_392; // C9H20-C10H22
        self.bvij[13][15] = 1.0;
        self.gvij[13][15] = 1.342_647_661;
        self.btij[13][15] = 1.0;
        self.gtij[13][15] = 2.234_354_04; // C9H20-H2
        self.bvij[13][16] = 1.0;
        self.gvij[13][16] = 1.0;
        self.btij[13][16] = 1.0;
        self.gtij[13][16] = 1.0; // C9H20-O2
        self.bvij[13][17] = 1.0;
        self.gvij[13][17] = 1.252_151_449;
        self.btij[13][17] = 1.0;
        self.gtij[13][17] = 1.294_070_556; // C9H20-CO
        self.bvij[13][18] = 1.0;
        self.gvij[13][18] = 1.0;
        self.btij[13][18] = 1.0;
        self.gtij[13][18] = 1.0; // C9H20-H2O
        self.bvij[13][19] = 1.0;
        self.gvij[13][19] = 1.082_905_109;
        self.btij[13][19] = 1.0;
        self.gtij[13][19] = 1.086_557_826; // C9H20-H2S
        self.bvij[13][20] = 1.0;
        self.gvij[13][20] = 1.0;
        self.btij[13][20] = 1.0;
        self.gtij[13][20] = 1.0; // C9H20-He
        self.bvij[13][21] = 1.0;
        self.gvij[13][21] = 1.0;
        self.btij[13][21] = 1.0;
        self.gtij[13][21] = 1.0; // C9H20-Ar
        self.bvij[14][15] = 1.695_358_382;
        self.gvij[14][15] = 1.120_233_729;
        self.btij[14][15] = 1.064_818_089;
        self.gtij[14][15] = 3.786_003_724; // C10H22-H2
        self.bvij[14][16] = 1.0;
        self.gvij[14][16] = 1.0;
        self.btij[14][16] = 1.0;
        self.gtij[14][16] = 1.0; // C10H22-O2
        self.bvij[14][17] = 1.0;
        self.gvij[14][17] = 0.870_184_96;
        self.btij[14][17] = 1.049_594_632;
        self.gtij[14][17] = 1.803_567_587; // C10H22-CO
        self.bvij[14][18] = 1.0;
        self.gvij[14][18] = 0.551_405_318;
        self.btij[14][18] = 0.897_162_268;
        self.gtij[14][18] = 0.740_416_402; // C10H22-H2O
        self.bvij[14][19] = 0.975_187_766;
        self.gvij[14][19] = 1.171_714_677;
        self.btij[14][19] = 0.973_091_413;
        self.gtij[14][19] = 1.103_693_489; // C10H22-H2S
        self.bvij[14][20] = 1.0;
        self.gvij[14][20] = 1.0;
        self.btij[14][20] = 1.0;
        self.gtij[14][20] = 1.0; // C10H22-He
        self.bvij[14][21] = 1.0;
        self.gvij[14][21] = 1.0;
        self.btij[14][21] = 1.0;
        self.gtij[14][21] = 1.0; // C10H22-Ar
        self.bvij[15][16] = 1.0;
        self.gvij[15][16] = 1.0;
        self.btij[15][16] = 1.0;
        self.gtij[15][16] = 1.0; // H2-O2
        self.bvij[15][17] = 1.0;
        self.gvij[15][17] = 1.121_416_201;
        self.btij[15][17] = 1.0;
        self.gtij[15][17] = 1.377_504_607; // H2-CO
        self.bvij[15][18] = 1.0;
        self.gvij[15][18] = 1.0;
        self.btij[15][18] = 1.0;
        self.gtij[15][18] = 1.0; // H2-H2O
        self.bvij[15][19] = 1.0;
        self.gvij[15][19] = 1.0;
        self.btij[15][19] = 1.0;
        self.gtij[15][19] = 1.0; // H2-H2S
        self.bvij[15][20] = 1.0;
        self.gvij[15][20] = 1.0;
        self.btij[15][20] = 1.0;
        self.gtij[15][20] = 1.0; // H2-He
        self.bvij[15][21] = 1.0;
        self.gvij[15][21] = 1.0;
        self.btij[15][21] = 1.0;
        self.gtij[15][21] = 1.0; // H2-Ar
        self.bvij[16][17] = 1.0;
        self.gvij[16][17] = 1.0;
        self.btij[16][17] = 1.0;
        self.gtij[16][17] = 1.0; // O2-CO
        self.bvij[16][18] = 1.0;
        self.gvij[16][18] = 1.143_174_289;
        self.btij[16][18] = 1.0;
        self.gtij[16][18] = 0.964_767_932; // O2-H2O
        self.bvij[16][19] = 1.0;
        self.gvij[16][19] = 1.0;
        self.btij[16][19] = 1.0;
        self.gtij[16][19] = 1.0; // O2-H2S
        self.bvij[16][20] = 1.0;
        self.gvij[16][20] = 1.0;
        self.btij[16][20] = 1.0;
        self.gtij[16][20] = 1.0; // O2-He
        self.bvij[16][21] = 0.999_746_847;
        self.gvij[16][21] = 0.993_907_223;
        self.btij[16][21] = 1.000_023_103;
        self.gtij[16][21] = 0.990_430_423; // O2-Ar
        self.bvij[17][18] = 1.0;
        self.gvij[17][18] = 1.0;
        self.btij[17][18] = 1.0;
        self.gtij[17][18] = 1.0; // CO-H2O
        self.bvij[17][19] = 0.795_660_392;
        self.gvij[17][19] = 1.101_731_308;
        self.btij[17][19] = 1.025_536_736;
        self.gtij[17][19] = 1.022_749_748; // CO-H2S
        self.bvij[17][20] = 1.0;
        self.gvij[17][20] = 1.0;
        self.btij[17][20] = 1.0;
        self.gtij[17][20] = 1.0; // CO-He
        self.bvij[17][21] = 1.0;
        self.gvij[17][21] = 1.159_720_623;
        self.btij[17][21] = 1.0;
        self.gtij[17][21] = 0.954_215_746; // CO-Ar
        self.bvij[18][19] = 1.0;
        self.gvij[18][19] = 1.014_832_832;
        self.btij[18][19] = 1.0;
        self.gtij[18][19] = 0.940_587_083; // H2O-H2S
        self.bvij[18][20] = 1.0;
        self.gvij[18][20] = 1.0;
        self.btij[18][20] = 1.0;
        self.gtij[18][20] = 1.0; // H2O-He
        self.bvij[18][21] = 1.0;
        self.gvij[18][21] = 1.038_993_495;
        self.btij[18][21] = 1.0;
        self.gtij[18][21] = 1.070_941_866; // H2O-Ar
        self.bvij[19][20] = 1.0;
        self.gvij[19][20] = 1.0;
        self.btij[19][20] = 1.0;
        self.gtij[19][20] = 1.0; // H2S-He
        self.bvij[19][21] = 1.0;
        self.gvij[19][21] = 1.0;
        self.btij[19][21] = 1.0;
        self.gtij[19][21] = 1.0; // H2S-Ar
        self.bvij[20][21] = 1.0;
        self.gvij[20][21] = 1.0;
        self.btij[20][21] = 1.0;
        self.gtij[20][21] = 1.0; // He-Ar

        for i in 1..=MAXFLDS {
            self.bvij[i][i] = 1.0;
            self.btij[i][i] = 1.0;
            self.gvij[i][i] = 1.0 / DC[i];
            self.gtij[i][i] = TC[i];
            for j in i + 1..=MAXFLDS {
                self.gvij[i][j] = self.gvij[i][j] * self.bvij[i][j] * (vc3[i] + vc3[j]).powi(3);
                self.gtij[i][j] = self.gtij[i][j] * self.btij[i][j] * tc2[i] * tc2[j];
                self.bvij[i][j] = self.bvij[i][j].powi(2);
                self.btij[i][j] = self.btij[i][j].powi(2);
            }
        }

        for (i, bijki) in bijk.iter().enumerate().skip(1) {
            for (j, bijkij) in bijki.iter().enumerate().skip(1) {
                self.gijk[i][j] =
                    -self.cijk[i][j] * self.eijk[i][j].powi(2) + bijkij * self.gijk[i][j];
                self.eijk[i][j] = 2.0 * self.cijk[i][j] * self.eijk[i][j] - bijkij;
                self.cijk[i][j] = -self.cijk[i][j];
            }
        }

        let t0 = 298.15;
        let d0 = 101.325 / RGERG / t0;
        for i in 1..=MAXFLDS {
            self.n0i[i][3] -= 1.0;
            self.n0i[i][2] += t0;
            for j in 1..=7 {
                self.n0i[i][j] *= RSR;
            }
            self.n0i[i][2] -= t0;
            self.n0i[i][1] -= d0.ln();
        }
    }

    pub fn molar_mass(&mut self) {
        self.mm = 0.0;
        for (i, mmi_gerg) in MMI_GERG.iter().enumerate().skip(1) {
            self.mm += self.x[i] * mmi_gerg;
        }
    }

    pub fn pressure(&mut self) -> f64 {
        self.alphar(0);
        self.z = 1.0 + self.ar[0][1];
        let p = self.d * RGERG * self.t * self.z;
        self.dpddsave = RGERG * self.t * (1.0 + 2.0 * self.ar[0][1] + self.ar[0][2]);
        p
    }

    pub fn density(&mut self, iflag: i32) {
        let mut nfail: i32 = 0;
        let mut ifail: i32 = 0;
        const TOLR: f64 = 0.000_000_1;

        let (dcx, _tcx) = self.pseudocriticalpoint();

        if self.d > -EPSILON {
            self.d = self.p / RGERG / self.t;
            if iflag == 2 {
                self.d = dcx * 3.0;
            }
        } else {
            self.d = self.d.abs();
        }

        let plog = self.p.ln();
        let mut vlog = -self.d.ln();

        for it in 1..=50 {
            if !(-7.0..=100.0).contains(&vlog) || it == 20 || it == 30 || it == 40 || ifail == 1 {
                //Current state is bad or iteration is taking too long.  Restart with completely different initial state
                ifail = 0;
                if nfail > 2 {
                    // Iteration failed (above loop did not find a solution or checks made below indicate possible 2-phase state)
                    //herr = "Calculation failed to converge in GERG method, ideal gas density returned.";
                    self.d = self.p / RGERG / self.t;
                }
                nfail += 1;
                if nfail == 1 {
                    self.d = dcx * 3.0; // If vapor phase search fails, look for root in liquid region
                } else if nfail == 2 {
                    self.d = dcx * 2.5; // If liquid phase search fails, look for root between liquid and critical regions
                } else if nfail == 3 {
                    self.d = dcx * 2.0; // If search fails, look for root in critical region
                }
                vlog = -self.d.ln();
            }
            self.d = (-vlog).exp();
            let p2 = self.pressure();
            if self.dpddsave < EPSILON || p2 < EPSILON {
                // Current state is 2-phase, try locating a different state that is single phase
                let mut vinc = if self.d > dcx { -0.1 } else { 0.1 };
                if it > 5 {
                    vinc /= 2.0;
                }
                if it > 10 && it < 20 {
                    vinc /= 5.0;
                }
                vlog += vinc;
            } else {
                // Find the next density with a first order Newton's type iterative scheme, with
                // log(P) as the known variable and log(v) as the unknown property.
                // See AGA 8 publication for further information.
                let dpdlv = -self.d * self.dpddsave; // d(p)/d[log(v)]
                let vdiff = (p2.ln() - plog) * p2 / dpdlv;
                vlog += -vdiff;
                if vdiff.abs() < TOLR {
                    // Check to see if state is possibly 2-phase, and if so restart
                    if self.dpddsave < 0.0 {
                        ifail = 1;
                    } else {
                        self.d = (-vlog).exp();

                        // If requested, check to see if point is possibly 2-phase
                        if iflag > 0 {
                            if (self.properties() <= 0.0
                                || self.dp_dd <= 0.0
                                || self.d2p_dtd <= 0.0)
                                || (self.cv <= 0.0 || self.cp <= 0.0 || self.w <= 0.0)
                            {
                                // Iteration failed (above loop did find a solution or checks made below indicate possible 2-phase state)
                                //herr = "Calculation failed to converge in GERG method, ideal gas density returned.";
                                self.d = self.p / RGERG / self.t;
                            }
                            return;
                        }
                        return; // Iteration converged
                    }
                }
            }
        }
        // Iteration failed (above loop did not find a solution or checks made below indicate possible 2-phase state)
        //herr = "Calculation failed to converge in GERG method, ideal gas density returned.";
        self.d = self.p / RGERG / self.t;
    }

    pub fn properties(&mut self) -> f64 {
        self.molar_mass();
        self.alpha0();
        self.alphar(1);

        let rt = RGERG * self.t;
        self.z = 1.0 + self.ar[0][1];
        let p = self.d * rt * self.z;
        self.dp_dd = rt * (1.0 + 2.0 * self.ar[0][1] + self.ar[0][2]);
        self.dp_dt = self.d * RGERG * (1.0 + self.ar[0][1] - self.ar[1][1]);
        self.d2p_dtd = RGERG
            * (1.0 + 2.0 * self.ar[0][1] + self.ar[0][2] - 2.0 * self.ar[1][1] - self.ar[1][2]);
        self.a = rt * self.a0[0] + self.ar[0][0];
        self.g = rt * (1.0 + self.ar[0][1] + self.a0[0] + self.ar[0][0]);
        self.u = rt * (self.a0[1] + self.ar[1][0]);
        self.h = rt * (1.0 + self.ar[0][1] + self.a0[1] + self.ar[1][0]);
        self.s = RGERG * (self.a0[1] + self.ar[1][0] - self.a0[0] - self.ar[0][0]);
        self.cv = -RGERG * (self.a0[2] + self.ar[2][0]);
        if self.d > EPSILON {
            self.cp = self.cv + self.t * (self.dp_dt / self.d) * (self.dp_dt / self.d) / self.dp_dd;
            self.d2p_dd2 =
                rt * (2.0 * self.ar[0][1] + 4.0 * self.ar[0][2] + self.ar[0][3]) / self.d;
            self.jt = (self.t / self.d * self.dp_dt / self.dp_dd - 1.0) / self.cp / self.d;
        //  '=(dB/dT*T-B)/Cp for an ideal gas, but dB/dT is not known
        } else {
            self.cp = self.cv + RGERG;
            self.d2p_dd2 = 0.0;
            self.jt = 1E+20;
        }
        self.w = 1000.0 * self.cp / self.cv * self.dp_dd / self.mm;
        if self.w < 0.0 {
            self.w = 0.0;
        }
        self.w = self.w.sqrt();
        self.kappa = self.w.powi(2) * self.mm / (rt * 1000.0 * self.z);
        p
    }

    fn reducingparameters(&mut self) -> (f64, f64) {
        let mut dr: f64 = 0.0;
        let mut tr: f64 = 0.0;
        let mut vr: f64 = 0.0;
        let mut xij: f64;
        let mut f: f64;
        let mut icheck: i32 = 0;

        // Check to see if a component fraction has changed.  If x is the same as the previous call, then exit.
        for i in 1..=NC_GERG {
            if (self.x[i] - self.xold[i]).abs() > 0.000_000_1 {
                icheck = 1;
            }
            self.xold[i] = self.x[i];
        }
        if icheck == 0 {
            return (self.drold, self.trold);
        }
        self.told = 0.0;
        self.trold2 = 0.0;

        // Calculate reducing variables for T and D
        for i in 1..=NC_GERG {
            if self.x[i] > EPSILON {
                f = 1.0;
                for j in i..=NC_GERG {
                    if self.x[j] > EPSILON {
                        xij = f * (self.x[i] * self.x[j]) * (self.x[i] + self.x[j]);
                        vr += xij * GVIJ[i][j] / (self.bvij[i][j] * self.x[i] + self.x[j]);
                        tr += xij * GTIJ[i][j] / (BTIJ[i][j] * self.x[i] + self.x[j]);
                        f = 2.0;
                    }
                }
            }
        }
        if vr > EPSILON {
            dr = 1.0 / vr;
        }
        self.drold = dr;
        self.trold = tr;
        (dr, tr)
    }

    fn alpha0(&mut self) {
        let mut loghyp: f64;
        let mut th0t: f64;
        let mut logxd: f64;
        let mut sumhyp0: f64;
        let mut sumhyp1: f64;
        let mut sumhyp2: f64;
        let mut em: f64;
        let mut ep: f64;
        let mut hcn: f64;
        let mut hsn: f64;

        self.a0[0] = 0.0;
        self.a0[1] = 0.0;
        self.a0[2] = 0.0;
        let logd = if self.d > EPSILON {
            self.d.ln()
        } else {
            EPSILON.ln()
        };
        let logt = self.t.ln();
        for (i, th0i) in TH0I.iter().enumerate().skip(1) {
            if self.x[i] > EPSILON {
                logxd = logd + self.x[i].ln();
                sumhyp0 = 0.0;
                sumhyp1 = 0.0;
                sumhyp2 = 0.0;
                for (j, th0ij) in th0i.iter().enumerate().take(8).skip(4) {
                    if th0ij > &EPSILON {
                        th0t = th0ij / self.t;
                        ep = th0t.exp();
                        em = 1.0 / ep;
                        hsn = (ep - em) / 2.0;
                        hcn = (ep + em) / 2.0;
                        if j == 4 || j == 6 {
                            loghyp = hsn.abs().ln();
                            sumhyp0 += self.n0i[i][j] * loghyp;
                            sumhyp1 += self.n0i[i][j] * th0t * hcn / hsn;
                            sumhyp2 += self.n0i[i][j] * (th0t / hsn) * (th0t / hsn);
                        } else {
                            loghyp = hcn.abs().ln();
                            sumhyp0 -= self.n0i[i][j] * loghyp;
                            sumhyp1 -= self.n0i[i][j] * th0t * hsn / hcn;
                            sumhyp2 += self.n0i[i][j] * (th0t / hcn) * (th0t / hcn);
                        }
                    }
                }
                self.a0[0] += self.x[i]
                    * (logxd + self.n0i[i][1] + self.n0i[i][2] / self.t - self.n0i[i][3] * logt
                        + sumhyp0);
                self.a0[1] += self.x[i] * (self.n0i[i][3] + self.n0i[i][2] / self.t + sumhyp1);
                self.a0[2] += -self.x[i] * (self.n0i[i][3] + sumhyp2);
            }
        }
    }

    fn alphar(&mut self, itau: i32) {
        let mut ex: f64;
        let mut ex2: f64;
        let mut ex3: f64;
        let mut cij0: f64;
        let mut eij0: f64;
        let mut ndt: f64;
        let mut ndtd: f64;
        let mut ndtt: f64;
        let mut xijf: f64;
        let mut delp: [f64; 7 + 1] = [0.0; 7 + 1];
        let mut expd: [f64; 7 + 1] = [0.0; 7 + 1];

        for i in 0..=3 {
            for j in 0..=3 {
                self.ar[i][j] = 0.0;
            }
        }

        //Set up del, tau, log(tau), and the first 7 calculations for del^i
        let (dr, tr) = self.reducingparameters();
        let del = self.d / dr;
        let tau = tr / self.t;
        let lntau = tau.ln();
        delp[1] = del;
        expd[1] = (-delp[1]).exp();
        for i in 2..8 {
            delp[i] = delp[i - 1] * del;
            expd[i] = (-delp[i]).exp();
        }

        // If temperature has changed, calculate temperature dependent parts
        if (self.t - self.told).abs() > 0.000_000_1 || (tr - self.trold2).abs() > 0.000_000_1 {
            self.tterms(lntau);
        }
        self.told = self.t;
        self.trold2 = tr;

        // Calculate pure fluid contributions
        for i in 1..=NC_GERG {
            if self.x[i] > EPSILON {
                for k in 1..=KPOL[i] {
                    ndt = self.x[i] * delp[self.doik[i][k]] * self.taup[i][k];
                    ndtd = ndt * self.doik[i][k] as f64;
                    self.ar[0][1] += ndtd;
                    self.ar[0][2] += ndtd * (self.doik[i][k] as f64 - 1.0);
                    if itau > 0 {
                        ndtt = ndt * self.toik[i][k];
                        self.ar[0][0] += ndt;
                        self.ar[1][0] += ndtt;
                        self.ar[2][0] += ndtt * (self.toik[i][k] - 1.0);
                        self.ar[1][1] += ndtt * self.doik[i][k] as f64;
                        self.ar[1][2] +=
                            ndtt * self.doik[i][k] as f64 * (self.doik[i][k] as f64 - 1.0);
                        self.ar[0][3] +=
                            ndtd * (self.doik[i][k] as f64 - 1.0) * (self.doik[i][k] as f64 - 2.0);
                    }
                }
                for k in 1 + KPOL[i]..=KPOL[i] + KEXP[i] {
                    ndt =
                        self.x[i] * delp[self.doik[i][k]] * self.taup[i][k] * expd[self.coik[i][k]];
                    ex = self.coik[i][k] as f64 * delp[self.coik[i][k]];
                    ex2 = self.doik[i][k] as f64 - ex;
                    ex3 = ex2 * (ex2 - 1.0);
                    self.ar[0][1] += ndt * ex2;
                    self.ar[0][2] += ndt * (ex3 - self.coik[i][k] as f64 * ex);
                    if itau > 0 {
                        ndtt = ndt * self.toik[i][k];
                        self.ar[0][0] += ndt;
                        self.ar[1][0] += ndtt;
                        self.ar[2][0] += ndtt * (self.toik[i][k] - 1.0);
                        self.ar[1][1] += ndtt * ex2;
                        self.ar[1][2] += ndtt * (ex3 - self.coik[i][k] as f64 * ex);
                        self.ar[0][3] += ndt
                            * (ex3 * (ex2 - 2.0)
                                - ex * (3.0 * ex2 - 3.0 + self.coik[i][k] as f64)
                                    * self.coik[i][k] as f64);
                    }
                }
            }
        }

        // Calculate mixture contributions
        for i in 1..NC_GERG {
            if self.x[i] > EPSILON {
                for j in i + 1..=NC_GERG {
                    if self.x[j] > EPSILON {
                        let mn = MNUMB[i][j];
                        if mn > 0 {
                            xijf = self.x[i] * self.x[j] * FIJ[i][j];
                            for k in 1..=KPOLIJ[mn] {
                                ndt = xijf * delp[self.dijk[mn][k]] * self.taupijk[mn][k];
                                ndtd = ndt * self.dijk[mn][k] as f64;
                                self.ar[0][1] += ndtd;
                                self.ar[0][2] += ndtd * (self.dijk[mn][k] as f64 - 1.0);
                                if itau > 0 {
                                    ndtt = ndt * self.tijk[mn][k] as f64;
                                    self.ar[0][0] += ndt;
                                    self.ar[1][0] += ndtt;
                                    self.ar[2][0] += ndtt * (self.tijk[mn][k] - 1.0);
                                    self.ar[1][1] += ndtt * self.dijk[mn][k] as f64;
                                    self.ar[1][2] += ndtt
                                        * self.dijk[mn][k] as f64
                                        * (self.dijk[mn][k] as f64 - 1.0);
                                    self.ar[0][3] += ndtd
                                        * (self.dijk[mn][k] as f64 - 1.0)
                                        * (self.dijk[mn][k] as f64 - 2.0);
                                }
                            }
                            for k in 1 + KPOLIJ[mn]..=KPOLIJ[mn] + KEXPIJ[mn] {
                                cij0 = self.cijk[mn][k] * delp[2];
                                eij0 = self.eijk[mn][k] * del;
                                ndt = xijf
                                    * self.nijk[mn][k]
                                    * delp[self.dijk[mn][k]]
                                    * (cij0 + eij0 + self.gijk[mn][k] + self.tijk[mn][k] * lntau)
                                        .exp();
                                ex = self.dijk[mn][k] as f64 + 2.0 * cij0 + eij0;
                                ex2 = ex * ex - self.dijk[mn][k] as f64 + 2.0 * cij0;
                                self.ar[0][1] += ndt * ex;
                                self.ar[0][2] += ndt * ex2;
                                if itau > 0 {
                                    ndtt = ndt * self.tijk[mn][k];
                                    self.ar[0][0] += ndt;
                                    self.ar[1][0] += ndtt;
                                    self.ar[2][0] += ndtt * (self.tijk[mn][k] - 1.0);
                                    self.ar[1][1] += ndtt * ex;
                                    self.ar[1][2] += ndtt * ex2;
                                    self.ar[0][3] += ndt
                                        * (ex
                                            * (ex2 - 2.0 * (self.dijk[mn][k] as f64 - 2.0 * cij0))
                                            + 2.0 * self.dijk[mn][k] as f64);
                                }
                            }
                        }
                    }
                }
            }
        }
    }

    fn tterms(&mut self, lntau: f64) {
        let i: usize = 5;
        let mut taup0: [f64; 12 + 1] = [0.0; 12 + 1];

        //i = 5;  // Use propane to get exponents for short form of EOS
        for (k, taup) in taup0.iter_mut().enumerate().skip(1) {
            *taup = (self.toik[i][k] * lntau).exp();
        }
        for i in 1..=NC_GERG {
            if self.x[i] > EPSILON {
                if i > 4 && i != 15 && i != 18 && i != 20 {
                    for (k, taup) in taup0.iter().enumerate().skip(1) {
                        self.taup[i][k] = NOIK[i][k] * taup;
                    }
                } else {
                    for k in 1..=KPOL[i] + KEXP[i] {
                        self.taup[i][k] = NOIK[i][k] * (self.toik[i][k] * lntau).exp();
                    }
                }
            }
        }

        for (i, mnumbi) in MNUMB.iter().enumerate().skip(1) {
            if self.x[i] > EPSILON {
                for (j, mnumbij) in mnumbi.iter().enumerate().skip(i + 1) {
                    if self.x[j] > EPSILON {
                        let mn = *mnumbij;
                        if mn > 0 {
                            for k in 1..=KPOLIJ[mn] {
                                self.taupijk[mn][k] =
                                    self.nijk[mn][k] * (self.tijk[mn][k] * lntau).exp();
                            }
                        }
                    }
                }
            }
        }
    }

    fn pseudocriticalpoint(&self) -> (f64, f64) {
        let mut dcx = 0.0;
        let mut tcx = 0.0;
        let mut vcx: f64 = 0.0;

        for i in 1..=NC_GERG {
            tcx += self.x[i] * TC[i];
            vcx += self.x[i] / DC[i];
        }
        if vcx > EPSILON {
            dcx = 1.0 / vcx;
        }
        (dcx, tcx)
    }
}
