// Based on the new Squares RNG algorthim
// <https://arxiv.org/pdf/2004.06278v3.pdf>
use crate::math::*;

pub struct Random {
	pub key: usize,
	pub counter: u64,
}

impl Random {

	/// Requires a key to be picked for number generation. A unique combo of counter and key will provide the same number each time.
	/// A single key can generate 2^64 random numbers
	/// I suggest picking a starting counter position by using the walltime of the OS, or the realtime of the program
	/// Or you can just start at zero and count up.
	pub fn new(counter: u64, key: usize) -> Random {
		Random {
			key: key % KEYS_TABLE.len(),
			counter,
		}
	}

	pub fn get_seed_from_time() -> u64 {
		std::time::SystemTime::now().duration_since(std::time::UNIX_EPOCH).unwrap().as_secs()
	}

	/// Returns a random 64-bit unsigned integer
	/// The output is uniform and will eventually produce all 2^64 numbers with the key.
	fn rand(&mut self) -> u64 {
		let mut x: u64 = self.counter * KEYS_TABLE[self.key];
		let y: u64 = x;
		let z: u64 = y + KEYS_TABLE[self.key];

		x = x*x + y; x = (x >> 32) | (x << 32);
		x = x*x + z; x = (x >> 32) | (x << 32);
		x = x*x + y; x = (x >> 32) | (x << 32);
		let rand_num_u64 = (x*x + z) >> 32;
		
		self.counter += 1;
		return rand_num_u64;
	}

	// I think I did something wrong with rand(), because it only generates under a certain number? Or is that normal?
	/// Returns a random float between 0 and 1
	pub fn randf_linear01(&mut self) -> f32 {
		let remainder_limit: u64 = 20000000;
		let rand: u64 = self.rand() % remainder_limit;
		return (rand as f64 / remainder_limit as f64) as f32;
	}

	/// Returns a random 32-bit float between min and max.
	pub fn randf_range(&mut self, min: f32, max: f32) -> f32 {
		lerpf(min, max, self.randf_linear01())
	}

	/// Returns a random 32-bit integer between min and max.
	pub fn randi_range(&mut self, min: i32, max: i32) -> i32 {
		lerpi(min, max, self.randf_linear01())
	}

	/// Test that checks the uniformality of the random number generator
	/// The result should be as close to 0.5 as possible.
	/// Using a counter of 0 and key index of 0 I get 0.4995544154831766
	pub fn test_randf_average(&mut self) -> f64 {
		let count = 1_000_000_000;
		let mut average: f64 = 0.0;
		for _ in 0..count {
			let remainder_limit: u64 = 30000000;
			let rand: u64 = self.rand() % remainder_limit;
			average += rand as f64 / remainder_limit as f64;
		}
		
		average / count as f64
	}
}

/// Keys are u64's that have half ones and half zeros in their bits.
/// Each key can make 2^64 random numbers. This should be enough, right?
/// 
/// Table was generated from the 'keys' program provided by the Squares V2 program included with the article.
/// Starting point of 65536 generating 1024 keys.
/// 
/// Sometime down the road I'd like to be able to generate this at runtime, but a table lookup
/// will be okay for now.
/// 
/// Just pick one! 
const KEYS_TABLE: [u64; 1024] = [
	0x3bac897217db8e35,
	0xf96a78b435ce8da1,
	0xeabd4f1765d2ab1f,
	0x1a7842d762e3da7b,
	0x23754f9981c5d9e7,
	0xb3ad4fe98ecaf965,
	0x4653cfbbaebd28c1,
	0xe7f938aedabe273f,
	0x8ad6f2921cd2459b,
	0x87c164a43bd67519,
	0x123cbd9547da9385,
	0xd485baf876cb93e1,
	0x5bc6d32984bda15f,
	0xd1265ac98391cfab,
	0x4bf5916dc1b4de29,
	0x245edafdcea6fd95,
	0x968741243ea92cf1,
	0x7eb9ad443d9c2b6f,
	0x58a91fe54a8f39db,
	0x5fc3ad687bc26a49,
	0x4382c5fba9c478b5,
	0xe5f8a21ed6c9a823,
	0x38a9b2fed39ba58f,
	0xbda94c3fe28da4fb,
	0x1587cd6324b1e369,
	0x3984275761a4f2d5,
	0xa28d6f387fb91253,
	0xbad3c8e76d8931af,
	0xd63f25987a8b4f1d,
	0xfc58e67cba8f4e79,
	0x7f9ba83ed8927ce5,
	0x7ec34a81f7a59c63,
	0x23e7dfa2147689bf,
	0xc86597f3247ab93d,
	0x2caef1b5426ed7a9,
	0xf45e81ba93a2f817,
	0x2c6e734cbfa51783,
	0x12c856dbac8725df,
	0xb51468afec8a345d,
	0x86c523d1fa7d53c9,
	0xbf4973a21b9f8237,
	0x6df4a3165b8491a3,
	0x92815fd435648def,
	0x1975eca98357ae6d,
	0x9f6ab8e87258bcd9,
	0xc9ef32dba16dec47,
	0xa2c5e47edf721cb3,
	0xd413c591fe864b31,
	0x824ed7521e56398d,
	0xc8bd53432c5a47f9,
	0x85942e743b4d5867,
	0x4e6d1c58795f86d3,
	0xa7596b4dc983b651,
	0xea97382dc465b3ad,
	0xf261874fe468d32b,
	0x6298ce41f25be197,
	0xd31678c3215bdfe3,
	0x94cefd865e723f61,
	0xed24fb876c543ebd,
	0xfe3a27598c475d3b,
	0x7c541b998b395ca7,
	0xf7b2a5edc94d8b15,
	0x718cf62ed64f9a81,
	0x93c85161f542a7ed,
	0x72d365b21634d75b,
	0xc6d2f5754238e5c7,
	0x73514be9814bf635,
	0x7e2d4c598f4e25a1,
	0xfce64b8bad62541f,
	0x6481de7dcd34527b,
	0x14853bcedb3861e7,
	0x5fb3194fe92a8f45,
	0xe4b76d33273c8eb1,
	0x5781a295463e9c2f,
	0xb1ced7976532dc8b,
	0x65947ea98324daf7,
	0xfe853a4cb128fa75,
	0x742df51baf3b29d1,
	0xf51d4accbc1d284f,
	0x2c35de11fd2146ab,
	0xc6e872a43d147629,
	0xd4195ba43c178495,
	0xca2e4b97682a94f1,
	0xd5a9b4f9871ca26f,
	0xf3a17bc9852fa1db,
	0x149afd5cb512df39,
	0x5fd3a98dc214fea5,
	0xbe5d283dcdf82e13,
	0x74396cd21de92c7f,
	0x4f1a3bc329fc3aeb,
	0xa6cd47e43aef4b59,
	0x97ced4b76af279c5,
	0x451a789cb8f6a943,
	0x85f3e76cb4e7a69f,
	0x2a786b1dc4fab61d,
	0xb127fa3dc2fed479,
	0xafe19c4324d1f3e5,
	0xfb914d3982f71463,
	0x2b4f68d65ed732bf,
	0xa6b4d8987cea513d,
	0x2ec43ba76adc4f89,
	0x8cb6d4aa98be6df5,
	0x34e8bfddc8e29d73,
	0xbaf4271fe5d48acf,
	0xce4b73a215e7ba4d,
	0x6c9da73323ead8b9,
	0x4db8c2e653cde927,
	0x47e1692ba4e21893,
	0xcf43e6d98db426ef,
	0xef1743bbaec8356d,
	0x9ad7b21febca54d9,
	0xc12498e1facd8347,
	0x724cdb8219cf82b3,
	0x95edcf7769d5b231,
	0x1af86b7435b4af7d,
	0xc13ea9d653a6bde9,
	0x63f215c982baed57,
	0x37fd2c69819dfbc3,
	0x168f524edfb23c41,
	0xb7e6c2d1feb43a9d,
	0x764cb3a32ec7591b,
	0x39e76fd32dcb5987,
	0x6712dfc549ac87e3,
	0x769ea8d658bfa761,
	0x962ec3498692a4bd,
	0x719345edc5a6d43b,
	0x934ebc2ed3b8e2a7,
	0xd83ba24fe3abf215,
	0x9738af4654be3281,
	0x5ac98be54e813fcd,
	0x29dcb8576d935e4b,
	0xd3bf59a54c965db7,
	0x4b916d387b9a8c25,
	0x91758addc67d8b91,
	0x36f7d91cb46e98fd,
	0x63e17ba1f492d86b,
	0x673d9ca21394e6d7,
	0x8eb271a432a9f745,
	0x8fcb9138739c26b1,
	0x29bd18587e8d352f,
	0x57316d8bae91538b,
	0x46e5732cbc9462f7,
	0xf583427feca89175,
	0xa1b6845fe7698fc1,
	0xadf8b392176b9d3f,
	0x68fd53a3245fad9b,
	0x32e5db476573ec19,
	0x56413dfa9274fb85,
	0x825394ecb3782ae1,
	0x7129debbae6b295f,
	0xf8692e3bad5e37cb,
	0xe427bc31fd816739,
	0xda82f6321d7386a5,
	0x3ef78da54c78a613,
	0x8b59cf654859a37f,
	0x8142ca68765ca2eb,
	0xd52176fba46fd159,
	0x1784952ba362efb5,
	0xce83742ed4871f23,
	0x34fb587dce472d8f,
	0xf56a8b921e493cfb,
	0xec8346532d3e4c69,
	0x4df153a76c417ad5,
	0x6b92a4c98c649a53,
	0xa4928e6cb83597af,
	0x4f8691edc649b72d,
	0x9ac85d7fe43cd589,
	0xb267f9edc23ed4f5,
	0xb735cfe326841573,
	0xbe53c2f8715623cf,
	0x63fd4a287f39524d,
	0x42ca3be87e3c51b9,
	0x536ced8a9b3d6f17,
	0xedf8b23dcb419e83,
	0xbe2931adc7238bdf,
	0x5c3d6e11f726ab5d,
	0x4d8e73c21527dac9,
	0x3a8c1275452cea37,
	0x5b74a9d7642ef8a3,
	0x3e172fdcb3654921,
	0x87654cfbaf15367d,
	0x5a4f3d9cbd1845e9,
	0x21879d5fed1c8457,
	0xc2e8b951fa1e84c3,
	0xc1bfe9576c52b341,
	0x48f391d65824b19d,
	0xc6a75fd54615bef9,
	0xd75a63c98419de67,
	0xec14b799841afcd3,
	0xf8c97a4cb43f2d51,
	0xc46b237a9ef13bad,
	0x782ab61fedf45a2b,
	0xfabe2c321df65a97,
	0x9e3fcba32bea78f3,
	0xf7c62d9549fda871,
	0x5c3d129546ef95cd,
	0xa286fdca97f2d54b,
	0xc9fb7deba4f5e3b7,
	0x1b42e39dc4e9f325,
	0x6d24958fe4fc2391,
	0xb2f9e5c542ec31fd,
	0x6c519d243ed14f5b,
	0x3c5817a54bd35ec7,
	0x183295c76be78d35,
	0xbea78f9767e98ca1,
	0xb9e76dc986dc9a1f,
	0x3154b7dba4dfc97b,
	0x42dbc831f4c2d8e7,
	0xfe48ab6213d4f865,
	0xa63bdef654e927c1,
	0x3cf452e871db263f,
	0x7dea9c376fce349b,
	0x9746382baed27419,
	0xed32cfacbcd49285,
	0x1c9f74aedac792e1,
	0xc15f832ed6b89e4f,
	0x4b293c62158c9eab,
	0xc18f2da435afcd29,
	0xb35e478764b2fc95,
	0xa7463bd984c62bf1,
	0xf73dcbea91c82a6f,
	0x2938ad187f9a38db,
	0xd51e63fdcdbe5849,
	0x3467b8e1fdc187b5,
	0x9fb5c2721dc5a723,
	0xe9d13c7329b6a48f,
	0xecd6ab754698a3fb,
	0x2ce4ab9875acd269,
	0x37ef1969849fe1d5,
	0x4fa98e3ed7c62153,
	0xa5b3491dc1a52e9f,
	0x4b26817dcf984e1d,
	0xfa453ec21f8b4d79,
	0x651dc3732c8d6be5,
	0x49df28a65da19b63,
	0x143facd65a7398bf,
	0xc2b836dba796b83d,
	0x5bfda16dc478d6a9,
	0xf4e3297ed49ce617,
	0x3487f1ced29ef583,
	0xe965b8c2159324df,
	0x31274be761a7435d,
	0xba9352654f7852c9,
	0xed13bf776e9c8137,
	0xba3fec698b6d7f93,
	0xc4765feba9618cef,
	0xcf14839fe873ac6d,
	0x297cad61f465cbd9,
	0xe695a3221569eb47,
	0x1d36b8a3246cf9b3,
	0xfeb16d28758f2a31,
	0xb8c3d51ba192378d,
	0xdc69b1ebae7546f9,
	0xa627e83dce4a8567,
	0x6b2da7cfec6b85d3,
	0x8db4ea71fb7ea451,
	0xd29fe4543961b2ad,
	0xd8973e654975d12b,
	0x591b42c43645df87,
	0x629b84d87458fde3,
	0x613d4cfba47c2e61,
	0xdb186eca915f2cbd,
	0xe853fb2edf625c3b,
	0x8bed2951fe345ba7,
	0x769cdab32e498a15,
	0x6481d5943d4ca981,
	0x389de7f5483c96ed,
	0xe94215a98941d65b,
	0x8d239c4ba853e4c7,
	0x9e12678dc768f435,
	0x87dac95fe56b24a1,
	0x93581bffe26c431f,
	0x456c28d4325f417b,
	0x5726bfa54e325fd7,
	0xb8ad19c87d268e45,
	0x87e523976c378db1,
	0xc259de68783a9b2f,
	0xb4ac7e6a962eca8b,
	0x815f9aedc631d9f7,
	0x3984bc2fe623f975,
	0xa7e1d294364728d1,
	0x9cdf1837653a274f,
	0x61bca4d8712d35ab,
	0xa56c13298e1f6529,
	0xdc985a7cbe128395,
	0x268ac5fcbc2693f1,
	0x172864fedb28a16f,
	0x5bc7f4dfe6189fcb,
	0x9c15ade3271dce39,
	0x89ac61e54721fda5,
	0xe73af5ca98462d13,
	0xb5c62dfa94162b7f,
	0xf7e4386ba21839eb,
	0x6e3c42d98efc4a59,
	0x92487bdbabfe68c5,
	0xc267b5ffecf1a843,
	0x5ae2c6721be3a59f,
	0x6bfc917439f7b51d,
	0x3e9ac87546fad379,
	0x19d3c2f876fbd2e5,
	0x271346aa95def163,
	0x4c8bf29ba5e431bf,
	0x164c3eacb1e64f2d,
	0xf8397a1dcfd74e89,
	0x83e6b5921eda6cf5,
	0xc6de87f32bde8c73,
	0x635dc7154ad189cf,
	0xe479ad8768e3b94d,
	0xe7b369d986e5d7b9,
	0x89d321acb5dae827,
	0x564efa2cb4dcf693,
	0x63bad79dc3bd15ef,
	0x7ba9356215e2346d,
	0x86f21ea541e653d9,
	0xce34b6265fda8247,
	0x2a8fd3976edc81b3,
	0xba692df87abd9f21,
	0xc673b2ea98b1ae7d,
	0x16c3b79ed6a3bce9,
	0x98742311f4b6ec57,
	0x2e84ba5215b8fac3,
	0x1436f27656cd2b41,
	0xe98b6af653bf289d,
	0xe7c5f4ba92d3581b,
	0x9d4eb86bafc45687,
	0x42e6f83badb986e3,
	0xa463c9dfedbca561,
	0xb658cd11f78e93bd,
	0x35fdae6549a1d23b,
	0x48efad9438b4e1a7,
	0x5cb768d54596def3,
	0x95342e6985ba2f71,
	0xc5a98468738a2ecd,
	0x3c47f1dba19f3d4b,
	0xc9f3862edf925cb7,
	0x34c7d8b1fea68b25,
	0x1845ad332eb78a91,
	0xe14f5c932b6a97fd,
	0xfa4d2c84398ec76b,
	0xb14e5dc76891e5d7,
	0x298ce35cb8a3f645,
	0xf3e645ced6a825b1,
	0x92c6357ed59a342f,
	0x56bc8e31f19d428b,
	0xd7c86ef4329e51f7,
	0x5b687ef43e928f65,
	0xc37db2854d758ec1,
	0xb3478cf65b679c3f,
	0xc127ba976758ac9b,
	0xf374d1cba76edb19,
	0xcb3759aed671fa85,
	0x1967b4a1f68529e1,
	0xc73dafb21497285f,
	0xbde9f155427936cb,
	0x9c41d688718d5639,
	0x1fde39a98e6f74a5,
	0xdc7ef18cbe84a513,
	0x58a4defcbc65a27f,
	0x8f7acb1ed957a1eb,
	0x297cd13fe75acf49,
	0x9de26472174cdeb5,
	0xb241a3e769841e23,
	0x2b951a3875642c8f,
	0xe74c2fd982563afb,
	0x26b3d5edc15a4b69,
	0x5287f4bbaf3c69d5,
	0x5e26f71dcd4f8953,
	0xd7bf684fec4296af,
	0x54f82d721c45b62d,
	0x731adc932b37d489,
	0xda7e4817684ad3f5,
	0x8aed32ba985df273,
	0x5fcd98eba76132cf,
	0xe4bf78adc376514d,
	0x52c3b6fed1364fa9,
	0x92f14e5edf3a6e17,
	0xaf36d5b21f2c7d83,
	0x56ba9cf43b2e7adf,
	0x5fc891a76c32ba5d,
	0x4c8def776a24d8c9,
	0x65a82eccba28e937,
	0xd2ab487dc62bf7a3,
	0xfab7489ed75e3821,
	0xac1d5b41f641357d,
	0x9475bf62162354e9,
	0xe4213cf761298357,
	0x819ea5776f1a82c3,
	0xba23ce798e3da241,
	0x7c94db176a1e9f8d,
	0xc195643ba812bdf9,
	0x3a9f8b2dc914ed67,
	0x9d26f15fe617fbd3,
	0x382f97c5483b2c51,
	0x8b7edc65451e29ad,
	0x32b56da98631592b,
	0xe9df341cb2135897,
	0x52ba8c7a9de587f3,
	0x43e589dcbdf9a671,
	0xfc3e51dcb8fa94cd,
	0xa915f3bed8fec34b,
	0x49abd87439f1e2b7,
	0x6a2ec9f549e6f125,
	0xe3a1b75767f83291,
	0x129b7f5764c82fed,
	0x91ed2f7982fc3e5b,
	0x8c5a7e2a91ef4dc7,
	0x58692fdcbfe28c35,
	0x19c5e7a1fde58ba1,
	0x1fd58e932cd7a91f,
	0x1a9fc43329eac87b,
	0x4c28169438bdc6e7,
	0x7d5e9ca989d1f765,
	0x15d86b7ba8e526c1,
	0xe67f429cb6d8253f,
	0xe579b8aed2ea439b,
	0xd93a14c1f2ed6319,
	0x2d7f61c432df7185,
	0x79d5ca154eb38fd1,
	0xa5fcd4743cb59d4f,
	0x827d59a548c69dab,
	0x9c5f7d8767badc29,
	0xbfd5e37986adeb95,
	0x785c9b3ed7c32af1,
	0x2bfc9dafe5c4296f,
	0x8475c6a214c637db,
	0x9a8e5c6432db5749,
	0x19d687f541cd76b5,
	0x4189d3287ebf9623,
	0xf368ad987db2a38f,
	0x24c16a798c95a2fb,
	0xa8479c6feab8d169,
	0x62f479dfe689dfc5,
	0xbde48592179cfe43,
	0x81a6c79546a12d9f,
	0xae35dfb876a54c1d,
	0x9325fd4982b64c79,
	0xb6e71dc981b96ae5,
	0x5df8a7ecbf9d8a63,
	0xfe48736bac6e87bf,
	0x42eb8ca1fc92b73d,
	0xb2d7afe21c84d5a9,
	0xb89af1443ba9e517,
	0x4e396cb6579bf483,
	0x6af3bd58768c13df,
	0x19f56aecb7a1425d,
	0xe51a879ed3a451c9,
	0xe19c43bed1a86f27,
	0x4b3ae82edf6a7e93,
	0xefa796521d5a7bef,
	0xdaf983132c5f9b6d,
	0xd2ae58354b62cad9,
	0xabd314876b85ea47,
	0x734b65eba967f8b3,
	0x1fe54b2ed78c2931,
	0xe436b9ddc47e268d,
	0x8a65d37fe38145f9,
	0xeb6438d212a38467,
	0xf8e432c6519784d3,
	0x1895ca376f7ba351,
	0xa5e9f4376e5c91ad,
	0x6d41ea28795eaf1b,
	0x84ed1f5cba52de87,
	0xf93cda8cb865fce3,
	0xe9a58641f8792d61,
	0x237c84e216592abd,
	0x8b56ae73256e4a3b,
	0x1ec5d249836159a7,
	0x4e957cdcb2748915,
	0x4e6f837cbf46a781,
	0x5e86fcbbac3895ed,
	0x289f164fea3dc45b,
	0x6d4c327fe94fd3c7,
	0x962d5fc21b74f235,
	0x63198d765b6723a1,
	0x6c3b94d87869421f,
	0x5dc169a9854a3f6b,
	0x1f2c3e8a923c4ed7,
	0x4d397b2dc2618d45,
	0xa6c9fb3edf348cb1,
	0x532a486fed369a2f,
	0x5cf63ba21e27c98b,
	0xe8a2bc443a2bc8f7,
	0x6318dca54a2fe875,
	0xa5e124ba9a5427d1,
	0xadc7361cb937264f,
	0xf3ba8c7ed62834ab,
	0x38bd165fe53c6429,
	0xd185a421f13e7295,
	0xb1247a97645392f1,
	0xbd2a18554d139e5f,
	0xb596afe54d159ecb,
	0x1958ead76b18cd39,
	0x9d1cfb88781beca5,
	0x3cb248fba82efb13,
	0xa69b2fced8132a7f,
	0x2154b9d1f61538eb,
	0x91a384c3261a4859,
	0x3cadb675431b67c5,
	0x7cb95e19821e9743,
	0xf7e918365cde849f,
	0xfd547ea98cf3b41d,
	0x2bc4d5ebabf5d279,
	0x6a9fd38cb9f8d1e5,
	0x9856412dc7daef53,
	0x4fbe753326dc1eaf,
	0xa421938657f14e2d,
	0x2a785ef765f34d89,
	0xbaec135982e76bf5,
	0xef24ba7a91fa8b73,
	0x7be519476edb78cf,
	0x5346bfca9bcfa84d,
	0x67352b81fae2d6b9,
	0xeac5bf321bd5e627,
	0xaf145e332ad8f593,
	0xaf3c67b657ca14ef,
	0x7c1362f987ce236d,
	0x3895e74ba6e152d9,
	0x38fd726dc3e58147,
	0x897f435dc1d67fa3,
	0xb85ce67edfca9e21,
	0xecf48ad21fab9c7d,
	0x5cf3a7932c8fabe9,
	0x2987e5154ac2eb57,
	0xa9f41b3768b5f9c3,
	0xd5ca182cb8d92a41,
	0x78d3451ba5cb279d,
	0x237bfa1ed5ce471b,
	0xca197e41f3d16587,
	0x29b4e36212c485e3,
	0x7eda6b9652d8a461,
	0xb97235d54fa892bd,
	0x84dc73a76e9dc13b,
	0xd7ca2f465a8ecf97,
	0x65742f3ba892edf3,
	0x6f3b78eed9b62e71,
	0x8a9b6411f5972bcd,
	0xe8c5bf32169a3c4b,
	0x8c7ed193239e4ab7,
	0x5d4a1b2874c18a25,
	0x629841b872c4a891,
	0x1896f5d87e8596fd,
	0xbf3561dbae89c56b,
	0x492f85dfea8bc4d7,
	0x29561c8eda9fe345,
	0x28b6f9543ca524b1,
	0xc8d4793659a7432f,
	0x1fd6e84656a9418b,
	0x24b58f77638a4fe7,
	0xe6f4179ba29d7e65,
	0x83de6f5cb3928dc1,
	0x8172ba6dce749b3f,
	0x31c5e961fe65ca9b,
	0x1c875de32e68da19,
	0x73849fe32b6ce985,
	0x6d8a35c5496df7e1,
	0xa2d6cbe98a93275f,
	0x497f5e6ba78635cb,
	0xb831627ed48a6539,
	0x3fba487fe28c73a5,
	0x235a9711f28e9413,
	0xe61abf854382a17f,
	0xdc936e132e539fdb,
	0x854c7ef43d56ce49,
	0x1a47d5b76a48edb5,
	0xedb29f49896bfc23,
	0x8b5cf7dba75e1b8f,
	0xe81bca4ed76239fb,
	0xc716eb41f3754a69,
	0xf62b89e2134968d5,
	0x617c8eb5426c9853,
	0x9acfdbe7614d85af,
	0xcfae94698f61b52d,
	0x8967edcbae54d389,
	0x8eb6c79bab47d2f5,
	0xa17ebf3edb5af173,
	0xa96c2d1fe85b21cf,
	0x79afd362185e2f3d,
	0x9148cd7437524ea9,
	0xb27341aa98456d17,
	0x648e5b1ba2497c83,
	0x472deaf9813a79df,
	0x1b5e7af98f2ea95d,
	0x31c759ecbe31d7c9,
	0x735ab621fc24e837,
	0xa7423c821c27f6a3,
	0x49d3a1b76c5b3721,
	0xa7c8f4b7693c247d,
	0x2d3a74f9872f43e9,
	0xa365b8ddc9438257,
	0xf56427ced54681c3,
	0x17c8e64ed2489f31,
	0xbd3857fdcf1a9e8d,
	0xa3f5d4632e1dacf9,
	0xcde576854d21ec67,
	0xe3d8b4276b14fad3,
	0xa1c3df4a9c382b51,
	0x6f79d8ecb71928ad,
	0x1e3f2b7dc71d482b,
	0x8e3756ced51f4697,
	0xa5692d8fe51286f3,
	0xde18b2943426a571,
	0x73c9b486521793cd,
	0xf87c96d65ef9c24b,
	0x347fdc965afcd1b7,
	0x4acfd9e879cedf15,
	0x4e21879cbae42f81,
	0xe9d257cba7f52ced,
	0x638f974fe6f83d5b,
	0xcd67ab4215fa4bc7,
	0xcbadf18324fe7b35,
	0x8925cf1764f289a1,
	0x6148a9c872e4a81f,
	0x4fc672e98fe5c67b,
	0xc26bf7498bd9c5e7,
	0x3c1452acbadce465,
	0x738b25aed8bef4c1,
	0x9c78ae621ad5243f,
	0x215ad49438e6429b,
	0x6491bf7657ea6219,
	0x169dcab983db6f75,
	0x239fcd6983bc7ed1,
	0x9f5367bba2d19c4f,
	0xd43e69f98fd39cab,
	0x694a5271fec6db29,
	0x3f4d95721dc8ea95,
	0x8d5bc97439abe8f1,
	0xb1f7c5e549be186f,
	0x25641b9878c236db,
	0x3c6e75acb8d75649,
	0x153d427dc3d974b5,
	0x86c19f7fe4dc9523,
	0xb548c3dfe1bd928f,
	0xa6b25d7543c2a1fb,
	0xdf7958132ea3cf59,
	0x91d2feb43b96dec5,
	0xfc4279876ba9fd43,
	0xdec75817678b1c9f,
	0x32d8cb1ba79f3b1d,
	0x31fab45fe6c24b79,
	0x9dc51ba1f3b469e5,
	0x2b3d945214ca8963,
	0x2ad9cef4329a86bf,
	0x9d874cf6519ea63d,
	0xd3f4b2a76f81d4a9,
	0xb5f4d3aa9da5e417,
	0x9f74e32cbca7f283,
	0x6f7539edca8912df,
	0x5b4c6d71f89c315d,
	0x36974fb2167e3fb9,
	0xad2b5f8548a36e27,
	0x9d8ea37764a57d93,
	0x497e2fb652867aef,
	0x6b2a9fe9817b9a6d,
	0x5634d8f87f5eb8d9,
	0x9581436dcd81e947,
	0x2fa4981fec84f7b3,
	0xaed728554da92831,
	0x2db7ca843b7a258d,
	0xcf48ab76587c34f9,
	0xce35fd8ba89f5367,
	0xf7258c3cb69482d3,
	0xb4861f9ed497a251,
	0xa5d391bdc1568f9d,
	0x7bc5182edf59ae1b,
	0xcb5649321f4ecd87,
	0x91a37dc54c61fbe3,
	0x25eb3f187d852c61,
	0x1528b9c87a5629bd,
	0xb2d965fdc86a493b,
	0x87d6b14dc65d48a7,
	0x7e159faed46f7815,
	0x53982eb1f473a681,
	0x37fb8e92126594ed,
	0xb7291de54168c35b,
	0x8cfbe4365f4ad2c7,
	0xc43a19e98c6de135,
	0x316cf94cbd7432a1,
	0x63cf845ba9542efd,
	0x276be9fed9473e6b,
	0xbe92a671f6394cd7,
	0xd42bf592183d7c45,
	0xa689eb14354f7ab1,
	0x213895487453a92f,
	0x5b38feaa9134c78b,
	0xbd5e4c3a9e28c6f7,
	0xd671e39bad2be675,
	0x95e2ac4cba3cf5d1,
	0x89613adfeb61254f,
	0x4a5cbd821c2543ab,
	0x91c62fb65b486329,
	0x3da9f7b7683b7195,
	0x249cb369853b7fe1,
	0xb817f6da932e8d5f,
	0x54b7d68ba1329dcb,
	0x7f4a51ecbf15dc39,
	0x6ac95de1fd17eba5,
	0x12ac97f54d2bfa13,
	0xf769a5e54b2d197f,
	0xf1bd92c76a2137eb,
	0x5127ef4bab134759,
	0x3e7d9a4cb91876c5,
	0xb5f26e3ed71b9643,
	0xc8ae9f3ed41c839f,
	0x5ba37fcfe42fa31d,
	0xef41d9865423d179,
	0x75986b132be3cfd5,
	0x862d5b354bd7fe53,
	0x8f197a4659e81daf,
	0xd5128fb878fb3c2d,
	0x954c67dba6ef3c89,
	0xa5fb17edc7e26af5,
	0x12859a3fe4f58a73,
	0x682ad53213e687cf,
	0x74d8a1b432fba74d,
	0xea96b3f541fec5b9,
	0xd1e567398fd1e527,
	0x63f2dce98cd5f493,
	0xe43b5c2a9ac713ef,
	0xa1b9758dc9ea326d,
	0x178a9bd1f7ec41d9,
	0x89b2c67218ce5f37,
	0xc53b9a1546d27ea3,
	0xcd194f8875e69d21,
	0x94e8c3f652e69b7d,
	0x39caedb761d8bae9,
	0xf376a5e98fbeda57,
	0xc13d967badb1f8c3,
	0x824ac5e1fcd62941,
	0xa279b4821bc7269d,
	0xb7f8512439da461b,
	0xeb6a851439cd5487,
	0x4b159e7986bf74e3,
	0x5716e82cb7d4a361,
	0x95367f4ba3c591bd,
	0x1bc4875ed1c7af2b,
	0x42eba3cedf8ace97,
	0x5839fa421e8bdcf3,
	0xf13ea9665eb32d71,
	0xe6af32165b942acd,
	0x1d4eba3769c73a4b,
	0x2f4b197986ca49b7,
	0xf93a72ddc6bd7925,
	0x5a9c342ed4bf8791,
	0x7364d9afe3a295fd,
	0x7a6be2d212a5c46b,
	0xc16b9f8541a8c3d7,
	0x2e45a9776f9be245,
	0x8af32e576c8df2b1,
	0xf18be4ca9ca3412f,
	0x65abd79dca943f7b,
	0xdf2a973dc8964de7,
	0x91d7b521f49a7d65,
	0xb95683c3258d7bc1,
	0x173c8b93247e8a3f,
	0x86dac1f65372c89b,
	0x124cefa872a5d819,
	0x41d3eb9baf69e785,
	0xa7fe136a9c7af6e1,
	0x623f7e4dcb8d165f,
	0x46e382a1fa8134cb,
	0xf59243a21ca56439,
	0xfd9c27843b9872a5,
	0x95af31e7678b9213,
	0x9438ca28735b8e6f,
	0xc2569a38725f8edb,
	0x6ed98c5cb172cd49,
	0x5c2416fbaf64ecb5,
	0xa3d6295fed68fb23,
	0x5b36a4132d5b1a8f,
	0x7a1829443a5d28fb,
	0x84faec665c714869,
	0xa2738dba9a8367d5,
	0x65418b9dca689753,
	0xe7a2159dc35984af,
	0xd859af4fe36ca42d,
	0x527cf81fe15fc289,
	0xc679d5854273d1f5,
	0x7e48dcb32d45ef63,
	0x26f8c3465c471ebf,
	0x3e8fda187b5a2e3d,
	0x2d348698784e3da9,
	0x6482b7aeda526c17,
	0xcd1a258fe9547b83,
	0x125bd49fe43578df,
	0x925b4df21439a85d,
	0x7639fb85423db6c9,
	0x6e7d42f9814fd637,
	0x47532ed98e42f5a3,
	0x1ca8e62cbe583621,
	0x6c478decbc49237d,
	0x85943f6eda3b42e9,
	0x6c294b81fa3e6157,
	0x41a7fde329417fb3,
	0xe5bfa94768459e31,
	0x3825d9c764259c8d,
	0xfbdc82576327abf9,
	0xa438b75a912cdb67,
	0xcfe3d2598c1fe9d3,
	0x3f9c6b4ede342a51,
	0x3958e6c1fc1527ad,
	0xef7462b43d19472b,
	0xe89425743b1c4597,
	0xa1295b87691d75f3,
	0x3259c78cb932a471,
	0x93d1cbfba61492cd,
	0xef1768aed527c14b,
	0x42c6a37ed218cfa7,
	0x158da6ecbcfade15,
	0x69e4f5b32becfd81,
	0x391c5d754af12bed,
	0x13af48c76af43c5b,
	0x8ac3eb1659f64ac7,
	0x8ae5c4fa98fb7a35,
	0x9d43f2ccb5fd78a1,
	0x59d873fcb4de971f,
	0x79b26c4fe3f2c57b,
	0xe425867212f5c4e7,
	0xc17da65541f9e365,
	0x47ac9db65edaf3c1,
	0xda69f1287bcd123f,
	0x9cf2e5198ce2419b,
	0x57cf496ba9c34ef7,
	0x4952738dc9d86e75,
	0x568793d1f5e97cd1,
	0x82b6c73215dc8b4f,
	0xbecf59a323df89ab,
	0xc4521b3764e3da29,
	0x4a16dfe871d4e895,
	0xe389dfa76db8e7f1,
	0x4afdc38a9dcb176f,
	0xbda89f1dc9ae25db,
	0xce287351fbd16549,
	0xd3496ea21ad473b5,
	0xfbd23a165ad89423,
	0x3bce259545ca918f,
	0xec72f3a6539a8feb,
	0xe8b12ac872afbe59,
	0x8ceb4fda91a2edc5,
	0x7c4e613cbea6fc43,
	0xa518629feda71b9f,
	0xbf75ad132e9b3a1d,
	0x3aed61f32cbe3a79,
	0xef9378654ab168e5,
	0xd2ab86187ac49863,
	0x687cdb3a94a685bf,
	0x2f6e4a5dc4b9a53d,
	0x47f6b59fe28dc3a9,
	0x29ab168fe2afd317,
	0x264f387652b4f183,
	0xfe3ba1754e9621df,
	0xf92d8ea65c982f4d,
	0xe1c478665c7a3eb9,
	0xb39acf598a9e5d27,
	0x593e46aed8a17c93,
	0xe7b8243dc58379ef,
	0x6b475c81f385a96d,
	0x476eab321368b7d9,
	0x5ecad8b4329cd847,
	0x6a943fc6518fe6b3,
	0x46fe2daa9fa52731,
	0x138f256bae96248d,
	0x632f185cbc7843f9,
	0x9d4a523feb9c5267,
	0xecd651b1f98e71d3,
	0x2a846cd43a829f41,
	0x19e5638325638e9d,
	0xa9b71ec65476ad1b,
	0x3c91ef276259dc87,
	0xabd943e9816bdae3,
	0x4ae31d5a9d5efa61,
	0x93564dadce6328bd,
	0xc4137a21fd76483b,
	0x78d5e4c21d5946a7,
	0x64c1bae43b6c8715,
	0xbad7c2f54a6e9581,
	0x42e3dfa9876193ed,
	0x761539fba674c25b,
	0x2471365cb356d1c7,
	0x2f753addc169df25,
	0x163b475dcd4bfe91,
	0x37b95ce32b4e1cfd,
	0x91e35b654e523d6b,
	0x31bd85976c354bd7,
	0x7eda31698c397b45,
	0x81d236eba84c79b1,
	0x897e6f4cb64d982f,
	0x32af475fe541c68b,
	0x39f64c8fe243c5f7,
	0x9765c3221238e475,
	0xfd523c176259f4d1,
	0x2ebf8c587e5c134f,
	0x8ad53bf87d4f32ab,
	0xf362157cbd546129,
	0x9bd574adcb246f85,
	0xd9f8741dc9387de1,
	0xbc145affe62a8c5f,
	0xeb9c74f2152e8acb,
	0x78a6b3e65641db39,
	0xfd34c6987423e9a5,
	0xe36f81bba247f913,
	0xfc75eb4a9e2a187f,
	0xa76b32ea9d1c26eb,
	0xce485a3edc1f3659,
	0x174cdab1fb1274c5,
	0x84c316d32c179543,
	0x286ecfa54a17829f,
	0x74f13ae7673ba21d,
	0x36bcd5e8741dbf69,
	0xe57abd3a9421ced5,
	0x1a9836fcb214fd53,
	0x6c34d21a9de51caf,
	0x854fd32fecf83b2d,
	0xdb7a54e21dfc3b89,
	0x9645efb32aed49f5,
	0xdceb43665bf18973,
	0xad2f714657e386cf,
	0x8cb5faea95f7a64d,
	0x465eb18ba3f8c4b9,
	0x256a1f7cb3fcd427,
	0xe1b487ddc1dfe293,
	0xa2ce39d653d412ef,
	0xf986c1754fe7316d,
	0x16afb8554bd73fc9,
	0x945b7e876adb5e37,
	0x2ad9e64878ce6da3,
	0x768df3acb8e39c21,
	0xa36bd49dc5e39a7d,
	0x83a76e41f3d5b8e9,
	0x2feb53d213ead957,
	0xe7ba132542dbe7c3,
	0x71a2345762cef741,
	0xae86b1587fc3259d,
	0xb2a74d5a9ed7451b,
	0xfa39e75cbcda5387,
	0x16fa2beed9cb72e3,
	0xadc67941f8ce9261,
	0x41be7cd326b18fad,
	0xd8eafc6436c4ae2b,
	0xc172f93543b5cd97,
	0xdb8357f762a8dbf3,
	0xbfc571e982acfb71,
	0x164a9c287e8f19cd,
	0x46ba975edec2394b,
	0xe4db31c1fdc548b7,
	0x765cad221dca7825,
	0x61fc4b943abc8691,
	0x72891af4369c84fd,
	0x9fd31b2987a1c36b,
	0xc573f64a95a4c2d7,
	0x8d7125edc3b8e145,
	0xd365f7bdc298efa1,
	0xd5b8f42dcd8b2e1f,
	0x95c3deb21d8f2e7b,
	0x64b9ced43d924ce7,
	0x9825ed143da47c65,
	0x6ad174e65b987ac1,
	0x1586cdfba67b893f,
	0x23714badc46ea79b,
	0xd7c48b1fe5a2d719,
	0x9eb7cda1f2a3e685,
	0x6f97b8c32397f5e1,
	0xf2748de7629a145f,
	0xc8e97b465f7d23cb,
	0xdebf13687d8f5239,
	0x6c14a32cbd9371a5,
	0x9c84e73cba857ef1,
	0xd84f3b9dc9578d6f,
	0xb41c3ad1f4598cdb,
	0xbfea6232156ebc49,
	0x725f68b43471eab5,
	0xb9426f387485fa23,
	0xecb258fba276198f,
	0xc659de498e5927fb,
	0x92fa685cbe5d3769,
	0x42f3b87dcb7f46d5,
	0x953edac1fc829653,
	0x68ad42c21b6583af,
	0x2e1dbaf54968a32d,
	0xc986f125465bc189,
	0x9238fb18734cbfe5,
	0x845e172ba471fe63,
	0x2c8d6b7ba2641dbf,
	0x18637aecbf572c3d,
	0x1e5a8791fe483ca9,
	0x35c1b7a32e4d5b17,
	0x68e9bc432c4f6a83,
	0xf98a6bd65a4287df,
	0x63cf1ea98934a75d,
	0x65d32cfba637b5c9,
	0xb3e6187dc64bd537,
	0x4ab5d18dc25de4a3,
	0x6a4b1e91f6843521,
	0x38e76cb54165327d,
	0xc62d3a165f3741e9,
	0xfb58a9e76d2a5f47,
	0xc54329a87a2c6eb3,
	0xf3cd87aa9a4f8d31,
	0x42975bddc7329b8d,
	0xfec5748ed624baf9,
	0xb216c581f428da67,
	0x2785e4b2152ae8d3,
	0xa749df27643df851,
	0x17c3f499834126ad,
	0x4ba3c78a9f15462b,
	0x2d84a15cbe185497,
	0xd92a347cbc1a74f3,
	0xd8a6b45feb2d9371,
	0xa48b126fe72f81cd,
	0xc1d627954812af3b,
	0x6b28fe443614cea7,
	0x8cbd93f87628ed15,
	0x3f4a9688742afc81,
	0x21c35eb9832c1aed,
	0x1ad964798def2a5b,
	0x14d52fccbdf249c7,
	0x42cd913fecf67935,
	0xb13d9e732cf987a1,
	0x6cae512438eb961f,
	0x96875c2436fea47b,
	0x145d6bc986f1c3e7,
	0xef531b4ba7f4e265,
	0x65cedf1ba4e7f2c1,
	0xda95634dc4ea213f,
	0x29d456bbaedc2f8b,
];